﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/server/nsd_ManagerImpl.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nsd/nsd_ImportMode.h>
#include <nn/nsd/nsd_DeleteMode.h>
#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/fs/nsd_Fs.h>
#include <nn/nsd/detail/nsd_FqdnResolver.h>
#include <nn/nsd/detail/nsd_DetailApiForMenu.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nsd/detail/nsd_Log.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    const char SettingsName[] = "nsd";
    const char SettingsKey[] = "test_mode";
#endif

    bool IsTestMode()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        return true;
#else
        // settings の test_mode をチェック
        size_t valueSize = nn::settings::fwdbg::GetSettingsItemValueSize( SettingsName, SettingsKey );
        if ( valueSize != sizeof(bool) )
        {
            // 値が存在しない / bool でない場合は 非test_mode 扱いとする
            return false;
        }

        bool value;
        valueSize = nn::settings::fwdbg::GetSettingsItemValue( &value, sizeof(value), SettingsName, SettingsKey );
        return value;
#endif
    }
}

namespace nn { namespace nsd { namespace detail { namespace server {

nn::Result ManagerImpl::GetSettingName(nn::sf::Out<nn::nsd::SettingName> pOut) NN_NOEXCEPT
{
    // Nsd 設定名読み込み
    SettingName name;
    NN_RESULT_DO( nn::nsd::detail::GetSettingName(&name) );
    pOut.Set( name );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::GetEnvironmentIdentifier(nn::sf::Out<nn::nsd::EnvironmentIdentifier> pOut) NN_NOEXCEPT
{
    // 環境識別子取得
    EnvironmentIdentifier env;
    NN_RESULT_DO( nn::nsd::detail::GetEnvironmentIdentifier(&env) );
    pOut.Set( env );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::GetDeviceId(nn::sf::Out<nn::nsd::DeviceId> pOut) NN_NOEXCEPT
{
    // このインタフェースは不要かも
    NN_UNUSED(pOut);
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::DeleteSettings(const int32_t deleteModeValue) NN_NOEXCEPT
{
    // 権限チェック
    if( ! IsAdminService() )
    {
        NN_DETAIL_NSD_WARN("No privilege to call DeleteSettings.\n" );
        NN_RESULT_THROW( nn::nsd::ResultNotPermitted() );
    }

    // DeleteMode値への変換
    nn::nsd::DeleteMode deleteMode;
    switch( deleteModeValue )
    {
        case nn::nsd::DeleteMode_All:
            deleteMode = nsd::DeleteMode_All;
            break;
        case nn::nsd::DeleteMode_ApplicationSettings:
            deleteMode = nn::nsd::DeleteMode_ApplicationSettings;
            break;
        default:
            NN_RESULT_THROW( nn::nsd::ResultInvalidArgument() );
    }

    // 実際の削除処理
    NN_RESULT_DO( nn::nsd::detail::DeleteSettings(deleteMode) );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::ImportSettings(const nn::sf::InBuffer& pBuffer, const nn::sf::OutBuffer& pWorkBuffer, const int32_t importModeValue) NN_NOEXCEPT
{
    NN_UNUSED(pBuffer.GetPointerUnsafe());
    NN_UNUSED(pWorkBuffer.GetPointerUnsafe());

    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    // ImportMode値へ変換
    nn::nsd::ImportMode importMode;
    switch( importModeValue )
    {
        case nn::nsd::ImportMode_All:
            importMode = nsd::ImportMode_All;
            break;
        case nn::nsd::ImportMode_ApplicationSettings:
            importMode = nn::nsd::ImportMode_ApplicationSettings;
            break;
        default:
            NN_RESULT_THROW( nn::nsd::ResultInvalidArgument() );
    }

    // 実際のインポート処理
    NN_RESULT_DO( nn::nsd::detail::ImportSettings( pBuffer.GetPointerUnsafe(), pBuffer.GetSize(), pWorkBuffer.GetPointerUnsafe(), pWorkBuffer.GetSize(), importMode ) );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::Resolve(nn::sf::Out<nn::nsd::Fqdn> pOut, const nn::nsd::Fqdn& fqdn) NN_NOEXCEPT
{
    nn::nsd::Fqdn outFqdn;
    NN_RESULT_DO( nn::nsd::detail::GetFqdnResolverPtr()->Resolve( &outFqdn, fqdn ) );

    NN_DETAIL_NSD_INFO("nn::nsd::Resolve in(%s) out(%s)\n", fqdn.value, outFqdn.value);
    pOut.Set( outFqdn );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::ResolveEx(nn::sf::Out<nn::nsd::InnerResult> pInnerResult, nn::sf::Out<nn::nsd::Fqdn> pOut, const nn::nsd::Fqdn& fqdn) NN_NOEXCEPT
{
    nn::nsd::InnerResult innerResult;
    nn::nsd::Fqdn outFqdn;
    auto result = nn::nsd::detail::GetFqdnResolverPtr()->ResolveEx(&innerResult.result, &outFqdn, fqdn);

    // result は内部で失敗していても成功扱いになっていることがあるので、
    // 実行結果の生値である innerResult.result でログを出しわける
    if(innerResult.result.IsSuccess())
    {
        NN_DETAIL_NSD_INFO("nn::nsd::ResolveEx in(%s) out(%s)\n", fqdn.value, outFqdn.value);
    }

    pInnerResult.Set( innerResult );
    pOut.Set( outFqdn );
    return result;
}
nn::Result ManagerImpl::GetNasServiceSetting(nn::sf::Out<nn::nsd::NasServiceSetting> pOutNasServiceSetting, const nn::nsd::NasServiceName& nasServiceName) NN_NOEXCEPT
{
    nn::nsd::NasServiceSetting outSetting;
    NN_RESULT_DO( nn::nsd::detail::GetFqdnResolverPtr()->GetNasServiceSetting(&outSetting, nasServiceName) );

    pOutNasServiceSetting.Set( outSetting );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::GetNasServiceSettingEx(nn::sf::Out<nn::nsd::InnerResult>pInnerResult, nn::sf::Out<nn::nsd::NasServiceSetting> pOutNasServiceSetting, const nn::nsd::NasServiceName& nasServiceName) NN_NOEXCEPT
{
    nn::nsd::InnerResult innerResult;
    nn::nsd::NasServiceSetting outSetting;
    auto result = nn::nsd::detail::GetFqdnResolverPtr()->GetNasServiceSettingEx(&innerResult.result, &outSetting, nasServiceName);

    if(innerResult.result.IsFailure())
    {
        NN_DETAIL_NSD_WARN("nn::nsd::GetNasServiceSettingEx outputs invalid FQDN. Inner error(%08x, %03d-%04d)\n",
                   innerResult.result.GetInnerValueForDebug(),
                   innerResult.result.GetModule(), innerResult.result.GetDescription());
    }

    pInnerResult.Set( innerResult );
    pOutNasServiceSetting.Set( outSetting );
    return result;
}
nn::Result ManagerImpl::GetNasRequestFqdn(nn::sf::Out<nn::nsd::Fqdn> pOut) NN_NOEXCEPT
{
    nn::nsd::Fqdn outFqdn;
    NN_RESULT_DO( nn::nsd::detail::GetFqdnResolverPtr()->GetNasRequestFqdn( &outFqdn ) );

    pOut.Set( outFqdn );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::GetNasRequestFqdnEx(nn::sf::Out<nn::nsd::InnerResult>pInnerResult, nn::sf::Out<nn::nsd::Fqdn> pOut) NN_NOEXCEPT
{
    nn::nsd::InnerResult innerResult;
    nn::nsd::Fqdn outFqdn;
    auto result = nn::nsd::detail::GetFqdnResolverPtr()->GetNasRequestFqdnEx(&innerResult.result, &outFqdn);

    if(innerResult.result.IsFailure())
    {
        NN_DETAIL_NSD_WARN("nn::nsd::GetNasRequestFqdnEx outputs invalid FQDN. Inner error(%08x, %03d-%04d)\n",
                   innerResult.result.GetInnerValueForDebug(),
                   innerResult.result.GetModule(), innerResult.result.GetDescription());
    }

    pInnerResult.Set( innerResult );
    pOut.Set( outFqdn );
    return result;
}
nn::Result ManagerImpl::GetNasApiFqdn(nn::sf::Out<nn::nsd::Fqdn> pOut) NN_NOEXCEPT
{
    nn::nsd::Fqdn outFqdn;
    NN_RESULT_DO( nn::nsd::detail::GetFqdnResolverPtr()->GetNasApiFqdn( &outFqdn ) );

    pOut.Set( outFqdn );
    NN_RESULT_SUCCESS;
}
nn::Result ManagerImpl::GetNasApiFqdnEx(nn::sf::Out<nn::nsd::InnerResult>pInnerResult, nn::sf::Out<nn::nsd::Fqdn> pOut) NN_NOEXCEPT
{
    nn::nsd::InnerResult innerResult;
    nn::nsd::Fqdn outFqdn;
    auto result = nn::nsd::detail::GetFqdnResolverPtr()->GetNasApiFqdnEx(&innerResult.result, &outFqdn);

    if(innerResult.result.IsFailure())
    {
            NN_DETAIL_NSD_WARN("nn::nsd::GetNasApiFqdnEx outputs invalid FQDN. Inner error(%08x, %03d-%04d)\n",
                   innerResult.result.GetInnerValueForDebug(),
                   innerResult.result.GetModule(), innerResult.result.GetDescription());
    }

    pInnerResult.Set( innerResult );
    pOut.Set( outFqdn );
    return result;
}
nn::Result ManagerImpl::GetCurrentSetting(nn::sf::Out<nn::nsd::SaveData> pOut) NN_NOEXCEPT
{
    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    nn::nsd::SaveData* pOutSaveData;
    NN_RESULT_DO( nn::nsd::detail::GetFqdnResolverPtr()->GetSaveData( &pOutSaveData ) );

    pOut.Set( *pOutSaveData );
    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::WriteSaveDataToFsForTest(const nn::nsd::SaveData& saveData) NN_NOEXCEPT
{
    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    // settings の test_mode をチェック
    NN_RESULT_THROW_UNLESS( IsTestMode(), nn::nsd::ResultNotImplemented() );

    NN_RESULT_DO(nn::nsd::detail::fs::WriteSaveData(saveData));
    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::ReadSaveDataFromFsForTest(nn::sf::Out<nn::nsd::SaveData> pOut) NN_NOEXCEPT
{
    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    NN_RESULT_DO(nn::nsd::detail::fs::ReadSaveData(pOut.GetPointer()));
    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::DeleteSaveDataOfFsForTest() NN_NOEXCEPT
{
    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    // settings の test_mode をチェック
    NN_RESULT_THROW_UNLESS( IsTestMode(), nn::nsd::ResultNotImplemented() );

    NN_RESULT_DO(nn::nsd::detail::fs::DeleteSaveData());
    NN_RESULT_SUCCESS;
}

Result ManagerImpl::SetChangeEnvironmentIdentifierDisabled(bool isDisabled) NN_NOEXCEPT
{
    // 権限チェック
    NN_RESULT_THROW_UNLESS( IsAdminService(), nn::nsd::ResultNotPermitted() );

    auto current = nn::nsd::detail::IsChangeEnvironmentIdentifierDisabled();
    if ((isDisabled && current) || (!isDisabled && !current))
    {
        NN_RESULT_SUCCESS;
    }

    nn::nsd::detail::SetChangeEnvironmentIdentifierDisabled(isDisabled);
    NN_RESULT_SUCCESS;
}

Result ManagerImpl::IsChangeEnvironmentIdentifierDisabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    outValue.Set(nn::nsd::detail::IsChangeEnvironmentIdentifierDisabled());
    NN_RESULT_SUCCESS;
}

}}}}  // nn::nsd::detail::server
