﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/nsd_FqdnResolver.h>

#include <nn/nsd/nsd_Version.h>
#include <nn/nsd/nsd_ResultPrivate.h>

#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/detail/nsd_Log.h>
#include <nn/nsd/detail/nsd_StaticMutex.h>

#include <nn/nsd/detail/fs/nsd_Fs.h>
#include <nn/nsd/detail/device/nsd_Device.h>
#include <nn/nsd/detail/config/nsd_Config.h>

#include <nn/time/time_StandardNetworkSystemClock.h>

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#include <limits>

namespace nn { namespace nsd { namespace detail {

namespace
{
    FqdnResolver g_Resolver;

    //!< バージョンミスマッチや FS エラーなどにかかわらず、
    // 強制的に Resolve を成功させる FQDN リスト
    const nn::nsd::Fqdn ForcibleSuccessFqdnList[] =
    {
        {"api.sect.srv.nintendo.net"}, // Sirius
        {"ctest.cdn.nintendo.net"}, // 接続テスト
    };

#if 0
    bool s_IsInitialized = false; // 初期化されているか(されていない場合 FqdnResolver 時にデフォルト値がセットされる可能性がある)
    void InitializeResolverByDefaultSettings() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(detail::StaticMutex, s_Mutex, = NN_NSD_DETAIL_STATIC_MUTEX_INITIALIZER(false));

        std::lock_guard<nn::nsd::detail::StaticMutex> lock(s_Mutex);

        if(s_IsInitialized)
        {
            return;
        }

        // 何も設定注入されてない場合に利用するデフォルト設定を作る
        const DeviceId deviceId = {};
        const EnvironmentIdentifier env = detail::device::GetEnvironmentIdentifierFromSettings();
        detail::GetDefaultSettings(&g_SaveData, deviceId, env);

        NN_ABORT_UNLESS_RESULT_SUCCESS( g_Resolver.SetSaveData(&g_SaveData) ); // 初期セーブデータなので失敗しない想定
    }
#endif

    // 無効な FQDN を入れる
    void SetInvalidFqdn( Fqdn* pFqdn )
    {
        nn::util::Strlcpy(pFqdn->value, "unknown.dummy.nintendo.net", pFqdn->Size);
    }

    // 無効な リダイレクトURI を入れる
    void SetInvalidRedirectUri( nn::nsd::NasServiceSetting::RedirectUri* pUri )
    {
        nn::util::Strlcpy(pUri->value, "nintendo://unknown.nx.sys", pUri->Size);
    }


    // 現在時間を取得
    // ただしnsd設定がない場合と、ネットワーク時間取得に失敗した場合は
    // int64 最小値となり、有効期限チェックを必ずパスするようにする.
    void GetCurrentTime( bool isSettingPresent, nn::time::PosixTime* pCurrentTime )
    {
        NN_SDK_ASSERT_NOT_NULL( pCurrentTime );

        nn::time::PosixTime currentTime;
        if ( isSettingPresent && // 設定が存在するか
             NN_STATIC_CONDITION(config::IsSettingsExpirationEnabled) ) // 失効のしくみが有効か
        {
            auto result = nn::time::StandardNetworkSystemClock::GetCurrentTime( &currentTime ); // time による取得
            if ( result.IsFailure() )
            {
                NN_DETAIL_NSD_WARN("[NSD] time error (result=%08x %03d-%04d) \n", result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription() );
                currentTime.value = std::numeric_limits<int64_t>::min(); // 取得に失敗した場合は最小値 (比較で必ず成功する)
            }
        }
        else // 設定がないか、失効のしくみが無効のときは最小値 (比較で必ず成功する)
        {
            currentTime.value = std::numeric_limits<int64_t>::min();
        }
        *pCurrentTime = currentTime;
    }

    //----------------------------------------------------------------
    // Expire result の確認
    // 与えられた result が時限失効の検出であれば、
    // saveData の有効フラグを落としてセーブデータ保存する。
    nn::Result CheckResultSettingExpiredDetected( SaveData& saveData, nn::Result result )
    {
        if ( NN_STATIC_CONDITION(config::IsSettingsExpirationEnabled) && // 失効のしくみが有効か
             nn::nsd::ResultSettingExpiredDetected::Includes( result ) )
        {
            // 時限失効により無効
            saveData.isAvailable = false;
            NN_RESULT_DO( detail::fs::WriteIsAvailable( saveData ) );
            result = nn::nsd::ResultSettingExpired();
            NN_DETAIL_NSD_ERROR("[NSD] Detect expiration. Disabled save data.\n");
        }
        return result;
    }
}

//================================================================
// FqdnResolver
//
//----------------------------------------------------------------
// コンストラクタ
//
FqdnResolver::FqdnResolver() NN_NOEXCEPT:
    m_LockOfApi(true),
    m_pSaveData(nullptr),
    m_FatalError( nn::nsd::ResultInitializeFailed() ),
    m_IsSettingPresent(false)
{
}

//----------------------------------------------------------------
// FqdnResolver に使用する SaveData を教える
//
Result FqdnResolver::SetSaveData(SaveData* pSaveData) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);

    NN_SDK_ASSERT_NOT_NULL(pSaveData);

    m_pSaveData = pSaveData;

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// Resolve
//
Result FqdnResolver::Resolve(
    Fqdn *pOut,
    const Fqdn& fqdn) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    // バージョンミスマッチや FS エラーなどにかかわらず、そのまま FQDN を返す判定
    for(const auto& forcibleSuccessFqdn : ForcibleSuccessFqdnList)
    {
        if(fqdn == forcibleSuccessFqdn)
        {
            // 必ず成功させる
            *pOut = fqdn;
            NN_RESULT_SUCCESS;
        }
    }

    detail::fs::DumpFirstFsErrorIfFailed();// エラー時の調査用

    // 不正状態のチェック
    NN_RESULT_DO( m_FatalError );

    Fqdn envInsertedFqdn;
    NN_RESULT_DO( detail::InsertEnvironmentIdentifier(&envInsertedFqdn, fqdn, m_pSaveData->environmentIdentifier) );

    // 時間の取得
    nn::time::PosixTime currentTime;
    GetCurrentTime( m_IsSettingPresent, &currentTime );

    // 時間を考慮して Resolve
    Fqdn resolvedFqdn;
    nn::Result result = detail::Resolve(&resolvedFqdn, envInsertedFqdn , *m_pSaveData, currentTime);
    NN_RESULT_DO( CheckResultSettingExpiredDetected( *m_pSaveData, result) );
    *pOut = resolvedFqdn;

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ResolveEx
// 失敗時に無効な FQDN が格納される
//
Result FqdnResolver::ResolveEx(
    Result* pOutInnerResult,
    Fqdn *pOut,
    const Fqdn& fqdn) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOutInnerResult);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    auto result = this->Resolve(pOut, fqdn);
    if(result.IsFailure())
    {
        SetInvalidFqdn( pOut ); // 不正FQDN セット
    }

    *pOutInnerResult = result;

    // インポート済の設定に互換性がないことが起因するエラーは、必ず成功を返す
    if( nn::nsd::ResultInvalidSettingVersion::Includes(result) ||
        nn::nsd::ResultInvalidSettingData::Includes(result) ||
        m_FatalError.IsFailure())
    {
        NN_RESULT_SUCCESS;
    }

    return result;
}

//----------------------------------------------------------------
// NAS の FQDN 取得
//
Result FqdnResolver::GetNasRequestFqdn(Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    detail::fs::DumpFirstFsErrorIfFailed();// CI エラー時の調査用

    // 不正状態のチェック
    NN_RESULT_DO( m_FatalError );

    // 時間の取得
    nn::time::PosixTime currentTime;
    GetCurrentTime( m_IsSettingPresent, &currentTime );

    // 時間を考慮して FQDN 取得
    nn::Result result = detail::GetNasRequestFqdn(pOut, *m_pSaveData, currentTime);
    NN_RESULT_THROW( CheckResultSettingExpiredDetected(*m_pSaveData, result) );
}

//----------------------------------------------------------------
// NAS request の FQDN 取得
//
Result FqdnResolver::GetNasRequestFqdnEx(
    Result* pOutInnerResult,
    Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOutInnerResult);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    auto result = this->GetNasRequestFqdn(pOut);
    if(result.IsFailure())
    {
        SetInvalidFqdn( pOut ); // 不正FQDN セット
    }

    *pOutInnerResult = result;
    NN_RESULT_SUCCESS; // 必ず成功
}

//----------------------------------------------------------------
// NAS API の FQDN 取得
//
Result FqdnResolver::GetNasApiFqdn(Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    detail::fs::DumpFirstFsErrorIfFailed();// エラー時の調査用

    // 不正状態のチェック
    NN_RESULT_DO( m_FatalError );

    // 時間の取得
    nn::time::PosixTime currentTime;
    GetCurrentTime( m_IsSettingPresent, &currentTime );

    // 時間を考慮して FQDN 取得
    nn::Result result = detail::GetNasApiFqdn(pOut, *m_pSaveData, currentTime);
    NN_RESULT_THROW( CheckResultSettingExpiredDetected(*m_pSaveData, result) );
}

//----------------------------------------------------------------
// NAS API の FQDN 取得
// 失敗時に無効な FQDN が格納される
//
Result FqdnResolver::GetNasApiFqdnEx(
    Result* pOutInnerResult,
    Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOutInnerResult);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    auto result = this->GetNasApiFqdn(pOut);
    if(result.IsFailure())
    {
        SetInvalidFqdn( pOut ); // 不正FQDN セット
    }

    *pOutInnerResult = result;
    NN_RESULT_SUCCESS; // 必ず成功
}

//----------------------------------------------------------------
// NAS サービス設定の取得
//
Result FqdnResolver::GetNasServiceSetting(
    NasServiceSetting* pOutNasServiceSetting,
    const NasServiceName& nasServiceName) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOutNasServiceSetting);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    detail::fs::DumpFirstFsErrorIfFailed();// エラー時の調査用

    // 不正状態のチェック
    NN_RESULT_DO( m_FatalError );

    // 時間の取得
    nn::time::PosixTime currentTime;
    GetCurrentTime( m_IsSettingPresent, &currentTime );

    // 時間を考慮してサービス設定取得
    nn::Result result = detail::GetNasServiceSetting(pOutNasServiceSetting, nasServiceName, *m_pSaveData, currentTime);
    NN_RESULT_THROW( CheckResultSettingExpiredDetected(*m_pSaveData, result) );
}

//----------------------------------------------------------------
// NAS サービス設定の取得
// 失敗時に無効な設定が格納される
//
Result FqdnResolver::GetNasServiceSettingEx(
    Result* pOutInnerResult,
    NasServiceSetting* pOutNasServiceSetting,
    const NasServiceName& nasServiceName) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);
    NN_SDK_ASSERT_NOT_NULL(pOutInnerResult);
    NN_SDK_ASSERT_NOT_NULL(pOutNasServiceSetting);
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    auto result = this->GetNasServiceSetting(pOutNasServiceSetting, nasServiceName);
    if(result.IsFailure())
    {
        pOutNasServiceSetting->clientId = 0;
        SetInvalidRedirectUri( &pOutNasServiceSetting->redirectUri ); // 不正 ResirectUri セット
    }

    *pOutInnerResult = result;
    NN_RESULT_SUCCESS; // 必ず成功
}


//----------------------------------------------------------------
// 設定名の取得
nn::Result FqdnResolver::GetSettingName(SettingName* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    if( m_FatalError.IsFailure() )
    {
        nn::util::Strlcpy(pOut->value, "", static_cast<int>(SettingName::Size));
        NN_RESULT_THROW(m_FatalError);
    }

    // 起動時に設定が存在していなかったらエラー
    NN_RESULT_THROW_UNLESS( m_IsSettingPresent, nn::nsd::ResultNotFound() );

    *pOut = m_pSaveData->settingName;
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// 環境識別子の取得
nn::Result FqdnResolver::GetEnvironmentIdentifier(EnvironmentIdentifier* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");

    if( m_FatalError.IsFailure() )
    {
        nn::util::Strlcpy(pOut->value, "err", static_cast<int>(EnvironmentIdentifier::Size));
        NN_RESULT_THROW(m_FatalError);
    }

    *pOut = m_pSaveData->environmentIdentifier;
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// FqdnResolver の無効化
// セーブファイル壊れや不正データの検出による
//
void FqdnResolver::SetFatalError( nn::Result result ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LockOfApi);

    // エラーを覚えておく
    m_FatalError = result;

    if(result.IsFailure())
    {
        NN_DETAIL_NSD_ERROR("[NSD] SetFatalError (%08x, %03d-%04d)\n",
                   result.GetInnerValueForDebug(),
                   result.GetModule(), result.GetDescription());
    }
}

//----------------------------------------------------------------
// セーブデータの取得
//
nn::Result FqdnResolver::GetSaveData( SaveData** pSaveData ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pSaveData != nullptr, "Not called SetSaveData().");
    NN_SDK_ASSERT_NOT_NULL(pSaveData);

    *pSaveData = m_pSaveData;
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// FqdnResolver 取得
//
FqdnResolver* GetFqdnResolverPtr() NN_NOEXCEPT
{
    return &g_Resolver;
}

}}} // nn::nsd::detail
