﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_UserResourceManagementApi.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include "ns_Initialize.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/ns/srv/ns_UserResourceManagerImpl.h>
#endif

#include <mutex>

#include <nn/account/account_Api.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

namespace nn { namespace ns {

namespace {

struct InitializationManager
{
    os::MutexType _mutex;
    bool initialized;

    void lock() NN_NOEXCEPT
    {
        os::LockMutex(&_mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        os::UnlockMutex(&_mutex);
    }
} g_InitMgr = {NN_OS_MUTEX_INITIALIZER(false), false};

void InitializeImplicitly() NN_NOEXCEPT
{
    std::lock_guard<InitializationManager> lock(g_InitMgr);
    if (!g_InitMgr.initialized)
    {
        Initialize();
        g_InitMgr.initialized = true;
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
const char MessageNotInitialized[] = "[nn::ns::ProgressMonitorForDeleteUserSaveDataAll] Object not initialized\n";
const char MessageNotFinished[] = "[nn::ns::ProgressMonitorForDeleteUserSaveDataAll] Operation not finished yet\n";
#endif

} // namespace nn::ns::<anonymous>

// ---------------------------------------------------------------------------------------------
// API

Result CalculateUserSaveDataStatistics(ns::UserSaveDataStatistics* pOut, const account::Uid& uid) NN_NOEXCEPT
{
    InitializeImplicitly();
    return GetApplicationManagerInterface()->CalculateUserSaveDataStatistics(pOut, uid);
}

Result DeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll* pOut, const account::Uid& uid) NN_NOEXCEPT
{
    InitializeImplicitly();
    sf::SharedPointer<detail::IProgressMonitorForDeleteUserSaveDataAll> p;
    NN_RESULT_DO(GetApplicationManagerInterface()->DeleteUserSaveDataAll(&p, uid));
    NN_ABORT_UNLESS(p, "[nn::ns::DeleteUserSaveDataAll] Failed to get object\n");
    *pOut = ProgressMonitorForDeleteUserSaveDataAll(p.Detach());
    NN_RESULT_SUCCESS;
}

Result DeleteUserSystemSaveData(const account::Uid& uid, fs::SystemSaveDataId sysSaveId) NN_NOEXCEPT
{
    InitializeImplicitly();
    return GetApplicationManagerInterface()->DeleteUserSystemSaveData(uid, sysSaveId);
}

Result UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT
{
    InitializeImplicitly();
    return GetApplicationManagerInterface()->UnregisterNetworkServiceAccount(uid);
}

Result UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT
{
    InitializeImplicitly();
    return GetApplicationManagerInterface()->UnregisterNetworkServiceAccountWithUserSaveDataDeletion(uid);
}

// ---------------------------------------------------------------------------------------------
// ProgressMonitorForDeleteUserSaveDataAll

ProgressMonitorForDeleteUserSaveDataAll::ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT
    : m_Ptr(nullptr)
{
}

ProgressMonitorForDeleteUserSaveDataAll::ProgressMonitorForDeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll&& rhs) NN_NOEXCEPT
    : m_Ptr(rhs.m_Ptr)
    , m_Data(rhs.m_Data)
{
    rhs.m_Ptr = nullptr;
}
ProgressMonitorForDeleteUserSaveDataAll& ProgressMonitorForDeleteUserSaveDataAll::operator=(ProgressMonitorForDeleteUserSaveDataAll&& rhs) NN_NOEXCEPT
{
    ProgressMonitorForDeleteUserSaveDataAll tmp(std::move(rhs));
    std::swap(this->m_Ptr, tmp.m_Ptr);
    std::swap(this->m_Data, tmp.m_Data);
    return *this;
}
ProgressMonitorForDeleteUserSaveDataAll::ProgressMonitorForDeleteUserSaveDataAll(detail::IProgressMonitorForDeleteUserSaveDataAll* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
    Update();
}
ProgressMonitorForDeleteUserSaveDataAll::~ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        NN_ABORT_UNLESS(IsFinished(), MessageNotFinished);
        sf::ReleaseSharedObject(m_Ptr);
    }
}
void ProgressMonitorForDeleteUserSaveDataAll::GetSystemEvent(os::SystemEvent* pOut) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);

    sf::NativeHandle h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetSystemEvent(&h));
    pOut->AttachReadableHandle(h.GetOsHandle(), h.IsManaged(), os::EventClearMode_ManualClear);
    h.Detach();
}
bool ProgressMonitorForDeleteUserSaveDataAll::IsFinished() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    bool isFinished;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->IsFinished(&isFinished));
    return isFinished;
}
Result ProgressMonitorForDeleteUserSaveDataAll::GetResult() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    return m_Ptr->GetResult();
}
TimeSpan ProgressMonitorForDeleteUserSaveDataAll::GetElapsedTime() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    return m_Data.GetElapsedTime();
}
void ProgressMonitorForDeleteUserSaveDataAll::Update() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetProgress(&m_Data));
}
UserSaveDataStatistics ProgressMonitorForDeleteUserSaveDataAll::GetStatistics() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    return m_Data.GetStatistics();
}
bool ProgressMonitorForDeleteUserSaveDataAll::IsSystemSaveData() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    return m_Data.IsSystemSaveData();
}
bool ProgressMonitorForDeleteUserSaveDataAll::TryGetCurrentApplicationId(ApplicationId* pOut) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Ptr != nullptr, MessageNotInitialized);
    return m_Data.TryGetCurrentApplicationId(pOut);
}

}}  // namespace ns
