﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include "ns_Initialize.h"

namespace nn { namespace ns {
    Result GetSystemDeliveryInfo(SystemDeliveryInfo* outValue) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetSystemDeliveryInfo(outValue);
    }

    Result CompareSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CompareSystemDeliveryInfo(outValue, lhs, rhs);
    }

    Result SelectLatestSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo& receiverSystemInfo, const ApplicationDeliveryInfo receiverAppInfoList[], int numAppInfoList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->SelectLatestSystemDeliveryInfo(outValue, sf::InArray<SystemDeliveryInfo>(list, numList), receiverSystemInfo, sf::InArray<ApplicationDeliveryInfo>(receiverAppInfoList, numAppInfoList));
    }

    Result SelectLatestSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
    {
        return SelectLatestSystemDeliveryInfo(outValue, list, numList, receiverSystemInfo, nullptr, 0);
    }

    Result VerifyDeliveryProtocolVersion(const SystemDeliveryInfo& info) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->VerifyDeliveryProtocolVersion(info);
    }

    Result GetApplicationDeliveryInfo(int* outCount, ApplicationDeliveryInfo* outList, int numList, ncm::ApplicationId id, ApplicationDeliveryAttribute attributes) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetApplicationDeliveryInfo(outCount, sf::OutArray<ApplicationDeliveryInfo>(outList, numList), id, attributes);
    }

    Result HasAllContentsToDeliver(bool* outValue, const ApplicationDeliveryInfo appInfoList[], int numList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->HasAllContentsToDeliver(outValue, sf::InArray<ApplicationDeliveryInfo>(appInfoList, numList));
    }

    Result CompareApplicationDeliveryInfo(int* outValue, const ApplicationDeliveryInfo lhsList[], int numLhsList, const ApplicationDeliveryInfo rhsList[], int numRhsList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CompareApplicationDeliveryInfo(outValue, sf::InArray<ApplicationDeliveryInfo>(lhsList, numLhsList), sf::InArray<ApplicationDeliveryInfo>(rhsList, numRhsList));
    }

    Result CanDeliverApplication(bool* outValue, const ApplicationDeliveryInfo receiverInfoList[], int numReceiverInfoList, const ApplicationDeliveryInfo senderInfoList[], int numSenderInfoList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CanDeliverApplication(outValue, sf::InArray<ApplicationDeliveryInfo>(receiverInfoList, numReceiverInfoList), sf::InArray<ApplicationDeliveryInfo>(senderInfoList, numSenderInfoList));
    }

    Result CanDeliverApplication(const ApplicationDeliveryInfo senderInfoList[], int numSenderInfoList) NN_NOEXCEPT
    {
        bool value;
        const ApplicationDeliveryInfo* receiverInfoList = nullptr;
        const size_t numReceiverInfoList = 0;
        auto result = GetApplicationManagerInterface()->CanDeliverApplication(&value, sf::InArray<ApplicationDeliveryInfo>(receiverInfoList, numReceiverInfoList), sf::InArray<ApplicationDeliveryInfo>(senderInfoList, numSenderInfoList));
        NN_RESULT_DO(result);
        NN_ABORT_UNLESS(!value);
        NN_RESULT_SUCCESS;
    }

    Result ListContentMetaKeyToDeliverApplication(int* outCount, ncm::ContentMetaKey outList[], int numList, int offset, const ApplicationDeliveryInfo infoList[], int numInfoList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->ListContentMetaKeyToDeliverApplication(outCount, sf::OutArray<ncm::ContentMetaKey>(outList, numList), offset, sf::InArray<ApplicationDeliveryInfo>(infoList, numInfoList));
    }

    Result NeedsSystemUpdateToDeliverApplication(bool* outValue, const ApplicationDeliveryInfo appList[], int numAppList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->NeedsSystemUpdateToDeliverApplication(outValue, sf::InArray<ApplicationDeliveryInfo>(appList, numAppList), receiverSystemInfo);
    }

    Result EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey keyList[], int numKey) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->EstimateRequiredSize(outValue, sf::InArray<ncm::ContentMetaKey>(keyList, numKey));
    }

    Result RequestReceiveApplication(AsyncResult* outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId id, const ncm::ContentMetaKey keyList[], int numList, ncm::StorageId storageId) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetApplicationManagerInterface()->RequestReceiveApplication(&nativeHandle, &sp, ipv4, port, sf::InArray<ncm::ContentMetaKey>(keyList, numList), id, storageId));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result CommitReceiveApplication(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CommitReceiveApplication(id);
    }

    Result GetReceiveApplicationProgress(ReceiveApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetReceiveApplicationProgress(outValue, id);
    }

    Result RequestSendApplication(AsyncResult* outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId id, const ncm::ContentMetaKey keyList[], int numList) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetApplicationManagerInterface()->RequestSendApplication(&nativeHandle, &sp, ipv4, port, sf::InArray<ncm::ContentMetaKey>(keyList, numList), id));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result GetSendApplicationProgress(SendApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetSendApplicationProgress(outValue, id);
    }

    int ListNotCommittedContentMeta(ncm::ContentMetaKey outList[], int listCount, ncm::ApplicationId id, int offset) NN_NOEXCEPT
    {
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->ListNotCommittedContentMeta(&count, sf::OutArray<ncm::ContentMetaKey>(outList, listCount), id, offset));
        return count;
    }

    void CreateDownloadTask(const ncm::ContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->CreateDownloadTask(sf::InArray<ncm::ContentMetaKey>(keyList, listCount), id));
    }

    Result GetApplicationDeliveryInfoHash(ApplicationDeliveryInfoHash* outValue, const ApplicationDeliveryInfo infoList[], int numList) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetApplicationDeliveryInfoHash(outValue, sf::InArray<ApplicationDeliveryInfo>(infoList, numList));
    }
}}
