﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/ns_Async.h>

namespace nn { namespace ns {
    namespace detail {

        template<typename T>
        AsyncBase<T>::~AsyncBase() NN_NOEXCEPT
        {
            if (m_Async)
            {
                Finalize();
            }
        }
        template<typename T>
        void AsyncBase<T>::Initialize(sf::SharedPointer<T> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
        {
            m_Event.emplace();
            m_Event->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_ManualClear);
            m_Async = async;
        }

        template<typename T>
        void AsyncBase<T>::Wait() NN_NOEXCEPT
        {
            m_Event->Wait();
        }

        template<typename T>
        bool AsyncBase<T>::TryWait() NN_NOEXCEPT
        {
            return m_Event->TryWait();
        }

        template<typename T>
        void AsyncBase<T>::Cancel() NN_NOEXCEPT
        {
            m_Async->Cancel();
        }

        template<typename T>
        void AsyncBase<T>::Finalize() NN_NOEXCEPT
        {
            m_Async->Cancel();
            m_Event->Wait();
            m_Event = util::nullopt;
            m_Async = nullptr;
        }

        template<typename T>
        os::SystemEvent& AsyncBase<T>::GetEvent() NN_NOEXCEPT
        {
            return *m_Event;
        }

        template<typename T>
        void AsyncBase<T>::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->GetErrorContext(outValue));
        }

        void AsyncValueImpl::Initialize(sf::SharedPointer<detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
        {
            BaseType::Initialize(async, nativeHandle);
        }

        AsyncValueImpl::~AsyncValueImpl() NN_NOEXCEPT {}

        Result AsyncValueImpl::Get(void* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            Wait();
            return m_Async->Get(sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize));
        }

        uint64_t AsyncValueImpl::GetSize() NN_NOEXCEPT
        {
            uint64_t size;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->GetSize(&size));
            return size;
        }

        void AsyncValueImpl::Wait() NN_NOEXCEPT
        {
            BaseType::Wait();
        }

        bool AsyncValueImpl::TryWait() NN_NOEXCEPT
        {
            return BaseType::TryWait();
        }

        void AsyncValueImpl::Cancel() NN_NOEXCEPT
        {
            BaseType::Cancel();
        }

        void AsyncValueImpl::Finalize() NN_NOEXCEPT
        {
            BaseType::Finalize();
        }

        os::SystemEvent& AsyncValueImpl::GetEvent() NN_NOEXCEPT
        {
            return BaseType::GetEvent();
        }

        void AsyncValueImpl::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
        {
            BaseType::GetErrorContext(outValue);
        }

        void ProgressAsyncResultImpl::Initialize(sf::SharedPointer<detail::IProgressAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
        {
            BaseType::Initialize(async, nativeHandle);
        }

        ProgressAsyncResultImpl::~ProgressAsyncResultImpl() NN_NOEXCEPT {}

        Result ProgressAsyncResultImpl::Get() NN_NOEXCEPT
        {
            Wait();
            NN_RESULT_DO(m_Async->Get());

            NN_RESULT_SUCCESS;
        }

        void ProgressAsyncResultImpl::Wait() NN_NOEXCEPT
        {
            BaseType::Wait();
        }

        bool ProgressAsyncResultImpl::TryWait() NN_NOEXCEPT
        {
            return BaseType::TryWait();
        }

        void ProgressAsyncResultImpl::Cancel() NN_NOEXCEPT
        {
            BaseType::Cancel();
        }

        void ProgressAsyncResultImpl::Finalize() NN_NOEXCEPT
        {
            BaseType::Finalize();
        }

        os::SystemEvent& ProgressAsyncResultImpl::GetEvent() NN_NOEXCEPT
        {
            return BaseType::GetEvent();
        }

        void ProgressAsyncResultImpl::GetProgress(void* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->GetProgress(sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize)));
        }

        Result ProgressAsyncResultImpl::GetDetailResult() NN_NOEXCEPT
        {
            return m_Async->GetDetailResult();
        }

        void ProgressAsyncResultImpl::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
        {
            BaseType::GetErrorContext(outValue);
        }
    }

    void AsyncResult::Initialize(sf::SharedPointer<detail::IAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        BaseType::Initialize(async, nativeHandle);
    }

    AsyncResult::~AsyncResult() NN_NOEXCEPT {}

    Result AsyncResult::Get() NN_NOEXCEPT
    {
        Wait();
        NN_RESULT_DO(m_Async->Get());

        NN_RESULT_SUCCESS;
    }

    void AsyncResult::Wait() NN_NOEXCEPT
    {
        BaseType::Wait();
    }

    bool AsyncResult::TryWait() NN_NOEXCEPT
    {
        return BaseType::TryWait();
    }

    void AsyncResult::Cancel() NN_NOEXCEPT
    {
        BaseType::Cancel();
    }

    void AsyncResult::Finalize() NN_NOEXCEPT
    {
        BaseType::Finalize();
    }

    os::SystemEvent& AsyncResult::GetEvent() NN_NOEXCEPT
    {
        return BaseType::GetEvent();
    }

    void AsyncResult::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        BaseType::GetErrorContext(outValue);
    }
}}
