﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/os/os_NativeHandle.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_ApplicationLanguage.h>
#include <nn/util/util_StringUtil.h>

#include "ns_Initialize.h"

namespace nn { namespace ns {
    Result CalculateApplicationOccupiedSize(ApplicationOccupiedSize* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetContentManagementInterface()->CalculateApplicationOccupiedSize(outValue, id);
    }

    const ApplicationControlProperty& ApplicationControlDataAccessor::GetProperty() const NN_NOEXCEPT
    {
        return *static_cast<ApplicationControlProperty*>(m_Data);
    }

    const void* ApplicationControlDataAccessor::GetIconData() const NN_NOEXCEPT
    {
        auto dataPtr = static_cast<Bit8*>(m_Data);
        return &dataPtr[sizeof(ApplicationControlProperty)];
    }

    size_t ApplicationControlDataAccessor::GetIconSize() const NN_NOEXCEPT
    {
        return m_Size - sizeof(ApplicationControlProperty);
    }

    Result LaunchApplication(os::ProcessId* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationLaunchInfo info;
        NN_RESULT_DO(GetApplicationLaunchInfo(&info, id));
        uint8_t programIndex;
        NN_RESULT_DO(GetMainApplicationProgramIndex(&programIndex, id));
        return GetApplicationManagerInterface()->LaunchApplication(outValue, info, programIndex);
    }

    Result LaunchApplication(os::ProcessId* outValue, ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT
    {
        ApplicationLaunchInfo info;
        NN_RESULT_DO(GetApplicationLaunchInfo(&info, id));
        return GetApplicationManagerInterface()->LaunchApplication(outValue, info, programIndex);
    }

    Result LaunchApplication(os::ProcessId* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->LaunchApplication(outValue, info, programIndex);
    }

    Result TerminateApplication(os::ProcessId pid) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->TerminateApplication(pid);
    }

    Result BeginInstallApplication(ncm::ApplicationId id, std::uint32_t version, ncm::StorageId storage) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->BeginInstallApplication(id, version, storage);
    }

    Result PushDownloadTaskList(const void* buffer, size_t size) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->PushDownloadTaskList(sf::InBuffer(static_cast<const char*>(buffer), size));
    }

    Result CheckApplicationLaunchVersion(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CheckApplicationLaunchVersion(id);
    }

    Result CheckApplicationLaunchRights(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CheckApplicationLaunchRights(id);
    }

    Result CheckApplicationResumeRights(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CheckApplicationResumeRights(id);
    }

    Result GetApplicationLogoData(size_t* outSize, void* buffer, size_t bufferSize, ncm::ApplicationId id, const char* logoPath) NN_NOEXCEPT
    {
        sf::OutBuffer outBuffer(reinterpret_cast<char*>(buffer), bufferSize);

        ncm::Path path;
        util::Strlcpy(path.string, logoPath, sizeof(path.string));

        int64_t readSize;
        NN_RESULT_TRY(GetApplicationManagerInterface()->GetApplicationLogoData(&readSize, outBuffer, id, path))
            NN_RESULT_CATCH_CONVERT(fs::ResultDataCorrupted, ResultApplicationLogoCorrupted());
        NN_RESULT_END_TRY

        *outSize = static_cast<size_t>(readSize);
        NN_RESULT_SUCCESS;
    }

    Result GetTotalSpaceSize(int64_t* pOutSize, ncm::StorageId storage) NN_NOEXCEPT
    {
        return GetContentManagementInterface()->GetTotalSpaceSize(pOutSize, storage);
    }

    Result GetFreeSpaceSize(int64_t* pOutSize, ncm::StorageId storage) NN_NOEXCEPT
    {
        return GetContentManagementInterface()->GetFreeSpaceSize(pOutSize, storage);
    }

    settings::LanguageCode GetApplicationSupportedLanguage(Bit32 supportedLanguageFlag) NN_NOEXCEPT
    {
        Bit8 language;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetReadOnlyApplicationControlDataInterface()->GetApplicationDesiredLanguage(&language, supportedLanguageFlag));
        settings::LanguageCode code;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetReadOnlyApplicationControlDataInterface()->ConvertApplicationLanguageToLanguageCode(&code, language));

        return code;
    }

    Result GetBackgroundDownloadStressTaskInfo(nim::BackgroundDownloadStressTaskInfo* outValue) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetBackgroundDownloadStressTaskInfo(outValue);
    }

    Result GetBackgroundApplyDeltaStressTaskInfo(nim::BackgroundDownloadStressTaskInfo* outValue) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetBackgroundApplyDeltaStressTaskInfo(outValue);
    }

    void GetRequestServerStopper(RequestServerStopper* outValue) NN_NOEXCEPT
    {
        sf::SharedPointer<detail::IRequestServerStopper> interface;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->GetRequestServerStopper(&interface));

        *outValue = RequestServerStopper(std::move(interface));
    }

    Result GetStorageSize(int64_t* pOutTotalSize, int64_t* pOutFreeSpaceSize, ncm::StorageId storageId) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetStorageSize(pOutTotalSize, pOutFreeSpaceSize, storageId);
    }

    sf::NativeHandle GetDynamicCommitEvent() NN_NOEXCEPT
    {
        sf::NativeHandle sfHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->GetDynamicCommitEvent(sf::Out<sf::NativeHandle>(&sfHandle)));
        return sfHandle;
    }

    Result BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->BoostSystemMemoryResourceLimit(boostSize);
    }

    void EnableApplicationCrashReport(bool isEnabled) NN_NOEXCEPT
    {
        GetApplicationManagerInterface()->EnableApplicationCrashReport(isEnabled);
    }

    bool IsApplicationCrashReportEnabled() NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->IsApplicationCrashReportEnabled();
    }

    void EnableApplicationAllThreadDumpOnCrash(bool isEnabled) NN_NOEXCEPT
    {
        GetApplicationManagerInterface()->EnableApplicationAllThreadDumpOnCrash(isEnabled);
    }

    Result GetMainApplicationProgramIndex(uint8_t* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetMainApplicationProgramIndex(outValue, id);
    }

    Result GetMainApplicationProgramIndex(uint8_t* outValue, const ApplicationLaunchInfo& info) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetMainApplicationProgramIndexByApplicationLaunchInfo(outValue, info);
    }

    Result GetApplicationLaunchInfo(ApplicationLaunchInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetApplicationLaunchInfo(outValue, id);
    }

    Result AcquireApplicationLaunchInfo(ApplicationLaunchInfo* outValue, os::ProcessId processId) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->AcquireApplicationLaunchInfo(outValue, processId);
    }

    // RightsEnvironment

    Result CreateRightsEnvironment(RightsEnvironmentHandle* pOut) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->CreateRightsEnvironment(pOut);
    }

    void DestroyRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->DestroyRightsEnvironment(handle));
    }

    Result AddTargetApplicationToRightsEnvironment(RightsEnvironmentHandle handle, ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->AddTargetApplicationToRightsEnvironment(handle, applicationId);
    }

    Result SetUsersToRightsEnvironment(RightsEnvironmentHandle handle, bool anyUser, util::Span<const account::Uid> uids) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->SetUsersToRightsEnvironment(handle, anyUser, uids);
    }

    Result ActivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->ActivateRightsEnvironment(handle);
    }

    void DeactivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->DeactivateRightsEnvironment(handle));
    }

    Result GetRightsEnvironmentStatus(RightsEnvironmentStatus* pOut, RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        return GetApplicationManagerInterface()->GetRightsEnvironmentStatus(pOut, handle);
    }
}}
