﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/ns/detail/ns_ApplicationLanguage.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationControlDataSystemApi.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ns/ns_Result.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

#include "ns_Initialize.h"

namespace nn { namespace ns {

namespace {
    const ApplicationTitle InvalidTitle {};
}

    bool IsValid(const ApplicationTitle& title) NN_NOEXCEPT
    {
        return std::strlen(title.name) > 0 || std::strlen(title.publisher) > 0;
    }

    const ApplicationTitle& ApplicationControlProperty::GetTitle(const settings::LanguageCode& language) const NN_NOEXCEPT
    {
        Bit8 titleLanguage;
        if (GetReadOnlyApplicationControlDataInterface()->ConvertLanguageCodeToApplicationLanguage(&titleLanguage, language).IsFailure())
        {
            return InvalidTitle;
        }
        return title[titleLanguage];
    }

    const ApplicationTitle& ApplicationControlProperty::GetDefaultTitle() const NN_NOEXCEPT
    {
        Bit32 existLanguageFlag = 0;
        for (int i = 0; i < MaxLanguageCount; i++)
        {
            if (IsValid(title[i]))
            {
                existLanguageFlag |= 0x1u << i;
            }
        }

        if (existLanguageFlag == 0)
        {
            return InvalidTitle;
        }

        Bit8 language;
        if (GetReadOnlyApplicationControlDataInterface()->GetApplicationDesiredLanguage(&language, existLanguageFlag).IsFailure())
        {
            return InvalidTitle;
        }

        auto& t = title[language];
        NN_ABORT_UNLESS(IsValid(t));

        return t;
    }

    bool ApplicationControlProperty::SupportsLanguage(const settings::LanguageCode& code) const NN_NOEXCEPT
    {
        Bit8 language;
        if (GetReadOnlyApplicationControlDataInterface()->ConvertLanguageCodeToApplicationLanguage(&language, code).IsFailure())
        {
            return false;
        }

        return supportedLanguageFlag == 0 || (supportedLanguageFlag & detail::ToSupportedLanguageFlag(static_cast<detail::ApplicationLanguage>(language))) != 0;
    }

    Result GetApplicationControlData(size_t* outValue, void* buffer, size_t bufferSize, ApplicationControlSource source, ncm::ApplicationId id) NN_NOEXCEPT
    {
        uint32_t outSize;
        NN_RESULT_DO(GetReadOnlyApplicationControlDataInterface()->GetApplicationControlData(&outSize, sf::OutBuffer(static_cast<char*>(buffer), bufferSize), source, id));
        *outValue = outSize;
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationControlProperty(ApplicationControlProperty* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetApplicationManagerInterface()->GetApplicationControlProperty(outValue, info, programIndex));
        NN_RESULT_SUCCESS;
    }

    Result RequestDownloadApplicationControlData(AsyncResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetApplicationManagerInterface()->RequestDownloadApplicationControlData(&nativeHandle, &sp, id));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    void InvalidateAllApplicationControlCache() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->InvalidateAllApplicationControlCache());
    }

    void InvalidateApplicationControlCache(ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->InvalidateApplicationControlCache(id));
    }

    int GetMaxApplicationControlCacheCount() NN_NOEXCEPT
    {
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->GetMaxApplicationControlCacheCount(&count));
        return count;
    }

    int ListApplicationControlCacheEntryInfo(ApplicationControlCacheEntryInfo outList[], int count) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetApplicationManagerInterface()->ListApplicationControlCacheEntryInfo(&outCount, sf::OutArray<ApplicationControlCacheEntryInfo>(outList, static_cast<size_t>(count))));
        return outCount;
    }

    // TORIAEZU: 暫定的な対応
    void GetCommonProgramControlProperty(CommonProgramControlProperty*, const ApplicationControlProperty&) NN_NOEXCEPT
    {
    }

    void GetIndividualProgramControlProperty(IndividualProgramControlProperty*, const ApplicationControlProperty&) NN_NOEXCEPT
    {
    }
}}
