﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_VirtualAccountStore.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace nim { namespace srv {

    Result VirtualAccountStore::Register(const account::Uid& uid, ec::system::VirtualAccountId vaId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_DataMutex);

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            if (entry.uid == uid)
            {
                entry.Set(uid, vaId);
                NN_RESULT_DO(Save());
                NN_RESULT_SUCCESS;
            }
        }

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            if (entry.IsEmpty())
            {
                entry.Set(uid, vaId);
                NN_RESULT_DO(Save());
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultOutOfMaxVirtualAccountRegistration());
    }

    Result VirtualAccountStore::Unregister(const account::Uid& uid) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_DataMutex);

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            if (entry.uid == uid)
            {
                entry.Clear();
                NN_RESULT_DO(Save());
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultRegisteredVirtualAccountNotFound());
    }

    Result VirtualAccountStore::Get(ec::system::VirtualAccountId* outValue, const account::Uid& uid) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_DataMutex);

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            if (entry.uid == uid)
            {
                *outValue = entry.vaId;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultRegisteredVirtualAccountNotFound());
    }

    Result VirtualAccountStore::GetAccount(account::Uid* outValue, ec::system::VirtualAccountId vaId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_DataMutex);

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            if (entry.vaId == vaId)
            {
                *outValue = entry.uid;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultRegisteredVirtualAccountNotFound());
    }

    void VirtualAccountStore::Clear() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_DataMutex);

        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = GetRef().entryList[i];
            entry.Clear();
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(Save());
    }

    void VirtualAccountStore::SetInitialData(VirtualAccountData* data) NN_NOEXCEPT
    {
        for (int i = 0; i < VirtualAccountData::MaxEntryCount; i++)
        {
            auto& entry = data->entryList[i];
            entry.Clear();
        }
    }

}}}
