﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/account_NintendoAccountAuthorizationRequest.h>

#include <nn/dauth/dauth_Types.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Execution.h>
#include <nn/nsd/nsd_BackboneSettingsTypes.h>
#include <nn/nim/srv/detail/nim_MutexAtomic.h>

namespace nn { namespace nim { namespace srv {

namespace TokenStore {

    /**
     * @brief   任意シグナルイベント型参照保持式キャンセラ基底。
     */
    template<typename TEvent>
    class SignalCanceler : public std::atomic<TEvent*>
    {
    private:
        typedef std::atomic<TEvent*> BaseType;

    public:
        /**
         * @brief   コンストラクタ。
         */
        SignalCanceler() NN_NOEXCEPT : BaseType(nullptr) {}

        /**
         * @brief   中断要求。
         *
         * @details 未実施状態では何も行われません。
         */
        void Cancel() NN_NOEXCEPT
        {
            auto pHandle = BaseType::exchange(nullptr, std::memory_order_acq_rel);
            if (nullptr != pHandle)
            {
                pHandle->Signal();
            }
        }

        /**
         * @brief   無効化。
         */
        void Invalidate() NN_NOEXCEPT
        {
            BaseType::store(nullptr, std::memory_order_release);
        }
    };

    /**
     * @brief   任意非同期ハンドル型実体保持式キャンセラ基底。
     */
    template<typename TAsyncHandle>
    class Canceler
    {
    private:
        ::nn::os::SdkRecursiveMutex         m_Lock;
        ::nn::util::optional<TAsyncHandle>  m_Handle;

    public:
        Canceler() NN_NOEXCEPT {}

        /**
         * @brief   中断要求。
         */
        void Cancel() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Lock)> guard(m_Lock);
            if (m_Handle)
            {
                m_Handle->Cancel();
            }
        }

        /**
         * @brief   無効化。
         */
        void Invalidate() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Lock)> guard(m_Lock);
            m_Handle = ::nn::util::nullopt;
        }

        /**
         * @brief   インスタンス再配置要求。
         */
        TAsyncHandle* Emplace() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Lock)> guard(m_Lock);
            m_Handle.emplace();
            return &(*m_Handle);
        }
    };

    /**
     * @brief   任意非同期ハンドル型参照保持式キャンセラ基底。
     */
    template<typename TAsyncHandle>
    class ReferredCanceler : public std::atomic<TAsyncHandle*>
    {
    private:
        typedef std::atomic<TAsyncHandle*> BaseType;

    public:
        /**
         * @brief   コンストラクタ。
         */
        ReferredCanceler() NN_NOEXCEPT : BaseType(nullptr) {}

        /**
         * @brief   中断要求。
         */
        void Cancel() NN_NOEXCEPT
        {
            auto pHandle = BaseType::exchange(nullptr, std::memory_order_acq_rel);
            if (nullptr != pHandle)
            {
                pHandle->Cancel();
            }
        }

        /**
         * @brief   無効化。
         */
        void Invalidate() NN_NOEXCEPT
        {
            BaseType::store(nullptr, std::memory_order_release);
        }
    };

    /**
     * @brief   トークン所有者情報( ユーザアカウント依存 )
     */
    struct OwnerUserAccount
    {
        ::nn::account::Uid                      uid;
        ::nn::account::NetworkServiceAccountId  nsa;

        /**
         * @brief       コンストラクタ。
         */
        explicit OwnerUserAccount(const ::nn::account::Uid& uid_) NN_NOEXCEPT : uid(uid_), nsa() {}

        /**
         * @brief       情報の無効化。
         */
        void Invalidate() NN_NOEXCEPT
        {
            uid = ::nn::account::InvalidUid;
            nsa = {};
        }

        /**
         * @brief       等価比較演算子のオーバーロードです。
         *
         * @param[in]   lhs     比較対象の左辺を指定します。
         * @param[in]   rhs     比較対象の右辺を指定します。
         *
         * @return
         *  比較した両者が等しければ true を、等しくなければ false を返します。
         */
        friend bool operator ==(const OwnerUserAccount& lhs, const OwnerUserAccount& rhs) NN_NOEXCEPT
        {
            return (lhs.uid == rhs.uid && lhs.nsa == rhs.nsa);
        }

        /**
         * @brief       非等価比較演算子のオーバーロードです。
         *
         * @param[in]   lhs     比較対象の左辺を指定します。
         * @param[in]   rhs     比較対象の右辺を指定します。
         *
         * @return
         *  比較した両者が等しければ true を、等しくなければ false を返します。
         */
        friend bool operator !=(const OwnerUserAccount& lhs, const OwnerUserAccount& rhs) NN_NOEXCEPT
        {
            return !(lhs == rhs);
        }
    };

    /**
     * @brief   NSA-ID トークン取得。
     */
    class NsaId
    {
        NN_DISALLOW_COPY(NsaId);
        NN_DISALLOW_MOVE(NsaId);

    public:
        typedef ::nn::account::AsyncContext IAsync;
        typedef Canceler<IAsync>            ICanceler;

        /**
         * @brief       トークン最大長 ( 3072 + 1 byte )。
         */
        static const size_t LengthMax = static_cast<size_t>(::nn::account::NetworkServiceAccountIdTokenLengthMax + 1);

        /**
         * @brief       コンストラクタ。
         */
        NsaId() NN_NOEXCEPT {}

        /**
         * @brief       トークン取得要求。
         *
         * @param[out]  pOutToken           出力先メモリ領域先頭アドレスへのポインタ。@n
         *                                  出力トークンは null終端されます。
         * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
         *                                  トークンサイズ最大は @ref LengthMax。
         * @param[in]   uid                 Accountライブラリから取得する NSA-ID-TOKEN の対象ユーザアカウントID。
         * @param[in]   info                トークンIDに含めるシステムプログラムの識別情報。
         * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         *
         * @pre
         *              - pCanceler != nullptr
         *              - pOutToken != nullptr
         *              - availableCapacity >= LengthMax
         *              - uid が示すユーザが存在する。
         */
        Result AcquireToken(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, const ::nn::account::SystemProgramIdentification& info, ICanceler* pCanceler) NN_NOEXCEPT;
    };

    /**
     * @brief   NA-ID トークンストア( 共通項 )。
     */
    struct NaId
    {
        /**
         * @brief   ニンテンドーアカウントサーバーとのOAuth 2.0/Open ID Connectの手続きにおいて使用するパラメータ型。
         */
        typedef ::nn::account::NintendoAccountAuthorizationRequestParameters AuthParameter;

        /**
         * @brief   アカウントライブラリ非同期取得ハンドル。
         */
        typedef ::nn::account::NintendoAccountAuthorizationRequestContext IAsync;

        /**
         * @brief   非同期取得ハンドルキャンセル用コンテナ。
         */
        typedef ReferredCanceler<IAsync> ICanceler;

        /**
         * @brief   トークン最大長 ( 1536 + 1 byte )。
         */
        static const size_t LengthMax = static_cast<size_t>(::nn::account::NintendoAccountIdTokenLengthMax + 1);

        /**
         * @brief   NA-ID トークン取得。( 単一領域での排他管理 )
         *
         * @details 純粋仮想関数を持つ基底クラスです。@n
         *          以下要素を基底に提供する派生クラスを作成する必要があります。
         */
        class SingleCacheBase
        {
            NN_DISALLOW_COPY(SingleCacheBase);
            NN_DISALLOW_MOVE(SingleCacheBase);

        public:
            /**
             * @brief       コンストラクタ。
             *
             * @details     以下の初期値が採用されます。@n
             *              @li ::nn::TimeSpan::FromHours(1)
             */
            explicit SingleCacheBase() NN_NOEXCEPT;

            /**
             * @brief       コンストラクタ。
             *
             * @param[in]   cachedTokenAvailabilityPeriod   トークンキャッシュ利用可能期間( Uptime単位  )。
             */
            explicit SingleCacheBase(const ::nn::TimeSpan& cachedTokenAvailabilityPeriod) NN_NOEXCEPT;

            /**
             * @brief       デストラクタ。
             */
            ~SingleCacheBase() NN_NOEXCEPT;

            /**
             * @brief       キャッシュ済トークンの無効化(破棄)。
             */
            void Invalidate() NN_NOEXCEPT;

            /**
             * @brief       トークン取得要求。
             *
             * @param[out]  pOutToken           出力先メモリ領域先頭アドレスへのポインタ。@n
             *                                  出力トークンは null終端されます。
             * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
             *                                  トークンサイズ最大は @ref LengthMax。
             * @param[in]   uid                 Accountライブラリから取得する NA-ID-TOKEN の対象ユーザアカウントID。
             * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
             *
             * @return      処理結果が返ります。
             * @retval      nn::nim::ResultBufferNotEnough  トークン出力領域の容量が不足しています。@n
             *                                              @ref LengthMax 以上を確保してください。
             *
             * @pre
             *              - pCanceler != nullptr
             *              - pOutToken != nullptr
             *              - uid が示すユーザが存在する。
             *
             * @details     nim 側でキャッシュするため、同時要求は排他されます。
             */
            Result AcquireToken(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, ICanceler* pCanceler) NN_NOEXCEPT;

        protected:
            /**
             * @brief       本クラスインスタンスがトークン取得のために利用する各種パラメータ設定を派生先実装経由で取得します。
             *
             * @param[out]  pOutService 利用対象サービス設定格納構造体。@n
             *                          NAS サービス設定。(@ref ::nn::nsd::GetNasServiceSetting() が利用される想定です。)@n
             *                          派生先実装で本コンテナへ設定を入力してください。
             * @param[out]  pOutAuth    利用対象認証手続きパラメータ格納構造体。@n
             *                          ニンテンドーアカウントサーバーとのOAuth 2.0/Open ID Connectの手続きパラメータ。@n
             *                          派生先実装で本コンテナへパラメータを入力してください。
             */
            virtual Result PrepareAuthorizationRequestSettings(::nn::nsd::NasServiceSetting* pOutService, AuthParameter* pOutAuth) NN_NOEXCEPT = 0;

        private:
            void InvalidateImpl() NN_NOEXCEPT;
            bool IsExpired() const NN_NOEXCEPT;

            mutable ::nn::os::SdkRecursiveMutex m_Lock;                 //!< 取得要求中ロック
            ::nn::util::optional<IAsync>        m_Async;                //!< 非同期要求ハンドル
            OwnerUserAccount                    m_Owner;                //!< キャッシュ済トークンオーナーユーザID
            ::nn::TimeSpan                      m_Expired;              //!< Uptime 失効期限
            const ::nn::TimeSpan                m_AvailabilityPeriod;   //!< キャッシュ済トークンの利用可能期間( Uptime単位 )
            Bit32                               m_CachedLength;         //!< 取得済トークン長( 終端文字を含めない )
            NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) char m_Token[LengthMax];
        };
    };

    /**
     * @brief   デバイス認証トークン管理
     */
    class DeviceAuth
    {
        NN_DISALLOW_COPY(DeviceAuth);
        NN_DISALLOW_MOVE(DeviceAuth);

    public:
        typedef Canceler<::nn::util::Cancelable> ICanceler;

        /**
         * @brief       トークン最大長 ( 1024 + 1 byte )。
         */
        static const size_t LengthMax = ::nn::dauth::RequiredBufferSizeForDeviceAuthenticationToken;

        /**
         * @biref       コンストラクタ。
         */
        DeviceAuth() NN_NOEXCEPT {}

        /**
         * @brief       トークン取得要求。
         *
         * @param[out]  pCanceler           外部からキャンセルする際のハンドルを割り当てるキャンセラコンテナへの参照。
         * @param[in]   clientId            デバイス認証トークン用クライアントID。
         * @param[out]  pOutToken           トークンをコピーする領域の先頭アドレスへのポインタ。@n
         *                                  最小容量要件として @ref LengthMax を満たしてください。
         * @param[in]   availableCapacity   @a pOutToken が示すメモリ領域の書き込み可能な容量。( バイトサイズ )@n
         *                                  トークンサイズ最大は @ref LengthMax。
         * @param[out]  pOutTokenEnd        トークンをコピーした領域の終端文字を示す末尾アドレスを格納する領域の先頭アドレスへのポインタ。@n
         *                                  nullptr 指定の場合、出力されません。
         *
         * @pre
         *              - pCanceler != nullptr
         *              - pOutToken != nullptr
         *              - sizeof(pOutToken[]) >= LengthMax
         *
         * @details     nim 側でキャッシュするため、同時要求は排他されます。
         */
        Result AcquireToken(ICanceler* pCanceler, const uint64_t clientId, char* pOutToken, const size_t availableCapacity, char** pOutTokenEnd = nullptr) NN_NOEXCEPT;
    };

    /**
     * @brief       キャンセラホルダー。
     */
    struct CancelerSet
    {
        NaId::ICanceler         na;
        NsaId::ICanceler        nsa;
        DeviceAuth::ICanceler   device;

        inline void Cancel() NN_NOEXCEPT
        {
            na.Cancel();
            nsa.Cancel();
            device.Cancel();
        }

        inline void Invalidate() NN_NOEXCEPT
        {
            na.Invalidate();
            nsa.Invalidate();
            device.Invalidate();
        }
    };

    /**
     * @brief   共有トークンストア。
     * @note    NaId は、接続先サーバーに応じて構成パラメータ及びキャッシュ保持可能期間が違うので共有化はしません。
     */
    struct SharedStore
    {
        NsaId        nsa;
        DeviceAuth   device;

        /**
         * @brief   共有シングルトンインスタンスの取得。
         * @details 共有管理可能なトークンストアインスタンスを提供します。
         */
        static SharedStore* GetSharedInstance() NN_NOEXCEPT;

    private:
        NN_IMPLICIT SharedStore() NN_NOEXCEPT {}

        NN_DISALLOW_COPY(SharedStore);
        NN_DISALLOW_MOVE(SharedStore);
    };

}   // ~TokenStore

}}}
