﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_ShopServiceManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include "nim_AsyncDeviceAccountImpl.h"
#include "nim_AsyncDeviceAuthenticationTokenImpl.h"
#include "nim_AsyncDeviceLinkImpl.h"
#include "nim_AsyncNotificationTokenImpl.h"
#include "nim_AsyncSyncTicketImpl.h"
#include "nim_AsyncDownloadTicketImpl.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

        template <typename IAsyncT, typename EmplacedRefT>
        Result RunAndGetAsyncObject(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<IAsyncT>> outAsync, EmplacedRefT emplacedRef) NN_NOEXCEPT
        {
            NN_RESULT_DO(emplacedRef.GetImpl().Run());
            *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
            *outAsync = emplacedRef;

            NN_RESULT_SUCCESS;
        }
    }

    Result ShopServiceManager::Initialize(DeviceContext* deviceContext, DeviceAccountStore* deviceAccountStore, VirtualAccountStore* virtualAccountStore) NN_NOEXCEPT
    {
        m_DeviceContext = deviceContext;
        m_DeviceAccountStore = deviceAccountStore;
        m_VirtualAccountStore = virtualAccountStore;
        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestDeviceAuthenticationToken(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceAuthenticationTokenImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(&m_CashedTokenInfo));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestCachedDeviceAuthenticationToken(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync, nn::TimeSpan remainingTime) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceAuthenticationTokenImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(&m_CashedTokenInfo));
        emplacedRef.GetImpl().EnableCachedToken(remainingTime);
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestRegisterDeviceAccount(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncRegisterDeviceAccountImpl>(m_DeviceAccountStore);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestUnregisterDeviceAccount(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncUnregisterDeviceAccountImpl>(m_DeviceAccountStore);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestDeviceAccountStatus(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceAccountStatusImpl>(m_DeviceAccountStore);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestDeviceRegistrationInfo(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceRegistrationInfoImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::GetDeviceAccountInfo(sf::Out<ec::system::DeviceAccountInfo> outValue) NN_NOEXCEPT
    {
        DeviceAccountInfo info;
        NN_RESULT_DO(m_DeviceAccountStore->Get(&info));

        ec::system::DeviceAccountInfo ecInfo = {};
        util::Strlcpy(ecInfo.id, info.id.data, static_cast<int>(sizeof(ecInfo.id)));

        *outValue = ecInfo;
        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestTransferDeviceAccount(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncTransferDeviceAccountImpl>(m_DeviceAccountStore);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::RequestSyncRegistration(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncSyncRegistrationImpl>(m_DeviceAccountStore);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result ShopServiceManager::IsOwnDeviceId(sf::Out<bool> outValue, uint64_t deviceId) NN_NOEXCEPT
    {
        uint64_t ownDeviceId = m_DeviceContext->GetDeviceId();
        *outValue = ownDeviceId == deviceId;

        NN_RESULT_SUCCESS;
    }

     Result ShopServiceManager::RequestRegisterNotificationToken(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const npns::NotificationToken& notificationToken) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncRegisterNotificationTokenImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(notificationToken, m_DeviceContext));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestUnlinkDevice(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const account::NintendoAccountId naUserId) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncUnlinkDeviceImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, naUserId));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestUnlinkDeviceIntegrated(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const account::Uid uid) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncUnlinkDeviceImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_VirtualAccountStore, uid));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestUnlinkDeviceAll(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncUnlinkDeviceAllImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_VirtualAccountStore));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestLinkDevice(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const account::Uid uid) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncLinkDeviceImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_VirtualAccountStore, uid));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::HasDeviceLink(sf::Out<bool> outValue, const account::Uid& uid) NN_NOEXCEPT
     {
         ec::system::VirtualAccountId vaId;
         NN_RESULT_TRY(m_VirtualAccountStore->Get(&vaId, uid))
             NN_RESULT_CATCH(ResultRegisteredVirtualAccountNotFound)
             {
                *outValue = false;
                NN_RESULT_SUCCESS;
             }
         NN_RESULT_END_TRY

         *outValue = true;
         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestCreateVirtualAccount(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const account::Uid uid) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncCreateVirtualAccountImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, uid));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestDeviceLinkStatus(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync, const account::Uid& uid) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceLinkStatusImpl>();
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, uid));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestDownloadTicket(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncDownloadTicketImpl>(m_DeviceAccountStore);
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, rightsId));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestDownloadTicketForPrepurchasedContents(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::nim::detail::IAsyncValue>> outAsync, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDownloadTicketForPrepurchasedContentsImpl>(m_DeviceAccountStore);
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, rightsId));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

     Result ShopServiceManager::RequestSyncTicket(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncProgressResult>> outAsync) NN_NOEXCEPT
     {
         auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncProgressResult, AsyncSyncTicketImpl>(m_DeviceAccountStore);
         NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
         NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
         NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

         NN_RESULT_SUCCESS;
     }

}}}
