﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <type_traits>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/account/account_Result.h>

#include "nim_ShopServiceAccessServiceServer.h"
#include "nim_HttpJsonUtil.h"

namespace nn { namespace nim { namespace srv {

namespace {

// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DEBUG_TRACE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::ServiceServer] " __VA_ARGS__ )
#define DEBUG_TRACE_AS(...) DebugTraceAs( __VA_ARGS__ )
void DebugTraceAs(const char* pPrefix, const char* pResponse, size_t responseSize) NN_NOEXCEPT
{
    if (nullptr != pResponse && 0 < responseSize)
    {
        auto pRear = const_cast<char*>(&pResponse[responseSize - 1]);
        const auto rearCode = *pRear;
        *pRear = '\0';
        DEBUG_TRACE("%s => %s%c\n", pPrefix, pResponse, rearCode);
        *pRear = rearCode;
    }
}
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#define DEBUG_TRACE_AS(...) static_cast<void>(0)
#endif

/**
 * @brief       各種認証定数。
 */
struct AuthenticationRequirement
{
    /**
     * @brief   デバイス認証トークン用クライアントID
     */
    static const uint64_t DeviceTokenClientId = 0x93af0acb26258de9ull;

    /**
     * @brief   NSA-ID 認証トークン用システムプログラムID
     */
    static const ::nn::account::SystemProgramIdentification ProgramIdentifierForNsaId;

    /**
     * @brief   NSA-ID 認証トークンリクエストヘッダキー
     */
    static const char HeaderPrefixForNsaIdToken[];

    /**
     * @brief   デバイス認証トークンリクエストヘッダキー
     */
    static const char HeaderPrefixForDeviceToken[];

    /**
     * @brief   NintendoAccount認証トークンリクエストヘッダキー
     */
    static const char HeaderPrefixForNaIdToken[];

    /**
     * @brief   アプリケーション詐称検証用呼び出し元アプリケーションID付与リクエストヘッダキー
     */
    static const char HeaderPrefixForApplicationIdVerification[];
};

//-----------------------------------------------------------------------------
const char AuthenticationRequirement::HeaderPrefixForDeviceToken[] = "X-DeviceAuthorization: Bearer ";
//-----------------------------------------------------------------------------
const char AuthenticationRequirement::HeaderPrefixForNsaIdToken[] = "X-NINTENDO-NSA-ID-TOKEN: Bearer ";
//-----------------------------------------------------------------------------
const char AuthenticationRequirement::HeaderPrefixForNaIdToken[] = "Authorization: Bearer ";
//-----------------------------------------------------------------------------
const char AuthenticationRequirement::HeaderPrefixForApplicationIdVerification[] = "X-Nintendo-Application-Id: ";
//-----------------------------------------------------------------------------
const ::nn::account::SystemProgramIdentification AuthenticationRequirement::ProgramIdentifierForNsaId = {::nn::ApplicationId{0x010000000000101E}, 0};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// リクエストヘッダ生成作業領域確保用定数値保全チェック
NN_STATIC_ASSERT(ShopServiceAccessServiceServer::Requirement::RequestHeaderPrefixLengthMaxForToken >= (sizeof(AuthenticationRequirement::HeaderPrefixForApplicationIdVerification) - 1));
NN_STATIC_ASSERT(ShopServiceAccessServiceServer::Requirement::RequestHeaderPrefixLengthMaxForToken >= (sizeof(AuthenticationRequirement::HeaderPrefixForDeviceToken) - 1));
NN_STATIC_ASSERT(ShopServiceAccessServiceServer::Requirement::RequestHeaderPrefixLengthMaxForToken >= (sizeof(AuthenticationRequirement::HeaderPrefixForNsaIdToken) - 1));
NN_STATIC_ASSERT(ShopServiceAccessServiceServer::Requirement::RequestHeaderPrefixLengthMaxForToken >= (sizeof(AuthenticationRequirement::HeaderPrefixForNaIdToken) - 1));
//-----------------------------------------------------------------------------

/**
 * @brief       エラーコードレスポンスボディを持つ HTTPステータスコード Result の判定。
 */
bool HasErrorCodeResponse(const Result result) NN_NOEXCEPT
{
    return ResultHttpStatus::Includes(result) && (
           ResultHttpStatus400BadRequest::Includes(result)
        || ResultHttpStatus401Unauthorized::Includes(result)
        || ResultHttpStatus403Forbidden::Includes(result)
        || ResultHttpStatus404NotFound::Includes(result)
        || ResultHttpStatus405MethodNotAllowed::Includes(result)
        || ResultHttpStatus406NotAcceptable::Includes(result)
        || ResultHttpStatus407ProxyAuthenticationRequired::Includes(result)
        || ResultHttpStatus408RequestTimeout::Includes(result)
        || ResultHttpStatus409Conflict::Includes(result)
        || ResultHttpStatus410Gone::Includes(result)
        || ResultHttpStatus411LengthRequired::Includes(result)
        || ResultHttpStatus412PreconditionFailed::Includes(result)
        || ResultHttpStatus413PayloadTooLarge::Includes(result)
        || ResultHttpStatus414UriTooLong::Includes(result)
        || ResultHttpStatus415UnsupportedMediaType::Includes(result)
        || ResultHttpStatus416RangeNotSatisfiable::Includes(result)
        || ResultHttpStatus417ExpectationFailed::Includes(result)
        || ResultHttpStatus500InternalServerError::Includes(result)
        || ResultHttpStatus501NotImplemented::Includes(result)
        || ResultHttpStatus502BadGateway::Includes(result)
        || ResultHttpStatus503ServiceUnavailable::Includes(result)
        || ResultHttpStatus504GatewayTimeout::Includes(result)
        || ResultHttpStatus505HttpVersionNotSupported::Includes(result)
        || ResultHttpStatus509BandwidthLimitExceeded::Includes(result)
    );
}

}   // ~unnamed

//!============================================================================
//! @name ShopServiceAccessServiceServer::Requirement 実装

//-----------------------------------------------------------------------------
const ::nn::err::ErrorCode ShopServiceAccessServiceServer::Requirement::ErrorCodeNone = ::nn::err::ErrorCode::GetInvalidErrorCode();



//!============================================================================
//! @name ShopServiceAccessServiceServer::Accessor 実装

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::Accessor::CreateHeaderForNsaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStoreNsa::ICanceler* pCanceler) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCanceler);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    const size_t lengthOfNsaIdTokenHeader = sizeof(AuthenticationRequirement::HeaderPrefixForNsaIdToken) - 1;
    NN_RESULT_THROW_UNLESS(temporarySize >= static_cast<size_t>(lengthOfNsaIdTokenHeader + TokenStoreNsa::LengthMax), ResultBufferNotEnough());

    std::memcpy(pTemporary, AuthenticationRequirement::HeaderPrefixForNsaIdToken, lengthOfNsaIdTokenHeader);
    const size_t avilableSize = temporarySize - lengthOfNsaIdTokenHeader;
    NN_RESULT_DO(ShopServiceAccessServerImpl::AcquireTokenForNsaId(&pTemporary[lengthOfNsaIdTokenHeader], avilableSize, uid, AuthenticationRequirement::ProgramIdentifierForNsaId, pCanceler));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::Accessor::CreateHeaderForNaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStoreNa::ICanceler* pCanceler) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCanceler);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    const size_t lengthOfNaIdTokenHeader = sizeof(AuthenticationRequirement::HeaderPrefixForNaIdToken) - 1;
    NN_RESULT_THROW_UNLESS(temporarySize >= static_cast<size_t>(lengthOfNaIdTokenHeader + TokenStoreNa::LengthMax), ResultBufferNotEnough());

    std::memcpy(pTemporary, AuthenticationRequirement::HeaderPrefixForNaIdToken, lengthOfNaIdTokenHeader);
    const size_t avilableSize = temporarySize - lengthOfNaIdTokenHeader;
    NN_RESULT_DO(ShopServiceAccessServerImpl::AcquireTokenForNaId(&pTemporary[lengthOfNaIdTokenHeader], avilableSize, uid, pCanceler));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::Accessor::CreateHeaderForDeviceAuthenticationToken(char* pTemporary, size_t temporarySize, TokenStoreDevice::ICanceler* pCanceler) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCanceler);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    const size_t lengthOfDeviceTokenHeader = sizeof(AuthenticationRequirement::HeaderPrefixForDeviceToken) - 1;
    NN_RESULT_THROW_UNLESS(temporarySize >= (lengthOfDeviceTokenHeader + TokenStoreDevice::LengthMax), ResultBufferNotEnough());

    // デバイス認証トークンの取得
    std::memcpy(pTemporary, AuthenticationRequirement::HeaderPrefixForDeviceToken, lengthOfDeviceTokenHeader);
    const size_t avilableSize = temporarySize - lengthOfDeviceTokenHeader;
    NN_RESULT_DO(ShopServiceAccessServerImpl::AcquireTokenForDeviceAuthentication(pCanceler, AuthenticationRequirement::DeviceTokenClientId, &pTemporary[lengthOfDeviceTokenHeader], avilableSize));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::Accessor::CreateHeaderForApplicationIdVerification(char* pTemporary, size_t temporarySize, const ::nn::ncm::ApplicationId& applicationId) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    const size_t lengthOfHeader = sizeof(AuthenticationRequirement::HeaderPrefixForApplicationIdVerification) - 1;
    NN_RESULT_THROW_UNLESS(temporarySize >= (lengthOfHeader + ShopServiceAccess::Requirement::LengthMaxForStringApplicationId + 1), ResultBufferNotEnough());

    std::memcpy(pTemporary, AuthenticationRequirement::HeaderPrefixForApplicationIdVerification, lengthOfHeader);
    const auto pEnd = ShopServiceAccess::ToHexStringFromApplicationId(&pTemporary[lengthOfHeader], &pTemporary[temporarySize - 1], applicationId);
    *pEnd = '\0';
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::Accessor::QueryRestrictUserAvailability(const ::nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_UNUSED(uid);
    NN_RESULT_SUCCESS;
}


//!============================================================================
//! @name ShopServiceAccessServiceServer::AccessorForSugar 実装

//-----------------------------------------------------------------------------
const char ShopServiceAccessServiceServer::AccessorForSugar::ServiceDiscoveryBaseUrl[] = "https://pearljam.hac.%.eshop.nintendo.net/sugar";

//-----------------------------------------------------------------------------
ShopServiceAccessServiceServer::AccessorForSugar::AccessorForSugar() NN_NOEXCEPT
    : Accessor(ServiceDiscoveryBaseUrl, sizeof(ServiceDiscoveryBaseUrl) - 1)
    , m_Restriction(CountOfRestrictInOneMinutes, ::nn::TimeSpan::FromMinutes(1))
{
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForSugar::RegisterRequestHeaders(RequestHeaderGenerationHandle* const pHandle
    , const ::nn::account::Uid& uid, CancelerSet* pCanceler) const NN_NOEXCEPT
{
    // 基本コード実装時でインスタンス確定されるはず( ASSERT で検知可能 )ですが念のため.
    NN_ABORT_UNLESS_NOT_NULL(pHandle);
    NN_ABORT_UNLESS_NOT_NULL(pCanceler);

    const auto pTemporary = pHandle->pTemporary;
    const auto pConnection = pHandle->pConnection;
    const auto pTransaction = pHandle->pTransaction;
    const auto temporarySize = pHandle->temporarySize;
    NN_RESULT_DO(CreateHeaderForDeviceAuthenticationToken(pTemporary, temporarySize, &pCanceler->device));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_DO(CreateHeaderForNsaIdToken(pTemporary, temporarySize, uid, &pCanceler->nsa));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_DO(CreateHeaderForApplicationIdVerification(pTemporary, temporarySize, pHandle->applicationId));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForSugar::ResolveServerErrorResult(::nn::err::ErrorCode* pOutErrorCode, const Result& nowResult, const char* pReceivedResponse, size_t responseSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutErrorCode);    // 基本コード実装時でインスタンス確定されるはず( ASSERT で検知可能 )ですが念のため.

    *pOutErrorCode = Requirement::ErrorCodeNone;
    if (nullptr != pReceivedResponse && responseSize > 0 && HasErrorCodeResponse(nowResult))
    {
        DEBUG_TRACE_AS("Detected an error response by sugar", pReceivedResponse, responseSize);
        HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(pReceivedResponse, responseSize);
        JsonParser::ErrorReponseAdaptor<JsonPathType> adaptor("$.error.code");
        NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(nullptr)));
        pOutErrorCode->category = static_cast<::nn::err::ErrorCodeCategory>(ErrorCodeModuleId);
        pOutErrorCode->number = static_cast<::nn::err::ErrorCodeNumber>(adaptor.GetErrorCode());
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForSugar::AcquireRestrictPermission(RestrictionCanceler* pCanceler) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Restriction.AcquirePermission(pCanceler), ResultShopServiceAccessCanceled());
    NN_RESULT_SUCCESS;
}



//!============================================================================
//! @name ShopServiceAccessServiceServer::AccessorForCivil 実装

//-----------------------------------------------------------------------------
const char ShopServiceAccessServiceServer::AccessorForCivil::ServiceDiscoveryBaseUrl[] = "https://pearljam.hac.%.eshop.nintendo.net/civil";

//-----------------------------------------------------------------------------
ShopServiceAccessServiceServer::AccessorForCivil::AccessorForCivil() NN_NOEXCEPT
    : Accessor(ServiceDiscoveryBaseUrl, sizeof(ServiceDiscoveryBaseUrl) - 1)
    , m_Restriction(CountOfRestrictInOneMinutes, ::nn::TimeSpan::FromMinutes(1))
{
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForCivil::RegisterRequestHeaders(RequestHeaderGenerationHandle* pHandle
    , const ::nn::account::Uid& uid, CancelerSet* pCanceler) const NN_NOEXCEPT
{
    // 基本コード実装時でインスタンス確定されるはず( ASSERT で検知可能 )ですが念のため.
    NN_ABORT_UNLESS_NOT_NULL(pHandle);
    NN_ABORT_UNLESS_NOT_NULL(pCanceler);

    const auto pTemporary = pHandle->pTemporary;
    const auto pConnection = pHandle->pConnection;
    const auto pTransaction = pHandle->pTransaction;
    const auto temporarySize = pHandle->temporarySize;
    NN_RESULT_DO(CreateHeaderForDeviceAuthenticationToken(pTemporary, temporarySize, &pCanceler->device));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_DO(CreateHeaderForNaIdToken(pTemporary, temporarySize, uid, &pCanceler->na));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_DO(CreateHeaderForApplicationIdVerification(pTemporary, temporarySize, pHandle->applicationId));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTransaction, pTemporary));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForCivil::ResolveServerErrorResult(::nn::err::ErrorCode* pOutErrorCode, const Result& nowResult, const char* pReceivedResponse, size_t responseSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutErrorCode);    // 基本コード実装時でインスタンス確定されるはず( ASSERT で検知可能 )ですが念のため.

    *pOutErrorCode = Requirement::ErrorCodeNone;
    if (nullptr != pReceivedResponse && responseSize > 0 && HasErrorCodeResponse(nowResult))
    {
        DEBUG_TRACE_AS("Detected an error response by civil", pReceivedResponse, responseSize);
        HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(pReceivedResponse, responseSize);
        JsonParser::ErrorReponseAdaptor<JsonPathType> adaptor("$.error.code");
        NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(nullptr)));
        pOutErrorCode->category = static_cast<::nn::err::ErrorCodeCategory>(ErrorCodeModuleId);
        pOutErrorCode->number = static_cast<::nn::err::ErrorCodeNumber>(adaptor.GetErrorCode());
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForCivil::AcquireRestrictPermission(RestrictionCanceler* pCanceler) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Restriction.AcquirePermission(pCanceler), ResultShopServiceAccessCanceled());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServiceServer::AccessorForCivil::QueryRestrictUserAvailability(const ::nn::account::Uid& uid) NN_NOEXCEPT
{
    return ShopServiceAccess::VerifyQuebecAgeRestriction(uid);
}



//!============================================================================
//! @name ShopServiceAccessServiceServer 実装

//-----------------------------------------------------------------------------
ShopServiceAccessServiceServer::Accessor* ShopServiceAccessServiceServer::GetPresetAccessor(const ShopServiceAccessTypes::Server server) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(AccessorForSugar, s_SugarServerAccessor);
    NN_FUNCTION_LOCAL_STATIC(AccessorForCivil, s_CivilServerAccessor);
    Accessor* const pAccessors[] =
    {
        &s_SugarServerAccessor,
        &s_CivilServerAccessor,
    };
    const auto index = static_cast<size_t>(server);
    if (index >= std::extent<decltype(pAccessors)>::value)
    {
        return nullptr;
    }
    return pAccessors[index];
}

}}}
