﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <nn/os/os_Thread.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Result.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/account/account_Result.h>
#include <nn/util/util_ScopeExit.h>

#include "nim_ShopServiceAccessDebug.h"
#include "nim_ShopServiceAccessServerImpl.h"
#include <nn/result/result_HandlingUtility.h>

#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DEBUG_TRACE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::Server] " __VA_ARGS__ )
#define DEBUG_TRACE_ACCESSOR(...) NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::Accessor] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#define DEBUG_TRACE_ACCESSOR(...)    static_cast<void>(0)
#endif

namespace nn { namespace nim { namespace srv {

namespace {

    /**
     * @brief   コンフィグエイリアス
     */
    typedef ShopServiceAccessServer::Configuration Configuration;

    /**
     * @brief   指定プロセスIDのアプリケーションIDの取得。
     */
    Result QueryLaunchedApplicationId(::nn::ncm::ApplicationId* pOutValue, const ::nn::os::ProcessId& clientProcessId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        arp::ApplicationLaunchProperty property;
        NN_RESULT_DO(::nn::arp::GetApplicationLaunchProperty(&property, clientProcessId));

        *pOutValue = property.id;
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   プリセットスレッドスタックアロケータ。
     *
     * @tparam  EnsureStackCapacity １つのモジュール用スタック容量( バイトサイズ )。
     */
    template<size_t EnsureStackCapacity>
    class PresetStackAllocator
    {
        struct NN_OS_ALIGNAS_THREAD_STACK Storage
        {
            char pStack[EnsureStackCapacity];
        };

    public:
        typedef ::nn::util::TypedStorage<Storage, sizeof(Storage), NN_ALIGNOF(Storage)> StorageType;

        explicit PresetStackAllocator(StorageType* pStorages, int presetCount) NN_NOEXCEPT
            : m_pStorages(pStorages)
            , m_BitStorage()
            , m_BitArray(&m_BitStorage, sizeof(m_BitStorage), presetCount)
        {
            NN_SDK_ASSERT_NOT_NULL(pStorages);
            NN_SDK_ASSERT(presetCount > 0);
            m_BitArray.reset();
        }

        /**
         * @brief   獲得要求。
         */
        char* Acquire() NN_NOEXCEPT
        {
            for (int i = 0; i < m_BitArray.size(); ++i)
            {
                if (!m_BitArray[i])
                {
                    m_BitArray[i] = true;
                    auto& r = ::nn::util::Get(m_pStorages[i]);
                    return r.pStack;
                }
            }
            return nullptr;
        }

        /**
         * @brief   解放要求。
         */
        void Release(const char* const pStack) NN_NOEXCEPT
        {
            for (int i = 0; i < m_BitArray.size(); ++i)
            {
                auto& r = ::nn::util::Get(m_pStorages[i]);
                if (pStack == r.pStack)
                {
                    m_BitArray[i] = false;
                    return;
                }
            }
            NN_ABORT("must not come here");
        }

    private:
        StorageType* const  m_pStorages;
        Bit64               m_BitStorage;
        util::BitArray      m_BitArray;
    };

    /**
     * @brief   リクエストスケジューラスタックアロケータ
     */
    typedef PresetStackAllocator<ShopServiceAccessServerImpl::EnsureExpectStackSize> SchedulerStackHeap;

    /**
     * @brief   スレッドスタック実体。
     * @details アライメントが os::PageSize になるのでクラス外単独定義。
     */
    NN_OS_ALIGNAS_THREAD_STACK SchedulerStackHeap::StorageType g_PresetStack[Configuration::AvailableServiceClientMax];

    /**
     * @brief   サービスオブジェクトヒープ定義。
     *
     * @details 以下のサービスオブジェクトのアロケートヒープを定義します。
     *
     *          @li IShopServiceAccessServer>ShopServiceAccessServerImpl>
     *          @li IShopServiceAccessor<ShopServiceAccessor>
     *          @li IShopServiceAsync<ShopServiceAccessAsyncImpl>
     *
     *          以下のサービスオブジェクト同時利用を想定し、ExpHeap管理ブロック + 余剰分を含めて概算確保しています。@n
     *          IServiceObject のサイズは定義へのアクセス権限の都合上 sizeof(IServiceObject) で算出できないので、@n
     *          @ref Configuration::IServiceObjectSize で余裕を持たせています。@n
     *
     * @note   サービスオブジェクトサイズ算出は以下を利用。
     *          @code
     *          typedef nim::detail::IShopServiceAsync Interface;
     *          typedef nim::srv::ShopServiceAccessAsyncImpl Impl;
     *          typedef sf::impl::detail::ImplTemplateBase<Interface, Interface, sf::detail::EmplacedImplHolder<Impl>, sf::detail::EmplacedImplHolder<Impl>> Base;
     *          sizeof(sf::detail::ObjectImplFactoryWithStatefulAllocator<Base, sf::ExpHeapAllocator>::Object);
     *          @codeend
     */
    struct SharedServiceHeap
    {
    private:
        static const size_t AvailableAccessorMax = Configuration::AvailableAccessorMax * Configuration::AvailableServiceClientMax;
        static const size_t AvailableAsyncAccessMax = Configuration::AvailableAsyncAccessMax * Configuration::AvailableServiceClientMax;

    public:
        static const size_t CapacityRequirement = Configuration::ExpHeapManagedBlockSize +
            ((PRIVATE_ALIGNED_SIZEOF(ShopServiceAccessorImpl) + Configuration::IServiceObjectSize) * AvailableAccessorMax) +
            ((PRIVATE_ALIGNED_SIZEOF(ShopServiceAccessAsyncImpl) + Configuration::IServiceObjectSize) * AvailableAsyncAccessMax) +
            ((PRIVATE_ALIGNED_SIZEOF(ShopServiceAccessServerImpl) + Configuration::IServiceObjectSize) * Configuration::AvailableServiceClientMax);

        typedef HeapUtil::ServiceFactoryHeap<CapacityRequirement>   Storage;    // サービスオブジェクトインスタンスメモリストレージ
        typedef Storage::AllocatorType                              Allocator;  // サービスオブジェクトインスタンスメモリアロケータ
        typedef Storage::Factory                                    Factory;    // サービスオブジェクトファクトリ
    };

    /**
     * @brief  NaId トークンキャッシュ
     */
    class NaIdTokenImpl : public ::nn::nim::srv::TokenStore::NaId::SingleCacheBase
    {
        NN_DISALLOW_COPY(NaIdTokenImpl);
        NN_DISALLOW_MOVE(NaIdTokenImpl);

    public:
        typedef ::nn::nim::srv::TokenStore::NaId::SingleCacheBase BaseType;

        explicit NaIdTokenImpl() NN_NOEXCEPT : BaseType() {}

    protected:
        virtual Result PrepareAuthorizationRequestSettings(::nn::nsd::NasServiceSetting* pOutService, TokenStore::NaId::AuthParameter* pOutAuth) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_ASSERT_NOT_NULL(pOutAuth);
            NN_SDK_ASSERT_NOT_NULL(pOutService);
            NN_RESULT_DO(::nn::nsd::GetNasServiceSetting(pOutService, ::nn::nsd::NasServiceNameOfNxShop));

            NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->scope, "nxeshop", sizeof(pOutAuth->scope)) < sizeof(pOutAuth->scope));
            NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->state, " ", sizeof(pOutAuth->state)) < sizeof(pOutAuth->state));
            NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->nonce, " ", sizeof(pOutAuth->nonce)) < sizeof(pOutAuth->nonce));
            NN_RESULT_SUCCESS;
        }
    };

    /**
     * @brief   共有トークンストア。
     */
    struct SharedTokenStore
    {
        NaIdTokenImpl   na;
    };

}   // ~::unnamed

/**
 * @brief   サービスポートインタフェース用実装。
 * @details このクラスインスタンスは @ref sf::UnmanagedServiceObject として静的領域に確保されます。
 *          システムプロセスでの運用は事実上、デストラクタ及び Finalizer が呼ばれません。
 *          本クラスにプロキシクライアントの生存に左右されるリソースは配置及び適用しないでください。
 */
class ShopServiceAccessServerInterfaceImpl
{
    NN_DISALLOW_COPY(ShopServiceAccessServerInterfaceImpl);
    NN_DISALLOW_MOVE(ShopServiceAccessServerInterfaceImpl);

public:
    /**
     * @brief   シングルトンインスタンス取得。
     */
    static ShopServiceAccessServerInterfaceImpl& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ShopServiceAccessServerInterfaceImpl, s_Impl, (g_PresetStack, NN_ARRAY_SIZE(g_PresetStack)));
        return s_Impl;
    }

    /**
     * @brief   サービスオブジェクトインスタンスヒープオブジェクトの取得。
     */
    NN_FORCEINLINE SharedServiceHeap::Allocator* GetServiceObjectAllocator() NN_NOEXCEPT
    {
        return m_ServiceHeap.GetAllocator();
    }

    /**
     * @brief   nim サービス共通資源デバイスコンテキストオブジェクトの取得。
     */
    NN_FORCEINLINE DeviceContext* GetDeviceContext() NN_NOEXCEPT
    {
        return m_pDeviceContext;
    }

    /**
     * @brief   共有トークンストアオブジェクトの取得。
     */
    NN_FORCEINLINE SharedTokenStore* GetSharedTokenStore() NN_NOEXCEPT
    {
        return &m_SharedTokenStore;
    }

    /**
     * @brief   スケジューラスタックの獲得要求。
     */
    char* AcquireStack() NN_NOEXCEPT
    {
        return m_SchedulerStackHeap.Acquire();
    }

    /**
     * @brief   獲得したスタックの返却。
     */
    void ReleaseStack(char* pStack) NN_NOEXCEPT
    {
        m_SchedulerStackHeap.Release(pStack);
    }

    /**
     * @brief   初期化。
     */
    Result Initialize(DeviceContext* pContext) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pContext, ResultNotSupported());
        m_pDeviceContext = pContext;
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief   終了。
    */
    NN_FORCEINLINE void Finalize() NN_NOEXCEPT {}

private:
    explicit ShopServiceAccessServerInterfaceImpl(SchedulerStackHeap::StorageType* pStorages, int presetCount) NN_NOEXCEPT
        : m_pDeviceContext(nullptr)
        , m_SchedulerStackHeap(pStorages, presetCount)
    {
    }

    ~ShopServiceAccessServerInterfaceImpl() NN_NOEXCEPT
    {
        Finalize();
    }

    DeviceContext*              m_pDeviceContext;
    SharedServiceHeap::Storage  m_ServiceHeap;
    SharedTokenStore            m_SharedTokenStore;
    SchedulerStackHeap          m_SchedulerStackHeap;
};

///============================================================================
/// サービスポートインタフェース

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::Initialize(DeviceContext* pContext) NN_NOEXCEPT
{
    return ShopServiceAccessServerInterfaceImpl::GetInstance().Initialize(pContext);
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::Finalize() NN_NOEXCEPT
{
    ShopServiceAccessServerInterfaceImpl::GetInstance().Finalize();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::CreateServerInterface(::nn::sf::Out<::nn::sf::SharedPointer<::nn::nim::detail::IShopServiceAccessServer>> outValue
    , ::nn::Bit64 clientProcessId
    , ::nn::sf::NativeHandle inTransferHandle
    , uint64_t inTransferTotalSize) NN_NOEXCEPT
{
    auto& root = ShopServiceAccessServerInterfaceImpl::GetInstance();
    auto pStack = root.AcquireStack();
    NN_RESULT_THROW_UNLESS(pStack, ResultAllocationMemoryFailed());

    auto pAllocator = root.GetServiceObjectAllocator();
    auto emplacedRef = SharedServiceHeap::Factory::CreateSharedEmplaced<::nn::nim::detail::IShopServiceAccessServer, ShopServiceAccessServerImpl>(pAllocator, pStack);
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultAllocationMemoryFailed());

    NN_RESULT_DO(emplacedRef.GetImpl().Initialize(::nn::os::ProcessId{clientProcessId}, inTransferHandle, inTransferTotalSize));
    inTransferHandle.Detach();

    // NA-ID-TOKEN オンメモリキャッシュ無効化(破棄)。
    //  ec::InitializeShopServiceAccessors() の呼び出し時にオンメモリキャッシュを破棄します。
    // NOTE:
    //  現在、TokenStore::NaId のインスタンスはプロセス内に一つの共有インスタンスとして管理されます。
    //  ShopServiceAccessor をマルチクライアント動作へ対応させる際に、クライアント別に NA-ID-TOKEN のキャッシュ管理を行うか再検討する必要があります。
    //  ※ここで共有インスタンスのキャッシュ破棄を行うと、新しいクライアントからのポート初期化要求が来た時点で破棄が発生します。
    //  ※クライアント別対応する場合、NA-ID-TOKEN のキャッシュ管理に必要な構造体サイズ( sizeof(ShopServiceAccess::TokenStore::NaId) )がクライアント分必要になります。
    //  ※Invalidate() と取得処理自体は排他されているため、呼び出し競合による不正は発生しません。
    root.GetSharedTokenStore()->na.Invalidate();

    *outValue = emplacedRef;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::RefreshDebugAvailability() NN_NOEXCEPT
{
    ShopServiceAccessDebug::RefreshAvailability();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::ClearDebugResponse() NN_NOEXCEPT
{
    return ShopServiceAccessDebug::Response::Clear();
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServer::Interface::RegisterDebugResponse(const ::nn::nim::ShopServiceAccessTypes::Server& inKeyTarget, const ::nn::sf::InArray<char>& inKeyPath, const ::nn::sf::InArray<char>& inValueResponse, const uint32_t inExpectResult, const uint32_t inHappenedRate) NN_NOEXCEPT
{
    return ShopServiceAccessDebug::Response::Register(inKeyTarget, inKeyPath, inValueResponse, inExpectResult, inHappenedRate);
}



///============================================================================
/// アクセッササーバーインタフェース

//-----------------------------------------------------------------------------
ShopServiceAccessServerImpl::ShopServiceAccessServerImpl(char* pStackTop) NN_NOEXCEPT
    : m_Scheduler(pStackTop)
{
    DEBUG_TRACE("Construction: done( SchedulerStack[%p] ).\n", pStackTop);
}

//-----------------------------------------------------------------------------
ShopServiceAccessServerImpl::~ShopServiceAccessServerImpl() NN_NOEXCEPT
{
    DEBUG_TRACE("Destruction: called, Thread[ %llu ].\n", ::nn::os::GetThreadId(::nn::os::GetCurrentThread()));

    m_Scheduler.Finalize();         // スケジューラ終了
    m_TransferStorage.Finalize();   // トランスファーメモリ利用解除

    auto pStack = m_Scheduler.GetStackTop();
    ShopServiceAccessServerInterfaceImpl::GetInstance().ReleaseStack(pStack);

    DEBUG_TRACE("Destruction: done( SchedulerStack[%p] ).\n", pStack);
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::Initialize(const ::nn::os::ProcessId& clientProcessId, const ::nn::sf::NativeHandle& inTransferHandle, uint64_t inTransferTotalSize) NN_NOEXCEPT
{
    // 起動中アプリケーションID取得
    NN_RESULT_DO(QueryLaunchedApplicationId(&m_OwnerApplicationId, clientProcessId));

    // スケジューラ起動
    const RequestExecutorPool::ThreadAttributes threadAttributes =
    {
        NN_SYSTEM_THREAD_NAME(nim, ShopServiceAccessTask),
        NN_SYSTEM_THREAD_PRIORITY(nim, ShopServiceAccessTask)
    };
    NN_RESULT_DO(m_Scheduler.Initialize(threadAttributes, threadAttributes));

    // トランスファーメモリ利用開始
    NN_RESULT_DO(m_TransferStorage.Initialize(inTransferHandle, inTransferTotalSize));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::CreateAccessorInterface(::nn::sf::Out<::nn::sf::SharedPointer<::nn::nim::detail::IShopServiceAccessor>> outValue, const ::nn::nim::ShopServiceAccessTypes::Server& inServer) NN_NOEXCEPT
{
    auto pAllocator = ShopServiceAccessServerInterfaceImpl::GetInstance().GetServiceObjectAllocator();
    auto emplacedRef = SharedServiceHeap::Factory::CreateSharedEmplaced<::nn::nim::detail::IShopServiceAccessor, ShopServiceAccessorImpl>(pAllocator, this, inServer);
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());

    *outValue = emplacedRef;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::Prepare(ShopServiceAccessAsyncImpl* pRequest, const ::nn::nim::ShopServiceAccessTypes::FixedParams& inFixedParams) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    pRequest->SetParameter(m_TransferStorage.GetAllocator(), inFixedParams);
    NN_RESULT_SUCCESS;
}
//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::Request(ShopServiceAccessAsyncImpl* pRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    // スケジューラへ登録。
    NN_RESULT_DO(m_Scheduler.Request(pRequest));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessServerImpl::Finish() NN_NOEXCEPT
{
    m_Scheduler.Finish();
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::AcquireTokenForDeviceAuthentication(TokenStore::DeviceAuth::ICanceler* pCanceler, const uint64_t clientId, char* pOutToken, const size_t availableCapacity, char** pOutTokenEnd) NN_NOEXCEPT
{
    NN_RESULT_TRY(TokenStore::SharedStore::GetSharedInstance()->device.AcquireToken(pCanceler, clientId, pOutToken, availableCapacity, pOutTokenEnd))
    NN_RESULT_CATCH_CONVERT(::nn::dauth::ResultCancelled, ResultShopServiceAccessCanceled())
    NN_RESULT_CATCH(::nn::dauth::detail::ResultProgramError)
    {
        NN_ABORT("[ShopServiceAccess::Server] Unexpected failure as `dauth::detail::ResultProgramError` at the `AcquireDeviceAuthenticationToken`.\n");
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::AcquireTokenForNaId(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, TokenStore::NaId::ICanceler* pCanceler) NN_NOEXCEPT
{
    NN_RESULT_TRY(ShopServiceAccessServerInterfaceImpl::GetInstance().GetSharedTokenStore()->na.AcquireToken(pOutToken, availableCapacity, uid, pCanceler))
    NN_RESULT_CATCH_CONVERT(::nn::account::ResultCancelled, ResultShopServiceAccessCanceled())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessServerImpl::AcquireTokenForNsaId(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, const ::nn::account::SystemProgramIdentification& info, TokenStore::NsaId::ICanceler* pCanceler) NN_NOEXCEPT
{
    NN_RESULT_TRY(TokenStore::SharedStore::GetSharedInstance()->nsa.AcquireToken(pOutToken, availableCapacity, uid, info, pCanceler))
    NN_RESULT_CATCH_CONVERT(::nn::account::ResultCancelled, ResultShopServiceAccessCanceled())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}



///============================================================================
/// アクセッサインタフェース

//-----------------------------------------------------------------------------
ShopServiceAccessorImpl::ShopServiceAccessorImpl(ShopServiceAccessServerImpl* pServer, const ::nn::nim::ShopServiceAccessTypes::Server& inServer) NN_NOEXCEPT
    : m_pServer(pServer, true), m_ConnectMapLock(), m_ConnectMapArray(&m_ConnectMapValue, sizeof(m_ConnectMapValue), Configuration::ParallelExecutionCount), m_TargetServer(inServer)
{
    NN_SDK_ASSERT_NOT_NULL(pServer);
    NN_SDK_ASSERT(Configuration::ParallelExecutionCount <= NN_BITSIZEOF(decltype(m_ConnectMapValue)));
    m_ConnectMapArray.reset();
}

//-----------------------------------------------------------------------------
ShopServiceAccessorImpl::~ShopServiceAccessorImpl() NN_NOEXCEPT
{
    DEBUG_TRACE_ACCESSOR("Destruction: called, Thread[ %llu ].\n", ::nn::os::GetThreadId(::nn::os::GetCurrentThread()));

    // 強制中断要求
    //  現状はサービスフレームワークの親子階層, 寿命管理により、この時点で全ての要求は完了している。
    //  - 本デストラクタよりも先に ShopServiceAccessAsyncImpl デストラクションが全て呼ばれる。
    //  - ShopServiceAccessAsyncImpl デストラクション内で自身への Cancel and Wait を行っている。
    m_pServer->Finish();

    // HttpConnection リソース解放
    NN_ABORT_UNLESS(FinalizeConnection());

    DEBUG_TRACE_ACCESSOR("Destruction: done.\n");
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessorImpl::CreateAsyncInterface(::nn::sf::Out<::nn::sf::NativeHandle> outEventHandle
    , ::nn::sf::Out<::nn::sf::SharedPointer<::nn::nim::detail::IShopServiceAsync>> outAsync
    , const ::nn::nim::ShopServiceAccessTypes::FixedParams& inFixedParams) NN_NOEXCEPT
{
    auto pAllocator = ShopServiceAccessServerInterfaceImpl::GetInstance().GetServiceObjectAllocator();
    auto emplacedRef = SharedServiceHeap::Factory::CreateSharedEmplaced<::nn::nim::detail::IShopServiceAsync, ShopServiceAccessAsyncImpl>(pAllocator, this);
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());

    // Prepare the request handle.
    auto& impl = emplacedRef.GetImpl();
    NN_RESULT_DO(m_pServer->Prepare(&impl, inFixedParams));

    *outEventHandle = ::nn::sf::NativeHandle(impl.GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessorImpl::Request(ShopServiceAccessAsyncImpl* pRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    // Queueing the request handle.
    NN_RESULT_DO(m_pServer->Request(pRequest));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessorImpl::AcquireConnection(ConnectionHandle* pOutConnection) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ConnectMapLock)> lock(m_ConnectMapLock);

    *pOutConnection = nullptr;
    auto& maps = m_ConnectMapArray;
    for (int i = 0; i < maps.size(); ++i)
    {
        if (!maps[i])
        {
            auto& con = m_Connection[i];
            if (!con.IsInitialized())
            {
                // 未初期化なら HttpConnection 初期化.
                NN_RESULT_DO(con.Initialize(ShopServiceAccessServerInterfaceImpl::GetInstance().GetDeviceContext()));
            }
            maps[i] = true;
            con.Invalidate();
            *pOutConnection = &con;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultAllocationMemoryFailed());
}

//-----------------------------------------------------------------------------
void ShopServiceAccessorImpl::ReleaseConnection(ConnectionHandle pConnection) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ConnectMapLock)> lock(m_ConnectMapLock);

    auto& maps = m_ConnectMapArray;
    for (int i = 0; i < maps.size(); ++i)
    {
        if (pConnection == &m_Connection[i])
        {
            pConnection->Invalidate();
            maps[i] = false;
            // KeepAlive のため、このタイミングでは Finalizeしない。
            return;
        }
    }
}

//-----------------------------------------------------------------------------
bool ShopServiceAccessorImpl::FinalizeConnection() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ConnectMapLock)> lock(m_ConnectMapLock);

    auto& maps = m_ConnectMapArray;
    if (maps.any())
    {
        // 全て解放状態でなければ失敗とする。
        return false;
    }
    for (int i = 0; i < maps.size(); ++i)
    {
        m_Connection[i].Finalize();
    }
    return true;
}

//-----------------------------------------------------------------------------
::nn::ncm::ApplicationId ShopServiceAccessorImpl::GetOwnerApplicationId() const NN_NOEXCEPT
{
    return m_pServer->GetOwnerApplicationId();
}

//-----------------------------------------------------------------------------
void ShopServiceAccessServerImpl::RequestScheduler::RunOnThread(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT
{
    DEBUG_TRACE("RunOnThread: called( %p ), Thread[ %llu ].\n", pRequest, ::nn::os::GetThreadId(::nn::os::GetCurrentThread()));

    // 対象リクエストの処理実施。
    pRequest->RunOnThread();
}

void ShopServiceAccessServerImpl::RequestScheduler::FinishOnThread(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT
{
    DEBUG_TRACE("FinishOnThread: called( %p ), Thread[ %llu ].\n", pRequest, ::nn::os::GetThreadId(::nn::os::GetCurrentThread()));

    // 対象リクエストの処理完了。
    pRequest->FinishOnThread();
}

void ShopServiceAccessServerImpl::RequestScheduler::OnCancelRequestCalled(ShopServiceAccessAsyncImpl* const pRequest) NN_NOEXCEPT
{
    DEBUG_TRACE("OnCancelRequestCalled: called( %p ).\n", pRequest);

    // 対象リクエストの処理中断要求。
    //  現状はサービスフレームワークの親子階層, 寿命管理により、この要求が来る事はない。
    //  - 本デストラクタよりも先に ShopServiceAccessAsyncImpl デストラクションが全て呼ばれる。
    //  - ShopServiceAccessAsyncImpl デストラクション内で自身への Cancel and Wait を行っている。
    NN_UNUSED(pRequest);
}

void ShopServiceAccessServerImpl::RequestScheduler::OnWorkerThreadDestroyed(int index) NN_NOEXCEPT
{
    DEBUG_TRACE("Detected the worker thread has been released, index at [ %d ].\n", index);
    NN_UNUSED(index);
}

void ShopServiceAccessServerImpl::RequestScheduler::OnObserverFinished() NN_NOEXCEPT
{
    DEBUG_TRACE("Detected the finish observer.( Any run : %zu )\n", AnyRun());
}


}}}
