﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>
#include <type_traits>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_ShopServiceAccessDebug.h"
#include "nim_ShopServiceAccessServiceServer.h"
#include <curl/curl.h>

namespace nn { namespace nim { namespace srv {

namespace {

// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DEBUG_TRACE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::Async] " __VA_ARGS__ )
#define DEBUG_TRACE_AS(...) DebugTraceAs( __VA_ARGS__ )
void DebugTraceAs(const curl_slist* pHeaders) NN_NOEXCEPT
{
    DEBUG_TRACE("Request headers:\n");
    while (nullptr != pHeaders)
    {
        const auto pData = pHeaders->data;
        if (pData)
        {
            DEBUG_TRACE("\"%s\"\n", pData);
        }
        pHeaders = pHeaders->next;
    }
}
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#define DEBUG_TRACE_AS(...) static_cast<void>(0)
#endif

/**
 * @brief       カスタムHTTPメソッド文字列の取得。
 * @param[in]   method  ターゲットメソッド定数。
 * @return      HTTPメソッドを示す文字列を返します。@n
 *              GET / POST は、curl においてカスタムではないので nullptr が返されます。
 */
const char* GetCustomMethod(const ShopServiceAccessTypes::Method method) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(const char* const, s_Methods, [] =
    {
        nullptr,    // "GET",
        nullptr,    // "POST",
        "PUT",
        "DELETE",
    });
    return s_Methods[static_cast<int>(method)];
}

/**
 * @brief       key=value 形式のデータとして用いることができる文字かどうかを返します。
 */
NN_FORCEINLINE bool IsValidCharForPathAndQuery(const char ch) NN_NOEXCEPT
{
    // 参考: RFC 3986:
    //   query         = *( pchar / "/" / "?" )
    //   pchar         = unreserved / pct-encoded / sub-delims / ":" / "@"
    //   unreserved    = ALPHA / DIGIT / "-" / "." / "_" / "~"
    //   pct-encoded   = "%" HEXDIG HEXDIG
    //   sub-delims    = "!" / "$" / "&" / "'" / "(" / ")"
    //                 / "*" / "+" / "," / ";" / "="
    // ※ RFC 2234:
    //   ALPHA         = %x41-5A / %x61-7A   ; A-Z / a-z
    //   DIGIT         = %x30-39             ; 0-9
    //   HEXDIG        =  DIGIT / "A" / "B" / "C" / "D" / "E" / "F"
    return
        // unreserved
        (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') || (ch >= '0' && ch <= '9')
        || ch == '.' || ch == '-' || ch == '_' || ch == '~'
        // pct-encoded ( 「"%" HEXDIG HEXDIG」という書式までは簡潔化のためチェックしない )
        || ch == '%'
        // sub-delims
        || ch == '!' || ch == '$' || ch == '\'' || ch == '(' || ch == ')'
        || ch == '*' || ch == '+' || ch == ',' || ch == ';' || ch == '&' || ch == '='
        // pchar
        || ch == ':' || ch == '@'
        // query
        || ch == '/' || ch == '?'
        ;
}

/**
 * @brief   リプレースエイリアスシンボルの検出と変換実施クラス。
 */
class ReplaceExecutor
{
public:
    typedef ShopServiceAccessServiceServer::Accessor                    ServerAccessor;
    typedef ShopServiceAccessAsyncImpl::RequestParameter                Source;
    typedef ::nn::account::CachedNintendoAccountInfoForSystemService    AccountProperty;

    /**
     * @brief       リプレース対象ソースを提供するオーナーに関するプロパティ。
     */
    struct RequestOwnerProperty
    {
        ::nn::ncm::ApplicationId    applicationId;
    };

    /**
     * @brief       コンストラクタ。
     *
     * @details     リプレース後の値情報を初期値として指定します。
     */
    explicit ReplaceExecutor(const Source& source, const RequestOwnerProperty& owner) NN_NOEXCEPT
        : m_Source(source), m_Owner(owner) {}

    /**
     * @brief       URLパスに対するリプレース処理と、FQDN生成の実施。
     *
     * @param[out]  pTemporary      リプレース後URL文字列の出力先バッファ。
     * @param[in]   temporarySize   リプレース後URL文字列の出力先バッファ容量。( バイト単位, null終端を含めたバッファ長を設定します )
     * @param[in]   server          エンドポイントベースURL取得用サーバーアクセッサ。
     *
     * @details     リプレース前のURL文字列には、コンストラクタで指定した @ref Source が提供する[URLパス文字列]を採用します。
     */
    Result GenerateForUrl(char* const pTemporary, const size_t temporarySize, const ServerAccessor& server) const NN_NOEXCEPT;

    /**
     * @brief       URL文字列に対するリプレース処理の実施。
     *
     * @param[out]  pTemporary      リプレース後URL文字列の出力先バッファ。
     * @param[in]   temporarySize   リプレース後URL文字列の出力先バッファ容量。( バイト単位, null終端を含めたバッファ長を設定します )
     * @param[in]   pSource         リプレース前URL文字列( null 終端必要 )。
     */
    Result ExecuteForUrl(char* const pTemporary, const size_t temporarySize, const char* pSource) const NN_NOEXCEPT;

private:
    struct InternalState
    {
        AccountProperty info;
        Result          result;

        InternalState() NN_NOEXCEPT : info(), result(::nn::ResultSuccess()) {}
    };

    static Result EnsureCachedUserProperty(AccountProperty* pOutInfo, const ::nn::account::Uid& uid) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);
        if (::nn::account::InvalidNintendoAccountId == pOutInfo->GetId())
        {
            NN_RESULT_DO(ShopServiceAccess::GetCachedUserProperty(pOutInfo, uid));
        }
        NN_RESULT_SUCCESS;
    }

    char* DetectReplaceAliasCallback(char* pOut, const char* pOutEnd, char* pAlias, char* pAliasEnd, InternalState* pState) const NN_NOEXCEPT;

    Result ExecuteImpl(const char** pOutEnd, char* const pTemporary, const size_t temporarySize, const char* pSource) const NN_NOEXCEPT;

    const Source&               m_Source;
    const RequestOwnerProperty  m_Owner;
};

//-----------------------------------------------------------------------------
char* ReplaceExecutor::DetectReplaceAliasCallback(char* pOut, const char* pOutEnd, char* pAlias, char* pAliasEnd, InternalState* pState) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pState);

    const char aliasCountry[] = "country";
    const char aliasApplicationId[] = "applicationid";
    std::transform(pAlias, pAliasEnd, pAlias, ::nn::util::ToLower<char>);
    if (0 == ::nn::util::Strncmp(aliasApplicationId, pAlias, sizeof(aliasApplicationId) - 1))
    {
        const auto pEnd = ShopServiceAccess::ToHexStringFromApplicationId(pOut, pOutEnd, m_Owner.applicationId);
        pState->result = (pEnd >= pOutEnd) ? ResultBufferNotEnough() : pState->result;
        return pEnd;
    }
    else if (0 == ::nn::util::Strncmp(aliasCountry, pAlias, sizeof(aliasCountry) - 1))
    {
        const auto ensure = EnsureCachedUserProperty(&pState->info, m_Source.GetFixedParameter().uid);
        if (ensure.IsFailure())
        {
            pState->result = ensure;
            return pOut;
        }
        size_t codeSize;
        const auto result = ShopServiceAccess::GetUserCountryCode(&codeSize, pOut, pOutEnd, pState->info);
        pState->result = (result.IsFailure()) ? result : pState->result;
        return &pOut[codeSize];
    }
    return pOut;
}

//-----------------------------------------------------------------------------
Result ReplaceExecutor::ExecuteImpl(const char** pOutEnd, char* const pTemporary, const size_t temporarySize, const char* pSource) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEnd);
    NN_ABORT_UNLESS_NOT_NULL(pSource);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultAllocationMemoryFailed());

    InternalState state;
    const auto pEnd = ShopServiceAccess::Replacer::Execute(pTemporary, temporarySize, pSource,
        [&](char* pOut, const char* pOutEnd, char* pAlias, char* pAliasEnd) NN_NOEXCEPT -> char*{
            return DetectReplaceAliasCallback(pOut, pOutEnd, pAlias, pAliasEnd, &state);
        }
    );
    NN_RESULT_TRY(state.result)
        // ResultBufferNotEnough は出力バッファ不足なのでリソース不足のエラー( ec 側で `ec::ResultShopServiceAccessOutOfResource` に変換される )を返却.
        // 従来は Assert チェックのみだったが、URLはアプリケーションからの入力なので正しく返却すべき。
        // 機能互換としてはシステムAbortが、エラービューア行き Result に変わった形。
        NN_RESULT_CATCH_CONVERT(ResultBufferNotEnough, ResultAllocationMemoryFailed());
    NN_RESULT_END_TRY;

    *pOutEnd = pEnd;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ReplaceExecutor::ExecuteForUrl(char* const pTemporary, const size_t temporarySize, const char* pSource) const NN_NOEXCEPT
{
    const char* pEnd = nullptr;
    NN_RESULT_DO(ExecuteImpl(&pEnd, pTemporary, temporarySize, pSource));
    NN_RESULT_THROW_UNLESS(pEnd == std::find_if_not(static_cast<const char*>(pTemporary), pEnd, IsValidCharForPathAndQuery), ResultShopServiceAccessInvalidCharacter());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ReplaceExecutor::GenerateForUrl(char* const pTemporary, const size_t temporarySize, const ServerAccessor& server) const NN_NOEXCEPT
{
    // pPathSource の内容をScan & replace しつつ &pTemporary[pathRootIndex] 以降へコピー
    const auto pathRootIndex = server.QueryTargetBaseUrl(pTemporary);
    NN_SDK_ASSERT(temporarySize > pathRootIndex);

    const char* pEnd = nullptr;
    const auto capacity = static_cast<size_t>(temporarySize - pathRootIndex);
    const auto pPathSource = m_Source.GetVariableParameter().GetPathTop<char>();
    const auto result = ExecuteImpl(&pEnd, &pTemporary[pathRootIndex], capacity, pPathSource);
    DEBUG_TRACE("Url: \"%s\"\n", pTemporary);
    NN_RESULT_DO(result);

    // ExecuteImpl() は、URL特化ではないので、ここでURL無効文字のチェック。
    NN_RESULT_THROW_UNLESS(pEnd == std::find_if_not(static_cast<const char*>(pTemporary), pEnd, IsValidCharForPathAndQuery), ResultShopServiceAccessInvalidCharacter());
    NN_RESULT_SUCCESS;
}

/**
 * @brief   デバッグレスポンス検出結果受信用コンテナクラス。
 */
class DebugResponseReceiver : public ShopServiceAccessDebug::Response::EmulationResult
{
    NN_DISALLOW_COPY(DebugResponseReceiver);
    NN_DISALLOW_MOVE(DebugResponseReceiver);

    typedef ShopServiceAccessAsyncImpl::RequestParameter::VariableParams BufferAllocatorType;
    typedef ShopServiceAccessTypes::Server ServerType;

public:
    explicit DebugResponseReceiver(BufferAllocatorType* pAllocator, ReplaceExecutor* pReplacer) NN_NOEXCEPT
        : m_pAllocator(pAllocator), m_pReplacer(pReplacer) {}

    bool Find(const char* pVerifyPath, const ServerType& inServer) NN_NOEXCEPT
    {
        return (ShopServiceAccessDebug::Response::DoUseDebugResponse(this, pVerifyPath, inServer).IsSuccess() && IsApplied());
    }

    virtual void* AllocateResponseStore(size_t responseSize) NN_NOEXCEPT NN_OVERRIDE
    {
        BufferAllocatorType* const pAllocator = m_pAllocator;
        return (nullptr != pAllocator && pAllocator->PrepareResponse(responseSize)) ? pAllocator->GetReponseTop<void>() : nullptr;
    }

    virtual Result ExecuteAliasReplaceForUrl(char* const pOut, const size_t outCapacity, const char* pSource) NN_NOEXCEPT NN_OVERRIDE
    {
        ReplaceExecutor* const pReplacer = m_pReplacer;
        NN_RESULT_THROW_UNLESS(nullptr != pReplacer, ResultSuccess());
        return pReplacer->ExecuteForUrl(pOut, outCapacity, pSource);
    }

private:
    BufferAllocatorType* const  m_pAllocator;   //!< レスポンスバッファをユーザ指定ワークメモリから確保するためのハンドル。
    ReplaceExecutor* const      m_pReplacer;    //!< デバッグレスポンスのキーURLをリプレースするためのリプレースプロバイダ。
};

}   // ~unnamed

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::RequestParameter::VariableParams::Initialize(Allocator* const pAllocator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES(nullptr == m_pAllocator);
    NN_SDK_REQUIRES(nullptr == path.pTop);
    NN_SDK_REQUIRES(nullptr == post.pTop);
    NN_SDK_REQUIRES(nullptr == response.pTop);
    m_pAllocator = pAllocator;
}

//-----------------------------------------------------------------------------
bool ShopServiceAccessAsyncImpl::RequestParameter::VariableParams::Prepare(const ::nn::sf::InArray<char>& inPath, const ::nn::sf::InArray<char>& inPostData) NN_NOEXCEPT
{
    auto pAllocator = m_pAllocator;
    NN_SDK_ASSERT_NOT_NULL(pAllocator);

    // path, post の先頭を 8 アラインしてるのは、memcpy が最適化されるかもしれない希望で行ってます。
    const auto alignment = sizeof(uintptr_t);
    const auto sourcePathLength = inPath.GetLength();
    const auto sourcePostLength = inPostData.GetLength();
    const auto isAvailablePostData = (sourcePostLength > 0 && nullptr != inPostData.GetData());
    const auto allocatePathSize = ::nn::util::align_up(sourcePathLength + 1, alignment);
    const auto allocatePostSize = ::nn::util::align_up((isAvailablePostData) ? (sourcePostLength + 1) : 0, alignment);
    const auto totalAllocateSize = ::nn::util::align_up(allocatePathSize + allocatePostSize, alignment);

    Bit8* const pBuffer = static_cast<Bit8*>(pAllocator->Allocate(totalAllocateSize, alignment));
    if (nullptr == pBuffer)
    {
        return false;
    }

    // 以前の確保分があるなら破棄。
    FinalizeMemory(pAllocator, path);

    // path 設定
    auto pPathTop = pBuffer;
    path.Reset(pPathTop, sourcePathLength);
    std::memcpy(pPathTop, inPath.GetData(), sourcePathLength);
    pPathTop[sourcePathLength] = '\0';

    // post 設定
    auto pPostTop = (isAvailablePostData) ? &pBuffer[allocatePathSize] : nullptr;
    post.Reset(pPostTop, sourcePostLength);
    if (isAvailablePostData)
    {
        std::memcpy(pPostTop, inPostData.GetData(), sourcePostLength);
        pPostTop[sourcePostLength] = '\0';
    }
    return true;
}

//-----------------------------------------------------------------------------
bool ShopServiceAccessAsyncImpl::RequestParameter::VariableParams::PrepareResponse(const size_t expectResponseSize) NN_NOEXCEPT
{
    auto pAllocator = m_pAllocator;
    NN_SDK_ASSERT_NOT_NULL(pAllocator);

    const auto alignment = sizeof(uintptr_t);
    const auto responseSize = ::nn::util::align_up(expectResponseSize, alignment);

    auto pBuffer = pAllocator->Allocate(responseSize, alignment);
    if (nullptr == pBuffer)
    {
        return false;
    }
    auto pPre = response.pTop;
    if (nullptr != pPre)
    {
        // 以前のレスポンスがあるならコピーして破棄する。
        const auto preSize = response.size;
        const auto actualCopySize = (expectResponseSize > preSize) ? preSize : expectResponseSize;
        std::memcpy(pBuffer, pPre, actualCopySize);
        FinalizeMemory(pAllocator, response);
    }
    response.Reset(pBuffer, expectResponseSize);
    return true;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::RequestParameter::VariableParams::FinalizeMemory(Allocator* const pAllocator, Memory& memory) NN_NOEXCEPT
{
    auto pPre = memory.pTop;
    if (nullptr != pPre)
    {
        memory.Reset();
        pAllocator->Deallocate(pPre);
    }
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::RequestParameter::VariableParams::Finalize() NN_NOEXCEPT
{
    auto pAllocator = m_pAllocator;
    if (nullptr != pAllocator)
    {
        m_pAllocator = nullptr;
        FinalizeMemory(pAllocator, path);
        FinalizeMemory(pAllocator, response);
        //FinalizeMemory(pAllocator, post); // post は path との連続領域で確保するので Deallocate 対象外.
    }
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::RequestParameter::Apply(Allocator* const pAllocator, const FixedParams& fixed) NN_NOEXCEPT
{
    m_Variable.Initialize(pAllocator);
    m_Fixed = fixed;
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::RequestParameter::~RequestParameter() NN_NOEXCEPT
{
    m_Variable.Finalize();
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::ResponseValues::ResponseValues() NN_NOEXCEPT
    : receivedSize(0), errorCode(ShopServiceAccessServiceServer::Requirement::ErrorCodeNone), result(ResultTaskStillRunning()), reserved(0)
{
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::ResponseValues::ResponseValues(const Result result_) NN_NOEXCEPT
    : receivedSize(0), errorCode(ShopServiceAccessServiceServer::Requirement::ErrorCodeNone), result(result_), reserved(0)
{
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::ResponseValues::ResponseValues(const Result result_, const Bit64 receivedSize_, const ::nn::err::ErrorCode& errorCode_) NN_NOEXCEPT
    : receivedSize(receivedSize_), errorCode(errorCode_), result(result_), reserved(0)
{
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::ShopServiceAccessAsyncImpl(ShopServiceAccessorImpl* pAccessor) NN_NOEXCEPT
    : m_pAccessor(pAccessor, true)
    , m_pConnection(nullptr)
    , m_IsRequested(false)
{
    DEBUG_TRACE("Construction: done( %p ).\n", this);
}

//-----------------------------------------------------------------------------
ShopServiceAccessAsyncImpl::~ShopServiceAccessAsyncImpl() NN_NOEXCEPT
{
    DEBUG_TRACE("Destruction: called( %p ).\n", this);
    if (!m_Event.TryWait() && m_IsRequested)
    {
        Cancel();
        m_Event.Wait();
    }
    DEBUG_TRACE("Destruction: done( %p ).\n", this);
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::GetSize(::nn::sf::Out<uint64_t> outValue) NN_NOEXCEPT
{
    // サーバー側での out 引数の初期値設定は不要、プロキシ側ですべき。
    // IPC は Result 値が Success でなければ、out引数の内容は reply しない。
    const auto response = m_ResponseValues.load(std::memory_order_acquire);
    NN_RESULT_DO(response.result);

    *outValue = response.receivedSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::Read(::nn::sf::Out<uint64_t> outValue, uint64_t offset, const ::nn::sf::OutBuffer& outBuffer) const NN_NOEXCEPT
{
    // サーバー側での out 引数の初期値設定は不要、プロキシ側ですべき。
    // IPC は Result 値が Success でなければ、out引数の内容は reply しない。
    const auto response = m_ResponseValues.load(std::memory_order_acquire);
    NN_RESULT_DO(response.result);

    const uint64_t receivedSize = response.receivedSize;
    if (offset >= receivedSize)
    {
        *outValue = 0;
        NN_RESULT_SUCCESS;
    }
    const uint64_t outCapacity = outBuffer.GetSize();
    const uint64_t requiredTotalSize = receivedSize - offset;
    if (0 >= requiredTotalSize || 0 >= outCapacity)
    {
        *outValue = 0;
        NN_RESULT_SUCCESS;
    }
    const uint64_t writeSize = (outCapacity >= requiredTotalSize) ? requiredTotalSize : outCapacity;
    auto pResponseTop = m_Data.GetReponseTop<Bit8>();
    const auto pTop = &pResponseTop[offset];
    std::memcpy(outBuffer.GetPointerUnsafe(), pTop, static_cast<size_t>(writeSize));
    *outValue = writeSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::GetErrorCode(::nn::sf::Out<::nn::err::ErrorCode> outCode) const NN_NOEXCEPT
{
    // サーバー側での out 引数の初期値設定は不要、プロキシ側ですべき。
    // IPC は Result 値が Success でなければ、out引数の内容は reply しない。
    *outCode = m_ResponseValues.load(std::memory_order_acquire).errorCode;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::RunOnThread() NN_NOEXCEPT
{
    size_t receivedResponseSize = 0;
    auto result = OnExecute(&receivedResponseSize);

    // サーバーエラーの解決。
    ::nn::err::ErrorCode errorCode = ShopServiceAccessServiceServer::Requirement::ErrorCodeNone;
    ShopServiceAccessServiceServer::Accessor* pServerAccessor;
    if (ResultHttpStatus::Includes(result) && nullptr != (pServerAccessor = ShopServiceAccessServiceServer::GetPresetAccessor(m_pAccessor->GetTargetServer())))
    {
        // レスポンス内容にエラーコードが含まれた場合該当する Result 値に置き換えます。
        const char* const pResponse = m_Data.GetReponseTop<const char>();
        if (nullptr != pResponse)
        {
            pServerAccessor->ResolveServerErrorResult(&errorCode, result, pResponse, receivedResponseSize);
            result = (errorCode.IsValid()) ? ResultShopServiceAccessServiceSpecificError() : result;
        }
    }
    else if (ResultHttpConnectionCanceled::Includes(result))
    {
        // キャンセルの場合は受信サイズ 0 として設定。
        result = ResultShopServiceAccessCanceled();
        receivedResponseSize = 0;
    }

    // 要求開始状態での到着を期待とする。
    ResponseValues expectedState;
    ResponseValues reponse(result, static_cast<Bit64>(receivedResponseSize), errorCode);
    const auto& r = (m_ResponseValues.compare_exchange_strong(expectedState, reponse, std::memory_order_acq_rel)) ? reponse : expectedState;
    DEBUG_TRACE("Request( %p ) Completed => ( response[%llu byte], result[%u-%u] ).\n", this, r.receivedSize, r.result.GetModule(), r.result.GetDescription());
    NN_UNUSED(r);
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::FinishOnThread() NN_NOEXCEPT
{
    m_Event.Signal();
}

//-----------------------------------------------------------------------------
void ShopServiceAccessAsyncImpl::SetParameter(ShopServiceAccess::HeapAllocator* pAllocator, const ShopServiceAccessTypes::FixedParams& fixed) NN_NOEXCEPT
{
    m_Data.Apply(pAllocator, fixed);
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::Prepare(const ::nn::sf::InArray<char>& inPath, const ::nn::sf::InArray<char>& inPost) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Data.GetVariableParameter().Prepare(inPath, inPost), ResultShopServiceAccessInsufficientWorkMemory());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::Request() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAccessor->Request(this));

    // 通信開始要求成功した。
    m_IsRequested = true;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::OnCancel() NN_NOEXCEPT
{
    // 要求開始状態での到着を期待とする。
    ResponseValues expectedState;
    if (m_ResponseValues.compare_exchange_strong(expectedState, ResponseValues(ResultShopServiceAccessCanceled()), std::memory_order_acq_rel))
    {
        auto pConnection = m_pConnection.exchange(nullptr, std::memory_order_acq_rel);
        if (nullptr != pConnection)
        {
            pConnection->Cancel();
        }
        DEBUG_TRACE("OnCancel: done( %p ), Connection: %p\n", this, pConnection);
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::CanContinue() NN_NOEXCEPT
{
    // ハンドル状態が初期状態でなければ、キャンセルもしくは完了済として判断。
    const auto response = m_ResponseValues.load(std::memory_order_acquire);
    if (false == ResultTaskStillRunning::Includes(response.result))
    {
        NN_RESULT_THROW(response.result);
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessAsyncImpl::OnExecute(size_t* const pOutResponseSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponseSize);

    // ハンドル状態が初期状態でなければ、キャンセルもしくは完了済として即時リターン。
    NN_RESULT_DO(CanContinue());

    // プリセットサーバーアクセッサ取得
    const auto& inTargetServer = m_pAccessor->GetTargetServer();
    const auto pServerAccessor = ShopServiceAccessServiceServer::GetPresetAccessor(inTargetServer);
    NN_RESULT_THROW_UNLESS(nullptr != pServerAccessor, ResultShopServiceAccessCanceled());

    // 通信許可制限チェック
    NN_RESULT_DO(pServerAccessor->QueryRestrictUserAvailability(m_Data.GetFixedParameter().uid));

    // HttpConnection リソース獲得.
    ShopServiceAccessorImpl::ConnectionHandle pConnection;
    NN_RESULT_DO(m_pAccessor->AcquireConnection(&pConnection));
    m_pConnection.store(pConnection, std::memory_order_release);
    NN_UTIL_SCOPE_EXIT
    {
        // HttpConnection リソース解放.
        m_pConnection.store(nullptr, std::memory_order_release);
        m_pAccessor->ReleaseConnection(pConnection);
    };

    // リクエストパラメータ参照
    const auto& fixed = m_Data.GetFixedParameter();
    auto& variable = m_Data.GetVariableParameter();

    // トランザクション要求開始
    HttpConnection::TransactionHandle handle;
    NN_RESULT_DO(pConnection->BeginTransaction(&handle));
    NN_UTIL_SCOPE_EXIT{pConnection->CleanTransaction(&handle);};

    // カスタムメソッド設定( GET / POST は nullptr )
    NN_RESULT_DO(pConnection->SetTransactionCustomMethod(&handle, GetCustomMethod(fixed.method)));

    // キャンセルチェック。
    NN_RESULT_DO(CanContinue());

    // 作業用テンポラリ
    //  - デバイス認証トークンリクエストヘッダ生成
    //  - NSA-ID-TOKEN リクエストヘッダ生成
    //  - fqdn + path フィールド生成＆リプレース変換
    char pMakeParameterTemporary[4096];
    NN_STATIC_ASSERT(sizeof(pMakeParameterTemporary) >= ShopServiceAccessServiceServer::Accessor::LengthMaxForNaIdToken);
    NN_STATIC_ASSERT(sizeof(pMakeParameterTemporary) >= ShopServiceAccessServiceServer::Accessor::LengthMaxForNsaIdToken);
    NN_STATIC_ASSERT(sizeof(pMakeParameterTemporary) >= ShopServiceAccessServiceServer::Accessor::LengthMaxForDeviceToken);

    // 要求元オーナー情報
    ReplaceExecutor::RequestOwnerProperty requestOwnerProperty = {m_pAccessor->GetOwnerApplicationId()};

    // URL作成
    ReplaceExecutor replacer(m_Data, requestOwnerProperty);
    NN_RESULT_DO(replacer.GenerateForUrl(pMakeParameterTemporary, sizeof(pMakeParameterTemporary), *pServerAccessor));

    // デバッグレスポンスマッチング( Path フィールドを照合します )
    DebugResponseReceiver debugResponse(&variable, &replacer);
    if (debugResponse.Find(&pMakeParameterTemporary[pServerAccessor->GetTargetBaseUrlLength()], inTargetServer))
    {
        *pOutResponseSize = variable.response.size;
        return debugResponse.GetResult();
    }

    // URL登録
    NN_RESULT_DO(pConnection->SetTransactionUrl(&handle, pMakeParameterTemporary));

    // キャンセルチェック。
    NN_RESULT_DO(CanContinue());

    // システムリクエストヘッダ登録
    ShopServiceAccessServiceServer::Accessor::RequestHeaderGenerationHandle reqHeaderHandle =
    {
        pConnection,
        &handle,
        pMakeParameterTemporary,
        sizeof(pMakeParameterTemporary),
        requestOwnerProperty.applicationId
    };
    NN_RESULT_TRY(pServerAccessor->RegisterRequestHeaders(&reqHeaderHandle, fixed.uid, &pConnection->GetTokenCanceler()))
    // nim::ResultAllocationMemoryFailed は、ec側で ec::ResultShopServiceAccessOutOfResource に変換される対象.
    // 本スコープ中の運用であれば Debug, Developビルドで NN_STATIC_ASSERT(sizeof(pMakeParameterTemporary)) 検知できるので発生しないケース.
    // でも RegisterRequestHeaders の ResultBufferNotEnough がそのままクライアントに返却されるのは意図しないので.
    NN_RESULT_CATCH_CONVERT(ResultBufferNotEnough, ResultAllocationMemoryFailed())
    NN_RESULT_END_TRY;

    // キャンセルチェック。
    NN_RESULT_DO(CanContinue());

    // Setup the post-data field
    const auto postSize = static_cast<int64_t>(variable.post.size);
    const auto pPostSource = variable.GetPostDataTop<char>();
    if (nullptr != pPostSource && postSize > 0)
    {
        // 現状 postdata は replace しないので curl リソースへのクローンはせずに直接参照。
        NN_RESULT_DO(pConnection->SetTransactionPostFields(&handle, pPostSource, postSize, false));

        // "Content-Length" ヘッダはデフォルト付与。
        const char contentLengthHeader[] = "Content-Length:";
        const size_t lengthOfHeader = sizeof(contentLengthHeader) - 1;
        std::memcpy(pMakeParameterTemporary, contentLengthHeader, lengthOfHeader);
        auto pEnd = &pMakeParameterTemporary[sizeof(pMakeParameterTemporary) - 1];
        ShopServiceAccess::ToDecimalStringFrom(&pMakeParameterTemporary[lengthOfHeader], pEnd, static_cast<size_t>(postSize));
        NN_RESULT_DO(pConnection->SetTransactionHeader(&handle, pMakeParameterTemporary));
        NN_RESULT_DO(pConnection->SetTransactionHeader(&handle, "Content-Type:application/json"));
    }
    else
    {
        // POST, PUT( UPLOAD )指定でリクエストボディがない場合は空ボディの明示的な設定が必要( HttpConnection::TransactionHandle 仕様 )。
        const auto method = static_cast<::nn::ec::ShopService::Method>(fixed.method);
        if (::nn::ec::ShopService::Method_Put == method || ::nn::ec::ShopService::Method_Post == method)
        {
            NN_FUNCTION_LOCAL_STATIC(char, s_Empty, [] = "");
            NN_RESULT_DO(pConnection->SetTransactionPostFields(&handle, s_Empty));
            DEBUG_TRACE("The POST or PUT request without request body was detected, therefore we will append the empty body.\n");
        }
    }

    DEBUG_TRACE_AS(static_cast<curl_slist*>(handle.pHeaders));

    // キャンセルチェック。
    NN_RESULT_DO(CanContinue());

    // サーバー別要求制限許可申請
    NN_RESULT_DO(pServerAccessor->AcquireRestrictPermission(&pConnection->GetRestrictCanceler()));

    // perform
    size_t receivedSize = 0;
    const HttpConnection::HeaderCallback headerCallback = [&variable](const void* const pBuffer, const size_t bufferSize) NN_NOEXCEPT->Result
    {
        if (nullptr != pBuffer && bufferSize > 0 && !variable.response.IsAvailable())
        {
            size_t length;
            ::nn::util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "Content-Length", static_cast<const char*>(pBuffer), bufferSize));
            if (value && (length = static_cast<size_t>(std::strtoul(value->string, nullptr, 10))) > 0)
            {
                DEBUG_TRACE("Response header's [Content-Length( %zu )].\n", length);
                NN_RESULT_THROW_UNLESS(variable.PrepareResponse(length), ResultShopServiceAccessInsufficientWorkMemory());
            }
        }
        NN_RESULT_SUCCESS;
    };
    const HttpConnection::WriteCallback writeCallback = [&variable, &receivedSize](const void* const pBuffer, const size_t bufferSize) NN_NOEXCEPT -> Result
    {
        if (nullptr != pBuffer && bufferSize > 0)
        {
            const size_t nextReceivedSize = receivedSize + bufferSize;
            if (!variable.response.IsAvailable() || (nextReceivedSize > variable.response.size))
            {
                // ヘッダに Content-Length がない or 既存容量オーバーしたら再確保試行。
                DEBUG_TRACE("Unexpected over reseponse received: %zu -> %zu.\n", variable.response.size, nextReceivedSize);
                NN_RESULT_THROW_UNLESS(variable.PrepareResponse(nextReceivedSize), ResultShopServiceAccessInsufficientWorkMemory());
            }
            auto pResponseBuffer = variable.GetReponseTop<Bit8>();
            std::memcpy(&pResponseBuffer[receivedSize], pBuffer, bufferSize);
            receivedSize = nextReceivedSize;
        }
        NN_RESULT_SUCCESS;
    };
    const auto timeout = static_cast<int>(fixed.timeout.GetSeconds());
    const auto result = pConnection->EndTransaction(&handle, headerCallback, writeCallback, timeout);
    DEBUG_TRACE("Connection timeout(%d sec), response(%zu byte), Result[%u-%u].\n", timeout, receivedSize, result.GetModule(), result.GetDescription());
    *pOutResponseSize = receivedSize;
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
} // NOLINT(readability/fn_size)

}}}
