﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/os.h>
#include <nn/nim/nim_Result.h>
#include <nn/util/util_BitArray.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/srv/detail/nim_MutexAtomic.h>

namespace nn { namespace nim { namespace srv {

/**
 * @brief 再利用可能実行スレッドプール
 */
class RequestExecutorPool
{
    NN_DISALLOW_COPY(RequestExecutorPool);
    NN_DISALLOW_MOVE(RequestExecutorPool);

public:
    /**
     * @brief   スレッド属性情報
     */
    struct ThreadAttributes
    {
        const char* const   name;       //!< スレッド名称文字列。
        const int           priority;   //!< スレッド優先度。
    };

    /**
     * @brief   スレッドハンドルコンテキスト実体
     *
     * @details Exit移行通知用イベントを携帯したクラスです。@n
     *          多重イベント待ちを想定した多重待ちホルダーも提供されます。
     */
    class ThreadHandleImpl : public ::nn::os::ThreadType
    {
        friend class RequestExecutorPool;

    public:
        /**
         * @brief   スレッド実行中リクエストコンテキスト
         */
        typedef void*   RunnerContext;

        /**
         * @brief   デフォルトコンストラクタ
         *
         * @details スタックなどが未割当状態で初期化されます。@n
         *          運用する前に必ず @ref Initialize() を呼ぶ必要があります。
         *
         * @ref     Initialize()
         */
        NN_IMPLICIT ThreadHandleImpl() NN_NOEXCEPT
            : m_pOwner(nullptr)
            , m_Exit(::nn::os::EventClearMode_ManualClear)
            , m_Name(nullptr)
            , m_Stack(nullptr)
            , m_StackSize(0)
            , m_Priority(0)
            , m_Runner(nullptr)
        {
            ::nn::os::InitializeMultiWaitHolder(&m_ExitHolder, m_Exit.GetBase());
            ::nn::os::SetMultiWaitHolderUserData(&m_ExitHolder, reinterpret_cast<uintptr_t>(this));
        }

        /**
         * @brief   デストラクタ
         */
        ~ThreadHandleImpl() NN_NOEXCEPT
        {
            ::nn::os::FinalizeMultiWaitHolder(&m_ExitHolder);
        }

        /**
         * @brief   スレッド生成要求
         *
         * @param[in]   function    生成されたスレッドでのコードエントリポイントを指定します。
         *
         * @retresult
         *      @handleresult   {nn::ResultSuccess}
         *      @handleresult   {nn::os::ResultOutOfMemory}
         *      @handleresult   {nn::os::ResultOutOfResource}
         * @endresult
         */
        Result Create(::nn::os::ThreadFunction function) NN_NOEXCEPT
        {
            return ::nn::os::CreateThread(this, function, this, m_Stack, m_StackSize, m_Priority);
        }

        /**
         * @brief   スレッド破棄要求( Exit状態になるまでの待機含む )
         */
        void Destroy() NN_NOEXCEPT
        {
            ::nn::os::DestroyThread(this);
            m_Exit.Clear();
        }

        /**
         * @brief   スレッド開始要求
         */
        void Start() NN_NOEXCEPT
        {
            m_Exit.Clear();
            ::nn::os::SetThreadNamePointer(this, m_Name);
            ::nn::os::StartThread(this);
        }

        /**
         * @brief   スレッド利用終了通知。
         *
         * @details スレッドが Exit状態への遷移確定を通知するイベントをシグナルします。
         */
        void Finish() NN_NOEXCEPT
        {
            m_Exit.Signal();
        }

        /**
         * @brief   スレッドプールオーナーへのポインタ取得。
         */
        NN_FORCEINLINE RequestExecutorPool* GetOwner() const NN_NOEXCEPT
        {
            return m_pOwner;
        }

        /**
         * @brief   スレッド実行中リクエストコンテキストの登録。
         */
        NN_FORCEINLINE void SetActiveRunner(const RunnerContext runnerContext) NN_NOEXCEPT
        {
            m_Runner = runnerContext;
        }

        /**
         * @brief   スレッド実行中リクエストコンテキストの取得。
         */
        NN_FORCEINLINE const RunnerContext GetActiveRunner() const NN_NOEXCEPT
        {
            return m_Runner;
        }

    private:
        void Initialize(RequestExecutorPool* pOwner, char* pStackTop, const size_t stackSizeOnThread, const ThreadAttributes& threadAttributes) NN_NOEXCEPT
        {
            m_pOwner = pOwner;
            m_Stack = pStackTop;
            m_StackSize = stackSizeOnThread;
            m_Name = threadAttributes.name;
            m_Priority = threadAttributes.priority;
        }

        RequestExecutorPool*            m_pOwner;       //!< オーナーインスタンスへのポインタ
        ::nn::os::MultiWaitHolderType   m_ExitHolder;   //!< 多重待ち用ホルダー
        ::nn::os::Event                 m_Exit;         //!< Exit遷移確定通知用イベント
        const char*                     m_Name;         //!< スレッド名称
        char*                           m_Stack;        //!< 割り当てスタック先頭アドレス
        size_t                          m_StackSize;    //!< 割り当てスタックサイズ( byte )
        int                             m_Priority;     //!< スレッド優先度
        RunnerContext                   m_Runner;       //!< スレッド実行中リクエストへのポインタ
    };

    /**
     * @brief スレッドハンドル型
     */
    typedef ThreadHandleImpl* ThreadHandle;

    /**
     * @brief       必要なスタックサイズ算出ユーティリティ。
     *
     * @param[in]   stackSizeOnThread   １スレッドあたりのスタックサイズ。( byte単位 )@n
     *                                  ::nn::os::MemoryPageSize の倍数である必要があります。
     * @param[in]   expectThreadCount   割り当て可能プールスレッド上限数。
     */
    NN_FORCEINLINE static const size_t ComputeEnsureExpectStackSize(const size_t stackSizeOnThread, const int expectThreadCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(0 == (stackSizeOnThread % ::nn::os::MemoryPageSize));
        NN_SDK_ASSERT(0 < expectThreadCount);
        return (stackSizeOnThread * expectThreadCount);
    }

    /**
     * @brief コンストラクタ。
     *
     * @param[in]   poolCount           割り当て可能プールスレッド上限数。
     * @param[in]   pHandles            割り当て可能プールスレッドハンドル配列先頭アドレス。@n
     *                                  poolCount 分の連続領域が保証される必要があります。
     */
    explicit RequestExecutorPool(const Bit8 poolCount, ThreadHandleImpl* pHandles) NN_NOEXCEPT
        : m_BitStorage(), m_BitArray(&m_BitStorage, sizeof(m_BitStorage), poolCount), m_Lock()
        , m_CanFinish(::nn::os::EventClearMode_ManualClear)
    {
        NN_SDK_ASSERT(0 < poolCount);
        NN_SDK_ASSERT(nullptr != pHandles);
        NN_SDK_ASSERT(poolCount <= NN_BITSIZEOF(decltype(m_BitStorage)));

        m_pHandles = pHandles;
        m_BitArray.reset();
        m_CanFinish.Signal();
    }

    /**
     * @brief       スレッドコンテキストハンドルの初期化。
     *
     * @param[in]   pStackTop           @ref stackSizeOnThread * @ref poolCount で算出したサイズ分の連続領域が保証されたメモリ領域の先頭アドレスを指定します。@n
     *                                  算出サイズは @ref ComputeEnsureExpectStackSize() で求めた値と合致する事を想定しています。@n
     *                                  スタックに割り当てるため開始アドレスは @ref ::nn::os::ThreadStackAlignment でアラインされている必要があります。
     * @param[in]   stackSizeOnThread   １スレッドあたりのスタックサイズ。( byte単位 )@n
     *                                  ::nn::os::MemoryPageSize の倍数である必要があります。
     * @param[in]   threadAttributes    スレッド属性。
     *
     * @details     コンストラクタで登録したスレッドコンテキストハンドルを初期化します。
     */
    void Initialize(char* pStackTop, const size_t stackSizeOnThread, const ThreadAttributes& threadAttributes) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);

        NN_SDK_ASSERT(nullptr != pStackTop);
        NN_SDK_ASSERT(nullptr != m_pHandles);
        NN_SDK_ASSERT(0 == (stackSizeOnThread % ::nn::os::MemoryPageSize));

        auto pHandles = m_pHandles;
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            pHandles[i].Initialize(this, &pStackTop[i * stackSizeOnThread], stackSizeOnThread, threadAttributes);
        }
    }

    /**
     * @brief   スレッドプール中のアクティブなスレッドハンドルが全て解放されるまで待機します。
     *
     * @details 事前に対象スレッド処理が必ず終了するように中断要求などを実施してください。
     */
    void Finalize() NN_NOEXCEPT
    {
        m_CanFinish.Wait();
    }

    /**
     * @brief   スレッドプール中のスレッド利用状態の確認。
     *
     * @return  プールコンテキスト中の何れかのスレッドハンドルコンテキストが利用可能な場合、true が返ります。
     */
    bool CanAcquire() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        return !m_BitArray.all();
    }

    /**
     * @brief   スレッドプール中のスレッド利用状態の確認。
     *
     * @return  プールコンテキスト中の何れかのスレッドハンドルコンテキストが利用中の場合、利用中のコンテキスト数が返ります。
     */
    size_t AnyRun() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        return m_BitArray.count();
    }

    /**
     * @brief   スレッドハンドルの獲得要求。
     *
     * @return  要求結果を返します。
     * @retval  nn::ResultSuccess                   獲得に成功しました。
     * @retval  nn::nim::ResultOutOfMaxRunningTask  獲得可能なハンドルリソースがありません。
     */
    Result Acquire(ThreadHandle* pOutHandle) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            if (!m_BitArray[i])
            {
                *pOutHandle = &m_pHandles[i];
                m_BitArray[i] = true;
                m_CanFinish.Clear();
                NN_RESULT_SUCCESS;
            }
        }
        *pOutHandle = nullptr;
        NN_RESULT_THROW(ResultOutOfMaxRunningTask());
    }

    /**
     * @brief   スレッドハンドルの解放要求。
     */
    int Release(const ThreadHandle handle) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            if (handle == &m_pHandles[i])
            {
                m_BitArray[i] = false;
                if (m_BitArray.none())
                {
                    m_CanFinish.Signal();
                }
                return i;
            }
        }
        NN_ABORT("must not come here");
    }

    /**
     * @brief   Exit通知イベントの多重待ち登録
     */
    void LinkMultiWaitForExit(::nn::os::MultiWaitType* pWait) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            ::nn::os::LinkMultiWaitHolder(pWait, &m_pHandles[i].m_ExitHolder);
        }
    }

    /**
     * @brief   Exit通知イベントの多重待ち解除
     */
    void UnlinkMultiWaitForExit() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            ::nn::os::UnlinkMultiWaitHolder(&m_pHandles[i].m_ExitHolder);
        }
    }

    /**
     * @brief       アクティブな各スレッドハンドル毎に任意メソッドを実行するユーティリティ。
     *
     * @tparam      RequestExecutor スレッドハンドル毎に実行するメソッド型。( void (*RequestExecutor)(const ThreadHandle handle) 型 )
     *
     * @param[in]   スレッドハンドル毎に実行するメソッド実体。
     *
     * @details     指定されたメソッドは獲得処理排他期間中に呼び出されるため、呼び出しメソッド中での排他処理によるデッドロックなどに注意してください。
     */
    template<typename RequestExecutor>
    void RunForeachActiveHandles(RequestExecutor executor) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        for (int i = 0; i < m_BitArray.size(); ++i)
        {
            if (m_BitArray[i])
            {
                executor(&m_pHandles[i]);
            }
        }
    }

private:
    ThreadHandleImpl*                   m_pHandles;     //!< 事前準備済スレッドハンドルコンテキスト
    Bit64                               m_BitStorage;   //!< ハンドル割り当て管理用ビットマップストレージ領域
    util::BitArray                      m_BitArray;     //!< ハンドル割り当て管理ユーティリティ
    mutable ::nn::os::SdkRecursiveMutex m_Lock;         //!< 割り当てロック
    ::nn::os::LightEvent                m_CanFinish;
};

/**
 * @brief   リクエストスケジューラ
 * @details 以下の機能を提供します。@n
 *              - リクエストキュー@n
 *              - リクエストの並列実施@n
 *          並列用作業スレッドは @ref RequestExecutorPool で管理されます。@n
 *          作業スレッドとは別に作業状態監視用スレッドが生成され、作業スレッドからの Exit 通知を多重待ちします。
 */
template<typename TRequest, size_t ExpectSchedulerStackSize, size_t MessageQueueCapacity>
class RequestQueueScheduler : RequestExecutorPool
{
    NN_DISALLOW_COPY(RequestQueueScheduler);
    NN_DISALLOW_MOVE(RequestQueueScheduler);

    /**
     * @brief   スケジューラコンディション管理構造体。
     *
     * @details スケジューラ全体の状態管理は atomic<> でフラグ管理します。
     */
    struct NN_ALIGNAS(4) Condition
    {
        bool    isInitialized;  //!< 初期化済、稼働中
        bool    isFinishing;    //!< 終了処理中( 終了要求受理 )
        Bit8    reserved[2];    //!< compare_exchange() の値比較一致保証用。

        NN_IMPLICIT Condition() NN_NOEXCEPT : isInitialized(false), isFinishing(false), reserved() {}
        NN_IMPLICIT Condition(bool isInit_, bool isFinish_) NN_NOEXCEPT : isInitialized(isInit_), isFinishing(isFinish_), reserved() {}
        NN_FORCEINLINE bool IsIdle() const NN_NOEXCEPT
        {
            return false == isInitialized && false == isFinishing;
        }
        NN_FORCEINLINE bool IsInitialized() const NN_NOEXCEPT
        {
            return isInitialized && false == isFinishing;
        }
    };
    NN_STATIC_ASSERT(sizeof(Condition) == 4);

    /**
     * @brief   アトミック型宣言。
     */
    typedef detail::MutexAtomic<Condition> ConditionAtomic;

public:
    static const size_t EnsureExpectSchedulerStackSize = ExpectSchedulerStackSize;

    typedef RequestExecutorPool::ThreadHandleImpl   ThreadHandleImpl;
    typedef RequestExecutorPool::ThreadHandle       ThreadHandle;
    typedef RequestExecutorPool                     BaseType;

    /**
     * @brief       必要なスタックサイズ算出ユーティリティ。
     *
     * @param[in]   stackSizeOnThread   １スレッドあたりのスタックサイズ。( byte単位 )@n
     *                                  ::nn::os::MemoryPageSize の倍数である必要があります。
     * @param[in]   expectThreadCount   割り当て可能プールスレッド上限数。
     */
    NN_FORCEINLINE static const size_t ComputeEnsureExpectStackSize(const size_t stackSizeOnThread, const int expectThreadCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(0 == (stackSizeOnThread % ::nn::os::MemoryPageSize));
        NN_SDK_ASSERT(0 < expectThreadCount);
        return (stackSizeOnThread * expectThreadCount) + EnsureExpectSchedulerStackSize;
    }

    /**
     * @brief       コンストラクタ。
     *
     * @param[in]   poolCount           割り当て可能プールスレッド上限数。
     * @param[in]   pHandles            割り当て可能プールスレッドハンドル配列先頭アドレス。@n
     *                                  poolCount 分の連続領域が保証される必要があります。
     * @param[in]   pStackTop           @ref stackSizeOnThread * @ref poolCount + @ref EnsureExpectSchedulerStackSize で算出したサイズ分の連続領域が保証されたメモリ領域の先頭アドレスを指定します。@n
     *                                  算出サイズは @ref ComputeEnsureExpectStackSize() で求めた値と合致する事を想定しています。@n
     *                                  スタックに割り当てるため開始アドレスは @ref ::nn::os::ThreadStackAlignment でアラインされている必要があります。
     * @param[in]   stackSizeOnThread   １スレッドあたりのスタックサイズ。( byte単位 )@n
     *                                  ::nn::os::MemoryPageSize の倍数である必要があります。
     */
    explicit RequestQueueScheduler(const Bit8 poolCount, ThreadHandleImpl* pHandles, char* pStackTop, const size_t stackSizeOnThread) NN_NOEXCEPT
        : BaseType(poolCount, pHandles)
        , m_pStack(pStackTop)
        , m_StackSizeOnThread(stackSizeOnThread)
        , m_Exit(::nn::os::EventClearMode_ManualClear)
        , m_IsInitialized(Condition())
        , m_QueueLock()
    {
        ::nn::os::InitializeMessageQueue(&m_Queue, m_QueueStorage, NN_ARRAY_SIZE(m_QueueStorage));
        ::nn::os::InitializeMultiWaitHolder(&m_ExitHolder, m_Exit.GetBase());
        ::nn::os::SetMultiWaitHolderUserData(&m_ExitHolder, 0);
    }

    /**
     * @brief       デストラクタ。
     */
    ~RequestQueueScheduler() NN_NOEXCEPT
    {
        ::nn::os::FinalizeMultiWaitHolder(&m_ExitHolder);
        ::nn::os::FinalizeMessageQueue(&m_Queue);
    }

    /**
     * @brief       スケジューラ初期化。
     *
     * @param[in]   observer    監視用スレッド属性。
     * @param[in]   request     リクエスト実行用スレッド属性。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::os::ResultOutOfMemory}
     *      @handleresult   {nn::os::ResultOutOfResource}
     * @endresult
     *
     * @details     スケジューラコンディションを Initialized 状態にし、監視スレッドを起動します。@n
     *              呼び出しにはスケジューラが Idle 状態である必要があります。
     *
     * @pre
     *              - true == m_IsInitialized.load().IsIdle()
     *
     * @post
     *              - true == m_IsInitialized.load().IsInitialized()
     */
    Result Initialize(const ThreadAttributes& observer, const ThreadAttributes& request) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized.load(std::memory_order_acquire).IsIdle());
        BaseType::Initialize(&m_pStack[EnsureExpectSchedulerStackSize], m_StackSizeOnThread, request);
        NN_RESULT_DO(PrepareImpl(observer.name, observer.priority));
        m_IsInitialized.store(Condition(true, false), std::memory_order_release);
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       スケジューラ終了。
     *
     * @details     スケジューラコンディションを Finishing 状態にし、監視スレッドを停止/破棄します。@n
     *              監視スレッドの停止/破棄が完了した時点で Idle 状態になります。@n
     *              呼び出しにはスケジューラが Initialized 状態である必要があります。
     * @pre
     *              - true == m_IsInitialized.load().IsInitialized()
     *
     * @post
     *              - true == m_IsInitialized.load().IsIdle()
     */
    void Finalize() NN_NOEXCEPT
    {
        // 終了処理開始設定。
        Condition expected(true, false);
        if (!m_IsInitialized.compare_exchange_strong(expected, Condition(false, true), std::memory_order_acq_rel))
        {
            // 基本このケースは既に終了済でしか来ないので、Event化はインスタンス勿体ないので Sleep で代用。
            while (!m_IsInitialized.load(std::memory_order_acquire).IsIdle())
            {
                ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(100));
            }
            return; // 既に終了済 or 終了中。
        }

        // 作業スレッド停止要求
        Finish();

        // 監視スレッド終了要求。
        m_Exit.Signal();

        // 監視スレッド完了待ち。
        ::nn::os::DestroyThread(&m_ThreadHandle);
        m_Exit.Clear();

        m_IsInitialized.store(Condition(false, false), std::memory_order_release);
    }

    /**
     * @brief       全キュー/実行中リクエストに対してキャンセル要求を行い完了を待つ。
     */
    void Finish() NN_NOEXCEPT
    {
        // 全キュー/実行中リクエストに対してキャンセルを呼び出し。
        PrepareFinish();

        // 全キュー/作業スレッドの完了確認及び終了処理。
        BaseType::Finalize();
    }

    /**
     * @brief       リクエスト実施要求。
     *
     * @param[in]   pRequest    要求インスタンスへのポインタ。@n
     *                          本メソッド内でインスタンスのクローンは行われません。@n
     *                          本メソッドによる要求発行後も、要求インスタンスの確保は維持し続ける必要があります。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::os::ResultOutOfMemory}
     *      @handleresult   {nn::os::ResultOutOfResource}
     *      @handleresult   {nn::nim::ResultTaskNotFound}
     *      @handleresult   {nn::nim::ResultOutOfMaxRunningTask}
     * @endresult
     *
     * @details     並列可能な場合は、新規に作業スレッドを起動し要求を実施します。@n
     *              同時並列上限を超えた要求は @ref MessageQueueCapacity の上限までキューイングされます。@n
     *              キューイングは要求インスタンス @ref AsyncShopServiceAccessImpl のポインタを単位で行われるため、@n
     *              本メソッドによる要求発行後も、要求インスタンスの確保は維持し続ける必要があります。
     */
    Result Request(TRequest* const pRequest) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);
        if (!m_IsInitialized.load(std::memory_order_acquire).IsInitialized())
        {
            NN_RESULT_THROW(ResultTaskNotFound());  // 仮: スケジューラが終了しているのに呼ばれた
        }
        if (!::nn::os::TrySendMessageQueue(&m_Queue, reinterpret_cast<uintptr_t>(pRequest)))
        {
            NN_RESULT_THROW(ResultOutOfMaxRunningTask());
        }
        else if (BaseType::CanAcquire())
        {
            NN_RESULT_DO(CreateAndRunImpl());
        }
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   ワーカースレッド利用状態の確認。
     *
     * @return  何れかのワーカースレッドが利用中の場合、利用中のコンテキスト数が返ります。
     */
    size_t AnyRun() const NN_NOEXCEPT
    {
        return BaseType::AnyRun();
    }

    /**
     * @brief   割り当てられたスタック先頭アドレスを返します。
     */
    char* GetStackTop() const NN_NOEXCEPT
    {
        return m_pStack;
    }

protected:
    /**
     * @brief       ワーカースレッド上でのリクエスト実施コールバック。
     *
     * @param[in]   pRequest    呼び出されたスレッド上で実施すべき要求インスタンスへのポインタ。
     */
    virtual void RunOnThread(TRequest* const pRequest) NN_NOEXCEPT = 0;

    /**
     * @brief       ワーカースレッド上でのリクエスト終了処理実施コールバック。
     *
     * @param[in]   pRequest    呼び出されたスレッド上で終了処理を行うべき要求インスタンスへのポインタ。
     *
     * @details     このメソッド以降に本クラスが @a pRequest インスタンスを参照する事はありません。
     */
    virtual void FinishOnThread(TRequest* const pRequest) NN_NOEXCEPT = 0;

    /**
     * @brief       スケジューラからのキュー/実行中リクエストへのキャンセル要求コールバック。
     *
     * @param[in]   pRequest    キャンセル対象要求インスタンスへのポインタ。
     */
    virtual void OnCancelRequestCalled(TRequest* const pRequest) NN_NOEXCEPT = 0;

    /**
     * @brief       ワーカースレッドコンテキストの破棄検出時に呼び出される通知コールバック。
     *
     * @param[in]   index   ワーカースレッドコンテキストの一意なインデクスです。
     *
     * @details     監視スレッド上から呼び出されます。
     */
    virtual void OnWorkerThreadDestroyed(int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
    }

    /**
     * @brief       監視スレッドの終了時に呼び出される通知コールバック。
     *
     * @details     監視スレッド上から呼び出されます。
     */
    virtual void OnObserverFinished() NN_NOEXCEPT {}

private:
    /**
     * @brief       スケジューラ監視スレッド生成/開始。
     *
     * @param[in]   threadName          監視用スレッド名称文字列。
     * @param[in]   threadPriority      監視用スレッド優先度。
     */
    Result PrepareImpl(const char* threadName, int threadPriority) NN_NOEXCEPT
    {
        m_Exit.Clear();
        NN_RESULT_DO(::nn::os::CreateThread(&m_ThreadHandle, [](void* p)
        {
            reinterpret_cast<RequestQueueScheduler*>(p)->ObserveImpl();
        }, this, m_pStack, EnsureExpectSchedulerStackSize, threadPriority));
        ::nn::os::SetThreadNamePointer(&m_ThreadHandle, threadName);
        ::nn::os::StartThread(&m_ThreadHandle);
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       監視処理実体。
     *
     * @details     作業スレッドのExitイベントを多重イベント待ちします。@n
     *              Exitイベントを検知した場合、対象の作業スレッドを破棄して再利用可能な状態へ移行させます。
     */
    void ObserveImpl() NN_NOEXCEPT
    {
        ::nn::os::MultiWaitType observer;
        ::nn::os::InitializeMultiWait(&observer);
        ::nn::os::LinkMultiWaitHolder(&observer, &m_ExitHolder);
        LinkMultiWaitForExit(&observer);
        while (NN_STATIC_CONDITION(true))
        {
            auto pHolder = ::nn::os::WaitAny(&observer);
            auto pHandle = reinterpret_cast<ThreadHandle>(::nn::os::GetMultiWaitHolderUserData(pHolder));
            if (nullptr == pHandle)
            {
                break;
            }
            pHandle->Destroy();
            const auto index = Release(pHandle);
            OnWorkerThreadDestroyed(index);
        }
        UnlinkMultiWaitForExit();
        ::nn::os::UnlinkMultiWaitHolder(&m_ExitHolder);
        ::nn::os::FinalizeMultiWait(&observer);
        OnObserverFinished();
    }

    /**
     * @brief       作業スレッドの生成と開始。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::os::ResultOutOfMemory}
     *      @handleresult   {nn::os::ResultOutOfResource}
     *      @handleresult   {nn::nim::ResultOutOfMaxRunningTask}
     * @endresult
     */
    Result CreateAndRunImpl() NN_NOEXCEPT
    {
        ThreadHandle handle;
        NN_RESULT_DO(Acquire(&handle));
        NN_RESULT_TRY(handle->Create([](void* p)
        {
            auto pHandle = reinterpret_cast<ThreadHandle>(p);
            auto pScheduler = static_cast<RequestQueueScheduler*>(pHandle->GetOwner());
            pScheduler->RunOnWorker(pHandle);
            pHandle->Finish();
        }));
        NN_RESULT_CATCH_ALL
        {
            Release(handle);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_END_TRY;
        handle->Start();    // スレッド生成成功したのでスレッド開始。
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       終了準備要求。
     */
    void PrepareFinish() NN_NOEXCEPT
    {
        // キュー中要求へのキャンセル
        {
            std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);
            TRequest* pNext;
            while (nullptr != (pNext = NextImpl()))
            {
                OnCancelRequestCalled(pNext);
            }
        }

        // 実行中要求へのキャンセル
        RunForeachActiveHandles([this](const ThreadHandle handle) -> void
        {
            auto pActiveRunner = handle->GetActiveRunner();
            if (nullptr != pActiveRunner)
            {
                OnCancelRequestCalled(reinterpret_cast<TRequest*>(pActiveRunner));
            }
        });
    }

    /**
     * @brief       キューから次要求を取得(非スレッドセーフ)
     */
    TRequest* NextImpl() NN_NOEXCEPT
    {
        uintptr_t message;
        return (::nn::os::TryReceiveMessageQueue(&message, &m_Queue))
            ? reinterpret_cast<TRequest*>(message)
            : nullptr;
    }

    /**
     * @brief       キューから次要求を取得。
     */
    TRequest* Next(ThreadHandle pHandle) NN_NOEXCEPT
    {
        TRequest* pNext;
        std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);
        pHandle->SetActiveRunner(pNext = NextImpl());
        return pNext;
    }

    /**
     * @brief       実行中の要求の完了前処理。
     */
    void EndOfRunOnThread(ThreadHandle pHandle) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_QueueLock)> lock(m_QueueLock);
        pHandle->SetActiveRunner(nullptr);
    }

    /**
     * @brief       作業スレッド処理実体。
     */
    void RunOnWorker(ThreadHandle pHandle) NN_NOEXCEPT
    {
        TRequest* pNext;
        while (nullptr != (pNext = Next(pHandle)))
        {
            RunOnThread(pNext);

            EndOfRunOnThread(pHandle);

            FinishOnThread(pNext);
        }
    }

    // スケジューラスレッド
    char* const                     m_pStack;               //!< スレッドスタック先頭アドレス
    const size_t                    m_StackSizeOnThread;    //!< スレッドスタックサイズ( 1スレッド分 )
    ::nn::os::ThreadType            m_ThreadHandle;         //!< スレッドハンドル
    ::nn::os::MultiWaitHolderType   m_ExitHolder;           //!< 多重待ち用ホルダー
    ::nn::os::Event                 m_Exit;                 //!< Exit遷移確定通知用イベント
    ConditionAtomic                 m_IsInitialized;        //!< スケジューラコンディション

    // キュー
    ::nn::os::MessageQueueType          m_Queue;
    mutable ::nn::os::SdkRecursiveMutex m_QueueLock;
    uintptr_t                           m_QueueStorage[MessageQueueCapacity];
};

}}}
