﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/ncm/ncm_ContentIdUtil.h>

namespace nn { namespace nim { namespace srv {
    namespace
    {
        const char* GetAtumHostName(bool isSystem) NN_NOEXCEPT
        {
            return isSystem ? "atumn" : "atum";
        }

        void MakeContentUrlImpl(Url* outValue, const ncm::ContentId& contentId, char typeChar, bool isSystem) NN_NOEXCEPT
        {
            char contentIdString[33];
            ncm::GetStringFromContentId(contentIdString, sizeof(contentIdString), contentId);

            outValue->AssignFormat("https://%s.hac.%%.d4c.nintendo.net/c/%c/%s", GetAtumHostName(isSystem), typeChar, contentIdString);
        }

        void MakeContentMetaQueryUrlImpl(Url* outValue, Bit64 id, uint32_t version, char typeChar, Bit64 deviceId, bool isSystem) NN_NOEXCEPT
        {
            outValue->AssignFormat("https://%s.hac.%%.d4c.nintendo.net/t/%c/%016llx/%u?device_id=%016llx", GetAtumHostName(isSystem), typeChar, id, version, deviceId);
        }
    }

    void MakeSystemUpdateMetaQueryUrl(Url* outValue, Bit64 deviceId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://sun.hac.%%.d4c.nintendo.net/v1/system_update_meta?device_id=%016llx", deviceId);
    }

    void MakeContentMetaQueryUrl(Url* outValue, Bit64 id, uint32_t version, ncm::ContentMetaType type, Bit64 deviceId, bool isSystem) NN_NOEXCEPT
    {
        switch (type)
        {
        case ncm::ContentMetaType::SystemUpdate: MakeContentMetaQueryUrlImpl(outValue, id, version, 's', deviceId, isSystem); return;
        default: MakeContentMetaQueryUrlImpl(outValue, id, version, 'a', deviceId, isSystem); return;
        }
    }

    void MakeContentMetaUrl(Url* outValue, const ncm::ContentId& contentId, ncm::ContentMetaType type, bool isSystem) NN_NOEXCEPT
    {
        switch (type)
        {
        case ncm::ContentMetaType::SystemUpdate: MakeContentUrlImpl(outValue, contentId, 's', isSystem); return;
        default: MakeContentUrlImpl(outValue, contentId, 'a', isSystem); return;
        }
    }

    void MakeContentUrl(Url* outValue, const ncm::ContentId& contentId, bool isSystem) NN_NOEXCEPT
    {
        MakeContentUrlImpl(outValue, contentId, 'c', isSystem);
    }

    void MakeApplicationControlContentUrl(Url* outValue, const ncm::ApplicationId& appId, uint32_t version) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://atum.hac.%%.d4c.nintendo.net/a/d/%016llx/%u", appId.value, version);
    }

    void MakeLatestApplicationControlContentUrl(Url* outValue, const ncm::ApplicationId& appId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://atum.hac.%%.d4c.nintendo.net/a/d/%016llx", appId.value);
    }

    void MakeLatestVersionUrl(Url* outValue, Bit64 id) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://superfly.hac.%%.d4c.nintendo.net/v1/t/%016llx/dv", id);
    }

    void MakeLatestVersionArrayUrl(Url* outValue, Bit64 id) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://superfly.hac.%%.d4c.nintendo.net/v1/a/%016llx/dv", id);
    }

    void MakeGameCardRegistrationStatusUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/rom_cards");
    }

    void MakeRegisterGameCardUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/rom_cards/register");
    }

    void MakeNotificationTokenRegisterUrl(Url* outValue, Bit64 deviceId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://beach.hac.%%.eshop.nintendo.net/v1/devices/%016llx/notification_token/register", deviceId);
    }

    void MakeDownloadTaskListUrl(Url* outValue, Bit64 deviceId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://pushmo.hac.%%.eshop.nintendo.net/pushmo/d/%016llx.json", deviceId);
    }

    void MakeVersionListUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://tagaya.hac.%.eshop.nintendo.net/tagaya/hac_versionlist");
    }

    void MakeUnlinkDeviceUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://beach.hac.%%.eshop.nintendo.net/v1/devices/hac/unlink?lang=en");
    }

    void MakeUnlinkDeviceAllUrl(Url* outValue, Bit64 deviceId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://beach.hac.%%.eshop.nintendo.net/v1/my/devices/hac/unlink_all?device_id=%016llx", deviceId);
    }

    void MakeLinkDeviceUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/my/devices/hac/link?lang=en");
    }

    void MakeCreateVirtualAccountUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/my/account?lang=en");
    }

    void MakeDeviceLinkStatusUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/my/devices?device_type_id=6&lang=en");
    }

    void MakeVirtualAccountUrl(Url* outValue) NN_NOEXCEPT
    {
        outValue->Assign("https://beach.hac.%.eshop.nintendo.net/v1/my/virtual_account?lang=en");
    }

    void MakeDownloadCommonTicketUrl(Url* outValue, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        const int RightsIdStringLength = 33;
        char rightsIdString[RightsIdStringLength];

        for (size_t i = 0; i < sizeof(es::RightsIdIncludingKeyId); i++)
        {
            util::SNPrintf(&rightsIdString[i * 2], 3, "%02x", rightsId._data[i]);
        }
        rightsIdString[RightsIdStringLength - 1] = '\0';

        outValue->AppendFormat("https://atum.hac.%%.d4c.nintendo.net/r/t/%s", rightsIdString);
    }

    void MakeSystemUpdateMetaForContentDeliveryUrl(Url* outValue, Bit64 deviceId) NN_NOEXCEPT
    {
        outValue->AssignFormat("https://aqua.hac.%%.d4c.nintendo.net/required_system_update_meta?device_id=%016llx", deviceId);
    }

}}}
