﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nim/srv/nim_NetworkInstallTaskBase.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include "nim_NetworkInstallUtil.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {

    namespace {
        Result FindContentLength(util::optional<int64_t>* outValue, const char* buffer, size_t size) NN_NOEXCEPT
        {
            util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "Content-Length", buffer, size));
            if (!value)
            {
                *outValue = util::nullopt;
                NN_RESULT_SUCCESS;
            }

            *outValue =  NN_NIM_STR_TO_LL(value->string, nullptr, 10);
            NN_RESULT_SUCCESS;
        }

        Result FindContentId(util::optional<ncm::ContentId>* outValue, const char* buffer, size_t size) NN_NOEXCEPT
        {
            util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "X-Nintendo-Content-ID", buffer, size));
            if (!value)
            {
                *outValue = util::nullopt;
                NN_RESULT_SUCCESS;
            }

            *outValue = ncm::GetContentIdFromString(value->string, sizeof(value->string));
            NN_RESULT_SUCCESS;
        }

        Result FindContentHash(util::optional<ncm::Hash>* outValue, const char* buffer, size_t size) NN_NOEXCEPT
        {
            util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "X-Nintendo-Content-Hash", buffer, size));
            if (!value)
            {
                *outValue = util::nullopt;
                NN_RESULT_SUCCESS;
            }

            *outValue = ncm::GetHashFromString(value->string, sizeof(value->string));
            NN_RESULT_SUCCESS;
        }
    }

    void NetworkInstallTaskBase::Cancel() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        InstallTaskBase::Cancel();
    }

    void NetworkInstallTaskBase::ResetCancel() NN_NOEXCEPT
    {
        m_Connection.ResetCancel();
        InstallTaskBase::ResetCancel();
    }

    Result NetworkInstallTaskBase::Initialize(DeviceContext* deviceContext, ncm::StorageId storage, ncm::InstallTaskDataBase* data, bool isSystem, Bit32 config) NN_NOEXCEPT
    {
        m_DeviceContext = deviceContext;
        m_Data = data;
        NN_RESULT_DO(InstallTaskBase::Initialize(storage, m_Data, config));
        m_IsSystem = isSystem;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTaskBase::GetInstallContentMetaInfo(ncm::InstallContentMetaInfo* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        Url url;
        MakeContentMetaQueryUrl(&url, key.id, key.version, key.type, m_DeviceContext->GetDeviceId(), m_IsSystem);

        util::optional<ncm::ContentId> contentId;
        util::optional<int64_t> contentLength;
        util::optional<ncm::Hash> contentHash;
        NN_RESULT_DO(m_Connection.Head(url, [&contentId, &contentLength, &contentHash](const char* buffer, size_t size) -> Result
            {
                if (!contentLength)
                {
                    NN_RESULT_DO(FindContentLength(&contentLength, static_cast<const char*>(buffer), size));
                }
                if (!contentId)
                {
                    NN_RESULT_DO(FindContentId(&contentId, static_cast<const char*>(buffer), size));
                }
                if (!contentHash)
                {
                    NN_RESULT_DO(FindContentHash(&contentHash, static_cast<const char*>(buffer), size));
                }
                NN_RESULT_SUCCESS;
            }));
        NN_RESULT_THROW_UNLESS(contentId && contentLength && contentHash, ResultUnexpectedResponseContentMetaRequiredHeaderNotFound());

        *outValue = ncm::InstallContentMetaInfo::MakeVerifiable(*contentId, *contentLength, key.type, *contentHash);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTaskBase::PrepareInstallContentMetaData() NN_NOEXCEPT
    {
        auto count = CountInstallContentMetaKey();
        for (int i = 0; i < count; i++)
        {
            NN_RESULT_DO(CanContinue());
            ncm::ContentMetaKey key;
            NN_RESULT_DO(GetInstallContentMetaKey(&key, i));
            bool hasAlreadyData;
            NN_RESULT_DO(m_Data->Has(&hasAlreadyData, key.id));
            if (hasAlreadyData)
            {
                continue;
            }

            ncm::InstallContentMetaInfo info;
            NN_RESULT_DO(GetInstallContentMetaInfo(&info, key));
            NN_RESULT_DO(PrepareContentMeta(info, key));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTaskBase::OnWritePlaceHolder(ncm::InstallContentInfo* data) NN_NOEXCEPT
    {
        // 書き込みが終わった瞬間に HTTP がキャンセルされたあとのリトライ対応
        if (data->written == data->info.GetSize())
        {
            NN_RESULT_SUCCESS;
        }

        Url url;
        if (data->GetType() == ncm::ContentType::Meta)
        {
            MakeContentMetaUrl(&url, data->GetId(), data->metaType, m_IsSystem);
        }
        else
        {
            MakeContentUrl(&url, data->GetId(), m_IsSystem);
        }

        // GET が失敗した時のためのバッファクリア
        NN_UTIL_SCOPE_EXIT{ ClearBuffer(); };

        auto before = data->written;

        char rangeHeader[sizeof("Range: bytes=9223372036854775807")];
        nn::util::SNPrintf(rangeHeader, sizeof("Range: bytes=9223372036854775807"), "Range: bytes=%lld-", data->written);
        const char* HeaderList[] = { rangeHeader };
        NN_RESULT_TRY(m_Connection.Get(url,
            [this, data](const void* buffer, size_t bufferSize) -> Result
            {
                NN_RESULT_DO(WritePlaceHolderWithBuffer(data, buffer, bufferSize));

                NN_RESULT_SUCCESS;
            }, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            // コンテンツ DL 時に CURLE_PARTIAL_FILE が返ったら上位でリトライする
            NN_RESULT_CATCH_CONVERT(ResultCurlErrorPartialFile, ResultHttpConnectionContentPartialFile())
            NN_RESULT_CATCH(ResultHttpConnectionTimeout)
            {
                if (before == data->written)
                {
                    // 進捗が無かったので、そのまま投げる
                    NN_RESULT_RETHROW;
                }
                // 進捗があったので、上位でリトライさせる
                NN_RESULT_THROW(ResultHttpConnectionRetryableTimeout());
            }
        NN_RESULT_END_TRY
        NN_RESULT_DO(FlushBuffer(data));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTaskBase::GetLatestVersion(util::optional<uint32_t>* outValue, Bit64 id) NN_NOEXCEPT
    {
        return nn::nim::srv::GetLatestVersion(outValue, &m_Connection, id);
    }

    Result NetworkInstallTaskBase::WritePlaceHolderWithBuffer(ncm::InstallContentInfo* data, const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        if (m_Buffer)
        {
            if (bufferSize > m_BufferSize)
            {
                NN_RESULT_DO(FlushBuffer(data));
                NN_RESULT_DO(WritePlaceHolderBuffer(data, buffer, bufferSize));
                NN_RESULT_SUCCESS;
            }

            auto filledSize = FillBuffer(buffer, bufferSize);
            if (filledSize < bufferSize)
            {
                NN_RESULT_DO(FlushBuffer(data));
                FillBuffer(reinterpret_cast<const char*>(buffer) + filledSize, bufferSize - filledSize);
            }
        }
        else
        {
            NN_RESULT_DO(WritePlaceHolderBuffer(data, buffer, bufferSize));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallTaskBase::FlushBuffer(ncm::InstallContentInfo* data) NN_NOEXCEPT
    {
        if (m_Buffer && m_BufferFilledSize > 0)
        {
            NN_RESULT_DO(WritePlaceHolderBuffer(data, m_Buffer, m_BufferFilledSize));
            m_BufferFilledSize = 0;
        }

        NN_RESULT_SUCCESS;
    }

    size_t NetworkInstallTaskBase::FillBuffer(const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        auto fillSize = std::min(m_BufferSize - m_BufferFilledSize, bufferSize);

        std::memcpy(m_Buffer + m_BufferFilledSize, buffer, fillSize);
        m_BufferFilledSize += fillSize;

        return fillSize;
    }

    void NetworkInstallTaskBase::ClearBuffer() NN_NOEXCEPT
    {
        if (m_Buffer)
        {
            m_BufferFilledSize = 0;
        }
    }

}}}
