﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/fs/fs_SystemSaveData.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_StorageUtil.h>

#include <nn/nim/nim_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_NetworkInstallManagerServer.h>
#include <nn/util/util_Uuid.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/pm/pm_BootModeApi.h>
#include "nim_AsyncApplicationControlImpl.h"
#include "nim_AsyncDownloadTaskListImpl.h"
#include "nim_AsyncDeviceAuthenticationTokenImpl.h"
#include "nim_AsyncGameCardRegistrationImpl.h"
#include "nim_AsyncLatestVersionImpl.h"
#include "nim_AsyncNotificationTokenImpl.h"
#include "nim_AsyncSystemUpdateMetaImpl.h"
#include "nim_AsyncTaskResultImpl.h"
#include "nim_AsyncVersionListImpl.h"
#include "nim_DynamicRightsAvailableELicenses.h"
#include "nim_DynamicRightsAssignELicenses.h"
#include "nim_DynamicRightsDownloadETickets.h"
#include "nim_DynamicRightsExtendELicenses.h"
#include "nim_DynamicRightsNotificationToken.h"
#include "nim_DynamicRightsReportELicenses.h"
#include "nim_DynamicRightsRevokeReason.h"
#include "nim_DynamicRightsSyncELicenses.h"
#include "nim_DebugUtil.h"

namespace nn { namespace nim { namespace srv {

    namespace {
        const int64_t   SystemUpdateTaskSaveDataSize            = 0x108000; // 1MB + 32 KB
        const int64_t   SystemUpdateTaskSaveDataJournalSize     = 0x108000; // 1MB + 32 KB
        const int       SystemUpdateTaskSaveDataFlags           = 0;
        const int64_t   NetworkInstallTaskSaveDataSize          = 0x1008000; // 16MB + 32 KB
        const int64_t   NetworkInstallTaskSaveDataJournalSize   = 0x1008000; // 16MB + 32 KB
        const int64_t   NetworkInstallTaskSaveDataFlags         = 0;
        const int64_t   ApplyDeltaTaskSaveDataSize              = 0x280000; // 2.5MiB
        const int64_t   ApplyDeltaTaskSaveDataJournalSize       = 0x280000; // 2.5MiB
        const int64_t   ApplyDeltaTaskSaveDataFlags             = 0;
        const int64_t   TemporaryFileSaveDataSize               = 0x208000; // 2MB + 32 KB
        const int64_t   TemporaryFileSaveDataJournalSize        = 0x208000; // 2MB + 32 KB
        const int       TemporaryFileSaveDataFlags              = 0;
        const int64_t   LocalCommunicationReceiveApplicationTaskSaveDataSize              = 0x208000; // 2MB + 32 KB
        const int64_t   LocalCommunicationReceiveApplicationTaskSaveDataJournalSize       = 0x208000; // 2MB + 32 KB
        const int64_t   LocalCommunicationReceiveApplicationTaskSaveDataFlags             = 0;
        const int64_t   LocalCommunicationReceiveSystemUpdateTaskSaveDataSize             = 0x108000; // 1MB + 32 KB
        const int64_t   LocalCommunicationReceiveSystemUpdateTaskSaveDataJournalSize      = 0x108000; // 1MB + 32 KB
        const int64_t   LocalCommunicationReceiveSystemUpdateTaskSaveDataFlags            = 0;

        const ncm::ApplicationId StressTaskDataId = {0x010023d002b98000}; // Application ID を記載

        typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

        template <typename IAsyncT, typename EmplacedRefT>
        Result RunAndGetAsyncObject(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<IAsyncT>> outAsync, EmplacedRefT emplacedRef) NN_NOEXCEPT
        {
            NN_RESULT_DO(emplacedRef.GetImpl().Run());
            *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
            *outAsync = emplacedRef;

            NN_RESULT_SUCCESS;
        }

        template <typename TaskHolderT, typename TaskInfoT>
        Result VerifyDuplicatedTaskNotExist(TaskHolderT* holder, ncm::ApplicationId id) NN_NOEXCEPT
        {
            auto holderCount = holder->Count();
            for (int i = 0; i < holderCount; i++)
            {
                auto taskId = holder->GetId(i);

                TaskInfoT info;
                NN_RESULT_DO(holder->GetInfo(&info, taskId));
                NN_RESULT_THROW_UNLESS(info.applicationId != id, ResultApplicationAlreadyHasTask());
            }
            NN_RESULT_SUCCESS;
        }
        template <typename TaskDataStoreT>
        void CleanupDataStore(TaskDataStoreT* store, util::Uuid uuid, bool doCleanup) NN_NOEXCEPT
        {
            if (doCleanup)
            {
                auto result = store->Destroy(uuid);
                if (result.IsFailure())
                {
                    NN_DETAIL_NIM_TRACE("Clenaup failed: %08x\n", result.GetInnerValueForDebug());
                }
            }
        }
        struct SystemUpdateTaskMeta
        {
            ncm::ContentMetaKey key;
            Bit32 config;
            Bit8 reserved[4];
        };
    }

    Result NetworkInstallManagerServer::Initialize(DeviceContext* deviceContext, DeviceAccountStore* deviceAccountStore, bool isMaintenanceMode, bool isOvlnEnabled) NN_NOEXCEPT
    {
        m_DeviceContext = deviceContext;
        m_DeviceAccountStore = deviceAccountStore;

        if (isOvlnEnabled)
        {
            NN_RESULT_DO(ovln::InitializeSenderWithQueue(&m_Sender, MaxNetworkInstallTaskCount));
        }

        if (isMaintenanceMode)
        {
            NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] Delete Save for System Update\n");
            NN_RESULT_DO(TaskDataStore::DeleteSaveData(0x8000000000000070ull));
        }

        NN_RESULT_DO(m_SystemUpdateTaskDataStore.Initialize("nim_sys", 0x8000000000000070ull, SystemUpdateTaskSaveDataSize, SystemUpdateTaskSaveDataJournalSize, SystemUpdateTaskSaveDataFlags));
        NN_RESULT_DO(LoadFromSystemUpdateTaskStore());

        if (!isMaintenanceMode)
        {
            NN_RESULT_DO(m_NetworkInstallTaskDataStore.Initialize("nim_net", 0x8000000000000071ull, NetworkInstallTaskSaveDataSize, NetworkInstallTaskSaveDataJournalSize, NetworkInstallTaskSaveDataFlags));
            NN_RESULT_DO(LoadFromNetworkInstallTaskStore());

            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.Initialize("nim_delta", 0x8000000000000074ull, ApplyDeltaTaskSaveDataSize, ApplyDeltaTaskSaveDataJournalSize, ApplyDeltaTaskSaveDataFlags));
            NN_RESULT_DO(LoadFromApplyDeltaTaskStore());

            NN_RESULT_DO(m_TemporaryDataStore.Initialize("nim_tmp", 0x8000000000000072ull, TemporaryFileSaveDataSize, TemporaryFileSaveDataJournalSize, TemporaryFileSaveDataFlags));

            NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.Initialize("nim_local", 0x8000000000000076ull, LocalCommunicationReceiveApplicationTaskSaveDataSize, LocalCommunicationReceiveApplicationTaskSaveDataJournalSize, LocalCommunicationReceiveApplicationTaskSaveDataFlags));
            NN_RESULT_DO(LoadFromLocalCommunicationTaskStoreToCleanup());

            NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskDataStore.Initialize("nim_lsys", 0x8000000000000077ull, LocalCommunicationReceiveSystemUpdateTaskSaveDataSize, LocalCommunicationReceiveSystemUpdateTaskSaveDataJournalSize, LocalCommunicationReceiveSystemUpdateTaskSaveDataFlags));
            NN_RESULT_DO(LoadFromLocalCommunicationTaskStoreToCleanup());
        }

        SetupStressTask();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::SetupStressTask() NN_NOEXCEPT
    {
        {
            // download
            bool isEnabled = false;
            auto valueSize = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "systemupdate", "enable_background_download_stress_testing");
            if (valueSize != sizeof(isEnabled))
            {
                NN_DETAIL_NIM_WARN("[NetworkInstallManagerServer] Faild to get system settings. (enable_background_download_stress_testing)\n");
                isEnabled = false;
            }

            if (isEnabled)
            {
                ncm::StorageId storageId = ncm::StorageId::BuiltInSystem;
                int storageValue = 0;
                valueSize = nn::settings::fwdbg::GetSettingsItemValue(&storageValue, sizeof(storageValue), "systemupdate", "background_download_stress_testing_storage");
                if (valueSize != sizeof(storageValue))
                {
                    NN_DETAIL_NIM_WARN("[NetworkInstallManagerServer] Faild to get system settings. (background_download_stress_testing_storage)\n");
                }
                else
                {
                    storageId = static_cast<ncm::StorageId>(storageValue);
                }

                NN_RESULT_DO(m_BackgroundDownloadStressTask.Initialize(BackgroundDownloadStressTask::DefaultDownloadUrl, storageId, m_DeviceContext));
            }
        }
        {
            // apply delta
            int storage = 0;
            auto valueSize = nn::settings::fwdbg::GetSettingsItemValue(&storage, sizeof(storage), "nim.install", "apply_delta_stress_storage");
            if (valueSize == sizeof(storage))
            {
                ncm::StorageId storageId = static_cast<ncm::StorageId>(storage);
                if (storageId == ncm::StorageId::BuiltInUser || storageId == ncm::StorageId::SdCard)
                {
                    auto result = m_BackgroundApplyDeltaStressTask.Initialize(StressTaskDataId, storageId);
                    if (result.IsFailure())
                    {
                        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] Stress initialize failed %08x\n", result.GetInnerValueForDebug());
                    }
                }
            }
        }


        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::LoadFromSystemUpdateTaskStore() NN_NOEXCEPT
    {
        auto iterator = m_SystemUpdateTaskDataStore.CreateIdIterator();
        while (NN_STATIC_CONDITION(true))
        {
            util::optional<util::Uuid> uuid = iterator.Next();
            if (!uuid)
            {
                break;
            }

            SystemUpdateTaskMeta meta;
            size_t outSize;
            NN_RESULT_DO(m_SystemUpdateTaskDataStore.GetMeta(&outSize, *uuid, &meta, sizeof(meta)));

            TaskDataStore::Path path;
            NN_RESULT_DO(m_SystemUpdateTaskDataStore.GetDataFilePath(&path, *uuid));

            SystemUpdateTaskId id;
            NN_RESULT_DO(CreateSystemUpdateTaskImpl(&id, *uuid, meta.key, meta.config, path));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::LoadFromNetworkInstallTaskStore() NN_NOEXCEPT
    {
        auto iterator = m_NetworkInstallTaskDataStore.CreateIdIterator();
        while (NN_STATIC_CONDITION(true))
        {
            util::optional<util::Uuid> uuid = iterator.Next();
            if (!uuid)
            {
                break;
            }

            TaskDataStore::Path metaPath;
            NN_RESULT_DO(m_NetworkInstallTaskDataStore.GetMetaFilePath(&metaPath, *uuid));

            TaskDataStore::Path dataPath;
            NN_RESULT_DO(m_NetworkInstallTaskDataStore.GetDataFilePath(&dataPath, *uuid));

            NetworkInstallTaskId id;
            NN_RESULT_DO(CreateNetworkInstallTaskImpl(&id, *uuid, metaPath, dataPath));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::LoadFromApplyDeltaTaskStore() NN_NOEXCEPT
    {
        auto iterator = m_ApplyDeltaTaskDataStore.CreateIdIterator();
        while (NN_STATIC_CONDITION(true))
        {
            util::optional<util::Uuid> uuid = iterator.Next();
            if (!uuid)
            {
                break;
            }
            TaskDataStore::Path metaPath;
            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetMetaFilePath(&metaPath, *uuid));

            TaskDataStore::Path dataPath;
            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetDataFilePath(&dataPath, *uuid));

            ApplyDeltaTaskId id;
            NN_RESULT_DO(CreateApplyDeltaTaskImpl(&id, *uuid, metaPath, dataPath));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::LoadFromLocalCommunicationTaskStoreToCleanup() NN_NOEXCEPT
    {
        auto iterator = m_LocalCommunicationReceiveApplicationTaskDataStore.CreateIdIterator();
        while (NN_STATIC_CONDITION(true))
        {
            util::optional<util::Uuid> uuid = iterator.Next();
            if (!uuid)
            {
                break;
            }
            TaskDataStore::Path metaPath;
            NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.GetMetaFilePath(&metaPath, *uuid));

            TaskDataStore::Path dataPath;
            NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.GetDataFilePath(&dataPath, *uuid));

            LocalCommunicationReceiveApplicationTaskId id;
            NN_RESULT_DO(CreateLocalCommunicationReceiveApplicationTaskForCleanup(&id, *uuid, metaPath, dataPath));
        }

        // 残ったプレースホルダを消すためにタスクを削除する
        LocalCommunicationReceiveApplicationTaskId idList[MaxLocalCommunicationReceiveApplicationTaskCount];
        auto count = m_LocalCommunicationReceiveApplicationTaskHolder.List(idList, MaxLocalCommunicationReceiveApplicationTaskCount);
        for (auto i = 0; i < count; i++)
        {
            NN_RESULT_DO(DestroyLocalCommunicationReceiveApplicationTask(idList[i]));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::LoadFromLocalCommunicationSystemUpdateTaskStoreToCleanup() NN_NOEXCEPT
    {
        auto iterator = m_LocalCommunicationReceiveSystemUpdateTaskDataStore.CreateIdIterator();
        while (NN_STATIC_CONDITION(true))
        {
            util::optional<util::Uuid> uuid = iterator.Next();
            if (!uuid)
            {
                break;
            }

            SystemUpdateTaskMeta meta;
            size_t outSize;
            NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskDataStore.GetMeta(&outSize, *uuid, &meta, sizeof(meta)));

            TaskDataStore::Path path;
            NN_RESULT_DO(m_SystemUpdateTaskDataStore.GetDataFilePath(&path, *uuid));

            LocalCommunicationReceiveSystemUpdateTaskId id;
            NN_RESULT_DO(CreateLocalCommunicationReceiveSystemUpdateTaskForCleanup(&id, *uuid, meta.key, meta.config, path));
        }

        // 残ったプレースホルダを消すためにタスクを削除する
        LocalCommunicationReceiveSystemUpdateTaskId idList[MaxLocalCommunicationReceiveSystemUpdateTaskCount];
        auto count = m_LocalCommunicationReceiveSystemUpdateTaskHolder.List(idList, MaxLocalCommunicationReceiveSystemUpdateTaskCount);
        for (auto i = 0; i < count; i++)
        {
            NN_RESULT_DO(DestroyLocalCommunicationReceiveSystemUpdateTask(idList[i]));
        }

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateSystemUpdateTask(sf::Out<SystemUpdateTaskId> outValue, const ncm::ContentMetaKey& key, Bit32 config) NN_NOEXCEPT
    {
        SystemUpdateTaskMeta meta = { key, config };

        auto uuid = util::GenerateUuid();
        NN_RESULT_DO(m_SystemUpdateTaskDataStore.Create(uuid, &meta, sizeof(meta)));

        bool success = false;
        NN_UTIL_SCOPE_EXIT
        {
            CleanupDataStore(&m_SystemUpdateTaskDataStore, uuid, !success);
        };

        TaskDataStore::Path path;
        NN_RESULT_DO(m_SystemUpdateTaskDataStore.GetDataFilePath(&path, uuid));

        NN_RESULT_DO(CreateSystemUpdateTaskImpl(outValue.GetPointer(), uuid, key, config, path));
        success = true;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateSystemUpdateTaskImpl(nim::SystemUpdateTaskId* outValue, const util::Uuid& uuid, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT
    {
        SystemUpdateTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_SystemUpdateTaskHolder.FindVacantHolder(&vacant));

        TaskDataStore::Path metaPath;
        NN_RESULT_DO(m_SystemUpdateTaskDataStore.GetMetaFilePath(&metaPath, uuid));

        NN_RESULT_DO(vacant->Initialize(m_DeviceContext, uuid, key, config, dataFilePath));
        *outValue = vacant->GetId();

        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] System update task %s created\n", uuid.ToString(uuidString, sizeof(uuidString)));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::DestroySystemUpdateTask(const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_SystemUpdateTaskHolder.Destroy(id));
        NN_RESULT_DO(m_SystemUpdateTaskDataStore.Destroy(id.uuid));

        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] System update task %s destroyed\n", id.uuid.ToString(uuidString, sizeof(uuidString)));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListSystemUpdateTask(sf::Out<std::int32_t> outCount, const sf::OutArray<SystemUpdateTaskId>& outList) NN_NOEXCEPT
    {
        *outCount = m_SystemUpdateTaskHolder.List(outList.GetData(), static_cast<int>(outList.GetLength()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestSystemUpdateTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] Request system update task %s run\n", id.uuid.ToString(uuidString, sizeof(uuidString)));

        SystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_SystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->ResetLastResult();

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncNetworkSystemUpdateTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetSystemUpdateTaskInfo(sf::Out<nim::SystemUpdateTaskInfo> outValue, const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return m_SystemUpdateTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::CommitSystemUpdateTask(const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return m_SystemUpdateTaskHolder.Commit(id);
    }

    Result NetworkInstallManagerServer::CreateNetworkInstallTask(sf::Out<nim::NetworkInstallTaskId> outValue, ncm::ApplicationId appId, const sf::InArray<ncm::ContentMetaKey>& keyList, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT
    {
        auto uuid = util::GenerateUuid();
        NN_RESULT_DO(m_NetworkInstallTaskDataStore.Create(uuid, nullptr, 0));

        bool success = false;
        NN_UTIL_SCOPE_EXIT
        {
            CleanupDataStore(&m_NetworkInstallTaskDataStore, uuid, !success);
        };

        TaskDataStore::Path metaPath;
        NN_RESULT_DO(m_NetworkInstallTaskDataStore.GetMetaFilePath(&metaPath, uuid));

        NN_RESULT_DO(FileNetworkInstallTaskMeta::Create(metaPath, appId, keyList.GetData(), static_cast<int>(keyList.GetLength()), storage, config));

        TaskDataStore::Path dataPath;
        NN_RESULT_DO(m_NetworkInstallTaskDataStore.GetDataFilePath(&dataPath, uuid));

        NN_RESULT_DO(CreateNetworkInstallTaskImpl(outValue.GetPointer(), uuid, metaPath, dataPath));
        success = true;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateNetworkInstallTaskImpl(nim::NetworkInstallTaskId* outValue, const util::Uuid& uuid, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->Initialize(m_DeviceContext, m_DeviceAccountStore, uuid, metaFilePath, dataFilePath, &m_Sender));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::DestroyNetworkInstallTask(const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_NetworkInstallTaskHolder.Destroy(id));
        NN_RESULT_DO(m_NetworkInstallTaskDataStore.Destroy(id.uuid));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::NetworkInstallTaskId>& outList) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_NetworkInstallTaskDataStore.CreateIdIterator();

        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            NetworkInstallTaskId taskId = { *uuid };
            outList[count] = taskId;
            count++;
        }

        *outCount = count;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestNetworkInstallTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncNetworkInstallTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetNetworkInstallTaskInfo(sf::Out<nim::NetworkInstallTaskInfo> outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return m_NetworkInstallTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTaskContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, std::int32_t offset, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTaskCommittedContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, std::int32_t offset, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset, ncm::ListContentMetaKeyFilter::Committed));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTaskNotCommittedContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, std::int32_t offset, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset, ncm::ListContentMetaKeyFilter::NotCommitted));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTaskContentMetaFromInstallMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::ContentMetaKey>& outList, std::int32_t offset, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKeyFromInstallMeta(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CommitNetworkInstallTask(const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return m_NetworkInstallTaskHolder.Commit(id);
    }

    Result NetworkInstallManagerServer::CommitNetworkInstallTaskPartially(const sf::InArray<ncm::StorageContentMetaKey>& keyList, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().Commit(keyList.GetData(), static_cast<int>(keyList.GetLength())));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListApplicationNetworkInstallTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::NetworkInstallTaskId>& outList, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_NetworkInstallTaskDataStore.CreateIdIterator();
        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            NetworkInstallTaskId taskId = { *uuid };
            NetworkInstallTaskInfo info;
            NN_RESULT_DO(m_NetworkInstallTaskHolder.GetInfo(&info, taskId));

            if (info.applicationId == appId)
            {
                outList[count] = taskId;
                count++;
            }
        }

        *outCount = count;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::SetNetworkInstallTaskAttribute(const NetworkInstallTaskIdAttribute& idAttribute) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, idAttribute.id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->SetAttribute(idAttribute.attribute));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLatestSystemUpdateMeta(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncSystemUpdateMetaImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestApplicationControl(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync, ncm::ApplicationId applicationId, std::uint32_t version) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncApplicationControlImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, applicationId, version));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLatestApplicationControl(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync, ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncApplicationControlImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().InitializeLatest(m_DeviceContext, applicationId));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLatestVersion(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::nim::detail::IAsyncValue>> outAsync, nn::ncm::ApplicationId appId) NN_NOEXCEPT

    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncLatestVersionImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, appId));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::AddNetworkInstallTaskContentMeta(const sf::InArray<ncm::ContentMetaKey>& list, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().Add(list.GetData(), static_cast<int>(list.GetLength()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetDownloadedSystemDataPath(sf::Out<ncm::Path> outValue, ncm::SystemDataId dataId, const nim::SystemUpdateTaskId& taskId) NN_NOEXCEPT
    {
        SystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_SystemUpdateTaskHolder.FindHolder(&holder, taskId));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().GetDownloadedSystemDataPath(outValue.GetPointer(), dataId));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestGameCardRegistrationStatus(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync, const nn::sf::InArray<nn::ncm::ApplicationId>& appIdList, const sf::InBuffer& token, const sf::InBuffer& cert) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(token.GetPointerUnsafe()[token.GetSize() - 1] == '\0');

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncGameCardRegistrationStatusImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, appIdList.GetData(), static_cast<int>(appIdList.GetLength()), static_cast<const char*>(token.GetPointerUnsafe()), cert.GetPointerUnsafe(), cert.GetSize()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CalculateNetworkInstallTaskRequiredSize(sf::Out<std::int64_t> outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().CalculateRequiredSize(outValue.GetPointer()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CalculateNetworkInstallTaskContentsSize(sf::Out<std::int64_t> outValue, const nim::NetworkInstallTaskId& id, const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().CalculateContentsSize(outValue.GetPointer(), key, storageId));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListNetworkInstallTaskOccupiedSize(sf::Out<int> outCount, sf::OutArray<ncm::InstallTaskOccupiedSize> outList, int offset, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().ListOccupiedSize(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::IsExFatDriverIncluded(sf::Out<bool> outValue, const nim::SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        SystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_SystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        bool includesExFatDriver;
        NN_RESULT_DO(holder->GetTask().IncludesExFatDriver(&includesExFatDriver));

        *outValue = includesExFatDriver;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetBackgroundDownloadStressTaskInfo(sf::Out<BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT
    {
        BackgroundDownloadStressTaskInfo info;
        m_BackgroundDownloadStressTask.GetInfo(&info);
        outValue.Set(info);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetBackgroundApplyDeltaStressTaskInfo(sf::Out<BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT
    {
        BackgroundDownloadStressTaskInfo info;
        m_BackgroundApplyDeltaStressTask.GetInfo(&info);
        outValue.Set(info);
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestDeviceAuthenticationToken(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, AsyncDeviceAuthenticationTokenImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestRegisterGameCard(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nn::sf::InArray<nn::ncm::ApplicationId>& appIdList, const sf::InBuffer& token, const sf::InBuffer& cert, nim::GameCardRegistrationStatus status) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(token.GetPointerUnsafe()[token.GetSize() - 1] == '\0');

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncRegisterGameCardImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, appIdList.GetData(), static_cast<int>(appIdList.GetLength()), static_cast<const char*>(token.GetPointerUnsafe()), cert.GetPointerUnsafe(), cert.GetSize(), status));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestRegisterNotificationToken(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::nim::detail::IAsyncResult>> outAsync, const nn::npns::NotificationToken& notificationToken) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncRegisterNotificationTokenImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultAllocationMemoryFailed());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(notificationToken, m_DeviceContext));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestDownloadTaskList(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outAsync, const nn::nim::ETag& eTag) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, AsyncDownloadTaskListImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_TemporaryDataStore.GenerateTemporaryFilePath(), eTag));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestVersionList(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outAsync, const nn::nim::ETag& eTag) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, AsyncVersionListImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_TemporaryDataStore.GenerateTemporaryFilePath(), eTag));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::CreateApplyDeltaTask(sf::Out<nim::ApplyDeltaTaskId> outValue, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, const sf::InArray<ncm::ContentMetaKey>& keyList, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(keyList.GetLength() > 0, ResultEmptyKeyList());
        NN_RESULT_DO((VerifyDuplicatedTaskNotExist<ApplyDeltaTaskHolder, ApplyDeltaTaskInfo>(&m_ApplyDeltaTaskHolder, appId)));

        nim::ApplyDeltaTaskId taskId;
        {
            auto uuid = util::GenerateUuid();
            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.Create(uuid, nullptr, 0));

            bool success = false;
            NN_UTIL_SCOPE_EXIT { CleanupDataStore(&m_ApplyDeltaTaskDataStore, uuid, !success); };

            TaskDataStore::Path metaPath;
            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetMetaFilePath(&metaPath, uuid));

            NN_RESULT_DO(FileApplyDownloadedDeltaTaskMeta::Create(metaPath, appId, sourceKey, keyList.GetData(), static_cast<int>(keyList.GetLength()), storageId));

            TaskDataStore::Path dataPath;
            NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetDataFilePath(&dataPath, uuid));
            // TODO: DataStore で一度 dataPath のファイルが作られているのを、されないようにしたい
            NN_RESULT_DO(FileApplyDownloadedDeltaTaskData::Create(dataPath, static_cast<int>(keyList.GetLength())));

            NN_RESULT_DO(CreateApplyDeltaTaskImpl(&taskId, uuid, metaPath, dataPath));
            success = true;
        }

        // Create の時点で、最初の処理も行う
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, taskId));
        NN_RESULT_TRY(holder->GetTask().InitialPrepare())
            // Initial Prepare 時にエラーとなっても、タスク自体は作成できているのでエラーとしない
            // ※タスクは NotEnoughSpace / Fatal 状態になっている
            NN_RESULT_CATCH_ALL {}
        NN_RESULT_END_TRY

        *outValue = taskId;

        NN_RESULT_SUCCESS;
    }

    // ダウンロードタスクから適用タスクを作成する
    // 失敗したときにパッチ間差分を削除するのは呼び出し元の責任
    Result NetworkInstallManagerServer::CreateApplyDeltaTaskFromDownloadTask(sf::Out<nim::ApplyDeltaTaskId> outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskInfo info;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.GetInfo(&info, id));

        // commit 済みのことを確認
        NN_RESULT_THROW_UNLESS(info.progress.state == ncm::InstallProgressState::Commited, ResultUnexpectedTaskState());
        NN_RESULT_DO((VerifyDuplicatedTaskNotExist<ApplyDeltaTaskHolder, ApplyDeltaTaskInfo>(&m_ApplyDeltaTaskHolder, info.applicationId)));

        nim::ApplyDeltaTaskId taskId;
        {
            // ダウンロードタスクの情報から、パッチ間差分の情報を構築
            ncm::StorageContentMetaKey keys[MaxUpdateStep];
            int countKeys;
            NN_RESULT_DO(CreateApplyDeltaTaskKeys(&countKeys, keys, MaxUpdateStep, id));
            NN_RESULT_THROW_UNLESS(countKeys > 0, ResultApplyDeltaNotNeeded());

            ncm::StorageContentMetaKey skey;
            NN_RESULT_DO(FindTargetPatch(&skey, info.applicationId));
            {
                auto uuid = util::GenerateUuid();
                NN_RESULT_DO(m_ApplyDeltaTaskDataStore.Create(uuid, nullptr, 0));

                bool success = false;
                NN_UTIL_SCOPE_EXIT { CleanupDataStore(&m_ApplyDeltaTaskDataStore, uuid, !success); };

                TaskDataStore::Path metaPath;
                NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetMetaFilePath(&metaPath, uuid));

                NN_RESULT_DO(FileApplyDownloadedDeltaTaskMeta::Create(metaPath, info.applicationId, skey.key, skey.storageId, keys, countKeys));

                TaskDataStore::Path dataPath;
                NN_RESULT_DO(m_ApplyDeltaTaskDataStore.GetDataFilePath(&dataPath, uuid));
                // TODO: DataStore で一度 dataPath のファイルが作られているのを、されないようにしたい
                NN_RESULT_DO(FileApplyDownloadedDeltaTaskData::Create(dataPath, countKeys));

                NN_RESULT_DO(CreateApplyDeltaTaskImpl(&taskId, uuid, metaPath, dataPath));
                success = true;
            }
        }

        // Create の時点で、最初の処理も行う
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, taskId));
        NN_RESULT_TRY(holder->GetTask().InitialPrepare())
            // Initial Prepare 時にエラーとなっても、タスク自体は作成できているのでエラーとしない
            // ※タスクは NotEnoughSpace / Fatal 状態になっている
            NN_RESULT_CATCH_ALL {}
        NN_RESULT_END_TRY

        *outValue = taskId;

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::CreateApplyDeltaTaskKeys(int* outKeys, ncm::StorageContentMetaKey* keys, int countKeys, NetworkInstallTaskId taskId) NN_NOEXCEPT
    {
        const int ReadOne = 16;
        ncm::StorageContentMetaKey skeys[ReadOne];

        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, taskId));

        int index = 0;
        int offset = 0;
        for(;;)
        {
            int count;
            NN_RESULT_DO(holder->GetTask().ListContentMetaKey(&count, skeys, ReadOne, offset));
            for (int i = 0; i < count; ++i)
            {
                if (skeys[i].key.type == ncm::ContentMetaType::Patch && skeys[i].key.installType == ncm::ContentInstallType::FragmentOnly)
                {
                    keys[index] = skeys[i];
                    index++;
                    NN_RESULT_THROW_UNLESS(index < countKeys, ResultTooManyDelta());
                }
            }
            offset += count;
            if (count != ReadOne)
            {
                break;
            }
        }
        *outKeys = index;
        // バージョン昇順へのソート
        std::sort(keys, keys + index);

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::FindTargetPatch(ncm::StorageContentMetaKey* outKey, ncm::ApplicationId id) NN_NOEXCEPT
    {
        // メモ: 差分ダウンロード時に参照したパッチとは異なる可能性があり、
        //       そうすると適用時にエラーになる可能性がある
        auto storageList = ncm::GetStorageList(ncm::StorageId::Any);
        for (int i = 0; i < storageList.Count(); ++i)
        {
            auto storageId = storageList[i];
            ncm::ContentMetaDatabase db;
            if (ncm::OpenContentMetaDatabase(&db, storageId).IsFailure())
            {
                continue;
            }

            ncm::ContentMetaKey patch;
            auto listcount = db.ListContentMeta(&patch, 1, ncm::ContentMetaType::Patch, id);
            if (listcount.listed == 1)
            {
                // W/A for SIGLO-63944: 最新版を使うため、改めて取得し直す
                ncm::ContentMetaKey latest;
                NN_RESULT_DO(db.GetLatest(&latest, patch.id));
                *outKey = {latest, storageId};
                NN_RESULT_SUCCESS;
            }
        }
        // 見つからない場合は、ここでは StorageId::None の結果を返す
        // タスクを作成して、タスクとしてのエラーにするため
        {
            ncm::ContentMetaKey patch = {0, 0, ncm::ContentMetaType::Patch};
            *outKey = {patch, ncm::StorageId::None};
            NN_RESULT_SUCCESS;
        }
    }

    Result NetworkInstallManagerServer::CreateApplyDeltaTaskImpl(nim::ApplyDeltaTaskId* outValue, const util::Uuid& uuid, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindVacantHolder(&vacant));
        NN_RESULT_DO(vacant->Initialize(uuid, metaFilePath, dataFilePath));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::DestroyApplyDeltaTask(const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.Destroy(id));
        NN_RESULT_DO(m_ApplyDeltaTaskDataStore.Destroy(id.uuid));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListApplyDeltaTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::ApplyDeltaTaskId>& outList) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_ApplyDeltaTaskDataStore.CreateIdIterator();

        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            ApplyDeltaTaskId taskId = { *uuid };
            outList[count] = taskId;
            count++;
        }

        *outCount = count;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListApplicationApplyDeltaTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::ApplyDeltaTaskId>& outList, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_ApplyDeltaTaskDataStore.CreateIdIterator();
        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            ApplyDeltaTaskId taskId = { *uuid };
            ApplyDeltaTaskInfo info;
            NN_RESULT_DO(m_ApplyDeltaTaskHolder.GetInfo(&info, taskId));

            if (info.applicationId == appId)
            {
                outList[count] = taskId;
                count++;
            }
        }

        *outCount = count;

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestApplyDeltaTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncApplyDeltaTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::ClearNotEnoughSpaceStateOfApplyDeltaTask(const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_TRY(holder->GetTask().ClearNotEnoughSpaceState())
            // InitialPrepare でエラーになっても、Success を返す
            NN_RESULT_CATCH_ALL {}
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::GetApplyDeltaTaskInfo(sf::Out<nim::ApplyDeltaTaskInfo> outValue, const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.GetInfo(outValue.GetPointer(), id));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::ListApplyDeltaTaskContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, std::int32_t offset, const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::CommitApplyDeltaTask(const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.Commit(id));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::CalculateApplyDeltaTaskRequiredSize(sf::Out<std::int64_t> outValue, const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().CalculateRequiredSize(outValue.GetPointer()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CalculateApplyDeltaTaskOccupiedSize(sf::Out<std::int64_t> outValue, const nim::ApplyDeltaTaskId& id, ncm::StorageId storageId) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().CalculateOccupiedSize(outValue.GetPointer(), storageId));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetApplyDeltaTaskRequiredStorage(sf::Out<ncm::StorageId> outValue, const nim::ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        ApplyDeltaTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_ApplyDeltaTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetRequiredStorage(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::PrepareShutdown() NN_NOEXCEPT
    {
        m_NetworkInstallTaskHolder.PrepareShutdown();
        m_ApplyDeltaTaskHolder.PrepareShutdown();
        m_LocalCommunicationReceiveApplicationTaskHolder.PrepareShutdown();
        m_LocalCommunicationSendApplicationTaskHolder.PrepareShutdown();

        {
            ncm::ContentStorage storage;
            if (ncm::OpenContentStorage(&storage, ncm::StorageId::SdCard).IsSuccess())
            {
                auto result = storage.FlushPlaceHolder();
                if (result.IsFailure())
                {
                    NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] failed to flush placeholder %08x\n", result.GetInnerValueForDebug());
                }
            }
        }

        NN_RESULT_DO(m_NetworkInstallTaskDataStore.Commit());
        NN_RESULT_DO(m_ApplyDeltaTaskDataStore.Commit());

        // LocalCommunication 通信のタスクは復旧させないが、ゴミを起動時に削除するため、情報を残す
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.Commit());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::PrepareShutdownForSystemUpdate() NN_NOEXCEPT
    {
        m_SystemUpdateTaskHolder.PrepareShutdown();
        NN_RESULT_DO(m_SystemUpdateTaskDataStore.Commit());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::FindMaxRequiredApplicationVersionOfTask(sf::Out<std::uint32_t> outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        uint32_t version;
        NN_RESULT_DO(holder->GetTask().FindMaxRequiredApplicationVersion(&version));

        *outValue = version;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::FindMaxRequiredSystemVersionOfTask(sf::Out<std::uint32_t> outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        uint32_t version;
        NN_RESULT_DO(holder->GetTask().FindMaxRequiredSystemVersion(&version));

        *outValue = version;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetNetworkInstallTaskErrorContext(sf::Out<err::ErrorContext> outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        NetworkInstallTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_NetworkInstallTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetErrorContext(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveApplicationTask(sf::Out<nim::LocalCommunicationReceiveApplicationTaskId> outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId appId, const sf::InArray<ncm::ContentMetaKey>& keyList, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT
    {
        auto uuid = util::GenerateUuid();
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.Create(uuid, nullptr, 0));

        bool success = false;
        NN_UTIL_SCOPE_EXIT
        {
            CleanupDataStore(&m_LocalCommunicationReceiveApplicationTaskDataStore, uuid, !success);
        };

        TaskDataStore::Path metaPath;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.GetMetaFilePath(&metaPath, uuid));

        NN_RESULT_DO(FileNetworkInstallTaskMeta::Create(metaPath, appId, keyList.GetData(), static_cast<int>(keyList.GetLength()), storage, config));

        TaskDataStore::Path dataPath;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.GetDataFilePath(&dataPath, uuid));

        NN_RESULT_DO(CreateLocalCommunicationReceiveApplicationTaskImpl(outValue.GetPointer(), uuid, ipv4, port, metaPath, dataPath));
        success = true;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveApplicationTaskImpl(nim::LocalCommunicationReceiveApplicationTaskId *outValue, const util::Uuid &uuid, uint32_t ipv4, uint16_t port, const char *metaFilePath, const char *dataFilePath) NN_NOEXCEPT
    {
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->Initialize(uuid, ipv4, port, metaFilePath, dataFilePath));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveApplicationTaskForCleanup(nim::LocalCommunicationReceiveApplicationTaskId *outValue, const util::Uuid &uuid, const char *metaFilePath, const char *dataFilePath) NN_NOEXCEPT
    {
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->InitializeForCleanup(uuid, metaFilePath, dataFilePath));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::DestroyLocalCommunicationReceiveApplicationTask(const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.Destroy(id));
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskDataStore.Destroy(id.uuid));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListLocalCommunicationReceiveApplicationTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::LocalCommunicationReceiveApplicationTaskId>& outList) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_LocalCommunicationReceiveApplicationTaskDataStore.CreateIdIterator();

        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            LocalCommunicationReceiveApplicationTaskId taskId = { *uuid };
            outList[count] = taskId;
            count++;
        }

        *outCount = count;
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLocalCommunicationReceiveApplicationTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncLocalCommunicationReceiveApplicationTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationReceiveApplicationTaskInfo(sf::Out<nim::LocalCommunicationReceiveApplicationTaskInfo> outValue, const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationReceiveApplicationTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::ListLocalCommunicationReceiveApplicationTaskContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, std::int32_t offset, const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_DO(holder->GetTask().ListContentMetaKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CommitLocalCommunicationReceiveApplicationTask(const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationReceiveApplicationTaskHolder.Commit(id);
    }

    Result NetworkInstallManagerServer::ListApplicationLocalCommunicationReceiveApplicationTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::LocalCommunicationReceiveApplicationTaskId>& outList, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto iter = m_LocalCommunicationReceiveApplicationTaskDataStore.CreateIdIterator();
        int count = 0;
        while (count < listCount)
        {
            auto uuid = iter.Next();
            if (!uuid)
            {
                break;
            }

            LocalCommunicationReceiveApplicationTaskId taskId = { *uuid };
            LocalCommunicationReceiveApplicationTaskInfo info;
            NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.GetInfo(&info, taskId));

            if (info.applicationId == appId)
            {
                outList[count] = taskId;
                count++;
            }
        }

        *outCount = count;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationReceiveApplicationTaskErrorContext(sf::Out<err::ErrorContext> outValue, const nim::LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetErrorContext(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationSendApplicationTask(sf::Out<nim::LocalCommunicationSendApplicationTaskId> outValue, uint32_t ipv4, uint16_t port, const sf::InArray<ncm::StorageContentMetaKey>& keyList, ncm::ApplicationId id) NN_NOEXCEPT
    {
        auto uuid = util::GenerateUuid();

        NN_RESULT_DO(CreateLocalCommunicationSendApplicationTaskImpl(outValue.GetPointer(), uuid, ipv4, port, keyList.GetData(), static_cast<int>(keyList.GetLength()), id));

        NN_RESULT_SUCCESS;

    }

    Result NetworkInstallManagerServer::CalculateLocalCommunicationReceiveApplicationTaskRequiredSize(sf::Out<int64_t> outValue, const nim::LocalCommunicationReceiveApplicationTaskId id) NN_NOEXCEPT
    {
        LocalCommunicationReceiveApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().CalculateRequiredSize(outValue.GetPointer()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationSendApplicationTaskImpl(nim::LocalCommunicationSendApplicationTaskId* outValue, const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::StorageContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT
    {
        LocalCommunicationSendApplicationTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationSendApplicationTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->Initialize(uuid, ipv4, port, keyList, listCount, id));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLocalCommunicationSendApplicationTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nim::LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationSendApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationSendApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncLocalCommunicationSendApplicationTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationSendApplicationTaskInfo(sf::Out<nim::LocalCommunicationSendApplicationTaskInfo> outValue, const nim::LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationSendApplicationTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::DestroyLocalCommunicationSendApplicationTask(const nim::LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LocalCommunicationSendApplicationTaskHolder.Destroy(id));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationSendApplicationTaskErrorContext(sf::Out<err::ErrorContext> outValue, const nim::LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationSendApplicationTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationSendApplicationTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetErrorContext(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListApplicationLocalCommunicationSendApplicationTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::LocalCommunicationSendApplicationTaskId>& outList, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        auto listCount = static_cast<int>(outList.GetLength());
        auto holderCount = m_LocalCommunicationSendApplicationTaskHolder.Count();
        int count = 0;

        for (int i = 0; i < holderCount; i++)
        {
            auto id = m_LocalCommunicationSendApplicationTaskHolder.GetId(i);

            LocalCommunicationSendApplicationTaskHolder::TaskHolderType* holder;
            NN_RESULT_DO(m_LocalCommunicationSendApplicationTaskHolder.FindHolder(&holder, id));
            if (appId == holder->GetTask().GetApplicationId())
            {
                outList[count] = id;
                count++;
                if (count >= listCount)
                {
                    break;
                }
            }
        }

        *outCount = count;

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveSystemUpdateTask(sf::Out<LocalCommunicationReceiveSystemUpdateTaskId> outValue, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key, Bit32 config) NN_NOEXCEPT
    {
        SystemUpdateTaskMeta meta = { key, config };

        auto uuid = util::GenerateUuid();
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskDataStore.Create(uuid, &meta, sizeof(meta)));

        TaskDataStore::Path path;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskDataStore.GetDataFilePath(&path, uuid));

        NN_RESULT_DO(CreateLocalCommunicationReceiveSystemUpdateTaskImpl(outValue.GetPointer(), uuid, ipv4, port, key, config, path));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveSystemUpdateTaskForCleanup(LocalCommunicationReceiveSystemUpdateTaskId* outValue, const util::Uuid &uuid, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT
    {
        LocalCommunicationReceiveSystemUpdateTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->InitializeForCleanup(uuid, key, config, dataFilePath));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationReceiveSystemUpdateTaskImpl(nim::LocalCommunicationReceiveSystemUpdateTaskId* outValue, const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT
    {
        LocalCommunicationReceiveSystemUpdateTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->Initialize(uuid, ipv4, port, key, config, dataFilePath));
        *outValue = vacant->GetId();

        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] System update task %s created\n", uuid.ToString(uuidString, sizeof(uuidString)));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::DestroyLocalCommunicationReceiveSystemUpdateTask(const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.Destroy(id));
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskDataStore.Destroy(id.uuid));

        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] System update task %s destroyed\n", id.uuid.ToString(uuidString, sizeof(uuidString)));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListLocalCommunicationReceiveSystemUpdateTask(sf::Out<std::int32_t> outCount, const sf::OutArray<LocalCommunicationReceiveSystemUpdateTaskId>& outList) NN_NOEXCEPT
    {
        *outCount = m_LocalCommunicationReceiveSystemUpdateTaskHolder.List(outList.GetData(), static_cast<int>(outList.GetLength()));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLocalCommunicationReceiveSystemUpdateTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        char uuidString[util::Uuid::StringSize];
        NN_UNUSED(uuidString);
        NN_DETAIL_NIM_TRACE("[NetworkInstallManagerServer] Request system update task %s run\n", id.uuid.ToString(uuidString, sizeof(uuidString)));

        LocalCommunicationReceiveSystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncLocalCommunicationReceiveSystemUpdateTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationReceiveSystemUpdateTaskInfo(sf::Out<nim::LocalCommunicationReceiveSystemUpdateTaskInfo> outValue, const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationReceiveSystemUpdateTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::CommitLocalCommunicationReceiveSystemUpdateTask(const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationReceiveSystemUpdateTaskHolder.Commit(id);
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationReceiveSystemUpdateTaskErrorContext(sf::Out<err::ErrorContext> outValue, const nim::LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationReceiveSystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetErrorContext(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetReceivedSystemDataPath(sf::Out<ncm::Path> outValue, ncm::SystemDataId dataId, const LocalCommunicationReceiveSystemUpdateTaskId& taskId) NN_NOEXCEPT
    {
        LocalCommunicationReceiveSystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationReceiveSystemUpdateTaskHolder.FindHolder(&holder, taskId));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        NN_RESULT_DO(holder->GetTask().GetReceivedSystemDataPath(outValue.GetPointer(), dataId));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationSendSystemUpdateTask(sf::Out<nim::LocalCommunicationSendSystemUpdateTaskId> outValue, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        auto uuid = util::GenerateUuid();

        NN_RESULT_DO(CreateLocalCommunicationSendSystemUpdateTaskImpl(outValue.GetPointer(), uuid, ipv4, port, key));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::CreateLocalCommunicationSendSystemUpdateTaskImpl(nim::LocalCommunicationSendSystemUpdateTaskId* outValue, const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        LocalCommunicationSendSystemUpdateTaskHolder::TaskHolderType* vacant;
        NN_RESULT_DO(m_LocalCommunicationSendSystemUpdateTaskHolder.FindVacantHolder(&vacant));

        NN_RESULT_DO(vacant->Initialize(uuid, ipv4, port, key));
        *outValue = vacant->GetId();

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestLocalCommunicationSendSystemUpdateTaskRun(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outAsync, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationSendSystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationSendSystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncLocalCommunicationSendSystemUpdateTaskResultImpl>(holder);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outAsync, emplacedRef));

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationSendSystemUpdateTaskInfo(sf::Out<nim::LocalCommunicationSendSystemUpdateTaskInfo> outValue, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return m_LocalCommunicationSendSystemUpdateTaskHolder.GetInfo(outValue.GetPointer(), id);
    }

    Result NetworkInstallManagerServer::DestroyLocalCommunicationSendSystemUpdateTask(const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_LocalCommunicationSendSystemUpdateTaskHolder.Destroy(id));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::GetLocalCommunicationSendSystemUpdateTaskErrorContext(sf::Out<err::ErrorContext> outValue, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        LocalCommunicationSendSystemUpdateTaskHolder::TaskHolderType* holder;
        NN_RESULT_DO(m_LocalCommunicationSendSystemUpdateTaskHolder.FindHolder(&holder, id));
        NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());
        holder->GetTask().GetErrorContext(outValue.GetPointer());

        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::ListLocalCommunicationSendSystemUpdateTask(sf::Out<std::int32_t> outCount, const sf::OutArray<nim::LocalCommunicationSendSystemUpdateTaskId>& outList) NN_NOEXCEPT
    {
        *outCount = m_LocalCommunicationSendSystemUpdateTaskHolder.List(outList.GetData(), static_cast<int>(outList.GetLength()));
        NN_RESULT_SUCCESS;
    }

    Result NetworkInstallManagerServer::RequestQueryAvailableELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outValue, const nn::account::Uid& uid, const sf::InArray<nn::es::RightsId>& rightsIds) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, DynamicRights::AsyncAvailableELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, uid, rightsIds, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestAssignELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outValue, const nn::account::Uid& uid, const sf::InArray<nn::es::RightsId>& rightsIds, nn::nim::ELicenseType licenseType) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, DynamicRights::AsyncAssignELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, uid, rightsIds, licenseType, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestExtendELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outValue, const nn::account::Uid& uid, const sf::InArray<nn::es::ELicenseId>& elicenseIds) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, DynamicRights::AsyncExtendELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, uid, elicenseIds, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestSyncELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outValue, const nn::account::NintendoAccountId& naId) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, DynamicRights::AsyncSyncELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, naId));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestDownloadETickets(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outValue, const nn::account::Uid& uid, const sf::InArray<nn::es::ELicenseId>& elicenseIds) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, DynamicRights::AsyncDownloadETicketsImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_DeviceAccountStore, uid, elicenseIds, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestQueryRevokeReason(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncValue>> outValue, const nn::account::NintendoAccountId& naId, const nn::es::ELicenseId& eLicenseId) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncValue, DynamicRights::AsyncRevokeReasonImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, naId, eLicenseId));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;

    }
    Result NetworkInstallManagerServer::RequestReportActiveELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outValue, const nn::account::NintendoAccountId& naId, const sf::InArray<nn::es::ELicenseId>& elicenseIds) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, DynamicRights::AsyncReportELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, naId, elicenseIds, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestReportActiveELicensesPassively(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncResult>> outValue, const nn::account::NintendoAccountId& naId, const sf::InArray<nn::es::ELicenseId>& elicenseIds) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, DynamicRights::AsyncReportELicensesPassivelyImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, naId, elicenseIds, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestRegisterDynamicRightsNotificationToken(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::sf::Out<nn::sf::SharedPointer<nn::nim::detail::IAsyncResult>> outValue, const nn::npns::NotificationToken& notificationToken) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, DynamicRights::AsyncRegisterNotificationTokenImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultAllocationMemoryFailed());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, notificationToken));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }
    Result NetworkInstallManagerServer::RequestAssignAllDeviceLinkedELicenses(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<nim::detail::IAsyncData>> outValue) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncData, DynamicRights::AsyncAssignAllDeviceLinkedELicensesImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_DeviceContext, m_TemporaryDataStore.GenerateTemporaryFilePath()));
        NN_RESULT_DO(RunAndGetAsyncObject(outHandle, outValue, emplacedRef));
        NN_RESULT_SUCCESS;
    }

    void NetworkInstallManagerServer::ReloadErrorSimulation() NN_NOEXCEPT
    {
        ClearErrorSimulationCache();
    }
}}}
