﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <memory>

#include <nn/es.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_StorageUtil.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_LocalCommunicationReceiveApplicationTask.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/ovln/format/ovln_DownloadMessage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_ScopeExit.h>

#include "nim_ResultMappingUtil.h"

namespace nn { namespace nim { namespace srv {

namespace
{
} // namespace

    Result LocalCommunicationReceiveApplicationTask::Initialize(uint32_t ipv4, uint16_t port, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Meta.Initialize(metaFilePath));
        NN_RESULT_DO(m_Data.Initialize(dataFilePath));
        NN_RESULT_DO(LocalCommunicationReceiveContentTaskBase::Initialize(ipv4, port, m_Meta.GetInstallStorage(), &m_Data, m_Meta.GetConfig()));

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveApplicationTask::InitializeForCleanup(const char *metaFilePath, const char *dataFilePath) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Meta.Initialize(metaFilePath));
        NN_RESULT_DO(m_Data.Initialize(dataFilePath));
        NN_RESULT_DO(LocalCommunicationReceiveContentTaskBase::InitializeForCleanup(m_Meta.GetInstallStorage(), &m_Data, m_Meta.GetConfig()));

        NN_RESULT_SUCCESS;
    }

    ncm::InstallProgress LocalCommunicationReceiveApplicationTask::GetProgress() NN_NOEXCEPT
    {
        auto progress = InstallTaskBase::GetProgress();
        auto mappedResult = MapTaskResult(util::Get(progress.lastResult));
        std::memcpy(&(progress.lastResult), &mappedResult, sizeof(mappedResult));

        return progress;
    }

    int LocalCommunicationReceiveApplicationTask::CountInstallContentMetaKey() NN_NOEXCEPT
    {
        return m_Meta.CountContentMeta();
    }

    Result LocalCommunicationReceiveApplicationTask::GetInstallContentMetaKey(ncm::ContentMetaKey* outValue, int index) NN_NOEXCEPT
    {
        return m_Meta.GetContentMeta(outValue, index);
    }

    Result LocalCommunicationReceiveApplicationTask::InstallTicket(const fs::RightsId& rightsId, ncm::ContentMetaType contentMetaType) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if (contentMetaType == ncm::ContentMetaType::Application ||
            contentMetaType == ncm::ContentMetaType::Patch)
        {
            /* TORIAEZU: socketを使って適当に通信する */
            auto esRightsId = es::RightsIdIncludingKeyId::Construct(rightsId);
            auto header = MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag::GetCommonTicketTag, sizeof(esRightsId));
            NN_RESULT_DO(m_Connection.Send(header, &esRightsId, sizeof(esRightsId)));

            LocalCommunicationDeliveryProtocolHeader receiveHeader;
            NN_RESULT_DO(m_Connection.ReceiveHeader(&receiveHeader));
            NN_RESULT_THROW_UNLESS(receiveHeader.tag == header.tag, ResultLocalCommunicationUnexpectedTag());

            // TORIAEZU: ヒープから領域を取る

            LocalCommunicationDeliveryProtocolGetCommonTicketResponseHeader responseHeader;
            NN_RESULT_DO(m_Connection.ReceiveData(&responseHeader, sizeof(responseHeader), sizeof(responseHeader)));

            std::unique_ptr<Bit8> commonTicketBuffer(new Bit8[responseHeader.ticketSize]);
            NN_RESULT_THROW_UNLESS(commonTicketBuffer, ResultBufferNotEnough());
            NN_RESULT_DO(m_Connection.ReceiveData(commonTicketBuffer.get(), responseHeader.ticketSize, responseHeader.ticketSize));

            std::unique_ptr<Bit8> commonCertificateBuffer(new Bit8[responseHeader.certificateSize]);
            NN_RESULT_THROW_UNLESS(commonCertificateBuffer, ResultBufferNotEnough());
            NN_RESULT_DO(m_Connection.ReceiveData(commonCertificateBuffer.get(), responseHeader.certificateSize, responseHeader.certificateSize));

            NN_RESULT_DO(es::ImportTicket(commonTicketBuffer.get(), responseHeader.ticketSize, commonCertificateBuffer.get(), responseHeader.certificateSize));
        }
#else
        NN_UNUSED(contentMetaType);
        NN_UNUSED(rightsId);
#endif

        NN_RESULT_SUCCESS;
    }
}}}
