﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include "nim_HeapUtil.h"
#include <nn/nim/nim_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace nim { namespace srv {

namespace HeapUtil {

//!--------------------------------------------------------------------------------------
//! @brief  The implementation in C11 rules for aligned memory allocations.@n
//!         Memory allocator with alignment for start address of rules on the windows operating system.
//! @note   nim プロセスは @ref nn::init::InitializeAllocator() == @ref nn::mem::StandardAllocator を利用しているのでスレッドセーフである事が想定。
//!--------------------------------------------------------------------------------------
void* AllocateHeap(size_t beginAlignment, size_t size) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    return ::_aligned_malloc(size, beginAlignment);
#else
    return ::aligned_alloc(beginAlignment, size);
#endif
}

//!--------------------------------------------------------------------------------------
//! @brief The implementation for aligned memory release that has been imitated on the rules of the windows operating system.
//! @note   nim プロセスは @ref nn::init::InitializeAllocator() == @ref nn::mem::StandardAllocator を利用しているのでスレッドセーフである事が想定。
//!--------------------------------------------------------------------------------------
void DeallocateHeap(void* address) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    ::_aligned_free(address);
#else
    ::free(address);
#endif
}

//-----------------------------------------------------------------------------
OnetimeHeapSession::OnetimeHeapSession() NN_NOEXCEPT
    : pTop(nullptr), size(0)
{
}

//-----------------------------------------------------------------------------
OnetimeHeapSession::~OnetimeHeapSession() NN_NOEXCEPT
{
    Release();
}

//-----------------------------------------------------------------------------
Result OnetimeHeapSession::Acquire(size_t beginAlignment, size_t capacity) NN_NOEXCEPT
{
    Release();

    auto pWork = AllocateHeap(beginAlignment, capacity);
    NN_RESULT_THROW_UNLESS(pWork, ResultAllocationMemoryFailed());
    pTop = pWork;
    size = capacity;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result OnetimeHeapSession::Expand(size_t beginAlignment, size_t capacity) NN_NOEXCEPT
{
    auto pWork = AllocateHeap(beginAlignment, capacity);
    NN_RESULT_THROW_UNLESS(pWork, ResultAllocationMemoryFailed());

    auto pPre = pTop;
    if (nullptr != pPre)
    {
        // 以前の確保があるならコピーして破棄する。
        const auto preSize = size;
        const auto actualCopySize = std::min(capacity, preSize);
        std::memcpy(pWork, pPre, actualCopySize);
        Release();
    }
    pTop = pWork;
    size = capacity;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void OnetimeHeapSession::Release() NN_NOEXCEPT
{
    auto pWork = pTop;
    if (nullptr != pWork)
    {
        pTop = nullptr;
        size = 0;
        DeallocateHeap(pWork);
    }
}

}   // ~HeapUtil

}}}
