﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <limits>
#include <string>
#include <nn/crypto.h>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_EciAccessor.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "nim_Json.h"
#include "nim_EShopUtil.h"

namespace nn { namespace nim { namespace srv {

    namespace {
        const char Version[] = "2.0";

        const int MaxMessageIdLength = 124;
        const int MaxTivLength = 32;
        const int RightsIdStringLength = 33;

        const int SignatureLength = 60;
        const int DeviceCertificateForRegisterLength = 384;
        const int DeviceCertificateForETicketLength = 576;
        const int CertificateLength = 384;
        const int MaxETicketLength = 1024;
        const int MaxCertificateLength = 2048;

        const int SignatureBase64StringLength = SignatureLength * 4 / 3 + 1;
        const int DeviceCertificateBase64StringLength = DeviceCertificateForETicketLength * 4 / 3 + 1;
        const int CertificateBase64StringLength = CertificateLength * 4 / 3 + 1;
        const int MaxETicketBase64StringLength = MaxETicketLength * 4 / 3 + 1;

        // チケットのインポート用バッファ
        char g_ETicketBuffer[MaxETicketLength];
        char g_CertificateBuffer[MaxCertificateLength];

        // チケットのインポート用バッファの Mutex
        nn::os::Mutex g_Mutex(false);

        struct Response
        {
            char str[1024];
        };

        void CreateMessageId(char* outMessageId, int outMessageIdLength, Bit64 deviceId) NN_NOEXCEPT
        {
            nn::util::SNPrintf(outMessageId, outMessageIdLength, "EC-%lld-%d", deviceId, nn::os::GetSystemTick().ToTimeSpan().GetMicroSeconds());
        }

        void CreateRightsIdString(char* outRightsIdString, size_t outRightsIdStringLength, es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
        {
            NN_UNUSED(outRightsIdStringLength);
            NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(outRightsIdStringLength), RightsIdStringLength);

            for (size_t i = 0; i < sizeof(es::RightsIdIncludingKeyId); i++)
            {
                util::SNPrintf(&outRightsIdString[i * 2], 3, "%02x", rightsId._data[i]);
            }
            outRightsIdString[RightsIdStringLength - 1] = '\0';
        }

        es::RightsIdIncludingKeyId CreateRightsId(const char* rightsIdString) NN_NOEXCEPT
        {
            es::RightsIdIncludingKeyId rightsId = {};
            for (int i = 0; i < sizeof(es::RightsIdIncludingKeyId); i++)
            {
                char byte[3];
                util::Strlcpy(byte, &rightsIdString[i * 2], 3);
                rightsId._data[i] = static_cast<uint8_t>(std::strtol(byte, NULL, 16));
            }

            return rightsId;
        }

        void GetSerialNumber(nn::settings::system::SerialNumber* outSerialNumber) NN_NOEXCEPT
        {
#ifdef NN_BUILD_CONFIG_OS_WIN
            nn::util::SNPrintf(outSerialNumber->string, 24, "XAW00000000000");
#else
            nn::settings::system::GetSerialNumber(outSerialNumber);
#endif
        }

        bool IsRegistered(ec::system::DeviceAccountStatus status) NN_NOEXCEPT
        {
            return util::Strncmp(status.data, "R", 1) == 0 || util::Strncmp(status.data, "T", 1) == 0;
        }

        DeviceAccountToken GetWeakDeviceAccountToken(const DeviceAccountToken& deviceToken) NN_NOEXCEPT
        {
            const int Md5DigestLength = 16;

            DeviceAccountToken weakDeviceToken;
            unsigned char weakDeviceTokenBinary[Md5DigestLength];

            nn::crypto::GenerateMd5Hash(weakDeviceTokenBinary, sizeof(weakDeviceTokenBinary), deviceToken.data, util::Strnlen(deviceToken.data, MaxDeviceAccountTokenLength));

            nn::util::SNPrintf(&weakDeviceToken.data[0], MaxDeviceAccountTokenLength, "WT-");

            for (int i = 0; i < Md5DigestLength; i++)
            {
                nn::util::SNPrintf(&weakDeviceToken.data[3 + i * 2], MaxDeviceAccountTokenLength, "%02x", weakDeviceTokenBinary[i]);
            }

            weakDeviceToken.data[3 + Md5DigestLength * 2] = '\0';

            return weakDeviceToken;
        }

        // Challenge にのみ署名する場合
        void SignChallengeData(nn::es::Sign* outSign, nn::es::Certificate* outCertificate, int challengeData) NN_NOEXCEPT
        {
            const int ChallengeStringLength = 200;
            char challengeString[ChallengeStringLength];

            nn::util::SNPrintf(challengeString, ChallengeStringLength, "<Challenge>%d</Challenge>", challengeData);

            size_t challengeStringSize = strlen(challengeString);

            nn::es::SignData(outSign, outCertificate, &challengeString, challengeStringSize);
        }

        Result SignChallengeAndGetBase64String(char* outSignatureBase64String, int outSignatureBase64StringLength, char* outCertificateBase64String, int outCertificateBase64StringLength, int challenge)
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(outSignatureBase64StringLength, SignatureBase64StringLength);
            NN_ABORT_UNLESS_GREATER_EQUAL(outCertificateBase64StringLength, CertificateBase64StringLength);

            nn::es::Sign sign;
            nn::es::Certificate certificate;
            SignChallengeData(&sign, &certificate, challenge);

            nn::util::Base64::Status status = nn::util::Base64::ToBase64String(outSignatureBase64String, SignatureBase64StringLength, sign._data, SignatureLength, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            status = nn::util::Base64::ToBase64String(outCertificateBase64String, CertificateBase64StringLength, certificate._data, CertificateLength, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            NN_RESULT_SUCCESS;
        }

        // Challenge と SerialNumber に署名する場合
        void SignChallengeData(nn::es::Sign* outSign, nn::es::Certificate* outCertificate, int challengeData, nn::settings::system::SerialNumber serialNumber) NN_NOEXCEPT
        {
            const int ChallengeStringLength = 200;
            char challengeString[ChallengeStringLength];

            nn::util::SNPrintf(challengeString, ChallengeStringLength, "<Challenge>%d</Challenge><SerialNumber>%s</SerialNumber>", challengeData, serialNumber);

            size_t challengeStringSize = strlen(challengeString);

            nn::es::SignData(outSign, outCertificate, &challengeString, challengeStringSize);
        }

        Result SignChallengeAndGetBase64String(char* outSignatureBase64String, int outSignatureBase64StringLength, char* outCertificateBase64String, int outCertificateBase64StringLength, int challenge, nn::settings::system::SerialNumber serialNumber)
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(outSignatureBase64StringLength, SignatureBase64StringLength);
            NN_ABORT_UNLESS_GREATER_EQUAL(outCertificateBase64StringLength, CertificateBase64StringLength);

            nn::es::Sign sign;
            nn::es::Certificate certificate;
            SignChallengeData(&sign, &certificate, challenge, serialNumber);

            nn::util::Base64::Status status = nn::util::Base64::ToBase64String(outSignatureBase64String, SignatureBase64StringLength, sign._data, SignatureLength, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            status = nn::util::Base64::ToBase64String(outCertificateBase64String, CertificateBase64StringLength, certificate._data, CertificateLength, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            NN_RESULT_SUCCESS;
        }

        size_t GetDeviceCertificateForRegister(char* outDeviceCertificate, int outDeviceCertificateLength) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(outDeviceCertificateLength, DeviceCertificateForRegisterLength);

#ifdef NN_BUILD_CONFIG_OS_WIN
            // sdcard フォルダのマウント
            const char* MountName = "sdcard";
            const char* DeviceCertificateName = "sdcard:/device_cert_for_device_register.cert";

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSdCardForDebug(MountName));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName);
            };

            // 仮想デバイス証明書の読込み
            nn::fs::FileHandle deviceCertificateFileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&deviceCertificateFileHandle, DeviceCertificateName, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(deviceCertificateFileHandle);
            };

            int64_t deviceCertificateSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&deviceCertificateSize, deviceCertificateFileHandle));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(deviceCertificateFileHandle, 0, outDeviceCertificate, static_cast<size_t>(deviceCertificateSize)));

            return static_cast<size_t>(deviceCertificateSize);
#else
            nn::settings::factory::EccB233DeviceCertificate deviceCertificate;
            nn::settings::factory::GetEciDeviceCertificate(&deviceCertificate);

            std::memcpy(outDeviceCertificate, deviceCertificate.data, DeviceCertificateForRegisterLength);
            return DeviceCertificateForRegisterLength;
#endif
        }

        Result GetDeviceCertificateBase64StringForRegister(char* outDeviceCertificateBase64String, int outDeviceCertificateBase64StringLength)
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(outDeviceCertificateBase64StringLength, DeviceCertificateBase64StringLength);

            char deviceCertificate[DeviceCertificateForRegisterLength];
            size_t deviceCertificateSize = GetDeviceCertificateForRegister(deviceCertificate, DeviceCertificateForRegisterLength);

            nn::util::Base64::Status status = nn::util::Base64::ToBase64String(outDeviceCertificateBase64String, DeviceCertificateBase64StringLength, deviceCertificate, deviceCertificateSize, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            NN_RESULT_SUCCESS;
        }

        size_t GetDeviceCertificateForETicket(char* outDeviceCertificate, int outDeviceCertificateLength) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(outDeviceCertificateLength, DeviceCertificateForETicketLength);

#ifdef NN_BUILD_CONFIG_OS_WIN
            // sdcard フォルダのマウント
            const char* MountName = "sdcard";
            const char* DeviceCertificateName = "sdcard:/device_cert_for_eticket.cert";

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSdCardForDebug(MountName));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName);
            };

            // 仮想デバイス証明書の読込み
            nn::fs::FileHandle deviceCertificateFileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&deviceCertificateFileHandle, DeviceCertificateName, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(deviceCertificateFileHandle);
            };

            int64_t deviceCertificateSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&deviceCertificateSize, deviceCertificateFileHandle));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(deviceCertificateFileHandle, 0, outDeviceCertificate, static_cast<size_t>(deviceCertificateSize)));

            return static_cast<size_t>(deviceCertificateSize);
#else
            nn::settings::factory::Rsa2048DeviceCertificate deviceCertificate;
            nn::settings::factory::GetEticketDeviceCertificate(&deviceCertificate);

            std::memcpy(outDeviceCertificate, deviceCertificate.data, DeviceCertificateForETicketLength);
            return DeviceCertificateForETicketLength;
#endif
        }

        Result InterpretEciErrorCode(int errorCode)
        {
            switch (errorCode)
            {
            case 0:    NN_RESULT_SUCCESS;
            case 13:   NN_RESULT_THROW(ResultEciScNoProperCredential());
            case 14:   NN_RESULT_THROW(ResultEciUnderMaintenance());
            // 本体更新が必要な場合は上位でハンドリングされる必要があるため、ResultSystemUpdateRequired を返す
            case 22:   NN_RESULT_THROW(ResultSystemUpdateRequired());
            case 432:  NN_RESULT_THROW(ResultEciIasExceedsAllowedMaximumVirtualAccountPerDevice());
            case 434:  NN_RESULT_THROW(ResultEciIasLinkedVirtualAccountsExist());
            case 452:  NN_RESULT_THROW(ResultEciIasCannotIdentifyDeviceBySerialNo());
            case 527:  NN_RESULT_THROW(ResultEciEcsBadSerialNo());
            case 602:  NN_RESULT_THROW(ResultEciEcsBadTitleId());
            case 604:  NN_RESULT_THROW(ResultEciEcsBadDeviceType());
            case 621:  NN_RESULT_THROW(ResultEciEcsTitleAlreadyPurchased());
            case 630:  NN_RESULT_THROW(ResultEciEcsAppidTinMismatch());
            case 637:  NN_RESULT_THROW(ResultEciEcsNoEtkmTitleInfo());
            case 638:  NN_RESULT_THROW(ResultEciEcsNoEtkm());
            case 642:  NN_RESULT_THROW(ResultEciEcsInvalidAccount());
            case 643:  NN_RESULT_THROW(ResultEciEcsInvalidAccountStatus());
            case 685:  NN_RESULT_THROW(ResultEciEcsNoAccessRights());
            case 686:  NN_RESULT_THROW(ResultEciEcsIasConnectionError());
            case 687:  NN_RESULT_THROW(ResultEciEcsEtsConnectionError());
            case 688:  NN_RESULT_THROW(ResultEciEcsPasConnectionError());
            case 689:  NN_RESULT_THROW(ResultEciEcsDBConnectionError());
            case 691:  NN_RESULT_THROW(ResultEciEcsCacheError());
            case 696:  NN_RESULT_THROW(ResultEciEcsIasError());
            case 697:  NN_RESULT_THROW(ResultEciEcsEtsError());
            case 698:  NN_RESULT_THROW(ResultEciEcsPasError());
            case 699:  NN_RESULT_THROW(ResultEciEcsInternalError());
            case 701:  NN_RESULT_THROW(ResultEciEtsNoPermission());
            case 814:  NN_RESULT_THROW(ResultEciPasAccountNotUsable());
            case 819:  NN_RESULT_THROW(ResultEciPasNoAccountRecord());
            case 901:  NN_RESULT_THROW(ResultEciIasBadDeviceId());
            case 902:  NN_RESULT_THROW(ResultEciIasBadAccountId());
            // デバイストークンが無効な場合は上位でハンドリングされる必要があるため、ResultInvalidDeviceAccountToken を返す
            case 903:  NN_RESULT_THROW(ResultInvalidDeviceAccountToken());
            case 904:  NN_RESULT_THROW(ResultEciIasInvalidAccountTokenId());
            case 922:  NN_RESULT_THROW(ResultEciIasRegisterDeviceError());
            case 927:  NN_RESULT_THROW(ResultEciIasNoMsCert());
            case 928:  NN_RESULT_THROW(ResultEciIasInvalidDeviceCert());
            case 952:  NN_RESULT_THROW(ResultEciIasTransferRequestNotFoundError());
            case 971:  NN_RESULT_THROW(ResultEciIasDeviceOrAccountBusyError());
            case 980:  NN_RESULT_THROW(ResultEciIasMissingDeviceId());
            case 985:  NN_RESULT_THROW(ResultEciIasBackingStoreError());
            case 991:  NN_RESULT_THROW(ResultEciIasSalesTargetNotSetError());
            case 993:  NN_RESULT_THROW(ResultIasInvalidServiceLevel());
            case 999:  NN_RESULT_THROW(ResultEciIasInternalError());
            case 1704: NN_RESULT_THROW(ResultCasBadApplicationId());
            default:   NN_RESULT_THROW(ResultUnknownEcErrorCode());
            }
        }

        Result ParseErrorResponseDestructively(char* response) NN_NOEXCEPT
        {
            int errorCode;
            NN_RESULT_TRY(ParseEciErrorResponseDestructively(&errorCode, response))
                NN_RESULT_CATCH(ResultInvalidJson) { NN_RESULT_SUCCESS; }
            NN_RESULT_END_TRY

            NN_RESULT_DO(InterpretEciErrorCode(errorCode));

            NN_RESULT_SUCCESS;
        }
    }

    void CreateGetAccountStatusRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());

        if (deviceToken.data[0] != '\0')
        {
            document.AddMember("device_token", nne::rapidjson::StringRef(deviceToken.data), document.GetAllocator());
        }

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());
    }

    Result PerformGetAccountStatusRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        Url url("https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/GetAccountStatus");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 256;
        char postFields[PostFieldsLength];
        CreateGetAccountStatusRequestPostFields(postFields, PostFieldsLength, deviceId, deviceToken);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseGetAccountStatusTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformGetAccountStatusRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseGetAccountStatusResponse(ec::system::DeviceAccountStatus* outValue, char* readBuffer) NN_NOEXCEPT
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseGetAccountStatusParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto accountStatusObj = document.FindMember("account_status");
        NN_RESULT_THROW_UNLESS(accountStatusObj != document.MemberEnd() && accountStatusObj->value.IsString(), ResultUnexpectedResponseGetAccountStatusNotFound());

        ec::system::DeviceAccountStatus status = {};
        status.isDeviceTokenExpired = false; // 値を取得できないので無効値を入れておく
        util::Strlcpy(status.data, accountStatusObj->value.GetString(), static_cast<int>(sizeof(status.data)));

        if (IsRegistered(status))
        {
            auto ticketSyncFlagObj = document.FindMember("ticket_sync_flag");
            NN_RESULT_THROW_UNLESS(ticketSyncFlagObj != document.MemberEnd() && ticketSyncFlagObj->value.IsBool(), ResultUnexpectedResponseGetAccountStatusTicketSyncFlagNotFound());
            status.needsTicketSync = ticketSyncFlagObj->value.GetBool();
        }
        else
        {
            status.needsTicketSync = false;
        }

        *outValue = status;
        NN_RESULT_SUCCESS;
    }

    void CreateGetChallengeRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());
    }

    Result PerformGetChallengeRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/GetChallenge");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 256;
        char postFields[PostFieldsLength];
        CreateGetChallengeRequestPostFields(postFields, PostFieldsLength, deviceId);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseGetChallengeTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformGetChallengeRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseGetChallengeResponse(int* outValue, char* readBuffer) NN_NOEXCEPT
    {
        int challenge;

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseGetChallengeParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto challengeObj = document.FindMember("challenge");
        NN_RESULT_THROW_UNLESS(challengeObj != document.MemberEnd() && challengeObj->value.IsInt(), ResultUnexpectedResponseGetChallengeNotFound());

        challenge = challengeObj->value.GetInt();

        *outValue = challenge;
        NN_RESULT_SUCCESS;
    }

    Result CreateRegisterRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nn::settings::system::SerialNumber serialNumber;
        nn::nim::srv::GetSerialNumber(&serialNumber);

        char signatureBase64String[SignatureBase64StringLength];
        char certificateBase64String[CertificateBase64StringLength];
        NN_RESULT_DO(SignChallengeAndGetBase64String(signatureBase64String, SignatureBase64StringLength, certificateBase64String, CertificateBase64StringLength, challenge, serialNumber));

        char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
        NN_RESULT_DO(GetDeviceCertificateBase64StringForRegister(deviceCertificateBase64String, DeviceCertificateBase64StringLength));

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("signature", nne::rapidjson::StringRef(signatureBase64String), document.GetAllocator());
        document.AddMember("cert_chain", nne::rapidjson::StringRef(certificateBase64String), document.GetAllocator());
        document.AddMember("challenge", challenge, document.GetAllocator());
        document.AddMember("serial_number", nne::rapidjson::StringRef(serialNumber.string), document.GetAllocator());
        document.AddMember("device_cert", nne::rapidjson::StringRef(deviceCertificateBase64String), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());

        NN_RESULT_SUCCESS;
    }

    Result PerformRegisterRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/Register");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        NN_RESULT_DO(CreateRegisterRequestPostFields(postFields, PostFieldsLength, deviceId, challenge));

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseRegisterTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformRegisterRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseRegisterResponse(DeviceAccountToken* outDeviceToken, DeviceAccountId* outAccountId, char* readBuffer) NN_NOEXCEPT
    {
        DeviceAccountToken deviceToken;
        DeviceAccountId accountId;

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseRegisterParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto deviceTokenObj = document.FindMember("device_token");
        NN_RESULT_THROW_UNLESS(deviceTokenObj != document.MemberEnd() && deviceTokenObj->value.IsString(), ResultUnexpectedResponseRegisterDeviceTokenNotFound());
        NN_RESULT_THROW_UNLESS(deviceTokenObj->value.GetStringLength() < MaxDeviceAccountTokenLength, ResultUnexpectedResponseRegisterDeviceTokenTooLong());
        strcpy(deviceToken.data, deviceTokenObj->value.GetString());

        auto accountIdObj = document.FindMember("account_id");
        NN_RESULT_THROW_UNLESS(accountIdObj != document.MemberEnd() && accountIdObj->value.IsString(), ResultUnexpectedResponseRegisterAccountIdNotFound());
        NN_RESULT_THROW_UNLESS(accountIdObj->value.GetStringLength() < AccountIdLength, ResultUnexpectedResponseRegisterAccountIdTooLong());
        strcpy(accountId.data, accountIdObj->value.GetString());

        *outDeviceToken = deviceToken;
        *outAccountId = accountId;
        NN_RESULT_SUCCESS;
    }

    Result CreateUnregisterRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nn::settings::system::SerialNumber serialNumber;
        nn::nim::srv::GetSerialNumber(&serialNumber);

        char signatureBase64String[SignatureBase64StringLength];
        char certificateBase64String[CertificateBase64StringLength];
        NN_RESULT_DO(SignChallengeAndGetBase64String(signatureBase64String, SignatureBase64StringLength, certificateBase64String, CertificateBase64StringLength, challenge, serialNumber));

        char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
        NN_RESULT_DO(GetDeviceCertificateBase64StringForRegister(deviceCertificateBase64String, DeviceCertificateBase64StringLength));

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("signature", nne::rapidjson::StringRef(signatureBase64String), document.GetAllocator());
        document.AddMember("cert_chain", nne::rapidjson::StringRef(certificateBase64String), document.GetAllocator());
        document.AddMember("challenge", challenge, document.GetAllocator());
        document.AddMember("serial_number", nne::rapidjson::StringRef(serialNumber.string), document.GetAllocator());
        document.AddMember("device_cert", nne::rapidjson::StringRef(deviceCertificateBase64String), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());

        NN_RESULT_SUCCESS;
    }

    Result PerformUnregisterRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/Unregister");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        NN_RESULT_DO(CreateUnregisterRequestPostFields(postFields, PostFieldsLength, deviceId, challenge));

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseUnregisterTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformUnregisterRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    void CreateAccountListETicketIdsPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("account_id", nne::rapidjson::StringRef(accountId.data), document.GetAllocator());
        DeviceAccountToken weakDeviceToken = GetWeakDeviceAccountToken(deviceToken);
        document.AddMember("device_token", nne::rapidjson::StringRef(weakDeviceToken.data), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());
    }

    Result PerformAccountListETicketIdsRequest(char responseBuffer[], size_t bufferSize, HttpConnection* connection, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        Url url("https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/AccountListETicketIds");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 512;
        char postFields[PostFieldsLength];
        CreateAccountListETicketIdsPostFields(postFields, PostFieldsLength, deviceId, accountId, deviceToken);

        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &responseBuffer, &bufferSize](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < bufferSize, ResultUnexpectedResponseAccountListETicketIdsTooLong());
                std::memcpy(&responseBuffer[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                responseBuffer[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformAccountListETicketIdsRequest] Error response %s\n", responseBuffer);
                NN_RESULT_DO(ParseErrorResponseDestructively(responseBuffer));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        responseBuffer[readSize] = '\0';

        NN_RESULT_SUCCESS;
    }

    void CreateAccountListETicketIdsByRightsIdPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        char RightsIdString[RightsIdStringLength];
        CreateRightsIdString(RightsIdString, RightsIdStringLength, rightsId);

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("account_id", nne::rapidjson::StringRef(accountId.data), document.GetAllocator());
        document.AddMember("rights_id", nne::rapidjson::StringRef(RightsIdString), document.GetAllocator());
        DeviceAccountToken weakDeviceToken = GetWeakDeviceAccountToken(deviceToken);
        document.AddMember("device_token", nne::rapidjson::StringRef(weakDeviceToken.data), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());
    }

    Result PerformAccountListETicketIdsByRightsIdRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        Url url("https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/AccountListETicketIdsByRightsId");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 512;
        char postFields[PostFieldsLength];
        CreateAccountListETicketIdsByRightsIdPostFields(postFields, PostFieldsLength, deviceId, accountId, deviceToken, rightsId);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseAccountListETicketIdsByRightsIdTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformAccountListETicketIdsByRightsIdRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    // AccountListETicketIds と AccountListETicketIdsByRightsId 両方に使う
    Result ParseAccountListETicketIdsResponse(int* outTicketIdListCount, Bit64 outTicketIdList[], int outTicketIdListLength, char* readBuffer) NN_NOEXCEPT
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseAccountListETicketIdsParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto tivObj = document.FindMember("tiv");
        NN_RESULT_THROW_UNLESS(tivObj != document.MemberEnd(), ResultPersonalizedTicketNotFound());
        NN_RESULT_THROW_UNLESS(tivObj->value.IsArray(), ResultUnexpectedResponseAccountListETicketIdsTivNotFound());

        const nne::rapidjson::Value& tiv = document["tiv"];

        *outTicketIdListCount = tiv.Size();
        NN_RESULT_THROW_UNLESS(*outTicketIdListCount > 0, ResultUnexpectedResponseAccountListETicketIdsZeroTiv());
        NN_RESULT_THROW_UNLESS(outTicketIdListLength >= *outTicketIdListCount, ResultAccountListETicketIdsTicketIdListSizeSmall());

        for (int i = 0; i < *outTicketIdListCount; i++)
        {
            char tivString[MaxTivLength];

            NN_RESULT_THROW_UNLESS(tiv[i].IsString(), ResultUnexpectedResponseAccountListETicketIdsTivNotFound());
            NN_RESULT_THROW_UNLESS(tiv[i].GetStringLength() < MaxTivLength, ResultUnexpectedResponseAccountListETicketIdsTivTooLong());
            strcpy(tivString, tiv[i].GetString());

            char* ticketIdString;
            ticketIdString = strtok(tivString, ".");

            char* endPtr;
            outTicketIdList[i] = std::strtoull(ticketIdString, &endPtr, 10);
            NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultUnexpectedResponseAccountListETicketIdsInvalidTiv());
        }

        NN_RESULT_SUCCESS;
    }

    Result CreateAccountGetETicketsPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        char deviceCertificate[DeviceCertificateForETicketLength];
        size_t deviceCertificateSize = GetDeviceCertificateForETicket(deviceCertificate, DeviceCertificateForETicketLength);

        char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
        nn::util::Base64::Status status = nn::util::Base64::ToBase64String(deviceCertificateBase64String, DeviceCertificateBase64StringLength, deviceCertificate, deviceCertificateSize, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

        nne::rapidjson::Document document;
        document.SetObject();

        nne::rapidjson::Value ticketId(nne::rapidjson::kArrayType);
        for (int i = 0; i < ticketIdListLength; i++)
        {
            ticketId.PushBack(ticketIdList[i], document.GetAllocator());
        }

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("account_id", nne::rapidjson::StringRef(accountId.data), document.GetAllocator());
        document.AddMember("ticket_id", ticketId, document.GetAllocator());
        document.AddMember("device_cert", nne::rapidjson::StringRef(deviceCertificateBase64String), document.GetAllocator());
        DeviceAccountToken weakDeviceToken = GetWeakDeviceAccountToken(deviceToken);
        document.AddMember("device_token", nne::rapidjson::StringRef(weakDeviceToken.data), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());

        NN_RESULT_SUCCESS;
    }

    Result PerformAccountGetETicketsRequest(char responseBuffer[], size_t bufferSize, HttpConnection* connection, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT
    {
        Url url("https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/AccountGetETickets");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        NN_RESULT_DO(CreateAccountGetETicketsPostFields(postFields, PostFieldsLength, deviceId, accountId, deviceToken, ticketIdList, ticketIdListLength));

        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &responseBuffer, &bufferSize](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < bufferSize, ResultUnexpectedResponseAccountGetETicketsTooLong());
                std::memcpy(&responseBuffer[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                responseBuffer[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformAccountGetEticketsRequest] Error response %s\n", responseBuffer);
                NN_RESULT_DO(ParseErrorResponseDestructively(responseBuffer));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        responseBuffer[readSize] = '\0';

        NN_RESULT_SUCCESS;
    }

    Result ParseAccountGetETicketsResponseAndImportTicket(bool* outIsPrepurchaseRecordDownloaded, int* outComeBackAfter, char* readBuffer) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> guard(g_Mutex);

        *outComeBackAfter = 0;

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseAccountGetETicketsParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        bool isETicketFound = document.FindMember("etickets") != document.MemberEnd();
        bool isCertificateFound = document.FindMember("certs") != document.MemberEnd();
        bool isUnreleasedETicketFound = document.FindMember("unreleased_etickets") != document.MemberEnd();

        // etickets が存在するが certs が存在しないのはない
        if (isETicketFound && !isCertificateFound)
        {
            NN_RESULT_THROW(ResultUnexpectedResponseAccountGetETicketsCertificateNotFound());
        }

        // certs が存在するが etickets が存在しないのはない
        if (!isETicketFound && isCertificateFound)
        {
            NN_RESULT_THROW(ResultUnexpectedResponseAccountGetETicketsETicketNotFound());
        }

        size_t caCertificateSize = 0;
        size_t xsCertificateSize = 0;
        if (isCertificateFound)
        {
            const nne::rapidjson::Value& certs = document["certs"];
            NN_RESULT_THROW_UNLESS(certs[0].GetStringLength() < MaxCertificateLength, ResultUnexpectedResponseAccountGetETicketsCertificateTooLong());
            NN_RESULT_THROW_UNLESS(certs[1].GetStringLength() < MaxCertificateLength, ResultUnexpectedResponseAccountGetETicketsCertificateTooLong());

            nn::util::Base64::Status status = nn::util::Base64::FromBase64String(&caCertificateSize, g_CertificateBuffer, sizeof(g_CertificateBuffer), certs[0].GetString(), nn::util::Base64::Mode::Mode_Normal);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultUnexpectedResponseAccountGetETicketsInvalidCertificate());
            status = nn::util::Base64::FromBase64String(&xsCertificateSize, &g_CertificateBuffer[caCertificateSize], sizeof(g_CertificateBuffer) - caCertificateSize, certs[1].GetString(), nn::util::Base64::Mode_Normal);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultUnexpectedResponseAccountGetETicketsInvalidCertificate());
        }

        if (isETicketFound)
        {
            const nne::rapidjson::Value& eTicket = document["etickets"];
            int eTicketCount = eTicket.Size();

            for (int i = 0; i < eTicketCount; i++)
            {
                NN_RESULT_THROW_UNLESS(eTicket[i].GetStringLength() < MaxETicketLength, ResultUnexpectedResponseAccountGetETicketsETicketTooLong());

                size_t eTicketSize;
                nn::util::Base64::Status status = nn::util::Base64::FromBase64String(&eTicketSize, g_ETicketBuffer, sizeof(g_ETicketBuffer), eTicket[i].GetString(), nn::util::Base64::Mode::Mode_Normal);
                NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultUnexpectedResponseAccountGetETicketsInvalidETicket());

                NN_RESULT_DO(nn::es::ImportTicket(g_ETicketBuffer, eTicketSize, g_CertificateBuffer, caCertificateSize + xsCertificateSize));
            }
        }

        if (isUnreleasedETicketFound)
        {
            const nne::rapidjson::Value& unreleasedETicket = document["unreleased_etickets"];
            int unreleasedETicketCount = unreleasedETicket.Size();

            for (int i = 0; i < unreleasedETicketCount; i++)
            {
                es::PrepurchaseRecord record;

                auto rightsIdObj = unreleasedETicket[i].FindMember("rights_id");
                NN_RESULT_THROW_UNLESS(rightsIdObj != unreleasedETicket[i].MemberEnd() && rightsIdObj->value.IsString(), ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsRightsIdNotFound());
                NN_RESULT_THROW_UNLESS(rightsIdObj->value.GetStringLength() < RightsIdStringLength, ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsRightsIdTooLong());
                record.rightsId = CreateRightsId(rightsIdObj->value.GetString());

                auto accountIdObj = unreleasedETicket[i].FindMember("account_id");
                NN_RESULT_THROW_UNLESS(accountIdObj != unreleasedETicket[i].MemberEnd() && accountIdObj->value.IsString(), ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsAccountIdNotFound());
                NN_RESULT_THROW_UNLESS(accountIdObj->value.GetStringLength() < AccountIdLength, ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsAccountIdTooLong());
                char* endPtr;
                record.accountId = std::strtoul(accountIdObj->value.GetString(), &endPtr, 10);
                NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsInvalidAccountId());

                auto ticketIdObj = unreleasedETicket[i].FindMember("ticket_id");
                NN_RESULT_THROW_UNLESS(ticketIdObj != unreleasedETicket[i].MemberEnd() && ticketIdObj->value.IsUint64(), ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsTicketIdNotFound());
                record.ticketId = ticketIdObj->value.GetUint64();

                auto releaseDateObj = unreleasedETicket[i].FindMember("release_date");
                NN_RESULT_THROW_UNLESS(releaseDateObj != unreleasedETicket[i].MemberEnd() && releaseDateObj->value.IsUint64(), ResultUnexpectedResponseAccountGetETicketsUnreleasedETicketsComeBackAfterNotFound());
                record.deliveryScheduledTime = { releaseDateObj->value.GetInt64() };

                NN_RESULT_DO(nn::es::ImportPrepurchaseRecord(record));
            }

            // come_back_after は予約状態のチケットがある場合のみレスポンスに存在する
            auto comeBackAfterObj = document.FindMember("come_back_after");
            NN_RESULT_THROW_UNLESS(comeBackAfterObj != document.MemberEnd() && comeBackAfterObj->value.IsInt(), ResultUnexpectedResponseAccountGetETicketsComeBackAfterNotFound());
            *outComeBackAfter = comeBackAfterObj->value.GetInt();
        }

        *outIsPrepurchaseRecordDownloaded = isUnreleasedETicketFound;
        NN_RESULT_SUCCESS;
    }

    Result CreateGetRegistrationInforRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        char signatureBase64String[SignatureBase64StringLength];
        char certificateBase64String[CertificateBase64StringLength];
        NN_RESULT_DO(SignChallengeAndGetBase64String(signatureBase64String, SignatureBase64StringLength, certificateBase64String, CertificateBase64StringLength, challenge));

        char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
        NN_RESULT_DO(GetDeviceCertificateBase64StringForRegister(deviceCertificateBase64String, DeviceCertificateBase64StringLength));

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("signature", nne::rapidjson::StringRef(signatureBase64String), document.GetAllocator());
        document.AddMember("cert_chain", nne::rapidjson::StringRef(certificateBase64String), document.GetAllocator());
        document.AddMember("challenge", challenge, document.GetAllocator());
        document.AddMember("device_cert", nne::rapidjson::StringRef(deviceCertificateBase64String), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());

        NN_RESULT_SUCCESS;
    }

    Result PerformGetRegistrationInfoRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/GetRegistrationInfo");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        NN_RESULT_DO(CreateGetRegistrationInforRequestPostFields(postFields, PostFieldsLength, deviceId, challenge));

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseGetRegistrationInfoTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformGetRegistrationInfoRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseGetRegistrationInfoResponse(ec::system::DeviceAccountStatus* outStatus, DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken, char* readBuffer) NN_NOEXCEPT
    {
        ec::system::DeviceAccountStatus status = {};
        DeviceAccountToken deviceToken = {};
        DeviceAccountId accountId = {};

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseGetRegistrationInfoParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto accountStatusObj = document.FindMember("device_status");
        NN_RESULT_THROW_UNLESS(accountStatusObj != document.MemberEnd() && accountStatusObj->value.IsString(), ResultUnexpectedResponseGetRegistrationInfoDeviceStatusNotFound());
        util::Strlcpy(status.data, accountStatusObj->value.GetString(), static_cast<int>(sizeof(status.data)));

        // デバイス登録されている場合
        if (IsRegistered(status))
        {
            auto deviceTokenObj = document.FindMember("device_token");
            NN_RESULT_THROW_UNLESS(deviceTokenObj != document.MemberEnd() && deviceTokenObj->value.IsString(), ResultUnexpectedResponseGetRegistrationInfoDeviceTokenNotFound());
            NN_RESULT_THROW_UNLESS(deviceTokenObj->value.GetStringLength() < MaxDeviceAccountTokenLength, ResultUnexpectedResponseGetRegistrationInfoDeviceTokenTooLong());
            strcpy(deviceToken.data, deviceTokenObj->value.GetString());

            auto deviceTokenExpiredObj = document.FindMember("device_token_expired");
            NN_RESULT_THROW_UNLESS(deviceTokenExpiredObj != document.MemberEnd() && deviceTokenExpiredObj->value.IsBool(), ResultUnexpectedResponseGetRegistrationInfoDeviceTokenExpiredNotFound());
            status.isDeviceTokenExpired = deviceTokenExpiredObj->value.GetBool();

            auto accountIdObj = document.FindMember("account_id");
            NN_RESULT_THROW_UNLESS(accountIdObj != document.MemberEnd() && accountIdObj->value.IsString(), ResultUnexpectedResponseGetRegistrationInfoAccountIdNotFound());
            NN_RESULT_THROW_UNLESS(accountIdObj->value.GetStringLength() < AccountIdLength, ResultUnexpectedResponseGetRegistrationInfoAccountIdTooLong());
            strcpy(accountId.data, accountIdObj->value.GetString());
        }

        *outStatus = status;
        *outDeviceToken = deviceToken;
        *outAccountId = accountId;
        NN_RESULT_SUCCESS;
    }

    void CreateCompleteETicketSyncrRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        DeviceAccountToken weakDeviceToken = GetWeakDeviceAccountToken(deviceToken);
        document.AddMember("device_token", nne::rapidjson::StringRef(weakDeviceToken.data), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());
    }

    Result PerformCompleteETicketSyncRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, const DeviceAccountToken& deviceToken) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/CompleteETicketSync");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 256;
        char postFields[PostFieldsLength];
        CreateCompleteETicketSyncrRequestPostFields(postFields, PostFieldsLength, deviceId, deviceToken);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseCompleteETicketSyncTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformCompleteETicketSyncRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result PerformAccountTransferRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/AccountTransfer");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        // Register と同じ引数でよい
        CreateRegisterRequestPostFields(postFields, PostFieldsLength, deviceId, challenge);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseAccountTransferTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformAccountTransferRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseAccountTransferResponse(DeviceAccountToken* outDeviceToken, DeviceAccountId* outAccountId, char* readBuffer) NN_NOEXCEPT
    {
        DeviceAccountToken deviceToken;
        DeviceAccountId accountId;

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseAccountTransferParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto deviceTokenObj = document.FindMember("device_token");
        NN_RESULT_THROW_UNLESS(deviceTokenObj != document.MemberEnd() && deviceTokenObj->value.IsString(), ResultUnexpectedResponseAccountTransferDeviceTokenNotFound());
        NN_RESULT_THROW_UNLESS(deviceTokenObj->value.GetStringLength() < MaxDeviceAccountTokenLength, ResultUnexpectedResponseAccountTransferDeviceTokenTooLong());
        strcpy(deviceToken.data, deviceTokenObj->value.GetString());

        auto accountIdObj = document.FindMember("account_id");
        NN_RESULT_THROW_UNLESS(accountIdObj != document.MemberEnd() && accountIdObj->value.IsString(), ResultUnexpectedResponseAccountTransferAccountIdNotFound());
        NN_RESULT_THROW_UNLESS(accountIdObj->value.GetStringLength() < AccountIdLength, ResultUnexpectedResponseAccountTransferAccountIdTooLong());
        strcpy(accountId.data, accountIdObj->value.GetString());

        *outDeviceToken = deviceToken;
        *outAccountId = accountId;
        NN_RESULT_SUCCESS;
    }

    Result CreateSyncRegistrationRequestPostFields(char* outPostFields, int outPostFieldsLength, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        char messageId[MaxMessageIdLength];
        CreateMessageId(messageId, MaxMessageIdLength, deviceId);

        char signatureBase64String[SignatureBase64StringLength];
        char certificateBase64String[CertificateBase64StringLength];
        NN_RESULT_DO(SignChallengeAndGetBase64String(signatureBase64String, SignatureBase64StringLength, certificateBase64String, CertificateBase64StringLength, challenge));

        char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
        NN_RESULT_DO(GetDeviceCertificateBase64StringForRegister(deviceCertificateBase64String, DeviceCertificateBase64StringLength));

        nne::rapidjson::Document document;
        document.SetObject();

        document.AddMember("version", nne::rapidjson::StringRef(Version), document.GetAllocator());
        document.AddMember("message_id", nne::rapidjson::StringRef(messageId), document.GetAllocator());
        document.AddMember("device_id", nne::rapidjson::Value(deviceId), document.GetAllocator());
        document.AddMember("signature", nne::rapidjson::StringRef(signatureBase64String), document.GetAllocator());
        document.AddMember("cert_chain", nne::rapidjson::StringRef(certificateBase64String), document.GetAllocator());
        document.AddMember("challenge", challenge, document.GetAllocator());
        document.AddMember("device_cert", nne::rapidjson::StringRef(deviceCertificateBase64String), document.GetAllocator());

        nne::rapidjson::StringBuffer postFields;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(postFields);
        document.Accept(writer);

        nn::util::SNPrintf(outPostFields, outPostFieldsLength, "%s", postFields.GetString());

        NN_RESULT_SUCCESS;
    }

    Result PerformSyncRegistrationRequest(Response* outValue, HttpConnection* connection, Bit64 deviceId, int challenge) NN_NOEXCEPT
    {
        Url url("https://ias-%.hac.shop.nintendo.net/ias/services/rest/SyncRegistration");

        const char* HeaderList[] = {
            "Accept: application/json",
            "Content-Type: application/json",
        };

        const int PostFieldsLength = 2048;
        char postFields[PostFieldsLength];
        CreateSyncRegistrationRequestPostFields(postFields, PostFieldsLength, deviceId, challenge);

        Response response;
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Post(url, postFields,
            [&readSize, &response](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseSyncRegistrationTooLong());
                std::memcpy(&response.str[readSize], buffer, size);
                readSize += size;

                NN_RESULT_SUCCESS;
            },
            HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0])))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                response.str[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[PerformSyncRegistrationRequest] Error response %s\n", response.str);
                NN_RESULT_DO(ParseErrorResponseDestructively(response.str));

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        response.str[readSize] = '\0';

        *outValue = response;
        NN_RESULT_SUCCESS;
    }

    Result ParseSyncRegistrationResponse(DeviceAccountToken* outDeviceToken, DeviceAccountId* outAccountId, char* readBuffer) NN_NOEXCEPT
    {
        DeviceAccountToken deviceToken;
        DeviceAccountId accountId;

        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseSyncRegistrationParseError());

        auto codeObj = document.FindMember("error_code");
        NN_RESULT_THROW_UNLESS(codeObj != document.MemberEnd() && codeObj->value.IsInt(), ResultUnexpectedEShopErrorResponse());
        NN_RESULT_DO(InterpretEciErrorCode(codeObj->value.GetInt()));

        auto deviceTokenObj = document.FindMember("device_token");
        NN_RESULT_THROW_UNLESS(deviceTokenObj != document.MemberEnd() && deviceTokenObj->value.IsString(), ResultUnexpectedResponseSyncRegistrationDeviceTokenNotFound());
        NN_RESULT_THROW_UNLESS(deviceTokenObj->value.GetStringLength() < MaxDeviceAccountTokenLength, ResultUnexpectedResponseSyncRegistrationDeviceTokenTooLong());
        strcpy(deviceToken.data, deviceTokenObj->value.GetString());

        auto accountIdObj = document.FindMember("account_id");
        NN_RESULT_THROW_UNLESS(accountIdObj != document.MemberEnd() && accountIdObj->value.IsString(), ResultUnexpectedResponseSyncRegistrationAccountIdNotFound());
        NN_RESULT_THROW_UNLESS(accountIdObj->value.GetStringLength() < AccountIdLength, ResultUnexpectedResponseSyncRegistrationAccountIdTooLong());
        strcpy(accountId.data, accountIdObj->value.GetString());

        *outDeviceToken = deviceToken;
        *outAccountId = accountId;
        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::Initialize(DeviceContext* deviceContext, HttpConnection* connection) NN_NOEXCEPT
    {
        m_DeviceId = deviceContext->GetDeviceId();
        m_Connection = connection;

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::GetAccountStatus(ec::system::DeviceAccountStatus* outValue, const DeviceAccountToken& token) NN_NOEXCEPT
    {
        Response getAccountStatusResponse;
        NN_RESULT_DO(PerformGetAccountStatusRequest(&getAccountStatusResponse, m_Connection, m_DeviceId, token));

        NN_RESULT_DO(ParseGetAccountStatusResponse(outValue, getAccountStatusResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::Register(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT
    {
        Response getChallengeResponse;
        NN_RESULT_DO(PerformGetChallengeRequest(&getChallengeResponse, m_Connection, m_DeviceId));

        int challenge;
        NN_RESULT_DO(ParseGetChallengeResponse(&challenge, getChallengeResponse.str));

        Response registerResponse;
        NN_RESULT_DO(PerformRegisterRequest(&registerResponse, m_Connection, m_DeviceId, challenge));

        NN_RESULT_DO(ParseRegisterResponse(outDeviceToken, outAccountId, registerResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::Unregister() NN_NOEXCEPT
    {
        Response getChallengeResponse;
        NN_RESULT_DO(PerformGetChallengeRequest(&getChallengeResponse, m_Connection, m_DeviceId));

        int challenge;
        NN_RESULT_DO(ParseGetChallengeResponse(&challenge, getChallengeResponse.str));

        Response unregisterResponse;
        NN_RESULT_DO(PerformUnregisterRequest(&unregisterResponse, m_Connection, m_DeviceId, challenge));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::AccountListETicketIds(char buffer[], size_t bufferSize, int* outTicketIdListCount, Bit64 outTicketIdList[], int outTicketIdListLength, const DeviceAccountId& id, const DeviceAccountToken& token) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(bufferSize), 4096);

        *outTicketIdListCount = 0;
        NN_RESULT_DO(PerformAccountListETicketIdsRequest(buffer, bufferSize, m_Connection, m_DeviceId, id, token));

        NN_RESULT_DO(ParseAccountListETicketIdsResponse(outTicketIdListCount, outTicketIdList, outTicketIdListLength, buffer));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::AccountListETicketIdsByRightsId(int* outTicketIdListCount, Bit64 outTicketIdList[], int outTicketIdListLength, const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        Response accountListETicketIdsByRightsIdResponse;
        NN_RESULT_DO(PerformAccountListETicketIdsByRightsIdRequest(&accountListETicketIdsByRightsIdResponse, m_Connection, m_DeviceId, id, token, rightsId));

        NN_RESULT_DO(ParseAccountListETicketIdsResponse(outTicketIdListCount, outTicketIdList, outTicketIdListLength, accountListETicketIdsByRightsIdResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::AccountGetETickets(bool* outIsPrepurchaseRecordDownloaded, int* outComeBackAfter, char buffer[], size_t bufferSize, const DeviceAccountId& id, const DeviceAccountToken& token, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(bufferSize), 4096);

        // デバッグ用 ECI レスポンスのシミュレーション
        bool flag;
        size_t valueSize = settings::fwdbg::GetSettingsItemValue(&flag, sizeof(flag), "nim.install", "simulate_eci_delay_response");
        if (valueSize != sizeof(flag))
        {
            NN_DETAIL_NIM_WARN("[EciAccessor] Faild to get system settings. (simulate_eci_delay_response)\n");
            flag = false;
        }
        if (flag)
        {
            *outComeBackAfter = 30;
            *outIsPrepurchaseRecordDownloaded = true;
            NN_RESULT_SUCCESS;
        }

        valueSize = settings::fwdbg::GetSettingsItemValue(&flag, sizeof(flag), "nim.install", "simulate_eci_still_unavailable_response");
        if (valueSize != sizeof(flag))
        {
            NN_DETAIL_NIM_WARN("[EciAccessor] Faild to get system settings. (simulate_eci_still_unavailable_response)\n");
            flag = false;
        }
        if (flag)
        {
            *outIsPrepurchaseRecordDownloaded = true;
            NN_RESULT_SUCCESS;
        }

        // 与えられた buffer のサイズから一度に落としてくるチケットを計算(正確でないが暫定で計算)
        const int MinimumAccountGetETicketsResponseSize = 2730;
        const int OneRequestDownloadTicketCount = (static_cast<int>(bufferSize) - MinimumAccountGetETicketsResponseSize) / MaxETicketBase64StringLength;

        for (int i = 0; i < ticketIdListLength; i += OneRequestDownloadTicketCount)
        {
            int downloadTicketCount;

            if (i + OneRequestDownloadTicketCount <= ticketIdListLength)
            {
                downloadTicketCount = OneRequestDownloadTicketCount;
            }
            else
            {
                downloadTicketCount = ticketIdListLength - i;
            }

            NN_RESULT_DO(PerformAccountGetETicketsRequest(buffer, bufferSize, m_Connection, m_DeviceId, id, token, ticketIdList + i, downloadTicketCount));

            NN_RESULT_DO(ParseAccountGetETicketsResponseAndImportTicket(outIsPrepurchaseRecordDownloaded, outComeBackAfter, buffer));
        }

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::AccountGetETicketsForSyncTicket(char buffer[], size_t bufferSize, const DeviceAccountId& id, const DeviceAccountToken& token, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(bufferSize), 4096);

        NN_RESULT_DO(PerformAccountGetETicketsRequest(buffer, bufferSize, m_Connection, m_DeviceId, id, token, ticketIdList, ticketIdListLength));

        bool tmpIsPrepurchaseRecordDownloaded;
        int tmpComeBackAfter;
        NN_RESULT_DO(ParseAccountGetETicketsResponseAndImportTicket(&tmpIsPrepurchaseRecordDownloaded, &tmpComeBackAfter, buffer));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::GetRegistrationInfo(ec::system::DeviceAccountStatus* outStatus, DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT
    {
        Response getChallengeResponse;
        NN_RESULT_DO(PerformGetChallengeRequest(&getChallengeResponse, m_Connection, m_DeviceId));

        int challenge;
        NN_RESULT_DO(ParseGetChallengeResponse(&challenge, getChallengeResponse.str));

        Response getRegistrationInfoResponse;
        NN_RESULT_DO(PerformGetRegistrationInfoRequest(&getRegistrationInfoResponse, m_Connection, m_DeviceId, challenge));

        NN_RESULT_DO(ParseGetRegistrationInfoResponse(outStatus, outAccountId, outDeviceToken, getRegistrationInfoResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::CompleteETicketSync(const DeviceAccountToken& token) NN_NOEXCEPT
    {
        Response completeETicketSyncResponse;
        NN_RESULT_DO(PerformCompleteETicketSyncRequest(&completeETicketSyncResponse, m_Connection, m_DeviceId, token));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::AccountTransfer(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT
    {
        Response getChallengeResponse;
        NN_RESULT_DO(PerformGetChallengeRequest(&getChallengeResponse, m_Connection, m_DeviceId));

        int challenge;
        NN_RESULT_DO(ParseGetChallengeResponse(&challenge, getChallengeResponse.str));

        Response accountTransferResponse;
        NN_RESULT_DO(PerformAccountTransferRequest(&accountTransferResponse, m_Connection, m_DeviceId, challenge));

        NN_RESULT_DO(ParseAccountTransferResponse(outDeviceToken, outAccountId, accountTransferResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::SyncRegistration(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT
    {
        Response getChallengeResponse;
        NN_RESULT_DO(PerformGetChallengeRequest(&getChallengeResponse, m_Connection, m_DeviceId));

        int challenge;
        NN_RESULT_DO(ParseGetChallengeResponse(&challenge, getChallengeResponse.str));

        Response syncRegistrationResponse;
        NN_RESULT_DO(PerformSyncRegistrationRequest(&syncRegistrationResponse, m_Connection, m_DeviceId, challenge));

        NN_RESULT_DO(ParseSyncRegistrationResponse(outDeviceToken, outAccountId, syncRegistrationResponse.str));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::DownloadTicket(const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        int outTicketIdCount;
        nn::Bit64 ticketIdList[10];
        char buffer[4096];

        NN_RESULT_DO(AccountListETicketIdsByRightsId(&outTicketIdCount, ticketIdList, sizeof(ticketIdList) / sizeof(ticketIdList[0]), id, token, rightsId));

        bool tmpIsPrepurchaseRecordDownloaded;
        int tmpComeBackAfter;
        NN_RESULT_DO(AccountGetETickets(&tmpIsPrepurchaseRecordDownloaded, &tmpComeBackAfter, buffer, sizeof(buffer), id, token, ticketIdList, outTicketIdCount));

        NN_RESULT_SUCCESS;
    }

    Result EciAccessor::DownloadTicketForPrepurchasedContents(ec::system::TicketDownloadStatusForPrepurchasedContents* outStatus, const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
        int outTicketIdCount;
        nn::Bit64 ticketIdList[10];
        char buffer[4096];

        NN_RESULT_DO(AccountListETicketIdsByRightsId(&outTicketIdCount, ticketIdList, sizeof(ticketIdList) / sizeof(ticketIdList[0]), id, token, rightsId));
        NN_RESULT_DO(AccountGetETickets(&outStatus->isPrepurchaseRecordDownloaded, &outStatus->delayTime, buffer, sizeof(buffer), id, token, ticketIdList, outTicketIdCount));

        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------------------------------
    namespace DynamicETickets {

        //-------------------------------------------------------------------------------------
        //! @brief  デフォルトアライン
        constexpr static size_t DefaultAlignment = sizeof(Bit32);

        //-------------------------------------------------------------------------------------
        template<typename TYPE, size_t SIZE>
        inline constexpr static size_t LengthOf(const TYPE(&)[SIZE]) NN_NOEXCEPT
        {
            return SIZE - 1;
        }

        //-------------------------------------------------------------------------------------
        //! @brief  動的チケットトークン中のヘッダ要素に該当する領域サイズ上限。( null 終端含まず )
        //! @note   実装時想定では 230 byte.
        constexpr size_t MaxSizePerHeaderInToken = 384u;
        //-------------------------------------------------------------------------------------
        //! @brief  動的チケットトークン中のチケット配列要素一つに該当する領域サイズ上限。( null 終端含まず )
        //! @note   実装時想定では 200 byte.
        constexpr size_t MaxSizePerLicenseInToken = 384u;
        //-------------------------------------------------------------------------------------
        //! @brief  動的チケットトークンレスポンスデータ固定領域上限。
        constexpr size_t MaxSizePerFixedAreaInETicketTokenResponse = LengthOf("{\"eticket_token\": \"\"}") + MaxSizePerHeaderInToken;

        //-------------------------------------------------------------------------------------
        //! @brief  ECI要求時のポストデータ固定領域上限。( null 終端込み )
        constexpr size_t MaxSizePerFixedAreaInEciPostData =
            LengthOf("\"version\": \"\"") + LengthOf(Version) +
            LengthOf("\"message_id\": \"\"") + (MaxMessageIdLength - 1) +
            LengthOf("\"device_id\": ") + (std::numeric_limits<uint64_t>::digits10 + 1) +       // digits10 は「正常利用できる有効桁」、uint64_t の表現最大値は20桁なので +1。
            LengthOf("\"device_token\": \"\"") + (sizeof(DeviceAccountToken) - 1) +
            LengthOf("\"device_cert\": \"\"") + (DeviceCertificateBase64StringLength - 1) +
            LengthOf("\"account_id\": \"\"") + (sizeof(DeviceAccountId) - 1) +
            LengthOf("\"dynamic_eticket_token\": \"\"") +                                       // eTicketToken の値は可変長。
            sizeof("{}");                                                                       // null終端 + JSONブロック。

        //-------------------------------------------------------------------------------------
        //! @brief  ECI要求時のレスポンスデータ固定領域上限。( EciAccessor::AccountGetETickets() の定義を参照 )
        constexpr size_t MaxSizePerFixedAreaInEciResponse = 2730u;

        //-------------------------------------------------------------------------------------
        //! @brief  ECI要求時のレスポンスデータ１チケット領域上限。( null 終端込み )
        constexpr size_t MaxSizePerOneTicketInEciResponse = MaxETicketBase64StringLength;

        //-------------------------------------------------------------------------------------
        //! @brief  動的チケットトークンレスポンスの指定ライセンス数に基づく期待サイズ算出。( null 終端含まず )
        inline static const size_t ComputeExpectSizeForETicketTokenResponse(size_t nLicense) NN_NOEXCEPT
        {
            return ::nn::util::align_up(MaxSizePerFixedAreaInETicketTokenResponse + (MaxSizePerLicenseInToken * nLicense), DefaultAlignment);
        }

        //-------------------------------------------------------------------------------------
        //! @brief  ECI要求時のレスポンスデータの指定ライセンス数に基づく期待サイズ算出。( null 終端含まず )
        inline static const size_t ComputeExpectSizeForEciResponse(size_t nLicense) NN_NOEXCEPT
        {
            return ::nn::util::align_up(MaxSizePerFixedAreaInEciResponse + (MaxSizePerOneTicketInEciResponse * nLicense), DefaultAlignment);
        }

        //-------------------------------------------------------------------------------------
        //! @brief  ECI要求時のポストデータの指定ライセンス数に基づく期待サイズ算出。( null 終端含まず )
        inline static const size_t ComputeExpectSizeForEciPostData(size_t nLicense) NN_NOEXCEPT
        {
            return ::nn::util::align_up(MaxSizePerFixedAreaInEciPostData + MaxSizePerHeaderInToken + (MaxSizePerLicenseInToken * nLicense), DefaultAlignment);
        }

        //-------------------------------------------------------------------------------------
        //! @brief  "https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/AccountGetDynamicETickets" 用リクエストポストデータの作成。
        Result CreatePostFields(char* outPostFields, size_t postFieldsLength, Bit64 deviceId, const DeviceAccountId& accountId, const DeviceAccountToken& deviceToken, const char* pDynamicETicketToken) NN_NOEXCEPT
        {
            char messageId[MaxMessageIdLength];
            CreateMessageId(messageId, MaxMessageIdLength, deviceId);

            char deviceCertificate[DeviceCertificateForETicketLength];
            size_t deviceCertificateSize = GetDeviceCertificateForETicket(deviceCertificate, DeviceCertificateForETicketLength);

            char deviceCertificateBase64String[DeviceCertificateBase64StringLength];
            nn::util::Base64::Status status = nn::util::Base64::ToBase64String(deviceCertificateBase64String, DeviceCertificateBase64StringLength, deviceCertificate, deviceCertificateSize, nn::util::Base64::Mode::Mode_NormalNoLinefeed);
            NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultConvertBase64Failed());

            DeviceAccountToken weakDeviceToken = GetWeakDeviceAccountToken(deviceToken);

            constexpr char PostBase[] =
                "{"
                    "\"version\":\"%s\","
                    "\"message_id\":\"%s\","
                    "\"device_id\":%llu,"
                    "\"account_id\":\"%s\","
                    "\"device_cert\":\"%s\","
                    "\"device_token\":\"%s\","
                    "\"dynamic_eticket_token\":\"%s\""
                "}";

            const auto length = ::nn::util::SNPrintf(outPostFields, postFieldsLength, PostBase
                , Version
                , messageId
                , deviceId
                , accountId.data
                , deviceCertificateBase64String
                , weakDeviceToken.data
                , pDynamicETicketToken
            );
            NN_RESULT_THROW_UNLESS(length >= 0 && static_cast<size_t>(length) < postFieldsLength, ResultBufferNotEnough());
            NN_RESULT_SUCCESS;
        }

    }   // ~DynamicETickets

    //-------------------------------------------------------------------------------------
    size_t EciAccessor::DynamicETicketsContext::CalculateAvailableLicenseCount(const size_t requiredSize) NN_NOEXCEPT
    {
        // 最小要件( 1ライセンス )の条件チェック。
        //  NOTE: (ETicketTokenResponse + EciPostData) 領域と EciResponse 領域は用途契機上、排他共有できる想定で計算。
        //  TODO: 将来的には ETicketTokenResponse の $token 部のパース結果を直接 EciPostData の "dynamic_eticket_token" 要素部に出力するようにする。
        const size_t minimumEciResponse = DynamicETickets::ComputeExpectSizeForEciResponse(1);
        const size_t minimumEciPostData = DynamicETickets::ComputeExpectSizeForEciPostData(1);
        const size_t minimumTokenResponse = DynamicETickets::ComputeExpectSizeForETicketTokenResponse(1);
        const size_t requiredMinimumSize = std::max(minimumEciPostData + minimumTokenResponse, minimumEciResponse);
        if (requiredSize < requiredMinimumSize)
        {
            return 0;
        }
        // ここで 1ライセンスは保証された。
        size_t expectCount = 1;
        m_CapacityForEciResponse = minimumEciResponse;
        m_CapacityForEciPostData = minimumEciPostData;
        m_CapacityForTokenResponse = minimumTokenResponse;

        // 残量から簡易算出。
        // eci-post, eci-response, eticket-token-response の各変動値上限で残量を割ります。
        const size_t maximumSizeForTokenPerOneTicket = DynamicETickets::MaxSizePerLicenseInToken + DynamicETickets::MaxSizePerLicenseInToken;
        const size_t requiredMaximumSizePerOneTicket = std::max(maximumSizeForTokenPerOneTicket, DynamicETickets::MaxSizePerOneTicketInEciResponse);
        const size_t earlyCount = (requiredSize - requiredMinimumSize) / requiredMaximumSizePerOneTicket;

        // アライン含めて検証
        for (expectCount += earlyCount; expectCount > 1u; --expectCount)
        {
            const size_t expectEciResponse = DynamicETickets::ComputeExpectSizeForEciResponse(expectCount);
            const size_t expectEciPostData = DynamicETickets::ComputeExpectSizeForEciPostData(expectCount);
            const size_t expectTokenResponse = DynamicETickets::ComputeExpectSizeForETicketTokenResponse(expectCount);
            const size_t expectRequiredMaximumSize = std::max(expectEciPostData + expectTokenResponse, expectEciResponse);
            if (requiredSize >= expectRequiredMaximumSize)
            {
                // 指定容量内に期待値が収まったので、この時点の expectCount が要求可能なライセンス数。
                m_CapacityForEciResponse = expectEciResponse;
                m_CapacityForEciPostData = expectEciPostData;
                m_CapacityForTokenResponse = expectTokenResponse;
                break;
            }
        }
        return expectCount;
    }

    //-------------------------------------------------------------------------------------
    size_t EciAccessor::DynamicETicketsContext::Initialize(char* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
        NN_ABORT_UNLESS(0 == (reinterpret_cast<uintptr_t>(pBuffer) % DefaultAlignment));

        const auto availableCount = CalculateAvailableLicenseCount(bufferSize);
        if (availableCount > 0u)
        {
            m_AvailableLicenseCount = availableCount;
            m_BufferSize = bufferSize;
            m_pBuffer = pBuffer;
        }
        return availableCount;
    }

    //-------------------------------------------------------------------------------------
    char* EciAccessor::DynamicETicketsContext::GetBufferForETicketToken() NN_NOEXCEPT
    {
        return (IsAvailable()) ? &m_pBuffer[0] : nullptr;
    }

    //-------------------------------------------------------------------------------------
    char* EciAccessor::DynamicETicketsContext::GetBufferForEciPostData() NN_NOEXCEPT
    {
        return (IsAvailable()) ? &m_pBuffer[m_CapacityForTokenResponse] : nullptr;
    }

    //-------------------------------------------------------------------------------------
    char* EciAccessor::DynamicETicketsContext::GetBufferForEciResponse() NN_NOEXCEPT
    {
        // NOTE: EciPostData + ETicketToken 領域と排他共有を想定したバッファ先頭からの利用です。
        return (IsAvailable()) ? &m_pBuffer[0] : nullptr;
    }

    //-------------------------------------------------------------------------------------
    EciAccessor::DynamicETicketsContext::DynamicETicketsContext() NN_NOEXCEPT
        : m_pBuffer(nullptr)
        , m_BufferSize(0u)
        , m_AvailableLicenseCount(0u)
        , m_CapacityForEciPostData(0u)
        , m_CapacityForEciResponse(0u)
        , m_CapacityForTokenResponse(0u)
    {
    }

    //-------------------------------------------------------------------------------------
    Result EciAccessor::RequestDynamicETickets(DynamicETicketsContext& context, const DeviceAccountId& deviceAccountId, const DeviceAccountToken& deviceToken, const char* pDynamicETicketToken) NN_NOEXCEPT
    {
        // ポストデータ作成を呼び出す。
        const auto pPostData = context.GetBufferForEciPostData();
        const auto postCapacity = context.GetCapacityForEciPostData();
        NN_RESULT_DO(DynamicETickets::CreatePostFields(pPostData, postCapacity, m_DeviceId, deviceAccountId, deviceToken, pDynamicETicketToken));

        // 通信リクエスト。
        return RequestDynamicETicketsImpl(context.GetBufferForEciResponse(), context.GetCapacityForEciResponse(), pPostData);
    }

    //-------------------------------------------------------------------------------------
    Result EciAccessor::RequestDynamicETicketsImpl(char* pWorkBuffer, size_t bufferSize, char* pPostData) NN_NOEXCEPT
    {
        const char* HeaderList[] =
        {
            "Accept: application/json",
            "Content-Type: application/json",
        };
        const char url[] = "https://ecs-%.hac.shop.nintendo.net/ecs/services/rest/AccountGetDynamicETickets";

        size_t readSize = 0;
        NN_RESULT_TRY(m_Connection->Post(url, pPostData,
            [&readSize, &pWorkBuffer, &bufferSize](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < bufferSize, ResultUnexpectedResponseAccountGetETicketsTooLong());
                std::memcpy(&pWorkBuffer[readSize], buffer, size);
                readSize += size;
                NN_RESULT_SUCCESS;
            },
            HeaderList, NN_ARRAY_SIZE(HeaderList)))
            NN_RESULT_CATCH(ResultHttpStatus)
            {
                pWorkBuffer[readSize] = '\0';
                NN_DETAIL_NIM_TRACE("[RequestDynamicETickets] Error response %s\n", pWorkBuffer);
                NN_RESULT_DO(ParseErrorResponseDestructively(pWorkBuffer));
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;
        pWorkBuffer[readSize] = '\0';

        int comeBackAfter;
        bool isPrepurchaseRecordDownloaded;
        NN_RESULT_DO(ParseAccountGetETicketsResponseAndImportTicket(&isPrepurchaseRecordDownloaded, &comeBackAfter, pWorkBuffer));
        NN_RESULT_SUCCESS;
    }

}}}
