﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_DynamicRightsReportELicenses.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::ReportELicenses] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
}   //  ~unnamed
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
AsyncReportELicensesImplBase::AsyncReportELicensesImplBase() NN_NOEXCEPT
    : m_AccountId(::nn::account::InvalidNintendoAccountId)
{
}

//-----------------------------------------------------------------------------
AsyncReportELicensesImplBase::~AsyncReportELicensesImplBase() NN_NOEXCEPT
{
};

//-----------------------------------------------------------------------------
Result AsyncReportELicensesImplBase::Initialize(DeviceContext* pDeviceContext
    , const ::nn::account::NintendoAccountId& naId
    , const ::nn::sf::InArray<::nn::es::ELicenseId>& elicenseIds
    , const char* pTemporaryFilePath) NN_NOEXCEPT
{
    // 不正なパラメタチェック
    const auto nLicenses = elicenseIds.GetLength();
    NN_RESULT_THROW_UNLESS(nLicenses > 0, ResultContentNotFound());
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));

    m_AccountId = naId;
    m_ValueStore.Initialize(pTemporaryFilePath);

    // 入力 rigths をファイルへ
    m_ValueStore.SetInitialFileSize(nLicenses * sizeof(::nn::es::ELicenseId));
    NN_RESULT_DO(m_ValueStore.Append(elicenseIds.GetData(), nLicenses * sizeof(::nn::es::ELicenseId)));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncReportELicensesImplBase::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    NN_SDK_ASSERT_NOT_NULL(pOutPathUrl);

    NN_RESULT_DO(OnQueryAccessPathUrl(pOutPathUrl, availablePathUrlCapacity));
    pOut->uid = ::nn::account::InvalidUid;
    pOut->naId = m_AccountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncReportELicensesImplBase::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // eLicense データの有効性確認。
    auto nLicenses = static_cast<size_t>(m_ValueStore.GetWrittenSize() / sizeof(::nn::es::ELicenseId));
    NN_RESULT_THROW_UNLESS(nLicenses > 0U, ResultContentNotFound());

    char data[96];
    constexpr char PostBasePrefix[] = "{\"elicense_ids\": [";
    constexpr char PostBaseSuffix[] = "],\"account_ids\": [\"%016llx\"]}";
    const auto suffixLength = ::nn::util::SNPrintf(data, sizeof(data), PostBaseSuffix, m_AccountId.id);
    NN_ABORT_UNLESS(suffixLength < sizeof(data));

    // ポストデータバッファ確保。( null終端含めてざっくり確保、35 は "\"32桁\"," の文字数 )
    const size_t requiredPostSize = sizeof(PostBasePrefix) + suffixLength + static_cast<size_t>(35 * nLicenses);
    NN_RESULT_DO(m_Post.Acquire(sizeof(Bit64), requiredPostSize));
    const auto pOutPost = static_cast<char*>(m_Post.pTop);

    size_t postLength = 0;
    NN_RESULT_DO(AppendRequestBodyAsELicenses(&postLength, pOutPost, requiredPostSize, m_ValueStore, 0, nLicenses, PostBasePrefix, data));
    DEBUG_TRACE("Post filed generate complete, (heap: %zu, length: %zu)=>`%s`\n", requiredPostSize, postLength, pOutPost);
    NN_SDK_ASSERT(postLength == std::strlen(pOutPost)); // NOTE: デバッグ用の保険検証
    NN_RESULT_DO(pConnection->SetTransactionPostFields(pOutPost));

    // "Content-Length", "Content-Type" ヘッダ。
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncReportELicensesImplBase::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // 報告系 API は解析すべきレスポンスストリームはなし。
    NN_UNUSED(pInputStream);
    NN_RESULT_SUCCESS;
}


//!============================================================================
//! @name AsyncReportELicensesImpl 実装
//
// NOTE:
//  Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
AsyncReportELicensesImpl::AsyncReportELicensesImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator())
    , AsyncReportELicensesImplBase()
{
}

//-----------------------------------------------------------------------------
AsyncReportELicensesImpl::~AsyncReportELicensesImpl() NN_NOEXCEPT
{
    Join();
};

//-----------------------------------------------------------------------------
Result AsyncReportELicensesImpl::OnQueryAccessPathUrl(char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    constexpr char base[] = "/v1/elicenses/exercise";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);
    NN_RESULT_SUCCESS;
}


//!============================================================================
//! @name AsyncReportELicensesPassivelyImpl 実装
//
// NOTE:
//  Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
AsyncReportELicensesPassivelyImpl::AsyncReportELicensesPassivelyImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator())
    , AsyncReportELicensesImplBase()
{
}

//-----------------------------------------------------------------------------
AsyncReportELicensesPassivelyImpl::~AsyncReportELicensesPassivelyImpl() NN_NOEXCEPT
{
    Join();
};

//-----------------------------------------------------------------------------
Result AsyncReportELicensesPassivelyImpl::OnQueryAccessPathUrl(char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    constexpr char base[] = "/v1/elicenses/report";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);
    NN_RESULT_SUCCESS;
}


}   // ~DynamicRights
}}} // ~nn::nim::srv
