﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <type_traits>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_Result.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/util/util_Span.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_FormatString.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_DynamicRightsUrl.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_FileSystem.h>

#include "nim_StringUtil.h"
#include "nim_DynamicRightsCommon.h"
#include "nim_ShopServiceAccessDebug.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...)            NN_DETAIL_NIM_TRACE( "[DynamicRights] " __VA_ARGS__ )
#define DEBUG_TRACE_RESPONSE(...)   DebugTraceForResponse(__VA_ARGS__)
#else
#define DEBUG_TRACE(...)            static_cast<void>(0)
#define DEBUG_TRACE_RESPONSE(...)   static_cast<void>(0)
#endif

//-----------------------------------------------------------------------------
#if !defined(NN_SDK_BUILD_RELEASE)
void DebugTraceForResponse(const char* pBuffer, const size_t bufferSize) NN_NOEXCEPT
{
    auto pRear = const_cast<char*>(&pBuffer[bufferSize - 1]);
    const auto rearCode = *pRear;
    *pRear = '\0';
    DEBUG_TRACE("Response(%zu) =>\n%s%c\n\n", bufferSize, pBuffer, rearCode);
    *pRear = rearCode;
}
#endif

//-----------------------------------------------------------------------------
//! @brief  スレッド情報
constexpr int ThreadCount = 1;
constexpr size_t StackSize = 16 * 1024;

//-----------------------------------------------------------------------------
//! @brief  デバイス認証トークン用クライアントID ( for Dragons )
//! @note   http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=318897444
static constexpr uint64_t ClientIdForDeviceToken = 0xd5b6cac2c1514c56ull;

//-----------------------------------------------------------------------------
//! @brief  NA-ID リクエストヘッダキー
constexpr char HeaderPrefixForNaId[] = "Nintendo-Account-Id: ";

//-----------------------------------------------------------------------------
//! @brief  NA-ID トークンリクエストヘッダキー
constexpr char HeaderPrefixForNaIdToken[] = "AccountAuthorization: Bearer ";

//-----------------------------------------------------------------------------
//! @brief  デバイス認証トークンリクエストヘッダキー
constexpr char HeaderPrefixForDeviceToken[] = "DeviceAuthorization: Bearer ";

//-----------------------------------------------------------------------------
//! @brief  リクエストヘッダ NA-ID 上限長。
constexpr size_t LengthMaxForNaId = static_cast<size_t>((sizeof(HeaderPrefixForNaId) - 1) + 16);

//-----------------------------------------------------------------------------
//! @brief  リクエストヘッダ NA-ID-TOKEN 上限長。
constexpr size_t LengthMaxForNaIdToken = static_cast<size_t>((sizeof(HeaderPrefixForNaIdToken) - 1) + TokenStore::NaId::LengthMax);

//-----------------------------------------------------------------------------
//! @brief  リクエストヘッダ デバイス認証トークン上限長。
constexpr size_t LengthMaxForDeviceToken = static_cast<size_t>((sizeof(HeaderPrefixForDeviceToken) - 1) + TokenStore::DeviceAuth::LengthMax);

//-----------------------------------------------------------------------------
//! @brief  スレッドスタック
NN_OS_ALIGNAS_THREAD_STACK char g_Stack[StackSize * ThreadCount];

//-----------------------------------------------------------------------------
//! @brief  スレッドプール
class ThreadPool : public ThreadAllocator
{
private:
    ::nn::os::ThreadType m_List[ThreadCount];

public:
    // @note ShopServiceAccessTask は暫定( TODO )。
    NN_IMPLICIT ThreadPool() NN_NOEXCEPT
        : ThreadAllocator(m_List, NN_ARRAY_SIZE(m_List)
        , NN_SYSTEM_THREAD_PRIORITY(nim, ShopServiceAccessTask)
        , g_Stack, sizeof(g_Stack), StackSize
        , NN_SYSTEM_THREAD_NAME(nim, ShopServiceAccessTask)) {}
};

//-----------------------------------------------------------------------------
//! @brief      NaId トークンキャッシュ有効期間
//! @details    Dragons の NA ID トークン受付時間は 2時間( 2018/4/27 )。@n
//!             ShopServiceAccessor に倣って 1時間の利用期間とする。
//! @note       通常のNA-ID-TOKEN期限は15分。
const ::nn::TimeSpan NaIdCachedTokenAvailabilityPeriod = ::nn::TimeSpan::FromHours(1);

//-----------------------------------------------------------------------------
//! @brief  NaId トークンキャッシュ
class NaIdTokenImpl : public ::nn::nim::srv::TokenStore::NaId::SingleCacheBase
{
    NN_DISALLOW_COPY(NaIdTokenImpl);
    NN_DISALLOW_MOVE(NaIdTokenImpl);

public:
    typedef ::nn::nim::srv::TokenStore::NaId::SingleCacheBase BaseType;

    explicit NaIdTokenImpl() NN_NOEXCEPT : BaseType(NaIdCachedTokenAvailabilityPeriod) {}

protected:
    virtual Result PrepareAuthorizationRequestSettings(::nn::nsd::NasServiceSetting* pOutService, TokenStore::NaId::AuthParameter* pOutAuth) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAuth);
        NN_SDK_ASSERT_NOT_NULL(pOutService);
        NN_RESULT_DO(::nn::nsd::GetNasServiceSetting(pOutService, ::nn::nsd::NasServiceNameOfNxELicense));

        NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->scope, "openid user", sizeof(pOutAuth->scope)) < sizeof(pOutAuth->scope));
        NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->state, " ", sizeof(pOutAuth->state)) < sizeof(pOutAuth->state));
        NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutAuth->nonce, " ", sizeof(pOutAuth->nonce)) < sizeof(pOutAuth->nonce));
        NN_RESULT_SUCCESS;
    }

};

//-----------------------------------------------------------------------------
//! @brief  共有コンテキスト
class SharedContext
{
public:
    static SharedContext* GetInstance() NN_NOEXCEPT;

    Result CreateHeaderForNaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStore::NaId::ICanceler* pCanceler) NN_NOEXCEPT;
    Result CreateHeaderForDeviceAuthenticationToken(char* pTemporary, size_t temporarySize, TokenStore::DeviceAuth::ICanceler* pCanceler) NN_NOEXCEPT;

private:
    explicit SharedContext() NN_NOEXCEPT : m_NaIdToken() {}

    NaIdTokenImpl   m_NaIdToken;
};

//-----------------------------------------------------------------------------
SharedContext* SharedContext::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(SharedContext, s_SharedContext);
    return &s_SharedContext;
}

//-----------------------------------------------------------------------------
Result SharedContext::CreateHeaderForNaIdToken(char* pTemporary, size_t temporarySize, const ::nn::account::Uid& uid, TokenStore::NaId::ICanceler* pCanceler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCanceler);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    NN_RESULT_THROW_UNLESS(temporarySize >= LengthMaxForNaIdToken, ResultBufferNotEnough());

    // リクエストヘッダ Prefix 設定。
    const size_t lengthOfNaIdTokenHeader = sizeof(HeaderPrefixForNaIdToken) - 1;
    std::memcpy(pTemporary, HeaderPrefixForNaIdToken, lengthOfNaIdTokenHeader);
    const size_t avilableSize = temporarySize - lengthOfNaIdTokenHeader;

    // NA-ID トークンの取得。
    NN_RESULT_TRY(m_NaIdToken.AcquireToken(&pTemporary[lengthOfNaIdTokenHeader], avilableSize, uid, pCanceler))
        NN_RESULT_CATCH_CONVERT(::nn::account::ResultCancelled, ResultHttpConnectionCanceled())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result SharedContext::CreateHeaderForDeviceAuthenticationToken(char* pTemporary, size_t temporarySize, TokenStore::DeviceAuth::ICanceler* pCanceler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pCanceler);
    NN_RESULT_THROW_UNLESS(nullptr != pTemporary, ResultBufferNotEnough());
    NN_RESULT_THROW_UNLESS(temporarySize >= LengthMaxForDeviceToken, ResultBufferNotEnough());

    // リクエストヘッダ Prefix 設定。
    const size_t lengthOfDeviceTokenHeader = sizeof(HeaderPrefixForDeviceToken) - 1;
    std::memcpy(pTemporary, HeaderPrefixForDeviceToken, lengthOfDeviceTokenHeader);
    const size_t avilableSize = temporarySize - lengthOfDeviceTokenHeader;

    // デバイス認証トークンの取得。
    NN_RESULT_TRY(TokenStore::SharedStore::GetSharedInstance()->device.AcquireToken(pCanceler, ClientIdForDeviceToken, &pTemporary[lengthOfDeviceTokenHeader], avilableSize))
        NN_RESULT_CATCH_CONVERT(::nn::dauth::ResultCancelled, ResultHttpConnectionCanceled())
        NN_RESULT_CATCH(::nn::dauth::detail::ResultProgramError)
        {
            NN_ABORT("[DynamicRights::Server] Unexpected failure as `dauth::detail::ResultProgramError` at the `CreateHeaderForDeviceAuthenticationToken`.\n");
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief   デバッグレスポンス検出結果受信用コンテナクラス。
class DebugResponseReceiver : public ShopServiceAccessDebug::Response::EmulationResult
{
    NN_DISALLOW_COPY(DebugResponseReceiver);
    NN_DISALLOW_MOVE(DebugResponseReceiver);

    typedef HeapUtil::OnetimeHeapSession    BufferAllocatorType;

public:
    explicit DebugResponseReceiver(BufferAllocatorType* pAllocator) NN_NOEXCEPT : m_pAllocator(pAllocator) {}

    bool Find(const char* pVerifyPath) NN_NOEXCEPT
    {
        return (ShopServiceAccessDebug::Response::DoUseDebugResponse(this, pVerifyPath, ShopServiceAccessDebug::Response::DragonsServer).IsSuccess() && IsApplied());
    }

    virtual void* AllocateResponseStore(size_t responseSize) NN_NOEXCEPT NN_OVERRIDE
    {
        BufferAllocatorType* const pAllocator = m_pAllocator;
        return (nullptr != pAllocator && pAllocator->Acquire(sizeof(uintptr_t), responseSize).IsSuccess()) ? pAllocator->pTop : nullptr;
    }

    virtual Result ExecuteAliasReplaceForUrl(char* const pOut, const size_t outCapacity, const char* pSource) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(outCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

        const auto capacity = static_cast<int>(outCapacity);
        NN_RESULT_THROW_UNLESS(::nn::util::Strlcpy(pOut, pSource, capacity) < capacity, ResultBufferNotEnough());
        NN_RESULT_SUCCESS;
    }

private:
    BufferAllocatorType* const m_pAllocator;    //!< レスポンスバッファをユーザ指定ワークメモリから確保するためのハンドル。
};

//-----------------------------------------------------------------------------
//! @brief  エラーコードレスポンスボディを持つ HTTPステータスコード Result の判定。
//! @note   TODO: Dragons サーバーにおいて、対象ステータスコードが妥当かどうかの確認はできていない。
bool HasErrorCodeResponse(const Result result) NN_NOEXCEPT
{
    return ResultHttpStatus::Includes(result) && (
        ResultHttpStatus400BadRequest::Includes(result)
        || ResultHttpStatus401Unauthorized::Includes(result)
        || ResultHttpStatus403Forbidden::Includes(result)
        || ResultHttpStatus404NotFound::Includes(result)
        || ResultHttpStatus405MethodNotAllowed::Includes(result)
        || ResultHttpStatus406NotAcceptable::Includes(result)
        || ResultHttpStatus407ProxyAuthenticationRequired::Includes(result)
        || ResultHttpStatus408RequestTimeout::Includes(result)
        || ResultHttpStatus409Conflict::Includes(result)
        || ResultHttpStatus410Gone::Includes(result)
        || ResultHttpStatus411LengthRequired::Includes(result)
        || ResultHttpStatus412PreconditionFailed::Includes(result)
        || ResultHttpStatus413PayloadTooLarge::Includes(result)
        || ResultHttpStatus414UriTooLong::Includes(result)
        || ResultHttpStatus415UnsupportedMediaType::Includes(result)
        || ResultHttpStatus416RangeNotSatisfiable::Includes(result)
        || ResultHttpStatus417ExpectationFailed::Includes(result)
        || ResultHttpStatus500InternalServerError::Includes(result)
        || ResultHttpStatus501NotImplemented::Includes(result)
        || ResultHttpStatus502BadGateway::Includes(result)
        || ResultHttpStatus503ServiceUnavailable::Includes(result)
        || ResultHttpStatus504GatewayTimeout::Includes(result)
        || ResultHttpStatus505HttpVersionNotSupported::Includes(result)
        || ResultHttpStatus509BandwidthLimitExceeded::Includes(result)
    );
}

//-----------------------------------------------------------------------------
//! @brief  文字列定数
namespace StringConstants {

    template<typename ValueType>
    struct MappedValue
    {
        const char* pString;
        const int   length;
        ValueType   value;
    };

    namespace ELicenseType {
        constexpr char Temporary[] = "temporary";
        constexpr char Permanent[] = "permanent";
        constexpr char Unavailable[] = "unavailable";
        constexpr char DeviceLinkedPermanent[] = "device_linked_permanent";

        constexpr MappedValue<Dragons::ELicenseType> g_Values[] =
        {
            {Temporary, sizeof(Temporary) - 1, Dragons::ELicenseType::Temporary},
            {Permanent, sizeof(Permanent) - 1, Dragons::ELicenseType::Permanent},
            {Unavailable, sizeof(Unavailable) - 1, Dragons::ELicenseType::Unavailable},
            {DeviceLinkedPermanent, sizeof(DeviceLinkedPermanent) - 1, Dragons::ELicenseType::DeviceLinkedPermanent},
        };
    }

    namespace Reason {
        constexpr char UsedByAnother[] = "used_by_another";
        constexpr char LimitExceeded[] = "limit_exceeded";
        constexpr char NotDeviceLinked[] = "not_device_linked";
        constexpr char NotReleased[] = "not_released";
        constexpr char NoRights[] = "no_rights";
        constexpr char Expired[] = "expired";

        constexpr MappedValue<Dragons::Reason> g_Values[] =
        {
            {LimitExceeded, sizeof(LimitExceeded) - 1, Dragons::Reason::LimitExceeded},
            {NoRights, sizeof(NoRights) - 1, Dragons::Reason::NoRights},
            {NotDeviceLinked, sizeof(NotDeviceLinked) - 1, Dragons::Reason::NotDeviceLinked},
            {NotReleased, sizeof(NotReleased) - 1, Dragons::Reason::NoRights},
            {Expired, sizeof(Expired) - 1, Dragons::Reason::Expired},
            {UsedByAnother, sizeof(UsedByAnother) - 1, Dragons::Reason::UsedByAnother},
        };
    }
}

//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

namespace Dragons {

//!============================================================================
//! @name Dragons サーバー定数処理
//-----------------------------------------------------------------------------
Reason ParseReasonFrom(const char* const pSource, const int sourceLength) NN_NOEXCEPT
{
    for (const auto& v : StringConstants::Reason::g_Values)
    {
        if (v.length == sourceLength && 0 == ::nn::util::Strnicmp(v.pString, pSource, sourceLength))
        {
            return v.value;
        }
    }
    return Reason::Unknown;
}

//-----------------------------------------------------------------------------
ELicenseType ParseELicenseTypeFrom(const char* const pSource, const int sourceLength) NN_NOEXCEPT
{
    for (const auto& v : StringConstants::ELicenseType::g_Values)
    {
        if (v.length == sourceLength && 0 == ::nn::util::Strnicmp(v.pString, pSource, sourceLength))
        {
            return v.value;
        }
    }
    return ELicenseType::Unknown;

}

}   // ~Dragons

//!============================================================================
//! @name 共通処理実装
//-----------------------------------------------------------------------------
ThreadAllocator* GetSharedThreadAllocator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(ThreadPool, s_ThreadPool);
    return &s_ThreadPool;
}

//-----------------------------------------------------------------------------
nim::ELicenseStatus GetELicenseStatusFrom(bool isAvailable, const Dragons::ELicenseType type, const Dragons::Reason reason) NN_NOEXCEPT
{
    if (Dragons::ELicenseType::Unknown == type)
    {
        return nim::ELicenseStatus::Unknown;
    }
    else if (Dragons::ELicenseType::Unavailable == type || false == isAvailable)
    {
        switch (reason)
        {
        case Dragons::Reason::LimitExceeded:
            return nim::ELicenseStatus::NotAssignableSinceLimitExceeded;
        case Dragons::Reason::NoRights:
            return nim::ELicenseStatus::NotAssignableSinceNoRights;
        case Dragons::Reason::NotDeviceLinked:
            return nim::ELicenseStatus::NotAssignableSinceNotDeviceLinked;
        default:
            break;
        }
        return nim::ELicenseStatus::Unknown;
    }
    return nim::ELicenseStatus::Assignable;
}

//-----------------------------------------------------------------------------
RevokeReason GetRevokeReasonFrom(const Dragons::Reason reason) NN_NOEXCEPT
{
    switch (reason)
    {
    case Dragons::Reason::Expired:
        return RevokeReason::Expired;
    case Dragons::Reason::UsedByAnother:
        return RevokeReason::OtherDeviceAssigned;
    case Dragons::Reason::NoRights: NN_FALL_THROUGH;
    case Dragons::Reason::NotReleased: NN_FALL_THROUGH;
    case Dragons::Reason::LimitExceeded: NN_FALL_THROUGH;
    case Dragons::Reason::NotDeviceLinked: NN_FALL_THROUGH;
    default:
        break;
    }
    return RevokeReason::Unknown;
}

//-----------------------------------------------------------------------------
nim::ELicenseType GetELicenseTypeFrom(const Dragons::ELicenseType type) NN_NOEXCEPT
{
    switch (type)
    {
    case Dragons::ELicenseType::DeviceLinkedPermanent:
        return ELicenseType::DeviceLinkedPermanent;
    case Dragons::ELicenseType::Permanent:
        return ELicenseType::Permanent;
    case Dragons::ELicenseType::Temporary:
        return ELicenseType::Temporary;
    case Dragons::ELicenseType::Unavailable:
        return ELicenseType::Unavailable;
    default:
        break;
    }
    return ELicenseType::Unknown;
}

//-----------------------------------------------------------------------------
const char* GetStringFrom(const nim::ELicenseType& type) NN_NOEXCEPT
{
    for (const auto& v : StringConstants::ELicenseType::g_Values)
    {
        const auto elicenseType = GetELicenseTypeFrom(v.value);
        if (elicenseType != nim::ELicenseType::Unknown && elicenseType == type)
        {
            return v.pString;
        }
    }
    return nullptr;
}





//!============================================================================
//! @name DragonsErrorResponseAdaptor 実装
//-----------------------------------------------------------------------------
DragonsErrorResponseAdaptor::DragonsErrorResponseAdaptor() NN_NOEXCEPT
{
    m_Store.Reset(::nn::ResultSuccess());
    NN_ABORT_UNLESS(m_Store.CreateLookupPath("$.type"));
}

//-----------------------------------------------------------------------------
Result DragonsErrorResponseAdaptor::GetResult(const Result& invalidCase) const NN_NOEXCEPT
{
    // 検出可能なエラーレスポンスが見つからなかった。
    const auto storeValue = m_Store.GetValue();
    NN_RESULT_THROW_UNLESS(static_cast<bool>(m_Store) && storeValue.IsFailure(), invalidCase);
    return storeValue;
}

//-----------------------------------------------------------------------------
// パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
void DragonsErrorResponseAdaptor::Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
        m_Store.Store(jsonPath, pValue, valueLength,
        [](StoreResult::ValueType* pOut, const char* pValue, int valueLength) NN_NOEXCEPT -> Result
        {
            *pOut = ToResult(pValue, valueLength);
            NN_RESULT_SUCCESS;
        })
    ));
}

//-----------------------------------------------------------------------------
// エラーコード変換
Result DragonsErrorResponseAdaptor::ToResult(const char* pValue, int valueLength) NN_NOEXCEPT
{
    // 前方一致で検索
    constexpr char prefix[] = "https://dragons.nintendo.com/errors/v1/";
    constexpr int prefixLength = static_cast<int>(sizeof(prefix) - 1);
    if (0 == ::nn::util::Strncmp(prefix, pValue, prefixLength) && valueLength > prefixLength)
    {
        typedef std::pair<const char*, Result> LookupPairTable;
        static const LookupPairTable ErrorTypesLookup[] =
        {
            {"400/invalid_parameter", ResultDragonsInvalidParameter()},
            {"401/authentication_required", ResultDragonsAuthenticationRequired()},
            {"403/invalid_token", ResultDragonsInvalidToken()},
        };
        const auto pNext = &pValue[prefixLength];
        const auto nextLength = valueLength - prefixLength;
        const auto pEnd = ErrorTypesLookup + std::extent<decltype(ErrorTypesLookup)>::value;    // Coverity 範囲外要素アクセス指摘対策.
        const auto pFind = std::find_if(ErrorTypesLookup, pEnd, [&](const LookupPairTable& pair) NN_NOEXCEPT
            {
                return (0 == ::nn::util::Strncmp(pair.first, pNext, nextLength));
            }
        );
        if (pEnd != pFind)
        {
            NN_RESULT_THROW(pFind->second);
        }
    }
    NN_RESULT_THROW(ResultDragonsUnknownError());
}



//!============================================================================
//! @name HttpConnectionForDragons 実装
//-----------------------------------------------------------------------------
HttpConnectionContext::HttpConnectionContext() NN_NOEXCEPT
{
}

//-----------------------------------------------------------------------------
HttpConnectionContext::~HttpConnectionContext() NN_NOEXCEPT
{
    CleanTransaction();
}

//-----------------------------------------------------------------------------
void HttpConnectionContext::Cancel() NN_NOEXCEPT
{
    HttpConnection::Cancel();
    m_TokenCanceler.Cancel();
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::Initialize(DeviceContext* pDeviceContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDeviceContext);
    return HttpConnection::Initialize(pDeviceContext);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::BeginTransaction() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(HttpConnection::IsInitialized(), ResultCurlEasyInitFailure());
    return HttpConnection::BeginTransaction(&m_Transaction);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::SetTransactionUrl(const char* pUrl) NN_NOEXCEPT
{
    return HttpConnection::SetTransactionUrl(&m_Transaction, pUrl);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::SetTransactionHeader(const char* pHeader) NN_NOEXCEPT
{
    return HttpConnection::SetTransactionHeader(&m_Transaction, pHeader);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::SetTransactionPostFields(const char* pFields, const int64_t size, bool isClone) NN_NOEXCEPT
{
    return HttpConnection::SetTransactionPostFields(&m_Transaction, pFields, size, isClone);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::SetTransactionCustomMethod(const char* pMethod) NN_NOEXCEPT
{
    return HttpConnection::SetTransactionCustomMethod(&m_Transaction, pMethod);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::EndTransaction(::nn::util::optional<HeaderCallback> headerCallback, ::nn::util::optional<WriteCallback> writeCallback, int timeoutSecond) NN_NOEXCEPT
{
    return HttpConnection::EndTransaction(&m_Transaction, headerCallback, writeCallback, timeoutSecond);
}

//-----------------------------------------------------------------------------
void HttpConnectionContext::CleanTransaction() NN_NOEXCEPT
{
    m_TokenCanceler.Invalidate();
    HttpConnection::CleanTransaction(&m_Transaction);
}

//-----------------------------------------------------------------------------
Result HttpConnectionContext::GetErrorContext(::nn::err::ErrorContext* pOutValue) NN_NOEXCEPT
{
    HttpConnection::GetErrorContext(pOutValue);
    NN_RESULT_SUCCESS;
}




//!============================================================================
//! @name DragonsAsyncAccessTaskBase 実装
//-----------------------------------------------------------------------------
DragonsAsyncAccessTaskBase::ResponseValue::ResponseValue() NN_NOEXCEPT
    : result(ResultTaskStillRunning())
{
}

//-----------------------------------------------------------------------------
DragonsAsyncAccessTaskBase::ResponseValue::ResponseValue(const Result result_) NN_NOEXCEPT
    : result(result_)
{
}

//-----------------------------------------------------------------------------
DragonsAsyncAccessTaskBase::DragonsAsyncAccessTaskBase() NN_NOEXCEPT
{
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::Initialize(DeviceContext* pDeviceContext) NN_NOEXCEPT
{
    return m_Connection.Initialize(pDeviceContext);
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::GetErrorContext(::nn::err::ErrorContext* pOutValue) NN_NOEXCEPT
{
    return m_Connection.GetErrorContext(pOutValue);
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::GetResult() const NN_NOEXCEPT
{
    NN_RESULT_DO(m_ResponseValue.load(std::memory_order_acquire).result);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::Cancel() NN_NOEXCEPT
{
    // 要求開始状態での到着を期待とする。
    ResponseValue expectedState;
    if (m_ResponseValue.compare_exchange_strong(expectedState, ResponseValue(ResultHttpConnectionCanceled()), std::memory_order_acq_rel))
    {
        m_Connection.Cancel();
        DEBUG_TRACE("Cancel: done( %p ).\n", this);
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::CanContinue() NN_NOEXCEPT
{
    // ハンドル状態が初期状態でなければ、キャンセルもしくは完了済として判断。
    const auto response = m_ResponseValue.load(std::memory_order_acquire);
    if (false == ResultTaskStillRunning::Includes(response.result))
    {
        NN_RESULT_THROW(response.result);
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::RequestTaskFinished(Result resultOfTask) NN_NOEXCEPT
{
    // 要求開始状態での到着を期待とする。
    ResponseValue expectedState;
    ResponseValue reponse(resultOfTask);
    const auto& r = (m_ResponseValue.compare_exchange_strong(expectedState, reponse, std::memory_order_acq_rel)) ? reponse : expectedState;
    DEBUG_TRACE("Request( %p ) Completed => ( result[%u-%u] ).\n", this, r.result.GetModule(), r.result.GetDescription());
    return r.result;
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::RegisterSystemRequestHeader(ConnectionType* pConnection, const AccessProfile& profile) NN_NOEXCEPT
{
    char pTemporary[2048];
    NN_STATIC_ASSERT(sizeof(pTemporary) >= LengthMaxForNaId);
    NN_STATIC_ASSERT(sizeof(pTemporary) >= LengthMaxForNaIdToken);
    NN_STATIC_ASSERT(sizeof(pTemporary) >= LengthMaxForDeviceToken);

    // システムリクエストヘッダ登録( デバイス認証トークン )
    NN_RESULT_DO(SharedContext::GetInstance()->CreateHeaderForDeviceAuthenticationToken(pTemporary, sizeof(pTemporary), &pConnection->GetTokenCanceler().device));
    NN_RESULT_DO(pConnection->SetTransactionHeader(pTemporary));

    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    const auto& uid = profile.uid;
    if (static_cast<bool>(uid))
    {
        // システムリクエストヘッダ登録( NA-ID トークン )
        NN_RESULT_DO(SharedContext::GetInstance()->CreateHeaderForNaIdToken(pTemporary, sizeof(pTemporary), uid, &pConnection->GetTokenCanceler().na));
        NN_RESULT_DO(pConnection->SetTransactionHeader(pTemporary));
    }
    else if (static_cast<bool>(profile.naId))
    {
        // システムリクエストヘッダ登録( NA-ID )
        NN_RESULT_THROW_UNLESS(::nn::util::SNPrintf(pTemporary, sizeof(pTemporary), "%s%016llx", HeaderPrefixForNaId, profile.naId) < sizeof(pTemporary), ResultBufferNotEnough());
        NN_RESULT_DO(pConnection->SetTransactionHeader(pTemporary));
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result DragonsAsyncAccessTaskBase::Connect(size_t* const pOutReceivedSize, HeapUtil::OnetimeHeapSession* const pOutResponse, RequestProfileHolder* pRequestProfile, const char* pUrl, const AccessProfile& profile) NN_NOEXCEPT
{
    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    // トランザクション要求開始
    auto& connection = m_Connection;
    NN_RESULT_DO(connection.BeginTransaction());
    NN_UTIL_SCOPE_EXIT{connection.CleanTransaction();};

    // URL登録
    NN_RESULT_DO(connection.SetTransactionUrl(pUrl));

    // システムリクエストヘッダ登録
    NN_RESULT_DO(RegisterSystemRequestHeader(&connection, profile));

    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    // 派生先タスク別リクエストボディ準備
    NN_RESULT_DO(pRequestProfile->OnSetupRequestBody(&connection));

    // @note
    //      派生先へは入力ストリームを OnResolveResponse() 経由で渡す。
    //          派生先は、同入力ストリームを使って JSONパースを行う。
    //          JSONパースに伴う出力ストリームは派生先で用意する。
    //
    //      入力ストリームの機能定義は以下のステップで拡充・機能強化していく。
    //          - 初期版:  全受信レスポンスがオンメモリのオンメモリストリームを渡す。
    //                      CURLレスポンスは全てヒープで受信する。( ヘッダの `Content-Length` で確保して受信 )
    //          - 次期版:  オンメモリ/ファイルストリームのハイブリッドストリームを渡す。
    //                      想定オンメモリ利用量超過の受信長の場合:   CURLレスポンスはファイルに書き込まれ、ファイル入力ストリームとして扱う。
    //                      想定オンメモリ利用量以下の受信長の場合:   オンメモリストリームとして扱う。
    //          - 最終版:  CURL受信と同期したストリーミングストリームを渡す。
    //                      curl API (CURLINFO_RESPONSE_CODE) 経由の HttpStatusCode 取得ができないのでレスポンスヘッダから取得する必要あり。

    size_t receivedSize = 0;
    const HttpConnection::HeaderCallback headerCallback = [&pOutResponse](const void* const pBuffer, const size_t bufferSize) NN_NOEXCEPT->Result
    {
        if (nullptr != pBuffer && bufferSize > 0 && !pOutResponse->IsAvailable())
        {
            size_t length;
            ::nn::util::optional<HttpHeaderValue> value;
            NN_RESULT_DO(FindHttpHeader(&value, "Content-Length", static_cast<const char*>(pBuffer), bufferSize));
            if (value && (length = static_cast<size_t>(std::strtoul(value->string, nullptr, 10))) > 0)
            {
                DEBUG_TRACE("Response header's [Content-Length( %zu )].\n", length);
                NN_RESULT_DO(pOutResponse->Acquire(sizeof(uintptr_t), length));
            }
        }
        NN_RESULT_SUCCESS;
    };
    const HttpConnection::WriteCallback writeCallback = [&pOutResponse, &receivedSize](const void* const pBuffer, const size_t bufferSize) NN_NOEXCEPT->Result
    {
        if (nullptr != pBuffer && bufferSize > 0)
        {
            const size_t nextReceivedSize = receivedSize + bufferSize;
            if (!pOutResponse->IsAvailable() || (nextReceivedSize > pOutResponse->size))
            {
                // ヘッダに Content-Length がない or 既存容量オーバーしたら再確保試行。
                DEBUG_TRACE("Unexpected over reseponse received: %zu -> %zu.\n", pOutResponse->size, nextReceivedSize);
                NN_RESULT_DO(pOutResponse->Expand(sizeof(uintptr_t), nextReceivedSize));
            }
            auto pResponseBuffer = static_cast<Bit8*>(pOutResponse->pTop);
            std::memcpy(&pResponseBuffer[receivedSize], pBuffer, bufferSize);
            receivedSize = nextReceivedSize;
        }
        NN_RESULT_SUCCESS;
    };

    // 0 以下指定は @ref DefaultTimeOutSeconds と同意とします。
    const auto timeoutSeconds = (profile.timeout > 0) ? profile.timeout : DefaultTimeOutSeconds;
    const auto result = connection.EndTransaction(headerCallback, writeCallback, timeoutSeconds);
    *pOutReceivedSize = receivedSize;
    return result;
}

//-----------------------------------------------------------------------------
// NOTE: Result の扱い。
//  ResultBufferNotEnough:          内部固定バッファ以上の要求が発生した。
//  ResultHttpConnectionCanceled:   タスクがキャンセルされた。
Result DragonsAsyncAccessTaskBase::Request(RequestProfileHolder* pRequestProfile) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pRequestProfile);

    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    // デバッグレスポンス適用チェックのためURLベースを作る。
    char pUrl[192];
    const size_t baseUrlLength = sizeof(Dragons::ServerBaseUrl) - 1;
    NN_STATIC_ASSERT(sizeof(pUrl) > (baseUrlLength + 128));
    ::nn::util::Strlcpy(pUrl, Dragons::ServerBaseUrl, sizeof(pUrl));

    // 派生先タスクにアクセスプロファイルを問い合わせる。
    AccessProfile profile;
    const auto pPathRoot = &pUrl[baseUrlLength];
    const size_t availableCapacity = sizeof(pUrl) - baseUrlLength;
    NN_RESULT_DO(pRequestProfile->OnQueryAccessProfile(&profile, pPathRoot, availableCapacity));
    DEBUG_TRACE("Request url =>`%s`\n", pUrl);

    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    // レスポンスヒープ取得。
    HeapUtil::OnetimeHeapSession response;

    // デバッグレスポンスマッチング( Path フィールドを照合します )
    Result result = {};
    size_t receivedSize = 0;
    DebugResponseReceiver debugResponse(&response);
    if (debugResponse.Find(pPathRoot))
    {
        // デバッグレスポンスひっかかった。
        receivedSize = response.size;
        result = debugResponse.GetResult();
    }
    else
    {
        // 実通信実施。
        result = Connect(&receivedSize, &response, pRequestProfile, pUrl, profile);
    }

    // キャンセルチェック
    NN_RESULT_DO(CanContinue());

    // レポート。
    DEBUG_TRACE("Connection response(%zu byte), Result[%u-%u].\n", receivedSize, result.GetModule(), result.GetDescription());

    // レスポンスボディがないなら、即 Result 返却。
    NN_RESULT_THROW_UNLESS(response.IsAvailable(), result);
    DEBUG_TRACE_RESPONSE(static_cast<char*>(response.pTop), response.size);

    // レスポンスストリーム作成。( 現状はオンメモリストリーム )
    InputStream stream(static_cast<char*>(response.pTop), response.size);

    // エラーコードを含む HttpStatusResult かどうかチェック。
    //  厳密にするならレスポンスヘッダの Content-Type が "application/problem+json" か調べる。
    //  TODO: ストリーミング対応になったら、レスポンスヘッダの HttpStatus コードをみてレスポンス解析分岐を行う必要がある。
    if (HasErrorCodeResponse(result))
    {
        // { "key": "value" } 要素は、1024 byte に収まる想定の Insitu パース用ストリーム。
        HttpJson::Stream::ProxyInputStream<InputStream, 1024> errorStream(&stream);

        // エラーレスポンス解析。
        DragonsErrorResponseAdaptor adaptor;
        NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, errorStream, static_cast<HttpJson::Canceler*>(nullptr)));
        result = adaptor.GetResult(result);
    }

    // 最終的にエラー状態だったので返却。
    NN_RESULT_DO(result);

    // 正常通信に成功したので派生先タスクへレスポンス解析依頼。
    NN_RESULT_DO(pRequestProfile->OnResolveResponse(&stream));
    NN_RESULT_SUCCESS;
}



//!============================================================================
//! @name DragonsAccessAsyncImpl 実装
//-----------------------------------------------------------------------------
Result DragonsAccessAsyncImpl::Execute() NN_NOEXCEPT
{
    return DragonsAsyncAccessTaskBase::RequestTaskFinished(DragonsAsyncAccessTaskBase::Request(this));
}




//!============================================================================
//! @name TemporaryFileStore 実装
//-----------------------------------------------------------------------------
TemporaryFileStore::TemporaryFileStore() NN_NOEXCEPT
    : m_Written(0), m_InitialFileSize(0)
{
}

//-----------------------------------------------------------------------------
TemporaryFileStore::~TemporaryFileStore() NN_NOEXCEPT
{
    if (m_File)
    {
        ::nn::fs::FlushFile(*m_File);
        ::nn::fs::CloseFile(*m_File);
        ::nn::fs::DeleteFile(m_Path);
    }
}

//-----------------------------------------------------------------------------
void TemporaryFileStore::Initialize(const char* pPath) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pPath);
    NN_ABORT_UNLESS(m_Path.Assign(pPath).GetLength() > 0);
}

//-----------------------------------------------------------------------------
void TemporaryFileStore::SetInitialFileSize(const FileSizeType fileSize) NN_NOEXCEPT
{
    m_InitialFileSize = fileSize;
}

//-----------------------------------------------------------------------------
Result TemporaryFileStore::Append(const void* pSource, size_t writeSize) NN_NOEXCEPT
{
    if (!m_File)
    {
        NN_ABORT_UNLESS(m_Path.GetLength() > 0);
        NN_RESULT_DO(::nn::fs::CreateFile(m_Path, m_InitialFileSize));
        ::nn::fs::FileHandle file;
        const auto flags = ::nn::fs::OpenMode_Read | ::nn::fs::OpenMode_Write | ::nn::fs::OpenMode_AllowAppend;
        NN_RESULT_DO(::nn::fs::OpenFile(&file, m_Path, flags));
        m_File = file;
        m_Written = 0;
    }
    NN_RESULT_DO(::nn::fs::WriteFile(*m_File, m_Written, pSource, writeSize, ::nn::fs::WriteOption()));
    m_Written += writeSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result TemporaryFileStore::Read(size_t* pOutReadedSize, int64_t offset, void* pOutData, size_t readSize) const NN_NOEXCEPT
{
    *pOutReadedSize = 0;
    NN_RESULT_THROW_UNLESS(m_File, ResultSuccess());
    NN_RESULT_DO(::nn::fs::ReadFile(pOutReadedSize, *m_File, offset, pOutData, readSize));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result TemporaryFileStore::Read(int64_t offset, void* pOutData, size_t readSize) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_File, ResultSuccess());
    NN_RESULT_DO(::nn::fs::ReadFile(*m_File, offset, pOutData, readSize));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void TemporaryFileStore::Rewind() NN_NOEXCEPT
{
    m_Written = 0;
}

//-----------------------------------------------------------------------------
Result AppendRequestBodyAsBit64(size_t* pOutWriteLength
    , char* pOutBuffer
    , const size_t availableBufferCapacity
    , const TemporaryFileStore& source
    , size_t startIndex
    , size_t sourceCount
    , const char* pPrefix
    , const char* pSuffix) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutWriteLength);
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
    NN_SDK_ASSERT(availableBufferCapacity > 0);

    *pOutWriteLength = 0;
    size_t writeLength = 0;

    if (nullptr != pPrefix)
    {
        // 先頭追加。
        writeLength += ::nn::util::SNPrintf(pOutBuffer, availableBufferCapacity, pPrefix);
        NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());
    }
    // 配列生成。
    constexpr size_t ReadBufferSize = 32;
    const char* pFormat = "\"%016llx\"";
    int64_t readPos = startIndex * sizeof(Bit64);
    Bit64 pReadBuffer[ReadBufferSize]; // 256 Byte
    while (sourceCount > 0)
    {
        size_t readedSize = 0;
        const size_t readCount = std::min(sourceCount, ReadBufferSize);
        const size_t readSize = readCount * sizeof(Bit64);
        NN_RESULT_DO(source.Read(&readedSize, readPos, pReadBuffer, readSize));
        NN_RESULT_THROW_UNLESS(readedSize == readSize, ResultContentNotFound());
        for (const auto value : ::nn::util::MakeSpan(pReadBuffer, readCount))
        {
            writeLength += ::nn::util::SNPrintf(&pOutBuffer[writeLength], availableBufferCapacity - writeLength, pFormat, value);
            NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());
            pFormat = ",\"%016llx\"";
        }
        sourceCount -= readCount;
        readPos += readedSize;
    }
    if (nullptr != pSuffix)
    {
        // 末尾追加。
        writeLength += ::nn::util::SNPrintf(&pOutBuffer[writeLength], availableBufferCapacity - writeLength, pSuffix);
        NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());
    }
    *pOutWriteLength = writeLength;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AppendRequestBodyAsELicenses(size_t* pOutWriteLength
    , char* pOutBuffer
    , const size_t availableBufferCapacity
    , const TemporaryFileStore& source
    , size_t startIndex
    , size_t sourceCount
    , const char* pPrefix
    , const char* pSuffix) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutWriteLength);
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
    NN_SDK_ASSERT(availableBufferCapacity > 0);

    *pOutWriteLength = 0;
    size_t writeLength = 0;

    if (nullptr != pPrefix)
    {
        // 先頭追加。
        writeLength += ::nn::util::SNPrintf(pOutBuffer, availableBufferCapacity, pPrefix);
        NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());
    }
    // 配列生成。
    bool separator = false;
    constexpr size_t ReadBufferSize = 32;
    int64_t readPos = startIndex * sizeof(es::ELicenseId);
    es::ELicenseId pReadBuffer[ReadBufferSize]; // 512 Byte
    while (sourceCount > 0)
    {
        size_t readedSize = 0;
        const size_t readCount = std::min(sourceCount, ReadBufferSize);
        const size_t readSize = readCount * sizeof(es::ELicenseId);
        NN_RESULT_DO(source.Read(&readedSize, readPos, pReadBuffer, readSize));
        NN_RESULT_THROW_UNLESS(readedSize == readSize, ResultContentNotFound());
        for (const auto value : ::nn::util::MakeSpan(pReadBuffer, readCount))
        {
            auto pOutBase = &pOutBuffer[writeLength];
            writeLength += es::ELicenseId::StringSize + 2 + (separator ? 1 : 0);
            NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());

            auto pOut = pOutBase;
            if (separator)
            {
                pOut[0] = ',';
                pOut = &pOut[1];
            }
            pOut[0] = '\"';
            value.ToString(&pOut[1], es::ELicenseId::StringSize + 1);
            pOut[1 + es::ELicenseId::StringSize] = '\"';
            pOut[2 + es::ELicenseId::StringSize] = '\0';
            separator = true;
        }
        sourceCount -= readCount;
        readPos += readedSize;
    }
    if (nullptr != pSuffix)
    {
        // 末尾追加。
        writeLength += ::nn::util::SNPrintf(&pOutBuffer[writeLength], availableBufferCapacity - writeLength, pSuffix);
        NN_RESULT_THROW_UNLESS(writeLength < availableBufferCapacity, ResultBufferNotEnough());
    }
    *pOutWriteLength = writeLength;
    NN_RESULT_SUCCESS;
}


}   // ~DynamicRights
}}} // ~nn::nim::srv
