﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_ELicenseTypes.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_DynamicRightsAssignELicenses.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::AssignELicenses] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//! @brief      結果格納要素値型。
typedef ::nn::nim::AssignedELicense ValueType;

//! @brief      レスポンスアダプタ。
//! @details    各要素の型は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=325012215 に基づきます。
class AssignELicensesAdaptor : public HttpJson::Canceler
{
    typedef HttpJson::EntryLookup<6, 64>                                EntryLookup;
    typedef HttpJson::ValueStore<es::ELicenseId, EntryLookup>           StoreLicenseId;
    typedef HttpJson::ValueStore<es::RightsId, EntryLookup>             StoreRightsId;
    typedef HttpJson::ValueStore<Bit64, EntryLookup>                    StoreAccountId;
    typedef HttpJson::ValueStore<Dragons::ELicenseType, EntryLookup>    StoreLicenseType;

    struct StoreResponse
    {
        StoreLicenseId      licenseId;
        StoreAccountId      accountId;
        StoreRightsId       rightsId;
        StoreLicenseType    type;

        bool CreateLookupPath(int index) NN_NOEXCEPT
        {
            return (
                licenseId.CreateLookupPath("$.elicenses[%u].elicense_id", index) &&
                accountId.CreateLookupPath("$.elicenses[%u].account_id", index) &&
                type.CreateLookupPath("$.elicenses[%u].elicense_type", index) &&
                rightsId.CreateLookupPath("$.elicenses[%u].rights_id", index)
            );
        }

        void Reset() NN_NOEXCEPT
        {
            licenseId.Reset(es::InvalidELicenseId);
            accountId.Reset(account::InvalidNintendoAccountId.id);
            rightsId.Reset();
            type.Reset(Dragons::ELicenseType::Unknown);
        }
    };

public:
    typedef unsigned int                CountType;
    typedef EntryLookup::JsonPathType   JsonPathType;

    explicit AssignELicensesAdaptor(TemporaryFileStore* pFileStore) NN_NOEXCEPT
        : m_pFileStore(pFileStore)
        , m_Index(0)
        , m_Result(::nn::ResultSuccess())
    {
        NN_ABORT_UNLESS_NOT_NULL(pFileStore);
        pFileStore->Rewind();
        m_Store.Reset();
    }

    CountType GetCount() const NN_NOEXCEPT
    {
        return m_Index;
    }

    //! @brief      内部異常値の取得。
    //!
    //! @details    本クラスインスタンスをキャンセラ登録する事で内部異常が発生した場合には、@ref ::nn::http::json::ImportJsonByRapidJson() の返値に http::ResultCanceled が返ります。@n
    //!             その場合、本メソッドで異常値を取得し適切に処理してください。@n
    //!             異常状態が発生していない場合は、ResultSuccess が返されます。
    Result GetFailure() const NN_NOEXCEPT
    {
        return m_Result;
    }

    // パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.licenseId.Store(jsonPath, pValue, valueLength,
            [](StoreLicenseId::ValueType* pOut, const char* pValue, int valueLength) NN_NOEXCEPT -> Result {
                if (valueLength != es::ELicenseId::StringSize || !pOut->FromString(pValue))
                {
                    *pOut = es::InvalidELicenseId;  // 変換できなかったら無効値を代入する
                };
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.accountId.Store(jsonPath, pValue, valueLength,
            [](StoreAccountId::ValueType* pOut, const char* pValue, int) NN_NOEXCEPT->Result{
                *pOut = static_cast<Bit64>(NN_NIM_STR_TO_ULL(pValue, nullptr, 16));  // 不正ならゼロ
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.rightsId.Store(jsonPath, pValue, valueLength,
            [](StoreRightsId::ValueType* pOut, const char* pValue, int) NN_NOEXCEPT->Result{
                *pOut = static_cast<es::RightsId>(NN_NIM_STR_TO_ULL(pValue, nullptr, 16));  // 不正ならゼロ
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.type.Store(jsonPath, pValue, valueLength,
            [](StoreLicenseType::ValueType* pOut, const char* pValue, int length) NN_NOEXCEPT->Result{
                *pOut = Dragons::ParseELicenseTypeFrom(pValue, length);
                NN_RESULT_SUCCESS;
            })
        ));
    }

    void NotifyObjectBegin(const JsonPathType& jsonPath) NN_NOEXCEPT
    {
        char path[64];
        const auto index = m_Index;
        if (::nn::util::SNPrintf(path, sizeof(path), "$.elicenses[%u]", index) < sizeof(path) && jsonPath.Match(path))
        {
            NN_ABORT_UNLESS(m_Store.CreateLookupPath(index));
        }
    }

    void NotifyObjectEnd(const JsonPathType& jsonPath) NN_NOEXCEPT
    {
        char path[64];
        const auto index = m_Index;
        if (::nn::util::SNPrintf(path, sizeof(path), "$.elicenses[%u]", index) < sizeof(path) && jsonPath.Match(path))
        {
            // ファイルへ保存。
            if (m_Result.IsSuccess())
            {
                // FAILの場合は、パースがキャンセルされる。
                m_Result = Write();
            }

            // 更新
            m_Index = index + 1;

            // ストアを再利用可能に。
            m_Store.Reset();
        }
    }

    // 以下検出対象外
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}

    // パース中に std::numeric_limit<int64_t>::max 以下の整数値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    // パース中に std::numeric_limit<int64_t>::max より大きい整数値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}

    // 内部処理で失敗があったらパースをキャンセルする。
    virtual bool IsCancelled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Result.IsFailure();
    }

private:
    Result Write() NN_NOEXCEPT
    {
        ValueType value;
        value.eLicenseId = m_Store.licenseId.GetValue();
        value.naId.id = m_Store.accountId.GetValue();
        value.rightsId = m_Store.rightsId.GetValue();
        value.licenseType = GetELicenseTypeFrom(m_Store.type.GetValue());
        NN_RESULT_DO(m_pFileStore->Append(&value, sizeof(value)));
        NN_RESULT_SUCCESS;
    }

private:
    TemporaryFileStore* const   m_pFileStore;
    StoreResponse               m_Store;
    CountType                   m_Index;
    Result                      m_Result;
};

//-----------------------------------------------------------------------------
// NOTE: コードサイズ共有化のためにテンプレにはしないです
Result ReadValues(uint64_t* pOutValue, int64_t offset, const ::nn::sf::OutBuffer& buffer, const TemporaryFileStore& valueStore, size_t totalCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    *pOutValue = 0;

    // NOTE: 負値のインデクスオフセットは不正引数扱い。
    NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());

    // NOTE: ValueType アライメント違反、及び 出力バッファの未指定は不正引数指定の扱い。
    auto pOutBuffer = buffer.GetPointerUnsafe();
    const uint64_t outCapacity = buffer.GetSize();
    NN_RESULT_THROW_UNLESS(nullptr != pOutBuffer && outCapacity > 0 && 0 == (offset % sizeof(ValueType)) && 0 == (outCapacity % sizeof(ValueType)), ResultInvalidOffsetArgument());

    // NOTE: オーバーランオフセット指定の場合は要素数ゼロを返却して成功扱い。
    const uint64_t requestOffset = static_cast<uint64_t>(offset);
    const uint64_t totalValueSize = static_cast<uint64_t>(sizeof(ValueType) * totalCount);
    NN_RESULT_THROW_UNLESS(requestOffset < totalValueSize, ResultSuccess());

    size_t readedSize;
    const uint64_t writeSize = std::min(outCapacity, static_cast<uint64_t>(totalValueSize - requestOffset));
    NN_RESULT_DO(valueStore.Read(&readedSize, static_cast<int64_t>(requestOffset), pOutBuffer, static_cast<size_t>(writeSize)));
    NN_RESULT_THROW_UNLESS(readedSize == static_cast<size_t>(writeSize), ResultContentNotFound());

    *pOutValue = writeSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// NOTE: コードサイズ共有化のためにテンプレにはしないです
Result ResolveResponse(uint32_t* pOutCount, TemporaryFileStore* pValueStore, DragonsAccessAsyncImpl::InputStream* pInputStream) NN_NOEXCEPT
{
    // JSON 要素の "key", "value" 各々の上限が 1024 byte に収まる想定のパースストリーム。
    HttpJson::Stream::ProxyInputStream<DragonsAccessAsyncImpl::InputStream, 1024> stream(pInputStream);

    AssignELicensesAdaptor adaptor(pValueStore);
    NN_RESULT_TRY(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(&adaptor)))
        NN_RESULT_CATCH_CONVERT(::nn::http::ResultCanceled, adaptor.GetFailure())
    NN_RESULT_END_TRY;

    *pOutCount = adaptor.GetCount();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name AsyncAssignELicensesImpl 実装
//
// NOTE:
//  処理結果リソース( mValueStore, m_Count )は非同期処理部との排他は行っていません。
//  これは、Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncAssignELicensesImpl::AsyncAssignELicensesImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_User(::nn::account::InvalidUid)
    , m_LicenseType(ELicenseType::EndOfContents)
    , m_Count(0)
{
}

//-----------------------------------------------------------------------------
AsyncAssignELicensesImpl::~AsyncAssignELicensesImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::Initialize(DeviceContext* pDeviceContext
    , const ::nn::account::Uid& uid
    , const ::nn::sf::InArray<::nn::es::RightsId>& rightsIds
    , const ELicenseType& licenseType
    , const char* pTemporaryFilePath) NN_NOEXCEPT
{
    // 不正なパラメタチェック
    const auto nRights = rightsIds.GetLength();
    NN_RESULT_THROW_UNLESS(nRights > 0, ResultContentNotFound());
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));

    m_User = uid;
    m_LicenseType = licenseType;
    m_ValueStore.Initialize(pTemporaryFilePath);

    // 入力 rigths をファイルへ
    m_ValueStore.SetInitialFileSize(nRights * sizeof(::nn::es::RightsId));
    NN_RESULT_DO(m_ValueStore.Append(rightsIds.GetData(), nRights * sizeof(::nn::es::RightsId)));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::GetSize(::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    *outValue = static_cast<int64_t>(sizeof(ValueType) * m_Count);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::Read(::nn::sf::Out<std::uint64_t> outValue, int64_t offset, const ::nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());
    uint64_t outCount;
    NN_RESULT_DO(ReadValues(&outCount, offset, buffer, m_ValueStore, m_Count));
    *outValue = outCount;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/rights/publish_elicenses";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);

    pOut->uid = m_User;
    pOut->naId = ::nn::account::InvalidNintendoAccountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // License type 文字列の取得。
    const auto pType = GetStringFrom(m_LicenseType);
    NN_RESULT_THROW_UNLESS(nullptr != pType, ResultNotSupported());

    // Rights データの有効性確認。
    auto nRights = static_cast<size_t>(m_ValueStore.GetWrittenSize() / sizeof(::nn::es::RightsId));
    NN_RESULT_THROW_UNLESS(nRights > 0U, ResultContentNotFound());

    char data[96];
    constexpr char PostBasePrefix[] = "{\"rights_ids\": [";
    constexpr char PostBaseSuffix[] = "],\"elicense_type\": \"%s\"}";
    const auto suffixLength = ::nn::util::SNPrintf(data, sizeof(data), PostBaseSuffix, pType);
    NN_ABORT_UNLESS(suffixLength < sizeof(data));

    // ポストデータバッファ確保。( null終端含めてざっくり確保、19 は "\"%016llx\"," の文字数 )
    const size_t requiredPostSize = sizeof(PostBasePrefix) + suffixLength + static_cast<size_t>(19 * nRights);
    NN_RESULT_DO(m_Post.Acquire(sizeof(Bit64), requiredPostSize));
    const auto pOutPost = static_cast<char*>(m_Post.pTop);

    size_t postLength = 0;
    NN_RESULT_DO(AppendRequestBodyAsBit64(&postLength, pOutPost, requiredPostSize, m_ValueStore, 0, nRights, PostBasePrefix, data));
    DEBUG_TRACE("Post filed generate complete, (heap: %zu, length: %zu)=>`%s`\n", requiredPostSize, postLength, pOutPost);
    NN_SDK_ASSERT(postLength == std::strlen(pOutPost)); // NOTE: デバッグ用の保険検証
    NN_RESULT_DO(pConnection->SetTransactionPostFields(pOutPost));

    // "Content-Length", "Content-Type" ヘッダ。
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignELicensesImpl::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    AssignELicensesAdaptor::CountType count;
    NN_RESULT_DO(ResolveResponse(&count, &m_ValueStore, pInputStream));
    m_Count = static_cast<CountType>(count);
    NN_RESULT_SUCCESS;
}



//!============================================================================
//! @name AsyncAssignAllDeviceLinkedELicensesImpl 実装
//
// NOTE:
//  処理結果リソース( mValueStore, m_Count )は非同期処理部との排他は行っていません。
//  これは、Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncAssignAllDeviceLinkedELicensesImpl::AsyncAssignAllDeviceLinkedELicensesImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_Count(0)
{
}

//-----------------------------------------------------------------------------
AsyncAssignAllDeviceLinkedELicensesImpl::~AsyncAssignAllDeviceLinkedELicensesImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::Initialize(DeviceContext* pDeviceContext, const char* pTemporaryFilePath) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));
    m_ValueStore.Initialize(pTemporaryFilePath);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::GetSize(::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    *outValue = static_cast<int64_t>(sizeof(ValueType) * m_Count);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::Read(::nn::sf::Out<std::uint64_t> outValue, int64_t offset, const ::nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());
    uint64_t outCount;
    NN_RESULT_DO(ReadValues(&outCount, offset, buffer, m_ValueStore, m_Count));
    *outValue = outCount;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    // AccessProfile の各メンバは初期値設定されており、本 API は初期値のままで良い。
    NN_UNUSED(pOut);
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/rights/publish_device_linked_elicenses";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);
    NN_FUNCTION_LOCAL_STATIC(char, s_Empty, [] = "");
    NN_RESULT_DO(pConnection->SetTransactionPostFields(s_Empty));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAssignAllDeviceLinkedELicensesImpl::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    AssignELicensesAdaptor::CountType count;
    NN_RESULT_DO(ResolveResponse(&count, &m_ValueStore, pInputStream));
    m_Count = static_cast<CountType>(count);
    NN_RESULT_SUCCESS;
}

}   // ~DynamicRights
}}} // ~nn::nim::srv
