﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_PriorityPrivate.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nim/srv/nim_BackgroundApplyDeltaStressTask.h>
#include <nn/nim/nim_BackgroundDownloadStressTaskInfo.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>

namespace nn { namespace nim { namespace srv {

    namespace {
        const size_t StackSize = 16 * 1024;
        NN_OS_ALIGNAS_THREAD_STACK char g_BackgroundApplyDeltaStressTaskTaskStack[StackSize];
        const int ThreadPriority = NN_SYSTEM_THREAD_PRIORITY(nim, BackgroundApplyDeltaStressTask);
        const char* ThreadName = NN_SYSTEM_THREAD_NAME(nim, BackgroundApplyDeltaStressTask);

        const char* MountName = "@stress";
        const char* FilePath = "@stress:/random";

        const size_t BufferSize = 256 * 1024;
    }

    BackgroundApplyDeltaStressTask::BackgroundApplyDeltaStressTask() NN_NOEXCEPT
        : m_StorageId(ncm::StorageId::None)
        , m_Thread()
        , m_Event(os::EventClearMode_AutoClear)
        , m_BeginMeasurement()
        , m_LastResult()
        , m_IsInitialized()
        , m_State(BackgroundDownloadStressTaskState::NotRunning)
    {
    }

    BackgroundApplyDeltaStressTask::~BackgroundApplyDeltaStressTask() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            m_Event.Signal();
            os::WaitThread(&m_Thread);
            os::DestroyThread(&m_Thread);
        }
    }

    Result BackgroundApplyDeltaStressTask::Initialize(ncm::ApplicationId id, ncm::StorageId storageId) NN_NOEXCEPT
    {
        m_ApplicationId = id;
        m_StorageId = storageId;
        m_BufferSize = BufferSize;
        m_State = BackgroundDownloadStressTaskState::Waiting;

        NN_RESULT_DO(os::CreateThread(&m_Thread, [](void* arg)
            {
                BackgroundApplyDeltaStressTask* pTask = reinterpret_cast<BackgroundApplyDeltaStressTask*>(arg);
                pTask->Run();
            }, this, g_BackgroundApplyDeltaStressTaskTaskStack, StackSize, ThreadPriority));
        os::SetThreadNamePointer(&m_Thread, ThreadName);
        os::StartThread(&m_Thread);

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    void BackgroundApplyDeltaStressTask::Run() NN_NOEXCEPT
    {
        // ncm で発生する fs アクセスの優先度を設定する
        // 注意:    IPC 先で fs アクセスが発生するときに影響するので、
        //          他のプロセスへの IPC を行うときには注意
        fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);

        // 起動待ち。ContentMetaDatabase などが初期化されないため
        const int WaitLaunch = 20;
        os::SleepThread(TimeSpan::FromSeconds(WaitLaunch));

        while (!m_Event.TryWait())
        {
            m_LastResult = Execute();
            if (m_LastResult.IsFailure())
            {
                m_State = BackgroundDownloadStressTaskState::Error;
                NN_DETAIL_NIM_TRACE("[BackgroundApplyDeltaStressTask] failed %08x\n", m_LastResult.GetInnerValueForDebug());
                break;
            }
        }
    }

    Result BackgroundApplyDeltaStressTask::Execute() NN_NOEXCEPT
    {
        m_LastResult = ResultSuccess();
        m_State = BackgroundDownloadStressTaskState::Downloading;

        m_Buffer = new Bit8[m_BufferSize];
        NN_RESULT_THROW_UNLESS(m_Buffer != nullptr, ResultApplyDeltaStressTestOutOfMemory());
        NN_UTIL_SCOPE_EXIT {delete[] m_Buffer; };

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, m_StorageId));

        ncm::ContentMetaKey key;
        auto count = db.ListContentMeta(&key, 1, ncm::ContentMetaType::AddOnContent, m_ApplicationId);
        NN_RESULT_THROW_UNLESS(count.listed == 1, ResultStressTestDataNotFound());

        ncm::ContentId     contentId;
        NN_RESULT_DO(db.GetContentIdByType(&contentId, key, ncm::ContentType::Data));
        ncm::PatchId patchId = {key.id};

        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, m_StorageId));

        ncm::Path path;
        storage.GetPath(&path, contentId);
        NN_RESULT_DO(fs::MountContent(MountName, path.string, patchId, fs::ContentType_Data));
        NN_UTIL_SCOPE_EXIT {fs::Unmount(MountName);};

        ResetMeasure();
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, FilePath, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT {fs::CloseFile(file);};

            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

            // 128MB 分書いて 1 ループは終わり
            // 一度 64MB くらいで作り、128MB にリサイズしてから始める
            // Register する予定はないので、とりあえず読み込み先と同じ ContentId を使っておく
            auto placeholder = storage.GeneratePlaceHolderId();
            const int64_t PlaceHolderSize = 128 << 20;
            NN_RESULT_DO(storage.CreatePlaceHolder(placeholder, contentId, 64 << 20));
            NN_UTIL_SCOPE_EXIT {storage.DeletePlaceHolder(placeholder);};
            NN_RESULT_DO(storage.SetPlaceHolderSize(placeholder, PlaceHolderSize));

            int64_t offset = 0;
            int64_t fileOffset = 0;
            while (offset < PlaceHolderSize)
            {
                size_t readSize;
                NN_RESULT_DO(fs::ReadFile(&readSize, file, fileOffset, m_Buffer, m_BufferSize));

                // TODO: もう少しアライメントがそろわない書き込みが発生するようにしたほうが良いのでは
                size_t writeSize = std::min(static_cast<size_t>(PlaceHolderSize - offset), readSize);
                NN_RESULT_DO(storage.WritePlaceHolder(placeholder, offset, m_Buffer, readSize));
                fileOffset += readSize;
                offset += writeSize;


                if (fileOffset >= fileSize)
                {
                    fileOffset = 0;
                }
            }
            Measure(PlaceHolderSize);
        }
        NN_RESULT_SUCCESS;
    }

    void BackgroundApplyDeltaStressTask::Measure(size_t downloadSize) NN_NOEXCEPT
    {
        auto endMeasurement = os::GetSystemTick();
        auto time = (endMeasurement - m_BeginMeasurement).ToTimeSpan().GetMilliSeconds();
        if (time > 0)
        {
            m_Throughput = static_cast<double>(downloadSize) / time;
        }
    }

    void BackgroundApplyDeltaStressTask::GetInfo(BackgroundDownloadStressTaskInfo* outValue) const NN_NOEXCEPT
    {
        *outValue = { m_IsInitialized, m_State, m_StorageId, {}, m_LastResult.GetInnerValueForDebug(), m_Throughput };
    }

}}}
