﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/nim/srv/detail/nim_BufferAllocator.h>
#include <nn/nim/srv/nim_NetworkInstallTask.h>
#include <nn/nim/srv/nim_NetworkInstallTaskHolder.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>
#include <nn/nim/srv/nim_ThreadAllocator.h>
#include "nim_AsyncImpl.h"

namespace nn { namespace nim { namespace srv {

    namespace detail {
        template<typename TaskHolderT>
        class AsyncTaskResultImpl : public AsyncResultImpl<AsyncTaskResultImpl<TaskHolderT>>
        {
            typedef AsyncResultImpl<AsyncTaskResultImpl<TaskHolderT>> AsyncTaskResultImplT;
        public:
            AsyncTaskResultImpl(ThreadAllocator* allocator, TaskHolderT* holder, BufferAllocator* bufferAllocator, BufferFlagSet flag, fs::PriorityRaw fsPriority) NN_NOEXCEPT : AsyncTaskResultImplT(this, allocator), m_BufferAllocator(bufferAllocator), m_Flag(flag), m_Holder(holder), m_FsPriority(fsPriority)
            {
                m_Holder->GetTask().ResetCancel();
                size_t outSize;
                m_Buffer = m_BufferAllocator->Allocate(&outSize, m_Flag);
                m_Holder->GetTask().SetBuffer(m_Buffer, outSize);
            }

            Result Initialize() NN_NOEXCEPT
            {
                return m_Holder->GetTask().InitializeBeforeRun();
            }

            ~AsyncTaskResultImpl() NN_NOEXCEPT
            {
                AsyncTaskResultImplT::Join();
                m_Holder->GetTask().FinalizeAfterRun();
                m_Holder->SetRunning(false);
                m_BufferAllocator->Free(m_Buffer, m_Flag);
            }

            Result Run() NN_NOEXCEPT
            {
                NN_RESULT_DO(AsyncTaskResultImplT::Run());
                m_Holder->SetRunning(true);
                NN_RESULT_SUCCESS;
            }

            Result Execute() NN_NOEXCEPT
            {
                // ncm で発生する fs アクセスの優先度を設定する
                // 注意:    IPC 先で fs アクセスが発生するときに影響するので、
                //          他のプロセスへの IPC を行うときには注意
                fs::SetPriorityRawOnCurrentThread(m_FsPriority);
                NN_RESULT_DO(m_Holder->GetTask().PrepareAndExecute());
                NN_RESULT_SUCCESS;
            }

            Result Cancel() NN_NOEXCEPT
            {
                m_Holder->GetTask().Cancel();
                NN_RESULT_SUCCESS;
            }

            virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
            {
                m_Holder->GetTask().GetErrorContext(outValue.GetPointer());
                NN_RESULT_SUCCESS;
            }

        private:
            BufferAllocator* m_BufferAllocator;
            void* m_Buffer;
            BufferFlagSet m_Flag;
            TaskHolderT* m_Holder;
            fs::PriorityRaw m_FsPriority;
        };
    }


    class AsyncNetworkSystemUpdateTaskResultImpl : public detail::AsyncTaskResultImpl<SystemUpdateTaskHolder>
    {
    public:
        explicit AsyncNetworkSystemUpdateTaskResultImpl(SystemUpdateTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncNetworkInstallTaskResultImpl : public detail::AsyncTaskResultImpl<NetworkInstallTaskHolder>
    {
    public:
        explicit AsyncNetworkInstallTaskResultImpl(NetworkInstallTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncApplyDeltaTaskResultImpl : public detail::AsyncTaskResultImpl<ApplyDeltaTaskHolder>
    {
    public:
        explicit AsyncApplyDeltaTaskResultImpl(ApplyDeltaTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncLocalCommunicationReceiveApplicationTaskResultImpl : public detail::AsyncTaskResultImpl<LocalCommunicationReceiveApplicationTaskHolder>
    {
    public:
        explicit AsyncLocalCommunicationReceiveApplicationTaskResultImpl(LocalCommunicationReceiveApplicationTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncLocalCommunicationSendApplicationTaskResultImpl : public detail::AsyncTaskResultImpl<LocalCommunicationSendApplicationTaskHolder>
    {
    public:
        explicit AsyncLocalCommunicationSendApplicationTaskResultImpl(LocalCommunicationSendApplicationTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncLocalCommunicationReceiveSystemUpdateTaskResultImpl : public detail::AsyncTaskResultImpl<LocalCommunicationReceiveSystemUpdateTaskHolder>
    {
    public:
        explicit AsyncLocalCommunicationReceiveSystemUpdateTaskResultImpl(LocalCommunicationReceiveSystemUpdateTaskHolder* holder) NN_NOEXCEPT;
    };

    class AsyncLocalCommunicationSendSystemUpdateTaskResultImpl : public detail::AsyncTaskResultImpl<LocalCommunicationSendSystemUpdateTaskHolder>
    {
    public:
        explicit AsyncLocalCommunicationSendSystemUpdateTaskResultImpl(LocalCommunicationSendSystemUpdateTaskHolder* holder) NN_NOEXCEPT;
    };
}}}
