﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/util/util_Optional.h>
#include <nn/nim/detail/nim_IAsync.sfdl.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_OsUtil.h>
#include <nn/nim/srv/nim_ThreadAllocator.h>

namespace nn { namespace nim { namespace srv {
    template <class TaskT>
    class AsyncResultImpl
    {
    public:
        explicit AsyncResultImpl(TaskT* task, ThreadAllocator* allocator) NN_NOEXCEPT : m_Task(task), m_Allocator(allocator) {}
        ~AsyncResultImpl() NN_NOEXCEPT
        {
            if (m_ThreadInfo)
            {
                os::DestroyThread(m_ThreadInfo->thread);
                m_Allocator->Free(*m_ThreadInfo);
            }
        }

        void Join() NN_NOEXCEPT
        {
            if (m_ThreadInfo)
            {
                m_Task->Cancel();
                os::WaitThread(m_ThreadInfo->thread);
            }
        }

        Result Run() NN_NOEXCEPT
        {
            ThreadInfo info;
            NN_RESULT_DO(m_Allocator->Allocate(&info));

            NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
            {
                auto t = reinterpret_cast<AsyncResultImpl<TaskT>*>(p);
                t->m_Result = t->m_Task->Execute();
                t->m_Event.Signal();
            }, this, info.stack, info.stackSize, info.priority));
            os::SetThreadNamePointer(info.thread, info.threadName);
            os::StartThread(info.thread);

            m_ThreadInfo = info;

            NN_RESULT_SUCCESS;
        }

        Result Get() const NN_NOEXCEPT
        {
            return m_Result;
        }

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT
        {
            *outValue = {};
            NN_RESULT_SUCCESS;
        }

    private:
        TaskT* m_Task;
        ThreadAllocator* m_Allocator;
        ManualClearSystemEvent m_Event;
        util::optional<ThreadInfo> m_ThreadInfo;
        Result m_Result;
    };

    template<class TaskT, typename T>
    class AsyncValueImpl : public AsyncResultImpl<AsyncValueImpl<TaskT, T>>
    {
    public:
        explicit AsyncValueImpl(TaskT* task, ThreadAllocator* allocator) NN_NOEXCEPT : AsyncResultImpl<AsyncValueImpl<TaskT, T>>(this, allocator), m_Task(task), m_Value(){}

        Result GetSize(sf::Out<std::uint64_t> outValue) NN_NOEXCEPT
        {
            *outValue = sizeof(m_Value);

            NN_RESULT_SUCCESS;
        }

        Result Get(const sf::OutBuffer& buffer) const NN_NOEXCEPT
        {
            NN_RESULT_DO((AsyncResultImpl<AsyncValueImpl<TaskT, T>>::Get()));
            NN_RESULT_THROW_UNLESS(sizeof(m_Value) <= buffer.GetSize(), ResultBufferNotEnough());
            std::memcpy(buffer.GetPointerUnsafe(), &m_Value, sizeof(m_Value));

            NN_RESULT_SUCCESS;
        }

        Result Execute() NN_NOEXCEPT
        {
            return m_Task->ExecuteAndValue(&m_Value);
        }

        Result Cancel() NN_NOEXCEPT
        {
            return m_Task->CancelImpl();
        }

    private:
        TaskT* m_Task;
        T m_Value;
    };
}}}
