﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_Base64.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nim/detail/nim_Log.h>
#include "nim_AsyncGameCardRegistrationImpl.h"
#include "nim_EShopUtil.h"
#include "nim_Json.h"

namespace nn { namespace nim { namespace srv {
    namespace
    {
        static const int GameCardThreadCount = 1;
        static const size_t StackSize = 16 * 1024;

        os::ThreadType g_GameCardTaskThreadList[GameCardThreadCount];
        NN_OS_ALIGNAS_THREAD_STACK char g_GameCardTaskStack[StackSize * GameCardThreadCount];

        ThreadAllocator g_GameCardTaskThreadAllocator(g_GameCardTaskThreadList, GameCardThreadCount, NN_SYSTEM_THREAD_PRIORITY(nim, GameCardTask), g_GameCardTaskStack, sizeof(g_GameCardTaskStack), StackSize, NN_SYSTEM_THREAD_NAME(nim, GameCardTask));

        typedef kvdb::BoundedString<1024 + 512> PostFields;

        Result ParseGameCardResponse(GameCardRegistrationStatus* outValue, char* readBuffer, bool needGoldpointResponse) NN_NOEXCEPT
        {
            GameCardRegistrationStatus status;

            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseGameCardParseError());

            auto registeredObj = document.FindMember("registered");
            NN_RESULT_THROW_UNLESS(registeredObj != document.MemberEnd() && registeredObj->value.IsBool(), ResultUnexpectedResponseGameCardRegisteredMemberNotFound());
            status.isRegistered = registeredObj->value.GetBool();

            if (!status.isRegistered && needGoldpointResponse)
            {
                auto pointObj = document.FindMember("gold_point");
                NN_RESULT_THROW_UNLESS(pointObj != document.MemberEnd() && pointObj->value.IsUint(), ResultUnexpectedResponseGameCardGoldPointMemberNotFound());
                status.goldPoint = pointObj->value.GetUint();
            }
            else
            {
                status.goldPoint = 0;
            }

            *outValue = status;
            NN_RESULT_SUCCESS;
        }

        struct Response
        {
            char str[1024];
        };

        Result PerformGameCardRequest(Response* outValue, HttpConnection* connection, const Url& url, const PostFields& postFields, const char* headerList[], int headerCount) NN_NOEXCEPT
        {
            Response response = {};
            size_t readSize = 0;
            auto result = connection->Post(url, postFields,
                [&readSize, &response](const void* buffer, size_t size) -> Result
                {
                    NN_RESULT_THROW_UNLESS(readSize + size < sizeof(response), ResultUnexpectedResponseGameCardTooLong());
                    std::memcpy(&response.str[readSize], buffer, size);
                    readSize += size;

                    NN_RESULT_SUCCESS;
                },
                headerList, headerCount);
            if (result <= ResultHttpStatus())
            {
                NN_RESULT_TRY(ToEShopResultResponseDestructively(response.str))
                    NN_RESULT_CATCH(ResultInvalidJson) { NN_RESULT_THROW(result); }
                NN_RESULT_END_TRY
            }
            NN_DETAIL_NIM_TRACE("[PerformGameCardRequest] Response %s\n", response.str);
            NN_RESULT_DO(result);

            *outValue = response;
            NN_RESULT_SUCCESS;
        }

        void ConvertApplicationIdListToQueryParameter(PostFields* out, const ApplicationIdList& appIdList) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(appIdList.GetCount() > 0);

            out->AppendFormat("application_id=%016llx", appIdList[0]);
            for (int i = 1; i < appIdList.GetCount(); ++i)
            {
                out->AppendFormat(",%016llx", appIdList[i]);
            }
        }
    }

    AsyncGameCardRegistrationStatusImpl::AsyncGameCardRegistrationStatusImpl() NN_NOEXCEPT : AsyncValueImpl<AsyncGameCardRegistrationStatusImpl, GameCardRegistrationStatus>(this, &g_GameCardTaskThreadAllocator) {}

    Result AsyncGameCardRegistrationStatusImpl::Initialize(DeviceContext* deviceContext, const ncm::ApplicationId appIdList[], int appIdCount, const char* token, const void * cert, size_t certSize) NN_NOEXCEPT
    {
        m_AppIdList.Initialize(appIdList, appIdCount);
        auto base64Status = util::Base64::ToBase64String(m_CertBase64, sizeof(m_CertBase64), cert, certSize, util::Base64::Mode_NormalNoLinefeed);
        NN_DETAIL_NIM_TRACE("[AsyncGameCardRegistrationStatusImpl] certificate base64 %s\n", m_CertBase64);
        NN_RESULT_THROW_UNLESS(base64Status == util::Base64::Status_Success, ResultBufferNotEnough());
        NN_RESULT_DO(HttpConnection::Escape(m_CertBase64, sizeof(m_CertBase64), m_CertBase64));
        InitializeAuthorizationHeaderAndNeedGoldpointResponseFlag(token);
        NN_RESULT_DO(m_Connection.Initialize(deviceContext));

        NN_RESULT_SUCCESS;
    }

    void AsyncGameCardRegistrationStatusImpl::InitializeAuthorizationHeaderAndNeedGoldpointResponseFlag(const char* token) NN_NOEXCEPT
    {
        if (token[0] != '\0')
        {
            util::SNPrintf(m_AuthorizationHeader, sizeof(m_AuthorizationHeader), "%s%s", detail::AuthorizationHeaderPrefix, token);
            m_NeedGoldpointResponse = true;
        }
        else
        {
            m_AuthorizationHeader[0] = '\0';
            m_NeedGoldpointResponse = false;
        }
    }

    Result AsyncGameCardRegistrationStatusImpl::ExecuteAndValue(GameCardRegistrationStatus* outValue) NN_NOEXCEPT
    {
        Url url;
        MakeGameCardRegistrationStatusUrl(&url);

        PostFields postFields;
        ConvertApplicationIdListToQueryParameter(&postFields, m_AppIdList);
        postFields.AppendFormat("&rom_card_cert=%s", m_CertBase64);
        const char* headerList[] = { "Accept: application/json", m_AuthorizationHeader };

        Response response;
        NN_RESULT_DO(PerformGameCardRequest(&response, &m_Connection, url, postFields, headerList, sizeof(headerList) / sizeof(headerList[0])));
        NN_RESULT_DO(ParseGameCardResponse(outValue, response.str, m_NeedGoldpointResponse));

        NN_RESULT_SUCCESS;
    }

    Result AsyncRegisterGameCardImpl::Initialize(DeviceContext* deviceContext, const ncm::ApplicationId appIdList[], int appIdCount, const char* token, const void* cert, size_t certSize, GameCardRegistrationStatus status) NN_NOEXCEPT
    {
        m_AppIdList.Initialize(appIdList, appIdCount);
        auto base64Status = util::Base64::ToBase64String(m_CertBase64, sizeof(m_CertBase64), cert, certSize, util::Base64::Mode_NormalNoLinefeed);
        NN_DETAIL_NIM_TRACE("[AsyncRegisterGameCardImpl] certificate base64 %s\n", m_CertBase64);
        NN_RESULT_THROW_UNLESS(base64Status == util::Base64::Status_Success, ResultBufferNotEnough());
        NN_RESULT_DO(HttpConnection::Escape(m_CertBase64, sizeof(m_CertBase64), m_CertBase64));
        util::SNPrintf(m_AuthorizationHeader, sizeof(m_AuthorizationHeader), "%s%s", detail::AuthorizationHeaderPrefix, token);
        m_Status = status;
        NN_RESULT_DO(m_Connection.Initialize(deviceContext));

        NN_RESULT_SUCCESS;
    }

    AsyncRegisterGameCardImpl::AsyncRegisterGameCardImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncRegisterGameCardImpl>(this, &g_GameCardTaskThreadAllocator) {}

    Result AsyncRegisterGameCardImpl::Execute() NN_NOEXCEPT
    {
        Url url;
        MakeRegisterGameCardUrl(&url);

        PostFields postFields;
        ConvertApplicationIdListToQueryParameter(&postFields, m_AppIdList);
        postFields.AppendFormat("&rom_card_cert=%s", m_CertBase64);
        postFields.AppendFormat("&expected_gold_point=%d", m_Status.goldPoint);
        const char* headerList[] = { "Accept: application/json", m_AuthorizationHeader };

        Response response;
        NN_RESULT_DO(PerformGameCardRequest(&response, &m_Connection, url, postFields, headerList, sizeof(headerList) / sizeof(headerList[0])));
        GameCardRegistrationStatus status;
        NN_RESULT_DO(ParseGameCardResponse(&status, response.str, false));
        NN_RESULT_THROW_UNLESS(status.isRegistered, ResultRegisterGameCardFailed());

        NN_RESULT_SUCCESS;
    }

    void ApplicationIdList::Initialize(const ncm::ApplicationId appIdList[], int count) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(count > 0 && count <= MaxRegisterableApplicationCount);
        m_Count = count;
        std::copy(appIdList, appIdList + count, m_AppIdList.begin());
    }

    ncm::ApplicationId ApplicationIdList::operator[](int index) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Count > 0 && m_Count <= MaxRegisterableApplicationCount);
        NN_ABORT_UNLESS(index >= 0 && index < m_Count);

        return m_AppIdList[index];
    }

    int ApplicationIdList::GetCount() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Count > 0 && m_Count <= MaxRegisterableApplicationCount);
        return m_Count;
    }

}}}
