﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ec/system/ec_DeviceAccountTypes.h>
#include <nn/nim/srv/nim_DeviceAccountStore.h>
#include <nn/nim/srv/nim_EciAccessor.h>
#include <nn/result/result_HandlingUtility.h>
#include "nim_AsyncImpl.h"

namespace nn { namespace nim { namespace srv {

    template<typename TaskT>
    class AsyncDeviceAccountResultImplBase : public AsyncResultImpl<TaskT>
    {
    public:
        Result Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
        {
            m_DeviceContext = deviceContext;
            return m_Connection.Initialize(deviceContext);
        }
        Result Cancel() NN_NOEXCEPT
        {
            m_Connection.Cancel();
            NN_RESULT_SUCCESS;
        }
        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Connection.GetErrorContext(outValue.GetPointer());
            NN_RESULT_SUCCESS;
        }

    protected:
        explicit AsyncDeviceAccountResultImplBase(TaskT* self) NN_NOEXCEPT;
        DeviceContext* m_DeviceContext;
        HttpConnection m_Connection;
    };

    template<typename TaskT, typename T>
    class AsyncDeviceAccountValueImplBase : public AsyncValueImpl<TaskT, T>
    {
    public:
        Result Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
        {
            m_DeviceContext = deviceContext;
            return m_Connection.Initialize(deviceContext);
        }
        Result CancelImpl() NN_NOEXCEPT
        {
            m_Connection.Cancel();
            NN_RESULT_SUCCESS;
        }

    protected:
        explicit AsyncDeviceAccountValueImplBase(TaskT* self) NN_NOEXCEPT;
        DeviceContext* m_DeviceContext;
        HttpConnection m_Connection;
    };

    class AsyncRegisterDeviceAccountImpl : public AsyncDeviceAccountResultImplBase<AsyncRegisterDeviceAccountImpl>
    {
    public:
        explicit AsyncRegisterDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT;
        ~AsyncRegisterDeviceAccountImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Execute() NN_NOEXCEPT;

    private:
        DeviceAccountStore* m_Store;
    };

    class AsyncUnregisterDeviceAccountImpl : public AsyncDeviceAccountResultImplBase<AsyncUnregisterDeviceAccountImpl>
    {
    public:
        explicit AsyncUnregisterDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT;
        ~AsyncUnregisterDeviceAccountImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Execute() NN_NOEXCEPT;

    private:
        DeviceAccountStore* m_Store;
    };

    class AsyncDeviceAccountStatusImpl : public AsyncDeviceAccountValueImplBase<AsyncDeviceAccountStatusImpl, ec::system::DeviceAccountStatus>
    {
    public:
        explicit AsyncDeviceAccountStatusImpl(DeviceAccountStore* store) NN_NOEXCEPT;
        ~AsyncDeviceAccountStatusImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
        {
            NN_RESULT_DO(AsyncDeviceAccountValueImplBase::Initialize(deviceContext));
            NN_RESULT_DO(m_Store->Get(&m_DeviceAccountInfo));
            NN_RESULT_SUCCESS;
        }
        Result ExecuteAndValue(ec::system::DeviceAccountStatus* outValue) NN_NOEXCEPT;

    private:
        DeviceAccountInfo m_DeviceAccountInfo;
        DeviceAccountStore* m_Store;
    };

    class AsyncDeviceRegistrationInfoImpl : public AsyncDeviceAccountValueImplBase<AsyncDeviceRegistrationInfoImpl, ec::system::DeviceRegistrationInfo>
    {
    public:
        AsyncDeviceRegistrationInfoImpl() NN_NOEXCEPT;
        ~AsyncDeviceRegistrationInfoImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
        {
            NN_RESULT_DO(AsyncDeviceAccountValueImplBase::Initialize(deviceContext));
            NN_RESULT_SUCCESS;
        }
        Result ExecuteAndValue(ec::system::DeviceRegistrationInfo* outValue) NN_NOEXCEPT;
    };

    class AsyncTransferDeviceAccountImpl : public AsyncDeviceAccountResultImplBase<AsyncTransferDeviceAccountImpl>
    {
    public:
        explicit AsyncTransferDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT;
        ~AsyncTransferDeviceAccountImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Execute() NN_NOEXCEPT;

    private:
        DeviceAccountStore* m_Store;
    };

    class AsyncSyncRegistrationImpl : public AsyncDeviceAccountResultImplBase<AsyncSyncRegistrationImpl>
    {
    public:
        explicit AsyncSyncRegistrationImpl(DeviceAccountStore* store) NN_NOEXCEPT;
        ~AsyncSyncRegistrationImpl() NN_NOEXCEPT
        {
            Join();
        };

        Result Execute() NN_NOEXCEPT;

    private:
        DeviceAccountStore* m_Store;
    };
}}}
