﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "nim_AsyncApplicationControlImpl.h"


namespace nn { namespace nim { namespace srv {

    namespace
    {
        static const int ApplicationControlThreadCount = 1;
        static const size_t StackSize = 16 * 1024;

        os::ThreadType g_ApplicationControlTaskThreadList[ApplicationControlThreadCount];
        NN_OS_ALIGNAS_THREAD_STACK char g_ApplicationControlTaskStack[StackSize * ApplicationControlThreadCount];

        ThreadAllocator g_ApplicationControlTaskThreadAllocator(g_ApplicationControlTaskThreadList, ApplicationControlThreadCount, NN_SYSTEM_THREAD_PRIORITY(nim, ApplicationControlTask), g_ApplicationControlTaskStack, sizeof(g_ApplicationControlTaskStack), StackSize, NN_SYSTEM_THREAD_NAME(nim, ApplicationControlTask));
    }

    AsyncApplicationControlImpl::AsyncApplicationControlImpl() NN_NOEXCEPT : AsyncValueImpl(this, &g_ApplicationControlTaskThreadAllocator), m_Version(), m_RequiresLatest(), m_Written(), m_BufferFilledSize() {}

    AsyncApplicationControlImpl::~AsyncApplicationControlImpl() NN_NOEXCEPT
    {
        Join();
        if (m_PlaceHolderId)
        {
            auto result = m_Storage.DeletePlaceHolder(*m_PlaceHolderId);
            if (result.IsFailure())
            {
                NN_DETAIL_NIM_TRACE("[AsyncApplicationControlImpl] Failed to delete placeholder as 0x%08x\n", result.GetInnerValueForDebug());
            }
        }
    }

    Result AsyncApplicationControlImpl::Initialize(DeviceContext* deviceContext, ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT
    {
        return InitializeImpl(deviceContext, id, version, false);
    }

    Result AsyncApplicationControlImpl::InitializeLatest(DeviceContext* deviceContext, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return InitializeImpl(deviceContext, id, 0, true);
    }

    Result AsyncApplicationControlImpl::InitializeImpl(DeviceContext* deviceContext, ncm::ApplicationId id, uint32_t version, bool requiresLatest) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Connection.Initialize(deviceContext));
        NN_RESULT_DO(ncm::OpenContentStorage(&m_Storage, ncm::StorageId::BuildInSystem));
        m_ApplicationId = id;
        m_Version = version;
        m_RequiresLatest = requiresLatest;
        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationControlImpl::ExecuteAndValue(ApplicationControlInfo* outValue) NN_NOEXCEPT
    {
        // ncm で発生する fs アクセスの優先度を設定する
        // 注意:    IPC 先で fs アクセスが発生するときに影響するので、
        //          他のプロセスへの IPC を行うときには注意
        fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);
        Url url;
        if (m_RequiresLatest)
        {
            MakeLatestApplicationControlContentUrl(&url, m_ApplicationId);
        }
        else
        {
            MakeApplicationControlContentUrl(&url, m_ApplicationId, m_Version);
        }

        NN_RESULT_DO(m_Connection.Get(url,
            [this](const void* buffer, size_t bufferSize)->Result
            {
                {
                    util::optional<HttpHeaderValue> value;
                    NN_RESULT_DO(FindHttpHeader(&value, "Content-Length", reinterpret_cast<const char*>(buffer), bufferSize));
                    if (!m_ReceivedContentLength && value)
                    {
                        m_ReceivedContentLength = std::strtoll(value->string, nullptr, 10);
                    }
                }

                {
                    util::optional<HttpHeaderValue> value;
                    NN_RESULT_DO(FindHttpHeader(&value, "X-Nintendo-Title-Version", reinterpret_cast<const char*>(buffer), bufferSize));
                    if (!m_ReceivedVersion && value)
                    {
                        m_ReceivedVersion = std::strtoul(value->string, nullptr, 10);
                    }
                }

                {
                    util::optional<HttpHeaderValue> value;
                    NN_RESULT_DO(FindHttpHeader(&value, "X-Nintendo-Content-ID", reinterpret_cast<const char*>(buffer), bufferSize));
                    if (!m_ReceivedContentId && value)
                    {
                        m_ReceivedContentId = ncm::GetContentIdFromString(value->string, sizeof(value->string));
                    }
                }

                {
                    util::optional<HttpHeaderValue> value;
                    NN_RESULT_DO(FindHttpHeader(&value, "X-Nintendo-Content-Hash", reinterpret_cast<const char*>(buffer), bufferSize));
                    if (!m_ReceivedHash && value)
                    {
                        m_ReceivedHash = ncm::GetHashFromString(value->string, sizeof(value->string));
                    }
                }

                if (!m_PlaceHolderId && m_ReceivedContentLength && m_ReceivedContentId)
                {
                    auto placeHolderId = m_Storage.GeneratePlaceHolderId();
                    NN_RESULT_DO(m_Storage.CreatePlaceHolder(placeHolderId, *m_ReceivedContentId, *m_ReceivedContentLength));
                    m_PlaceHolderId = placeHolderId;
                }

                NN_RESULT_SUCCESS;
            },
            [this](const void* buffer, size_t bufferSize)->Result
            {
                if (m_PlaceHolderId)
                {
                    NN_RESULT_DO(Write(buffer, bufferSize));
                }
                NN_RESULT_SUCCESS;
            }));
        NN_RESULT_DO(Flush());

        NN_RESULT_THROW_UNLESS(m_ReceivedHash, ResultUnexpectedResponseApplicationControlUnknownHash());
        NN_RESULT_THROW_UNLESS(m_ReceivedContentId, ResultUnexpectedResponseApplicationControlUnknownVersion());
        NN_RESULT_THROW_UNLESS(m_ReceivedVersion, ResultUnexpectedResponseApplicationControlUnknownVersion());
        NN_RESULT_THROW_UNLESS(m_PlaceHolderId, ResultUnexpectedResponseApplicationControlUnknownLength());
        m_Storage.GetPlaceHolderPath(&outValue->path, *m_PlaceHolderId);
        outValue->version = *m_ReceivedVersion;
        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationControlImpl::CancelImpl() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationControlImpl::Write(const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        if (bufferSize > sizeof(m_Buffer))
        {
            NN_RESULT_DO(Flush());
            NN_RESULT_DO(WritePlaceHolder(buffer, bufferSize));
            NN_RESULT_SUCCESS;
        }

        if (m_BufferFilledSize + bufferSize > sizeof(m_Buffer))
        {
            NN_RESULT_DO(Flush());
        }

        std::memcpy(&m_Buffer[m_BufferFilledSize], buffer, bufferSize);
        m_BufferFilledSize += bufferSize;

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationControlImpl::Flush() NN_NOEXCEPT
    {
        if (m_BufferFilledSize > 0)
        {
            NN_RESULT_DO(WritePlaceHolder(m_Buffer, m_BufferFilledSize));
            m_BufferFilledSize = 0;
        }

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationControlImpl::WritePlaceHolder(const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Storage.WritePlaceHolder(*m_PlaceHolderId, m_Written, buffer, bufferSize));
        m_Written += bufferSize;
        NN_RESULT_SUCCESS;
    }

}}}
