﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/nim/detail/nim_ServiceName.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nim/srv/nim_NetworkInstallManagerServer.h>
#endif

namespace nn { namespace nim {

    namespace
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        util::optional<srv::DeviceContext> g_DeviceContext;
        util::optional<srv::DeviceAccountStore> g_DeviceAccountStore;
        util::optional<sf::UnmanagedServiceObject<detail::INetworkInstallManager, srv::NetworkInstallManagerServer>> g_Server;

        const int64_t   DeviceAccountSaveDataSize = 0xc000; // 16KB + 32 KB
        const int64_t   DeviceAccountSaveDataJournalSize = 0xc000; // 16KB + 32 KB
        const int       DeviceAccountSaveDataFlags = 0;
#endif
        sf::ProxyObjectAllocator<16> g_ServiceObjectAllocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
        sf::SharedPointer<detail::INetworkInstallManager> g_Interface;

        template <typename AsyncT, typename IAsyncT, typename RequestFuncT>
        Result RequestAsyncValue(AsyncT* outValue, RequestFuncT func) NN_NOEXCEPT
        {
            sf::NativeHandle nativeHandle;
            sf::SharedPointer<IAsyncT> sp;
            NN_RESULT_DO(func(&nativeHandle, &sp));
            outValue->Initialize(sp, nativeHandle);
            nativeHandle.Detach();

            NN_RESULT_SUCCESS;
        }
    }

    int64_t AsyncData::GetSize() NN_NOEXCEPT
    {
        int64_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Async->GetSize(&size));
        return size;
    }

    Result AsyncData::Read(size_t* outValue, int64_t offset, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        uint64_t outSize;
        NN_RESULT_DO(m_Async->Read(&outSize, offset, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize)));
        *outValue = static_cast<size_t>(outSize);
        NN_RESULT_SUCCESS;
    }

    Result AsyncData::GetETag(ETag* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Async->GetETag(outValue));
        NN_RESULT_SUCCESS;
    }

    void InitializeForNetworkInstallManager()
    {
        NN_SDK_ASSERT(!g_Interface);

#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_DeviceContext.emplace();
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_DeviceContext->Initialize());
        g_DeviceAccountStore.emplace();
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_DeviceAccountStore->Initialize("nim_dac", 0x8000000000000073ull, DeviceAccountSaveDataSize, DeviceAccountSaveDataJournalSize, DeviceAccountSaveDataFlags));
        g_Server.emplace();
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->GetImpl().Initialize(&(*g_DeviceContext), &(*g_DeviceAccountStore), false, true));
        g_Interface = g_Server->GetShared();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
        g_ServiceObjectAllocator.Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS((sf::CreateHipcProxyByName<detail::INetworkInstallManager>(&g_Interface, g_ServiceObjectAllocator.GetMemoryResource(), detail::ServiceName)));

#else
    #error "unsupported os"
#endif
    }

    void FinalizeForNetworkInstallManager()
    {
        g_Interface = nullptr;
#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Server = util::nullopt;
        g_DeviceAccountStore = util::nullopt;
        g_DeviceContext = util::nullopt;
#endif
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        g_ServiceObjectAllocator.Finalize();
#endif
    }

    Result CreateSystemUpdateTask(SystemUpdateTaskId* outValue, const ncm::ContentMetaKey& key, bool requiresExFatDriver) NN_NOEXCEPT
    {
        auto config = requiresExFatDriver ? ncm::InstallConfig_SystemUpdateRecursive | ncm::InstallConfig_RequiresExFatDriver :
                                            ncm::InstallConfig_SystemUpdateRecursive;
        return g_Interface->CreateSystemUpdateTask(outValue, key, config);
    }

    Result DestroySystemUpdateTask(const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroySystemUpdateTask(id);
    }

    int ListSystemUpdateTask(SystemUpdateTaskId outList[], int count) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListSystemUpdateTask(&outCount, sf::OutArray<SystemUpdateTaskId>(outList, count)));

        return outCount;
    }

    Result RequestSystemUpdateTaskRun(AsyncResult* outValue, const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestSystemUpdateTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetSystemUpdateTaskInfo(SystemUpdateTaskInfo* outValue, const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetSystemUpdateTaskInfo(outValue, id);
    }

    Result CommitSystemUpdateTask(const SystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitSystemUpdateTask(id);
    }

    Result CreateNetworkInstallTask(NetworkInstallTaskId* outValue, ncm::ApplicationId appId, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storage) NN_NOEXCEPT
    {
        return g_Interface->CreateNetworkInstallTask(outValue, appId, sf::InArray<ncm::ContentMetaKey>(keyList, static_cast<size_t>(count)), storage, ncm::InstallConfig_LatestAddOnContentsAuto);
    }

    Result CreateNetworkInstallTask(NetworkInstallTaskId* outValue, ncm::ApplicationId appId, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storage, Bit32 config) NN_NOEXCEPT
    {
        return g_Interface->CreateNetworkInstallTask(outValue, appId, sf::InArray<ncm::ContentMetaKey>(keyList, static_cast<size_t>(count)), storage, config);
    }

    Result DestroyNetworkInstallTask(const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyNetworkInstallTask(id);
    }

    int ListNetworkInstallTask(NetworkInstallTaskId outList[], int count) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListNetworkInstallTask(&outCount, sf::OutArray<NetworkInstallTaskId>(outList, count)));

        return outCount;
    }

    Result AddNetworkInstallTaskContentMeta(const ncm::ContentMetaKey list[], int count, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->AddNetworkInstallTaskContentMeta(sf::InArray<ncm::ContentMetaKey>(list, static_cast<size_t>(count)), id);
    }

    Result RequestNetworkInstallTaskRun(AsyncResult* outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestNetworkInstallTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetNetworkInstallTaskInfo(NetworkInstallTaskInfo* outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetNetworkInstallTaskInfo(outValue, id);
    }

    Result GetNetworkInstallTaskErrorContext(err::ErrorContext* outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetNetworkInstallTaskErrorContext(outValue, id);
    }

    Result ListNetworkInstallTaskContentMeta(int* outValue, ncm::StorageContentMetaKey* outList, int count, int offset, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListNetworkInstallTaskContentMeta(outValue, sf::OutArray<ncm::StorageContentMetaKey>(outList, count), offset, id);
    }

    Result ListNetworkInstallTaskCommittedContentMeta(int* outValue, ncm::StorageContentMetaKey* outList, int count, int offset, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListNetworkInstallTaskCommittedContentMeta(outValue, sf::OutArray<ncm::StorageContentMetaKey>(outList, count), offset, id);
    }

    Result ListNetworkInstallTaskNotCommittedContentMeta(int* outValue, ncm::StorageContentMetaKey* outList, int count, int offset, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListNetworkInstallTaskNotCommittedContentMeta(outValue, sf::OutArray<ncm::StorageContentMetaKey>(outList, count), offset, id);
    }

    Result ListNetworkInstallTaskContentMetaFromInstallMeta(int* outValue, ncm::ContentMetaKey* outList, int count, int offset, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListNetworkInstallTaskContentMetaFromInstallMeta(outValue, sf::OutArray<ncm::ContentMetaKey>(outList, count), offset, id);
    }

    Result CommitNetworkInstallTask(const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitNetworkInstallTask(id);
    }

    Result CommitNetworkInstallTaskPartially(const ncm::StorageContentMetaKey list[], int count, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitNetworkInstallTaskPartially(sf::InArray<ncm::StorageContentMetaKey>(list, static_cast<size_t>(count)), id);
    }

    int ListApplicationNetworkInstallTask(NetworkInstallTaskId outList[], int count, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListApplicationNetworkInstallTask(&outCount, sf::OutArray<NetworkInstallTaskId>(outList, count), appId));

        return outCount;
    }

    Result SetNetworkInstallTaskAttribute(const NetworkInstallTaskId& id, Bit64 attribute) NN_NOEXCEPT
    {
        NetworkInstallTaskIdAttribute idAttribute = { id, attribute };
        return g_Interface->SetNetworkInstallTaskAttribute(idAttribute);
    }

    Result RequestSystemUpdateMeta(AsyncContentMetaKey* outValue) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncContentMetaKey, detail::IAsyncValue>(outValue,
            [](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
            {
                return g_Interface->RequestLatestSystemUpdateMeta(outHandle, outResult);
            }
        );
    }

    Result RequestApplicationControl(AsyncApplicationControlInfo* outValue, ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncApplicationControlInfo, detail::IAsyncValue>(outValue,
            [id, version](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
        {
            return g_Interface->RequestApplicationControl(outHandle, outResult, id, version);
        }
        );
    }

    Result RequestLatestApplicationControl(AsyncApplicationControlInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncApplicationControlInfo, detail::IAsyncValue>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
        {
            return g_Interface->RequestLatestApplicationControl(outHandle, outResult, id);
        }
        );
    }

    Result RequestLatestVersion(AsyncLatestKeyList* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncLatestKeyList, detail::IAsyncValue>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
        {
            return g_Interface->RequestLatestVersion(outHandle, outResult, id);
        }
        );
    }

    Result GetDownloadedSystemDataPath(ncm::Path* outValue, ncm::SystemDataId dataId, const nim::SystemUpdateTaskId& taskId) NN_NOEXCEPT
    {
        return g_Interface->GetDownloadedSystemDataPath(outValue, dataId, taskId);
    }

    Result CalculateNetworkInstallTaskRequiredSize(int64_t* outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CalculateNetworkInstallTaskRequiredSize(outValue, id);
    }

    Result CalculateNetworkInstallTaskContentsSize(int64_t* outValue, const NetworkInstallTaskId& id, const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
    {
        return g_Interface->CalculateNetworkInstallTaskContentsSize(outValue, id, key, storageId);
    }

    Result ListNetworkInstallTaskOccupiedSize(int* outCount, ncm::InstallTaskOccupiedSize* outList, int numList, int offset, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListNetworkInstallTaskOccupiedSize(outCount, sf::OutArray<ncm::InstallTaskOccupiedSize>(outList, numList), offset, id);
    }

    Result IsExFatDriverIncluded(bool* outValue, const SystemUpdateTaskId& taskId) NN_NOEXCEPT
    {
        return g_Interface->IsExFatDriverIncluded(outValue, taskId);
    }

    Result GetBackgroundDownloadStressTaskInfo(BackgroundDownloadStressTaskInfo* outValue) NN_NOEXCEPT
    {
        return g_Interface->GetBackgroundDownloadStressTaskInfo(outValue);
    }

    Result GetBackgroundApplyDeltaStressTaskInfo(BackgroundDownloadStressTaskInfo* outValue) NN_NOEXCEPT
    {
        return g_Interface->GetBackgroundApplyDeltaStressTaskInfo(outValue);
    }

    Result RequestGameCardRegistrationStatus(AsyncGameCardRegistrationStatus* outValue, const ncm::ApplicationId appIdList[], int appIdCount, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncGameCardRegistrationStatus, detail::IAsyncValue>(outValue,
            [appIdList, appIdCount, token, cert, certSize](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
            {
                sf::InArray<ncm::ApplicationId> appIdArray(appIdList, appIdCount);
                return g_Interface->RequestGameCardRegistrationStatus(outHandle, outResult, appIdArray, sf::InBuffer(token, strlen(token) + 1), sf::InBuffer(static_cast<const char*>(cert), certSize));
            }
            );
    }

    Result RequestRegisterGameCard(AsyncResult* outValue, const ncm::ApplicationId appIdList[], int appIdCount, const char* token, const void* cert, size_t certSize, GameCardRegistrationStatus status) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [appIdList, appIdCount, token, cert, certSize, status](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                sf::InArray<ncm::ApplicationId> appIdArray(appIdList, appIdCount);
                return g_Interface->RequestRegisterGameCard(outHandle, outResult, appIdArray, sf::InBuffer(token, strlen(token) + 1), sf::InBuffer(static_cast<const char*>(cert), certSize), status);
            }
            );
    }

    Result RequestRegisterNotificationToken(AsyncResult* outValue, const npns::NotificationToken& token) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [token](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestRegisterNotificationToken(outHandle, outResult, token);
            }
            );
    }

    Result RequestDownloadTaskList(AsyncData* outValue, const ETag& eTag) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncData, detail::IAsyncData>(outValue,
            [&eTag](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestDownloadTaskList(outHandle, outAsync, eTag);
            }
            );
    }

    Result RequestVersionList(AsyncData* outValue, const ETag& eTag) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncData, detail::IAsyncData>(outValue,
            [&eTag](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestVersionList(outHandle, outAsync, eTag);
            }
            );
    }

    Result CreateApplyDeltaTask(ApplyDeltaTaskId* outValue, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storageId) NN_NOEXCEPT
    {
        return g_Interface->CreateApplyDeltaTask(outValue, appId, sourceKey, sf::InArray<ncm::ContentMetaKey>(keyList, static_cast<size_t>(count)), storageId);
    }

    Result CreateApplyDeltaTaskFromDownloadTask(ApplyDeltaTaskId* outValue, const NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CreateApplyDeltaTaskFromDownloadTask(outValue, id);
    }

    Result DestroyApplyDeltaTask(const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyApplyDeltaTask(id);
    }

    Result RequestApplyDeltaTaskRun(AsyncResult* outValue, const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestApplyDeltaTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result ClearNotEnoughSpaceStateOfApplyDeltaTask(const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ClearNotEnoughSpaceStateOfApplyDeltaTask(id);
    }

    Result GetApplyDeltaTaskInfo(ApplyDeltaTaskInfo* outValue, const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetApplyDeltaTaskInfo(outValue, id);
    }

    Result ListApplyDeltaTaskContentMeta(int* outValue, ncm::StorageContentMetaKey* outList, int count, int offset, const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListApplyDeltaTaskContentMeta(outValue, sf::OutArray<ncm::StorageContentMetaKey>(outList, count), offset, id);
    }

    Result CommitApplyDeltaTask(const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitApplyDeltaTask(id);
    }

    int ListApplyDeltaTask(ApplyDeltaTaskId outList[], int count) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListApplyDeltaTask(&outCount, sf::OutArray<ApplyDeltaTaskId>(outList, count)));

        return outCount;
    }

    int ListApplicationApplyDeltaTask(ApplyDeltaTaskId outList[], int count, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListApplicationApplyDeltaTask(&outCount, sf::OutArray<ApplyDeltaTaskId>(outList, count), appId));

        return outCount;
    }

    Result CalculateApplyDeltaTaskRequiredSize(int64_t* outValue, const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CalculateApplyDeltaTaskRequiredSize(outValue, id);
    }

    Result CalculateApplyDeltaTaskOccupiedSize(int64_t* outValue, const ApplyDeltaTaskId& id, ncm::StorageId storageId) NN_NOEXCEPT
    {
        return g_Interface->CalculateApplyDeltaTaskOccupiedSize(outValue, id, storageId);
    }

    Result GetApplyDeltaTaskRequiredStorage(ncm::StorageId* outValue, const ApplyDeltaTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetApplyDeltaTaskRequiredStorage(outValue, id);
    }

    Result FindMaxRequiredApplicationVersionOfTask(uint32_t* outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->FindMaxRequiredApplicationVersionOfTask(outValue, id);
    }

    Result FindMaxRequiredSystemVersionOfTask(uint32_t* outValue, const nim::NetworkInstallTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->FindMaxRequiredSystemVersionOfTask(outValue, id);
    }

    Result PrepareShutdown() NN_NOEXCEPT
    {
        return g_Interface->PrepareShutdown();
    }

    Result PrepareShutdownForSystemUpdate() NN_NOEXCEPT
    {
        return g_Interface->PrepareShutdownForSystemUpdate();
    }

    Result CreateLocalCommunicationReceiveApplicationTask(LocalCommunicationReceiveApplicationTaskId* outValue, uint32_t ipv4, uint16_t port, ncm::ApplicationId appId, const ncm::ContentMetaKey keyList[], int count, ncm::StorageId storage) NN_NOEXCEPT
    {
        return g_Interface->CreateLocalCommunicationReceiveApplicationTask(outValue, ipv4, port, appId, sf::InArray<ncm::ContentMetaKey>(keyList, static_cast<size_t>(count)), storage, ncm::InstallConfig_LatestAddOnContentsAuto);
    }

    Result DestroyLocalCommunicationReceiveApplicationTask(const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyLocalCommunicationReceiveApplicationTask(id);
    }

    int ListLocalCommunicationReceiveApplicationTask(LocalCommunicationReceiveApplicationTaskId outList[], int count) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListLocalCommunicationReceiveApplicationTask(&outCount, sf::OutArray<LocalCommunicationReceiveApplicationTaskId>(outList, count)));

        return outCount;
    }

    Result RequestLocalCommunicationReceiveApplicationTaskRun(AsyncResult* outValue, const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestLocalCommunicationReceiveApplicationTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetLocalCommunicationReceiveApplicationTaskInfo(LocalCommunicationReceiveApplicationTaskInfo* outValue, const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationReceiveApplicationTaskInfo(outValue, id);
    }

    Result ListLocalCommunicationReceiveApplicationTaskContentMeta(int* outValue, ncm::StorageContentMetaKey* outList, int count, int offset, const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->ListLocalCommunicationReceiveApplicationTaskContentMeta(outValue, sf::OutArray<ncm::StorageContentMetaKey>(outList, count), offset, id);
    }

    Result CommitLocalCommunicationReceiveApplicationTask(const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitLocalCommunicationReceiveApplicationTask(id);
    }

    Result GetLocalCommunicationReceiveApplicationTaskErrorContext(err::ErrorContext* outValue, const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationReceiveApplicationTaskErrorContext(outValue, id);
    }

    Result CalculateLocalCommunicationReceiveApplicationTaskRequiredSize(int64_t* outValue, const LocalCommunicationReceiveApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CalculateLocalCommunicationReceiveApplicationTaskRequiredSize(outValue, id);
    }

    int ListApplicationLocalCommunicationReceiveApplicationTask(LocalCommunicationReceiveApplicationTaskId outList[], int listCount, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListApplicationLocalCommunicationReceiveApplicationTask(&outCount, sf::OutArray<LocalCommunicationReceiveApplicationTaskId>(outList, listCount), appId));
        return outCount;
    }

    Result CreateLocalCommunicationSendApplicationTask(LocalCommunicationSendApplicationTaskId* outValue, uint32_t ipv4, uint16_t port, const ncm::StorageContentMetaKey keyList[], int count, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return g_Interface->CreateLocalCommunicationSendApplicationTask(outValue, ipv4, port, sf::InArray<ncm::StorageContentMetaKey>(keyList, static_cast<size_t>(count)), id);
    }

    Result RequestLocalCommunicationSendApplicationTaskRun(AsyncResult* outValue, const LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestLocalCommunicationSendApplicationTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetLocalCommunicationSendApplicationTaskInfo(LocalCommunicationSendApplicationTaskInfo* outValue, const LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationSendApplicationTaskInfo(outValue, id);
    }

    Result DestroyLocalCommunicationSendApplicationTask(const LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyLocalCommunicationSendApplicationTask(id);
    }

    Result GetLocalCommunicationSendApplicationTaskErrorContext(err::ErrorContext* outValue, const LocalCommunicationSendApplicationTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationSendApplicationTaskErrorContext(outValue, id);
    }

    int ListApplicationLocalCommunicationSendApplicationTask(LocalCommunicationSendApplicationTaskId outList[], int countList, ncm::ApplicationId appId) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListApplicationLocalCommunicationSendApplicationTask(&outCount, sf::OutArray<LocalCommunicationSendApplicationTaskId>(outList, countList), appId));
        return outCount;
    }

    Result CreateLocalCommunicationReceiveSystemUpdateTask(LocalCommunicationReceiveSystemUpdateTaskId* outValue, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key, Bit32 config) NN_NOEXCEPT
    {
        return g_Interface->CreateLocalCommunicationReceiveSystemUpdateTask(outValue, ipv4, port, key, config);
    }

    Result DestroyLocalCommunicationReceiveSystemUpdateTask(const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyLocalCommunicationReceiveSystemUpdateTask(id);
    }

    int ListLocalCommunicationReceiveSystemUpdateTask(LocalCommunicationReceiveSystemUpdateTaskId outList[], int listCount) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListLocalCommunicationReceiveSystemUpdateTask(&outCount, sf::OutArray<LocalCommunicationReceiveSystemUpdateTaskId>(outList, listCount)));
        return outCount;
    }

    Result RequestLocalCommunicationReceiveSystemUpdateTaskRun(AsyncResult* outValue, const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestLocalCommunicationReceiveSystemUpdateTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetLocalCommunicationReceiveSystemUpdateTaskInfo(nim::LocalCommunicationReceiveSystemUpdateTaskInfo* outValue, const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationReceiveSystemUpdateTaskInfo(outValue, id);
    }

    Result CommitLocalCommunicationReceiveSystemUpdateTask(const LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->CommitLocalCommunicationReceiveSystemUpdateTask(id);
    }

    Result GetLocalCommunicationReceiveSystemUpdateTaskErrorContext(err::ErrorContext* outValue, const nim::LocalCommunicationReceiveSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationReceiveSystemUpdateTaskErrorContext(outValue, id);
    }

    Result GetReceivedSystemDataPath(ncm::Path* outValue, ncm::SystemDataId dataId, const nim::LocalCommunicationReceiveSystemUpdateTaskId& taskId) NN_NOEXCEPT
    {
        return g_Interface->GetReceivedSystemDataPath(outValue, dataId, taskId);
    }

    Result CreateLocalCommunicationSendSystemUpdateTask(nim::LocalCommunicationSendSystemUpdateTaskId* outValue, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        return g_Interface->CreateLocalCommunicationSendSystemUpdateTask(outValue, ipv4, port, key);
    }

    Result RequestLocalCommunicationSendSystemUpdateTaskRun(AsyncResult* outValue, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [id](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return g_Interface->RequestLocalCommunicationSendSystemUpdateTaskRun(outHandle, outResult, id);
            }
        );
    }

    Result GetLocalCommunicationSendSystemUpdateTaskInfo(nim::LocalCommunicationSendSystemUpdateTaskInfo* outValue, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationSendSystemUpdateTaskInfo(outValue, id);
    }

    Result DestroyLocalCommunicationSendSystemUpdateTask(const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->DestroyLocalCommunicationSendSystemUpdateTask(id);
    }

    Result GetLocalCommunicationSendSystemUpdateTaskErrorContext(err::ErrorContext* outValue, const nim::LocalCommunicationSendSystemUpdateTaskId& id) NN_NOEXCEPT
    {
        return g_Interface->GetLocalCommunicationSendSystemUpdateTaskErrorContext(outValue, id);
    }

    int ListLocalCommunicationSendSystemUpdateTask(nim::LocalCommunicationSendSystemUpdateTaskId outList[], int listCount) NN_NOEXCEPT
    {
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ListLocalCommunicationSendSystemUpdateTask(&outCount, sf::OutArray<LocalCommunicationSendSystemUpdateTaskId>(outList, listCount)));
        return outCount;
    }
    void ReloadErrorSimulation() NN_NOEXCEPT
    {
        g_Interface->ReloadErrorSimulation();
    }

    //!------------------------------------------------------------------------
    //! @name 柔軟な権利管理
    //! @{
    Result RequestQueryAvailableELicenses(AsyncAvailableELicenses* outValue, const account::Uid& uid, const es::RightsId* rightsIds, int rightsCount) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAvailableELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestQueryAvailableELicenses(outHandle, outAsync, uid, sf::InArray<es::RightsId>(rightsIds, rightsCount));
            }
        );
    }

    Result RequestQueryAvailableELicenses(AsyncAvailableELicenses* outValue, const es::RightsId* rightsIds, int rightsCount) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAvailableELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestQueryAvailableELicenses(outHandle, outAsync, account::InvalidUid, sf::InArray<es::RightsId>(rightsIds, rightsCount));
            }
        );
    }

    Result RequestAssignELicenses(AsyncAssignedELicenses* outValue, const account::Uid& uid, const es::RightsId* rightsIds, int rightsCount, ELicenseType elicenseType) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAssignedELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestAssignELicenses(outHandle, outAsync, uid, sf::InArray<es::RightsId>(rightsIds, rightsCount), elicenseType);
            }
        );
    }

    Result RequestAssignELicenses(AsyncAssignedELicenses* outValue, const es::RightsId* rightsIds, int rightsCount, ELicenseType elicenseType) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAssignedELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestAssignELicenses(outHandle, outAsync, account::InvalidUid, sf::InArray<es::RightsId>(rightsIds, rightsCount), elicenseType);
            }
        );
    }

    Result RequestExtendELicenses(AsyncAssignedELicenses* outValue, const account::Uid& uid, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAssignedELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestExtendELicenses(outHandle, outAsync, uid, sf::InArray<es::ELicenseId>(elicenseIds, count));
            }
        );
    }

    Result RequestSyncELicenses(AsyncResult* outValue, const account::NintendoAccountId& naId) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outAsync)->Result
            {
                return g_Interface->RequestSyncELicenses(outHandle, outAsync, naId);
            }
        );
    }

    Result RequestDownloadETickets(AsyncResult* outValue, const account::Uid& uid, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outAsync)->Result
            {
                return g_Interface->RequestDownloadETickets(outHandle, outAsync, uid, sf::InArray<es::ELicenseId>(elicenseIds, count));
            }
        );
    }

    Result RequestQueryRevokeReason(AsyncRevokeReason* outValue, const account::NintendoAccountId& naId, const es::ELicenseId& eLicenseId) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncRevokeReason, detail::IAsyncValue>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outAsync)->Result
            {
                return g_Interface->RequestQueryRevokeReason(outHandle, outAsync, naId, eLicenseId);
            }
        );
    }

    Result RequestReportActiveELicenses(AsyncResult* outValue, const account::NintendoAccountId& naId, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outAsync)->Result
            {
                return g_Interface->RequestReportActiveELicenses(outHandle, outAsync, naId, sf::InArray<es::ELicenseId>(elicenseIds, count));
            }
        );
    }

    Result RequestReportActiveELicensesPassively(AsyncResult* outValue, const account::NintendoAccountId& naId, const es::ELicenseId* elicenseIds, int count) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outAsync)->Result
            {
                return g_Interface->RequestReportActiveELicensesPassively(outHandle, outAsync, naId, sf::InArray<es::ELicenseId>(elicenseIds, count));
            }
        );
    }
    Result RequestAssignAllDeviceLinkedELicenses(AsyncAssignedELicenses* outValue) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncAssignedELicenses, detail::IAsyncData>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncData>* outAsync)->Result
            {
                return g_Interface->RequestAssignAllDeviceLinkedELicenses(outHandle, outAsync);
            }
        );
    }
    Result RequestRegisterDynamicRightsNotificationToken(AsyncResult* outValue, const npns::NotificationToken& token) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [&](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outAsync)->Result
            {
                return g_Interface->RequestRegisterDynamicRightsNotificationToken(outHandle, outAsync, token);
            }
        );
    }
    //! @}
    //!------------------------------------------------------------------------

}}
