﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/nifm_ApiScan.h>
#include <nn/nifm/nifm_ApiScanForSystem.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_TypesScan.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/detail/util/nifm_SfUtility.h>

#include <nn/nifm/detail/nifm_ScanRequestClient.h>


namespace nn
{
namespace nifm
{

namespace detail
{
    nn::Result GetGeneralServicePointer(nn::sf::SharedPointer<detail::IGeneralService>* ppGeneralService) NN_NOEXCEPT;
    nn::Result GetScanRequestClientPointer(ScanRequestClient** ppScanRequestClient) NN_NOEXCEPT;
    nn::Result GetNetworkConnectionPointer(NetworkConnection** ppNetworkConnection) NN_NOEXCEPT;
}

// For Application
nn::Result Scan() NN_NOEXCEPT
{
    NN_RESULT_THROW(Scan({}, 0, RequirementPreset_Scan));
}

nn::Result Scan(const int16_t scanChannels[], int count) NN_NOEXCEPT
{
    NN_RESULT_THROW(Scan(scanChannels, count, RequirementPreset_Scan));
}

// For System
nn::Result Scan(RequirementPreset preset) NN_NOEXCEPT
{
    NN_RESULT_THROW(Scan({}, 0, preset));
}

nn::Result Scan(const int16_t scanChannels[], int count, RequirementPreset preset) NN_NOEXCEPT
{
    detail::ScanRequestClient* pScanRequestClient;
    NN_RESULT_DO(detail::GetScanRequestClientPointer(&pScanRequestClient));

    NetworkConnection* pNetworkConnection = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        if (pNetworkConnection != nullptr)
        {
            pNetworkConnection->CancelRequest();
        }
    };

    if (!pScanRequestClient->IsProcessing())
    {
        // インターネット通信接続状態以外の場合、切断状態に遷移してそれを維持
        InternetConnectionStatus internetConnectionStatus;
        if (GetInternetConnectionStatus(&internetConnectionStatus).IsFailure())
        {
            NN_RESULT_DO(detail::GetNetworkConnectionPointer(&pNetworkConnection));
            pNetworkConnection->CancelRequest();
            NN_RESULT_DO(SetRequestRequirementPreset(pNetworkConnection->GetRequestHandle(), preset));
            pNetworkConnection->SubmitRequestAndWait();
            NN_RESULT_DO(pNetworkConnection->GetResult());
        }

        NN_RESULT_DO(pScanRequestClient->SetChannels(scanChannels, count));
        pScanRequestClient->Submit();
    }

    while (pScanRequestClient->IsProcessing())
    {
        // 複数のクライアントが Scan() を呼んでもデッドロックしないように時限待ち
        // 待ち時間に深い意味はないが、スキャンにかかる見込みの時間と同じくらいで
        if (pScanRequestClient->GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)))
        {
            break;
        }
    }

    NN_RESULT_THROW(pScanRequestClient->GetResult());
}

nn::Result GetAllowedChannels(int16_t* pOutChannels, int* pOutCount, int bufferCount) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    NN_RESULT_THROW(pGeneralService->GetAllowedChannels(nn::sf::OutArray<int16_t>(pOutChannels, bufferCount), pOutCount));
}

nn::Result GetScanData(AccessPointData* pOutAccessPointData, int* pOutCount, int bufferCount) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(AccessPointData) >= sizeof(detail::sf::AccessPointData));
    NN_SDK_REQUIRES_NOT_NULL(pOutAccessPointData);

    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;

    NN_RESULT_DO(detail::GetGeneralServicePointer(&pGeneralService));

    int scanCount = 0;
    auto pOutMemory = reinterpret_cast<detail::sf::AccessPointData*>(pOutAccessPointData);

    NN_RESULT_DO(pGeneralService->GetScanData(nn::sf::OutArray<detail::sf::AccessPointData>(pOutMemory, bufferCount), &scanCount));

    *pOutCount = scanCount;
    int copyCount = std::min<int>(bufferCount, scanCount);

    for (int i = copyCount - 1; i >= 0; --i)
    {
        detail::sf::AccessPointData accessPointDataTemp = pOutMemory[i];
        detail::ConvertAccessPointDataToNifmFromSf(&pOutAccessPointData[i], accessPointDataTemp);
    }

    NN_RESULT_SUCCESS;
}

}
}
