﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ApiScan.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <nn/nifm/detail/nifm_RequestClient.h>
#include <nn/nifm/detail/nifm_ScanRequestClient.h>
#include <nn/nifm/detail/nifm_TemporaryNetworkProfileClient.h>
#include <nn/nifm/detail/nifm_TypesServiceObjectList.h>

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/detail/service/nifm_ServiceProviderClient.h>
#include <nn/nifm/detail/util/nifm_SfUtility.h>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/result/result_HandlingUtility.h>

#include <new>

namespace nn
{
namespace nifm
{

namespace
{
    bool g_IsInitialized = false;
    // 権限の強さで昇順
    enum class ClientMode
    {
        None,
        User,
        System,
        Admin,

        Count
    } g_ClientMode = ClientMode::None;
    nn::os::SdkRecursiveMutexType g_MutexInitialize = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();

    nn::sf::SharedPointer<detail::IGeneralService> g_pGeneralService;

    nn::util::TypedStorage<detail::ServiceProviderClient, sizeof(detail::ServiceProviderClient), NN_ALIGNOF(detail::ServiceProviderClient)> g_ServiceProviderClientStorage;
    detail::ServiceProviderClient* g_pServiceProviderClient = nullptr;

    nn::util::TypedStorage<NetworkConnection, sizeof(NetworkConnection), NN_ALIGNOF(NetworkConnection)> g_NetworkConnectionStorage;
    NetworkConnection* g_pNetworkConnection = nullptr;

    nn::util::TypedStorage<detail::ScanRequestClient, sizeof(detail::ScanRequestClient), NN_ALIGNOF(detail::ScanRequestClient)> g_ScanRequestClientStorage;
    detail::ScanRequestClient* g_pScanRequestClient = nullptr;

    bool IsInitialized() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_MutexInitialize);
        return g_IsInitialized;
    }

    Result InitializeImpl(ClientMode mode) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_MutexInitialize);

        if (IsInitialized())
        {
            // Windows では複数のクライアントモジュールが一つのプロセスにまとまるので、権限を区別しない
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            // Admin 権限は User 権限を包含し、一つのクライアントには一方しか付与されない前提
            NN_RESULT_THROW_UNLESS(g_ClientMode >= mode, ResultAlreadyInitialized());
#endif
            NN_RESULT_SUCCESS;
        }

        if (g_pServiceProviderClient == nullptr)
        {
            g_pServiceProviderClient = new(&g_ServiceProviderClientStorage) detail::ServiceProviderClient();
        }

        if (g_pGeneralService == nullptr)
        {
            NN_SDK_ASSERT(g_ClientMode == ClientMode::None);
            switch (mode)
            {
            case ClientMode::User:
                NN_RESULT_DO(g_pServiceProviderClient->GetUserServiceSharedPointer(&g_pGeneralService));
                break;
            case ClientMode::Admin:
                NN_RESULT_DO(g_pServiceProviderClient->GetAdminServiceSharedPointer(&g_pGeneralService));
                break;
            case ClientMode::System:
                NN_RESULT_DO(g_pServiceProviderClient->GetSystemServiceSharedPointer(&g_pGeneralService));
                break;
            default:
                NN_ABORT("[NIFM] Unknown Client mode.");
                break;
            }
        }

        g_ClientMode = mode;
        g_IsInitialized = true;

        NN_RESULT_SUCCESS;
    }

    Result FinalizeImpl(ClientMode mode) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_MutexInitialize);

        if (!IsInitialized())
        {
            NN_RESULT_SUCCESS;
        }

        // Windows では複数のクライアントモジュールが一つのプロセスにまとまるので、 User 権限と Admin 権限を区別しない
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        // Admin 権限は User 権限を包含し、一つのクライアントには一方しか付与されない前提
        if (g_ClientMode != mode)
        {
            NN_RESULT_SUCCESS;
        }
#endif
        if (g_pNetworkConnection != nullptr)
        {
            g_pNetworkConnection->~NetworkConnection();
            g_pNetworkConnection = nullptr;
        }

        if (g_pScanRequestClient != nullptr)
        {
            g_pScanRequestClient->~ScanRequestClient();
            g_pScanRequestClient = nullptr;
        }

        g_ClientMode = ClientMode::None;
        g_pGeneralService = nullptr;

        g_pServiceProviderClient->Finalize();

        g_IsInitialized = false;

        NN_RESULT_SUCCESS;
    }
}

namespace detail
{
    // nifm_Api*.h で呼ぶためのユーティリティ

    nn::Result GetGeneralServicePointer(nn::sf::SharedPointer<detail::IGeneralService>* ppGeneralService) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsInitialized());
        NN_RESULT_THROW_UNLESS(IsInitialized(), ResultNotInitialized());

        *ppGeneralService = g_pGeneralService;

        NN_RESULT_SUCCESS;
    }

    nn::Result GetNetworkConnectionPointer(NetworkConnection** ppNetworkConnection) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsInitialized());
        NN_RESULT_THROW_UNLESS(IsInitialized(), ResultNotInitialized());

        if (g_pNetworkConnection == nullptr)
        {
            g_pNetworkConnection = new(&g_NetworkConnectionStorage)NetworkConnection();
        }

        *ppNetworkConnection = g_pNetworkConnection;

        NN_RESULT_SUCCESS;
    }

    nn::Result GetScanRequestClientPointer(ScanRequestClient** ppScanRequestClient) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsInitialized());
        NN_RESULT_THROW_UNLESS(IsInitialized(), ResultNotInitialized());

        if (g_pScanRequestClient == nullptr)
        {
            g_pScanRequestClient = new(&g_ScanRequestClientStorage)ScanRequestClient(nn::os::EventClearMode_AutoClear);
        }

        *ppScanRequestClient = g_pScanRequestClient;

        NN_RESULT_SUCCESS;
    }
}

nn::Result Initialize() NN_NOEXCEPT
{
    return InitializeImpl(ClientMode::User);
}

nn::Result InitializeAdmin() NN_NOEXCEPT
{
    return InitializeImpl(ClientMode::Admin);
}

nn::Result InitializeSystem() NN_NOEXCEPT
{
    return InitializeImpl(ClientMode::System);
}

// For Test
nn::Result FinalizeForTest() NN_NOEXCEPT
{
    return FinalizeImpl(ClientMode::User);
}

nn::Result FinalizeAdminForTest() NN_NOEXCEPT
{
    return FinalizeImpl(ClientMode::Admin);
}

nn::Result FinalizeSystemForTest() NN_NOEXCEPT
{
    return FinalizeImpl(ClientMode::System);
}

}
}

