﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_SsidListUtility.h>

#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemData.h>

#include <nn/util/util_Base64.h>
#include <nn/util/util_StringUtil.h>

#include <nn/result/result_HandlingUtility.h>

#include <new>
#include <cstdio>

namespace nn
{
namespace nifm
{
namespace detail
{

/**
 * 単一のデリミタを取る，再入不可な Tokenizer
 * 連続するデリミタが表れても空文字列として処理する
 */
char* Strtokc(char *pBuffer, char delimiter) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(char*, s_Str, =nullptr);

    if (pBuffer)
    {
        s_Str = pBuffer;
    }
    else
    {
        pBuffer = s_Str;
    }

    if (!pBuffer)
    {
        return nullptr;
    }

    do
    {
        if (*s_Str == delimiter) // デリミタ
        {
            *s_Str++ = '\0';
            return pBuffer;
        }
    } while (*++s_Str);

    s_Str = nullptr;
    return pBuffer;
}

void RemoveChars(char* str, char const* cs) NN_NOEXCEPT
{
    while (*cs)
    {
        char *pr = str, *pw = str;
        while (*pr)
        {
            *pw = *pr++;
            pw += (*pw != *cs);
        }
        ++cs;
        *pw = '\0';
    }
}

namespace
{

// SSID LIST における定義
enum SsidListAuthentication : uint8_t
{
    SsidListAuthentication_Open    = 0,
    SsidListAuthentication_Shared  = 1,
    SsidListAuthentication_WpaPsk  = 2,
    SsidListAuthentication_Wpa2Psk = 3,
};

enum SsidListEncryption : uint8_t
{
    SsidListEncryption_None = 0,
    SsidListEncryption_Wep  = 1,
    SsidListEncryption_Aes  = 2,
};


Authentication ConvertAuthentication(const char* pAuthString) NN_NOEXCEPT
{
    if (std::strncmp(pAuthString, "0", sizeof("0")) == 0)
    {
        return Authentication_Open;
    }
    else if (std::strncmp(pAuthString, "1", sizeof("1")) == 0)
    {
        return Authentication_Shared;
    }
    else if (std::strncmp(pAuthString, "2", sizeof("2")) == 0)
    {
        return Authentication_WpaPsk;
    }
    else if (std::strncmp(pAuthString, "3", sizeof("3")) == 0)
    {
        return Authentication_Wpa2Psk;
    }

    return Authentication_Invalid; // 不正なデータ
}

Encryption ConvertEncryption(const char* pEncryptionString) NN_NOEXCEPT
{
    if (std::strncmp(pEncryptionString, "0", sizeof("0")) == 0)
    {
        return Encryption_None;
    }
    else if (std::strncmp(pEncryptionString, "1", sizeof("1")) == 0)
    {
        return Encryption_Wep;
    }
    else if (std::strncmp(pEncryptionString, "2", sizeof("2")) == 0)
    {
        return Encryption_Aes;
    }

    return Encryption_Invalid; // 不正なデータ
}

}


SsidListAccessor::SsidListAccessor() NN_NOEXCEPT
    : m_pCacheBuffer(nullptr),
      m_pFileBuffer(nullptr),
      m_pCurrentPos(nullptr)
{
}

SsidListAccessor::~SsidListAccessor() NN_NOEXCEPT
{
    if (m_pCacheBuffer != nullptr)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        delete[] m_pCacheBuffer;
#endif
        m_pCacheBuffer = nullptr;
    }

    if (m_pFileBuffer != nullptr)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        delete[] m_pFileBuffer;
#endif
        m_pFileBuffer = nullptr;
    }
}

Result SsidListAccessor::Load() NN_NOEXCEPT
{
    return Load(nullptr);
}

Result SsidListAccessor::Load(const char* pInString) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pCacheBuffer == nullptr);
    NN_SDK_ASSERT(m_pFileBuffer == nullptr);

    if (pInString == nullptr)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::ncm::SystemDataId id = { 0x0100000000000807 };
        size_t cacheSize = 0;

        NN_RESULT_DO(nn::fs::QueryMountSystemDataCacheSize(&cacheSize, id));

        m_pCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_RESULT_THROW_UNLESS(m_pCacheBuffer != nullptr, ResultOutOfMemory());

        NN_RESULT_DO(nn::fs::MountSystemData("systemData", id, m_pCacheBuffer, cacheSize));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("systemData");
        };

        // nn::fs::ResultPathNotFound 以外の原因で失敗した場合はライブラリ内でアボート
        nn::fs::FileHandle fileHandle;
        // nn::fs::ResultPathNotFound の場合は return
        NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, "systemData:/ssid_list.csv", nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, fileHandle));

        m_pFileBuffer = new(std::nothrow) char[fileSize + 1];
        NN_RESULT_THROW_UNLESS(m_pFileBuffer != nullptr, ResultOutOfMemory());

        size_t readSize = 0;

        // 読み込み失敗時はライブラリ内でアボートするため、エラーハンドリングは不要
        (void)nn::fs::ReadFile(&readSize, fileHandle, 0, m_pFileBuffer, fileSize);
        m_pFileBuffer[fileSize] = '\0';
#else
        NN_RESULT_THROW(ResultNotImplemented());
#endif
    }
    else
    {
        // Windows または Debug 用に外部から直接 SSID リストを渡された場合
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        int size = nn::util::Strnlen(pInString, MaxSsidListSize);
        auto ssidListBuffer = new(std::nothrow) char[size];
        NN_RESULT_THROW_UNLESS(ssidListBuffer != nullptr, ResultOutOfMemory());
#else
        static char ssidListBuffer[MaxSsidListSize]; // global new を使用しない
#endif
        nn::util::Strlcpy(ssidListBuffer, pInString, MaxSsidListSize);
        m_pFileBuffer = ssidListBuffer;
    }

    m_pCurrentPos = m_pFileBuffer;

    NN_RESULT_SUCCESS;
}

Result SsidListAccessor::GetVersion(char* pOutVersion) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutVersion);

    nn::util::Strlcpy(pOutVersion, m_pFileBuffer, sizeof("YYMMDD"));

    NN_RESULT_SUCCESS;
}

Result SsidListAccessor::GetNextEntry(
    nn::util::Uuid *pOutUuid, Ssid* pOutSsid, SharedKey* pOutPassphrase,
    Authentication *pOutAuthentication, Encryption *pOutEncryption, char* pOutName, bool *pOutIsAutoConnect) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutUuid);
    NN_SDK_ASSERT_NOT_NULL(pOutSsid);
    NN_SDK_ASSERT_NOT_NULL(pOutPassphrase);
    NN_SDK_ASSERT_NOT_NULL(pOutAuthentication);
    NN_SDK_ASSERT_NOT_NULL(pOutEncryption);
    NN_SDK_ASSERT_NOT_NULL(pOutName);
    NN_SDK_ASSERT_NOT_NULL(pOutIsAutoConnect);

    if (m_pCurrentPos == m_pFileBuffer)
    {
        (void)strtok(m_pFileBuffer, "\n\r"); // 1行目は読み飛ばし
        m_pCurrentPos = strtok(nullptr, "\n\r");
    }
    else
    {
        m_pCurrentPos = strtok(nullptr, "\n\r");
    }

    NN_RESULT_THROW_UNLESS(m_pCurrentPos != nullptr, ResultEndOfStream());

    RemoveChars(m_pCurrentPos, " \t\"");

    char* pItemPos = nullptr;

    //uuid
    pItemPos = Strtokc(m_pCurrentPos, ',');
    pOutUuid->FromString(pItemPos);

    size_t num;
    nn::util::Base64::Status status;

    //ssid
    pItemPos = Strtokc(nullptr, ',');
    char decodedSsid[Ssid::HexSize + 1] = {};
    status = nn::util::Base64::FromBase64String(&num, decodedSsid, Ssid::HexSize + 1, pItemPos, nn::util::Base64::Mode_NormalNoLinefeed);
    NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultInternalError());
    std::memcpy(pOutSsid->hex, decodedSsid, num);
    pOutSsid->length = static_cast<uint8_t>(num);

    //passphrase
    pItemPos = Strtokc(nullptr, ',');
    status = nn::util::Base64::FromBase64String(&num, pOutPassphrase->keyMaterial, SharedKey::KeyMaterialSize, pItemPos, nn::util::Base64::Mode_NormalNoLinefeed);
    NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultInternalError());
    pOutPassphrase->length = static_cast<uint8_t>(num);

    //authentication
    pItemPos = Strtokc(nullptr, ',');
    *pOutAuthentication = ConvertAuthentication(pItemPos);

    //encryption
    pItemPos = Strtokc(nullptr, ',');
    *pOutEncryption = ConvertEncryption(pItemPos);

    //name
    pItemPos = Strtokc(nullptr, ',');
    status = nn::util::Base64::FromBase64String(&num, pOutName, NetworkProfileBasicInfo::NameSize, pItemPos, nn::util::Base64::Mode_NormalNoLinefeed);
    NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, ResultInternalError());
    pOutName[num] = '\0';

    //isAutoConnect
    pItemPos = Strtokc(nullptr, ',');
    *pOutIsAutoConnect = std::strncmp(pItemPos, "0", 1) == 0 ? false : true;

    NN_RESULT_SUCCESS;
}

}
}
}
