﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_SocketHolder.h>

#include <cerrno>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/bsdsocket/cfg/cfg.h>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{

nn::Result ConvertErrorNoToResult(nn::socket::Errno errorNo) NN_NOEXCEPT
{
    switch (errorNo)
    {
    case nn::socket::Errno::EBadf:
        NN_RESULT_THROW(ResultSocketHolderInvalidDescriptor());
    case nn::socket::Errno::EAgain:
        NN_RESULT_THROW(ResultSocketHolderTemporarilyUnavailable());
    case nn::socket::Errno::EFault:
        NN_RESULT_THROW(ResultSocketInvalidAddress());
    case nn::socket::Errno::EInval:
        NN_RESULT_THROW(ResultSocketHolderInvalidArgument());
    case nn::socket::Errno::ENotSock:
        NN_RESULT_THROW(ResultSocketHolderNotSocket());
    case nn::socket::Errno::ENoProtoOpt:
        NN_RESULT_THROW(ResultSocketHolderNoSuchOption());
    case nn::socket::Errno::EConnReset:
        NN_RESULT_THROW(ResultSocketHolderConnectionAlreadyReset());
    case nn::socket::Errno::ENoBufs:
        NN_RESULT_THROW(ResultSocketHolderNoBuffer());
    default:
        NN_RESULT_THROW(ResultSocketHolderError());
    }
}

}

SocketHolder::SocketHolder(int descriptor, nn::Bit64 processId) NN_NOEXCEPT
    : m_Descriptor(descriptor),
    m_ProcessId(processId),
    m_IsCloseRequired(false)
{
}

SocketHolder::~SocketHolder() NN_NOEXCEPT
{
    if (m_IsCloseRequired)
    {
        NN_SDK_ASSERT(m_Descriptor >= 0);
        nn::socket::Close(m_Descriptor);
    }
}

SocketHolder::SocketHolder(const SocketHolder& socketHolder) NN_NOEXCEPT
    : m_Descriptor(nn::socket::DuplicateSocket(socketHolder.m_Descriptor, socketHolder.m_ProcessId)),
      m_ProcessId(0),
      m_IsCloseRequired(m_Descriptor >= 0)
{
}

SocketHolder::SocketHolder(SocketHolder&& socketHolder) NN_NOEXCEPT
    : m_Descriptor(socketHolder.m_Descriptor),
      m_ProcessId(socketHolder.m_ProcessId),
      m_IsCloseRequired(socketHolder.m_IsCloseRequired)
{
    socketHolder.m_Descriptor = -1;
    socketHolder.m_ProcessId = 0;
    socketHolder.m_IsCloseRequired = false;
}

Result SocketHolder::GetDestination(nn::socket::SockAddrIn* pOutSockAddrIn) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSockAddrIn);

    NN_RESULT_THROW_UNLESS(m_Descriptor >= 0, ResultSocketHolderInvalidDescriptor());

    nn::socket::SockLenT sockLen = sizeof(*pOutSockAddrIn);

    if (nn::socket::GetPeerName(m_Descriptor, reinterpret_cast<nn::socket::SockAddr*>(pOutSockAddrIn), &sockLen) < 0)
    {
        nn::socket::Errno errorNo = nn::socket::GetLastError();

        NN_DETAIL_NIFM_WARN(">GetPeerName failed (error %d)\n", errorNo);
        NN_UNUSED(errorNo);

        NN_RESULT_THROW(ConvertErrorNoToResult(errorNo));
    }

    NN_RESULT_SUCCESS;
}

Result SocketHolder::GetSource(nn::socket::SockAddrIn* pOutSockAddrIn) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSockAddrIn);

    NN_RESULT_THROW_UNLESS(m_Descriptor >= 0, ResultSocketHolderInvalidDescriptor());

    nn::socket::SockLenT sockLen = sizeof(*pOutSockAddrIn);

    if (nn::socket::GetSockName(m_Descriptor, reinterpret_cast<nn::socket::SockAddr*>(pOutSockAddrIn), &sockLen) < 0)
    {
        nn::socket::Errno errorNo = nn::socket::GetLastError();

        NN_DETAIL_NIFM_WARN(">GetSockName failed (error %d)\n", errorNo);
        NN_UNUSED(errorNo);

        NN_RESULT_THROW(ConvertErrorNoToResult(errorNo));
    }

    NN_RESULT_SUCCESS;
}

Result SocketHolder::GetTcpInfo(nn::socket::TcpInfo* pOutTcpInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutTcpInfo);

    NN_RESULT_THROW_UNLESS(m_Descriptor >= 0, ResultSocketHolderInvalidDescriptor());

    nn::socket::SockLenT sockLen = sizeof(*pOutTcpInfo);

    if (nn::socket::GetSockOpt(m_Descriptor, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info, reinterpret_cast<void*>(pOutTcpInfo), &sockLen) < 0)
    {
        nn::socket::Errno errorNo = nn::socket::GetLastError();

        NN_DETAIL_NIFM_WARN(">GetSockOpt failed (error %d)\n", errorNo);
        NN_UNUSED(errorNo);

        NN_RESULT_THROW(ConvertErrorNoToResult(errorNo));
    }

    NN_RESULT_SUCCESS;
}

nn::Result SocketHolder::LookUpArpEntry(MacAddress* pOutMacAddress, nn::socket::InAddr inAddr) NN_NOEXCEPT
{
    nn::Result result = nn::bsdsocket::cfg::LookupArpEntry(pOutMacAddress->data, MacAddress::Size, inAddr);
    if (result.IsFailure())
    {
        NN_DETAIL_NIFM_WARN(">LookupArpEntry failed (result 0x%08x)\n", result.GetInnerValueForDebug());

        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

}
}
}
