﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_DebugUtility.h>

#include <nn/nifm/detail/config/nifm_ConfigDetail.h>

#if defined(NN_DETAIL_NIFM_LOG_ENABLED) && (defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP))

#include <nn/nifm/nifm_TypesAccessPoint.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/detail/nifm_TypesDetail.h>
#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/nifm_RequestManager.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileManager.h>
#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>
#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/wlan/wlan_Types.h>


namespace nn
{
namespace nifm
{
namespace detail
{

void Dump(const RequestManager& requestManager) NN_NOEXCEPT
{
    requestManager.Dump();
}

void Dump(const AccessPointData& accessPoint) NN_NOEXCEPT
{
    const nn::nifm::Ssid& ssid = accessPoint.ssid;
    bool hasNonAscii = false;

    for (int i = 0; i < ssid.length; ++i)
    {
        if (ssid.hex[i] < 0x20 || 0x7e < ssid.hex[i])
        {
            hasNonAscii = true;
            break;
        }
    }
    NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
    NN_DETAIL_NIFM_TRACE_V2("  SSID(%d) : ", ssid.length);
    if (hasNonAscii)
    {
        for (int i = 0; i < ssid.length; ++i)
        {
            NN_DETAIL_NIFM_TRACE_V2("%02x", ssid.hex[i]);
        }
    }
    else
    {
        for (int i = 0; i < ssid.length; ++i)
        {
            NN_DETAIL_NIFM_TRACE_V2("%c", ssid.hex[i]);
        }
    }
    NN_DETAIL_NIFM_TRACE_V2("\n");

    switch (accessPoint.authentication)
    {
    case nn::nifm::Authentication_Open:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : Open\n");
        break;
    case nn::nifm::Authentication_Shared:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : Shared\n");
        break;
    case nn::nifm::Authentication_Unknown:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : Unknown\n");
        break;
    case nn::nifm::Authentication_WpaPsk:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : WPA-PSK\n");
        break;
    case nn::nifm::Authentication_Wpa2Psk:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : WPA2-PSK\n");
        break;
    case nn::nifm::Authentication_Wpa:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : WPA-EAP\n");
        break;
    case nn::nifm::Authentication_Wpa2:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : WPA2-EAP\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V2("  Authentication : Unsupported(%d)\n", accessPoint.authentication);
        break;
    }

    switch (accessPoint.encryption)
    {
    case nn::nifm::Encryption_None:
        NN_DETAIL_NIFM_TRACE_V2("  Encryption : None\n");
        break;
    case nn::nifm::Encryption_Wep:
        NN_DETAIL_NIFM_TRACE_V2("  Encryption : WEP\n");
        break;
    case nn::nifm::Encryption_Tkip:
        NN_DETAIL_NIFM_TRACE_V2("  Encryption : TKIP\n");
        break;
    case nn::nifm::Encryption_Aes:
        NN_DETAIL_NIFM_TRACE_V2("  Encryption : AES(CCMP)\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V2("  Encryption : Unsupported(%d)\n", accessPoint.encryption);
        break;
    }

    switch (accessPoint.groupEncryption)
    {
    case nn::nifm::Encryption_None:
        NN_DETAIL_NIFM_TRACE_V2("  groupEncryption : None\n");
        break;
    case nn::nifm::Encryption_Wep:
        NN_DETAIL_NIFM_TRACE_V2("  groupEncryption : WEP\n");
        break;
    case nn::nifm::Encryption_Tkip:
        NN_DETAIL_NIFM_TRACE_V2("  groupEncryption : TKIP\n");
        break;
    case nn::nifm::Encryption_Aes:
        NN_DETAIL_NIFM_TRACE_V2("  groupEncryption : AES(CCMP)\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V2("  groupEncryption : Unsupported(%d)\n", accessPoint.groupEncryption);
        break;
    }


    NN_DETAIL_NIFM_TRACE_V2("  BSSID : ");
    for (int i = 0; i < accessPoint.bssid.Size; ++i)
    {
        NN_DETAIL_NIFM_TRACE_V2("%02x", accessPoint.bssid.data[i]);
    }
    NN_DETAIL_NIFM_TRACE_V2("\n");

    NN_DETAIL_NIFM_TRACE_V2("  Channel : %d\n", accessPoint.channel);
    NN_DETAIL_NIFM_TRACE_V2("  RSSI : %d\n", accessPoint.rssi);
    NN_DETAIL_NIFM_TRACE_V2("  Link-Level : %d\n", accessPoint.linkLevel);
    NN_DETAIL_NIFM_TRACE_V2("  isSupported: %s\n", accessPoint.isSupported ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
} // NOLINT(impl/function_size)

void Dump(const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    const nn::nifm::Ssid& ssid = networkProfile.wirelessSetting.ssidConfig.ssid;
    bool hasNonAscii = false;

    for (int i = 0; i < ssid.length; ++i)
    {
        if (ssid.hex[i] < 0x20 || 0x7e < ssid.hex[i])
        {
            hasNonAscii = true;
            break;
        }
    }

    char idStr[nn::util::Uuid::StringSize];
    NN_UNUSED(idStr);

    NN_DETAIL_NIFM_TRACE_V2("------------- NetworkProfileData ----------\n");
    NN_DETAIL_NIFM_TRACE_V2("  Id : %s\n", networkProfile.id.ToString(idStr, nn::util::Uuid::StringSize));
    NN_DETAIL_NIFM_TRACE_V2("  Name : %s\n", networkProfile.name);
    switch (networkProfile.networkProfileType)
    {
    case NetworkProfileType_User:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkProfileType : User\n");
        break;
    case NetworkProfileType_SsidList:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkProfileType : SsidList\n");
        break;
    case NetworkProfileType_Temporary:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkProfileType : Temporary\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkProfileType : Unknown\n");
        break;
    }
    switch (networkProfile.networkInterfaceType)
    {
    case NetworkInterfaceType_Ethernet:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkInterfaceType : Ethernet\n");
        break;
    case NetworkInterfaceType_Ieee80211:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkInterfaceType : Ieee802.11\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V2("  NetworkInterfaceType : Unknown\n");
        break;
    }
    NN_DETAIL_NIFM_TRACE_V2("  isAutoConnect : %s\n", networkProfile.isAutoConnect ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("  isLargeCapacity : %s\n", networkProfile.isLargeCapacity ? "true" : "false");

    // 無線の場合
    if (networkProfile.networkInterfaceType == NetworkInterfaceType_Ieee80211)
    {
        NN_DETAIL_NIFM_TRACE_V2("  wirelsesSetting\n");
        NN_DETAIL_NIFM_TRACE_V2("\tSsid : ");
        if (hasNonAscii)
        {
            for (int i = 0; i < ssid.length; ++i)
            {
                NN_DETAIL_NIFM_TRACE_V2("%02x", ssid.hex[i]);
            }
        }
        else
        {
            for (int i = 0; i < ssid.length; ++i)
            {
                NN_DETAIL_NIFM_TRACE_V2("%c", ssid.hex[i]);
            }
        }
        NN_DETAIL_NIFM_TRACE_V2("\n");

        switch (networkProfile.wirelessSetting.security.authEncryption.authentication)
        {
        case Authentication_Open:
            NN_DETAIL_NIFM_TRACE_V2("\tAuthentication : Open\n");
            break;
        case Authentication_Shared:
            NN_DETAIL_NIFM_TRACE_V2("\tAuthentication : Shared\n");
            break;
        case Authentication_WpaPsk:
            NN_DETAIL_NIFM_TRACE_V2("\tAuthentication : WpaPsk\n");
            break;
        case Authentication_Wpa2Psk:
            NN_DETAIL_NIFM_TRACE_V2("\tAuthentication : Wpa2Psk\n");
            break;
        default:
            NN_DETAIL_NIFM_TRACE_V2("\tAuthentication : Unknown\n");
            break;
        }
        switch (networkProfile.wirelessSetting.security.authEncryption.encryption)
        {
        case Encryption_None:
            NN_DETAIL_NIFM_TRACE_V2("\tEncryption : None\n");
            break;
        case Encryption_Wep:
            NN_DETAIL_NIFM_TRACE_V2("\tEncryption : Wep\n");
            break;
        case Encryption_Tkip:
            NN_DETAIL_NIFM_TRACE_V2("\tEncryption : Tkip\n");
            break;
        case Encryption_Aes:
            NN_DETAIL_NIFM_TRACE_V2("\tEncryption : Aes\n");
            break;
        default:
            NN_DETAIL_NIFM_TRACE_V2("\tEncryption : Unknown\n");
            break;
        }
        NN_DETAIL_NIFM_TRACE_V2("\tPass : ");
#if defined(NN_DETAIL_NIFM_LOG_SECURITY_INFO_ENABLED)
        for (int i = 0; i < networkProfile.wirelessSetting.security.sharedKey.length; ++i)
        {
            NN_DETAIL_NIFM_TRACE_V2("%c", networkProfile.wirelessSetting.security.sharedKey.keyMaterial[i]);
        }
        NN_DETAIL_NIFM_TRACE_V2(" (%d)\n", networkProfile.wirelessSetting.security.sharedKey.length);
#else
        NN_DETAIL_NIFM_TRACE_V2("XXXXXXXXXX\n");
#endif
    }
    NN_DETAIL_NIFM_TRACE_V2("  ipSetting\n");
    NN_DETAIL_NIFM_TRACE_V2("\tip.isAuto : %s\n", networkProfile.ipSetting.ip.isAuto ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("\tip.ipAddress : %d.%d.%d.%d\n",
        networkProfile.ipSetting.ip.ipAddress.data[0],
        networkProfile.ipSetting.ip.ipAddress.data[1],
        networkProfile.ipSetting.ip.ipAddress.data[2],
        networkProfile.ipSetting.ip.ipAddress.data[3]);
    NN_DETAIL_NIFM_TRACE_V2("\tip.subnetMask : %d.%d.%d.%d\n",
        networkProfile.ipSetting.ip.subnetMask.data[0],
        networkProfile.ipSetting.ip.subnetMask.data[1],
        networkProfile.ipSetting.ip.subnetMask.data[2],
        networkProfile.ipSetting.ip.subnetMask.data[3]);
    NN_DETAIL_NIFM_TRACE_V2("\tip.defaultGateway : %d.%d.%d.%d\n",
        networkProfile.ipSetting.ip.defaultGateway.data[0],
        networkProfile.ipSetting.ip.defaultGateway.data[1],
        networkProfile.ipSetting.ip.defaultGateway.data[2],
        networkProfile.ipSetting.ip.defaultGateway.data[3]);
    NN_DETAIL_NIFM_TRACE_V2("\tdns.isAuto : %s\n", networkProfile.ipSetting.dns.isAuto ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("\tdns.preferredDns : %d.%d.%d.%d\n",
        networkProfile.ipSetting.dns.preferredDns.data[0],
        networkProfile.ipSetting.dns.preferredDns.data[1],
        networkProfile.ipSetting.dns.preferredDns.data[2],
        networkProfile.ipSetting.dns.preferredDns.data[3]);
    NN_DETAIL_NIFM_TRACE_V2("\tdns.alternateDns : %d.%d.%d.%d\n",
        networkProfile.ipSetting.dns.alternateDns.data[0],
        networkProfile.ipSetting.dns.alternateDns.data[1],
        networkProfile.ipSetting.dns.alternateDns.data[2],
        networkProfile.ipSetting.dns.alternateDns.data[3]);
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.isEnabled : %s\n", networkProfile.ipSetting.proxy.isEnabled ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.port : %d\n", networkProfile.ipSetting.proxy.port);
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.proxy : %s\n", networkProfile.ipSetting.proxy.proxy);
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.auth.isEnabled : %s\n", networkProfile.ipSetting.proxy.authentication.isEnabled ? "true" : "false");
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.auth.username : %s\n", networkProfile.ipSetting.proxy.authentication.username);
    NN_DETAIL_NIFM_TRACE_V2("\tproxy.auth.pass : ");
#if defined(NN_DETAIL_NIFM_LOG_SECURITY_INFO_ENABLED)
    NN_DETAIL_NIFM_TRACE_V2("%s\n", networkProfile.ipSetting.proxy.authentication.password);
#else
    NN_DETAIL_NIFM_TRACE_V2("XXXXXXXXXX\n");
#endif
    NN_DETAIL_NIFM_TRACE_V2("-------------------------------------------\n");
} // NOLINT(impl/function_size)

void Dump(const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT
{
    char uuidString[nn::util::Uuid::StringSize];
    char networkTypeStr[16] = {};
    char connectionConfirmationOptionStr[16] = {};
    char connectionConfirmationOptionSubStr[16] = {};
    NN_UNUSED(uuidString);

    switch (aggregatedRequestType.networkType)
    {
    case NetworkType_None:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "None");
        break;
    case NetworkType_Internet:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Internet");
        break;
    case NetworkType_Local:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Local");
        break;
    case NetworkType_NeighborDetection:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "Neighbor");
        break;
    default:
        nn::util::SNPrintf(networkTypeStr, sizeof(networkTypeStr), "%d", aggregatedRequestType.networkType);
        break;
    }

    switch (aggregatedRequestType.connectionConfirmationOption)
    {
    case nn::nifm::ConnectionConfirmationOption_Invalid:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Invalid");
        break;
    case nn::nifm::ConnectionConfirmationOption_Prohibited:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Prohibited");
        break;
    case nn::nifm::ConnectionConfirmationOption_Preferred:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Preferred");
        break;
    case nn::nifm::ConnectionConfirmationOption_Required:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Required");
        break;
    case nn::nifm::ConnectionConfirmationOption_Forced:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "Forced");
        break;
    default:
        nn::util::SNPrintf(connectionConfirmationOptionStr, sizeof(connectionConfirmationOptionStr), "%d", aggregatedRequestType.connectionConfirmationOption);
        break;
    }

    switch (aggregatedRequestType.connectionConfirmationOptionSub)
    {
    case nn::nifm::ConnectionConfirmationOption_Invalid:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "Invalid");
        break;
    case nn::nifm::ConnectionConfirmationOption_Prohibited:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "Prohibited");
        break;
    case nn::nifm::ConnectionConfirmationOption_Preferred:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "Preferred");
        break;
    case nn::nifm::ConnectionConfirmationOption_Required:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "Required");
        break;
    case nn::nifm::ConnectionConfirmationOption_Forced:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "Forced");
        break;
    default:
        nn::util::SNPrintf(connectionConfirmationOptionSubStr, sizeof(connectionConfirmationOptionSubStr), "%d", aggregatedRequestType.connectionConfirmationOption);
        break;
    }

    NN_DETAIL_NIFM_INFO_V1("\n-*-*-*-*-*-*-*- AggregatedRequestType -*-*-*-*-*-*-*\n");
    NN_DETAIL_NIFM_INFO_V1("  priority : %d\n", aggregatedRequestType.priority);
    NN_DETAIL_NIFM_INFO_V1("  isGreedy : %s\n", aggregatedRequestType.isGreedy ? "true"  : "false");
    NN_DETAIL_NIFM_INFO_V1("  isSharable : %s\n", aggregatedRequestType.isSharable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("  isForeground : %s\n", aggregatedRequestType.isForeground ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("  networkType : %s\n", networkTypeStr);
    NN_DETAIL_NIFM_INFO_V1("  profileId : %s\n", aggregatedRequestType.profileId.ToString(uuidString, nn::util::Uuid::StringSize));
    NN_DETAIL_NIFM_INFO_V1("  connectionConfirmationOption : %s\n", connectionConfirmationOptionStr);
    NN_DETAIL_NIFM_INFO_V1("  connectionConfirmationOptionSub : %s\n", connectionConfirmationOptionSubStr);
    NN_DETAIL_NIFM_INFO_V1("  revision : %d\n", aggregatedRequestType.revision);
    NN_DETAIL_NIFM_INFO_V1("  isAcceptedRejectable : %s\n", aggregatedRequestType.isAcceptedRejectable ? "true" : "false");
    NN_DETAIL_NIFM_INFO_V1("  lastRevision : %d\n", aggregatedRequestType.lastRevision);
    NN_DETAIL_NIFM_INFO_V1("  restrictedRevision : %d\n", aggregatedRequestType.restrictedRevision);
    NN_DETAIL_NIFM_INFO_V1("-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*\n\n");

} // NOLINT(impl/function_size)

void Dump(const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT
{
    char uuidString[nn::util::Uuid::StringSize];
    NN_UNUSED(uuidString);

    NN_DETAIL_NIFM_INFO_V1("\n=*=*=*=*=*=*=* TotalNetworkResourceInfo =*=*=*=*=*=*\n");
    switch (totalNetworkResourceInfo.networkType)
    {
    case NetworkType_None:
        NN_DETAIL_NIFM_INFO_V1("  networkType : None\n");
        break;
    case NetworkType_Internet:
        NN_DETAIL_NIFM_INFO_V1("  networkType : Internet\n");
        break;
    case NetworkType_Local:
        NN_DETAIL_NIFM_INFO_V1("  networkType : Local\n");
        break;
    case NetworkType_NeighborDetection:
        NN_DETAIL_NIFM_INFO_V1("  networkType : Neighbor\n");
        break;
    default:
        NN_DETAIL_NIFM_INFO_V1("  networkType : %d\n", totalNetworkResourceInfo.networkType);
        break;
    }

    switch (totalNetworkResourceInfo.networkResourceState)
    {
    case nn::nifm::detail::NetworkResourceState::Invalid:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : Invalid\n");
        break;
    case nn::nifm::detail::NetworkResourceState::Free:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : Free\n");
        break;
    case nn::nifm::detail::NetworkResourceState::ToBePrepared:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : ToBePrepared\n");
        break;
    case nn::nifm::detail::NetworkResourceState::Available:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : Available\n");
        break;
    case nn::nifm::detail::NetworkResourceState::Lost:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : Lost\n");
        break;
    default:
        NN_DETAIL_NIFM_INFO_V1("  networkResourceState : %d\n", totalNetworkResourceInfo.networkResourceState);
        break;
    }
    NN_DETAIL_NIFM_INFO_V1("  connectionResult : %d-%d\n", totalNetworkResourceInfo.connectionResult.GetModule(), totalNetworkResourceInfo.connectionResult.GetDescription());
    NN_DETAIL_NIFM_INFO_V1("  isDisconnectionBlocking : %s\n", totalNetworkResourceInfo.isDisconnectionBlocking ? "true"  : "false");
    switch (totalNetworkResourceInfo.internetAvailability)
    {
    case InternetAvailability_Invalid:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : Invalid\n");
        break;
    case InternetAvailability_LocalAreaOnly:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : LocalAreaOnly\n");
        break;
    case InternetAvailability_InProgress:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : InProgress\n");
        break;
    case InternetAvailability_NotConfirmed:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : NotConfirmed\n");
        break;
    case InternetAvailability_Confirmed:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : Confirmed\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V1("  internetAvailability : %d\n", totalNetworkResourceInfo.internetAvailability);
        break;
    }
    switch (totalNetworkResourceInfo.networkInterfaceType)
    {
    case NetworkInterfaceType_Ethernet:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : Ethernet\n");
        break;
    case NetworkInterfaceType_Ieee80211:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : Ieee802.11\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : %d\n", totalNetworkResourceInfo.networkInterfaceType);
        break;
    }
    NN_DETAIL_NIFM_INFO_V1("  networkProfileId : %s\n", totalNetworkResourceInfo.networkProfileId.ToString(uuidString, nn::util::Uuid::StringSize));
    NN_DETAIL_NIFM_INFO_V1("  requestRevision : %d\n", totalNetworkResourceInfo.requestRevision);
    NN_DETAIL_NIFM_INFO_V1("  isGreedy : %s\n", totalNetworkResourceInfo.isGreedy ? "true"  : "false");
    NN_DETAIL_NIFM_INFO_V1("  isSharable : %s\n", totalNetworkResourceInfo.isSharable ? "true"  : "false");
    NN_DETAIL_NIFM_INFO_V1("  restrictedRevision : %d\n", totalNetworkResourceInfo.restrictedRevision);
    NN_DETAIL_NIFM_INFO_V1("=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*\n\n");
} // NOLINT(impl/function_size)

void Dump(const NetworkProfileManager& networkProfileManager) NN_NOEXCEPT
{
    networkProfileManager.Dump();
}

void Dump(const AccessPointListBase& accessPointList) NN_NOEXCEPT
{
    NN_UNUSED(accessPointList);

#if defined(NN_DETAIL_NIFM_LOG_AP_ENABLED)
    for (auto&& accessPoint : accessPointList)
    {
        AccessPointData accessPointData;
        nn::Result result = accessPoint.TryExport(&accessPointData);
        if (result.IsSuccess())
        {
            if (!accessPointData.isSupported)
            {
                NN_DETAIL_NIFM_TRACE_V2("vvvv Unsupported Ap vvvv\n");
            }
            Dump(accessPointData);
        }
        else if (result <= ResultLocalConnection())
        {
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
            NN_DETAIL_NIFM_TRACE_V2("  Local Ap\n");
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
        }
        else if (result <= ResultEthernetConnection())
        {
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
            NN_DETAIL_NIFM_TRACE_V2("  Ethernet Ap\n");
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
        }
        else
        {
            // ここにくることはないはず
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
            NN_DETAIL_NIFM_TRACE_V2("  Invalid Ap\n");
            NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
        }
    }
#endif
}

void Dump(const NetworkInterfaceBase& networkInterface) NN_NOEXCEPT
{
    MacAddress macAddress;
    networkInterface.GetMacAddress(&macAddress);

    switch (networkInterface.GetNetworkInterfaceType())
    {
    case NetworkInterfaceType_Ethernet:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : Ethernet\n");
        break;
    case NetworkInterfaceType_Ieee80211:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : Ieee802.11\n");
        break;
    default:
        NN_DETAIL_NIFM_TRACE_V1("  networkInterfaceType : Unknown\n");
        break;
    }
    NN_DETAIL_NIFM_INFO_V1(
        "  MacAddress : %02x:%02x:%02x:%02x:%02x:%02x\n",
        macAddress.data[0], macAddress.data[1], macAddress.data[2],
        macAddress.data[3], macAddress.data[4], macAddress.data[5]
    );

}

void Dump(const MacAddress& macAddress) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO_V2(
        "MacAddress : %02x:%02x:%02x:%02x:%02x:%02x\n",
        macAddress.data[0], macAddress.data[1], macAddress.data[2],
        macAddress.data[3], macAddress.data[4], macAddress.data[5]
    );
}

void Dump(const NetworkType& networkType, const NetworkInterfaceType& networkInterfaceType, const InternetAvailability& internetAvailability,
    const ConnectionConfirmationOption& op, const ConnectionConfirmationOption& opSub) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_TRACE_V1("Internet Connection Confirmation:\n");
    NN_DETAIL_NIFM_TRACE_V1("  NetworkType: %d\n", networkType);
    NN_DETAIL_NIFM_TRACE_V1("  NetworkInterfaceType: %d\n", networkInterfaceType);
    NN_DETAIL_NIFM_TRACE_V1("  InternetAvailability: %d\n", internetAvailability);
    NN_DETAIL_NIFM_TRACE_V1("  ConnectionConfirmationOption: %d/%d\n", op, opSub);
}

void Dump(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& macAddress, uint16_t channel, const nn::wlan::Security& security) NN_NOEXCEPT
{
    bool hasNonAscii = false;
    int ssidLength = static_cast<int>(ssid.GetLength());
    const nn::Bit8* pSsidData = ssid.GetSsidData();

    for (int i = 0; i < ssidLength; ++i)
    {
        if (pSsidData[i] < 0x20 || 0x7e < pSsidData[i])
        {
            hasNonAscii = true;
            break;
        }
    }
    NN_DETAIL_NIFM_TRACE_V2("---------------------------------\n");
    NN_DETAIL_NIFM_TRACE_V2("  SSID(%d) : ", ssidLength);
    if (hasNonAscii)
    {
        for (int i = 0; i < ssidLength; ++i)
        {
            NN_DETAIL_NIFM_TRACE_V2("%02x", pSsidData[i]);
        }
    }
    else
    {
        for (int i = 0; i < ssidLength; ++i)
        {
            NN_DETAIL_NIFM_TRACE_V2("%c", pSsidData[i]);
        }
    }
    NN_DETAIL_NIFM_TRACE_V2("\n");

    char macAddressString[nn::wlan::MacAddress::MacStringSize];
    NN_DETAIL_NIFM_TRACE_V2("  BSSID: %s\n", macAddress.GetString(macAddressString));
    NN_DETAIL_NIFM_TRACE_V2("  Channel: %d\n", channel);
    NN_DETAIL_NIFM_TRACE_V2("  Security: %d, %d", security.privacyMode, security.groupPrivacyMode);
#if defined(NN_DETAIL_NIFM_LOG_SECURITY_INFO_ENABLED)
    NN_DETAIL_NIFM_TRACE_V2(", %s\n", security.key);
#else
    NN_DETAIL_NIFM_TRACE_V2(", XXXXXXXXXX\n");
#endif

}

#else

namespace nn
{
namespace nifm
{
namespace detail
{

void Dump(const RequestManager& requestManager) NN_NOEXCEPT { NN_UNUSED(requestManager); }
void Dump(const AccessPointData& accessPoint) NN_NOEXCEPT { NN_UNUSED(accessPoint); }
void Dump(const NetworkProfileData& networkProfile) NN_NOEXCEPT { NN_UNUSED(networkProfile); }
void Dump(const AggregatedRequestType& aggregatedRequestType) NN_NOEXCEPT { NN_UNUSED(aggregatedRequestType); }
void Dump(const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT { NN_UNUSED(totalNetworkResourceInfo); }
void Dump(const NetworkProfileManager& networkProfileManager) NN_NOEXCEPT { NN_UNUSED(networkProfileManager); }
void Dump(const AccessPointListBase& accessPointList) NN_NOEXCEPT { NN_UNUSED(accessPointList); }
void Dump(const NetworkInterfaceBase& networkInterface) NN_NOEXCEPT { NN_UNUSED(networkInterface); }
void Dump(const MacAddress& macAddress) NN_NOEXCEPT { NN_UNUSED(macAddress); }
void Dump(const NetworkType& networkType, const NetworkInterfaceType& networkInterfaceType, const InternetAvailability& internetAvailability,
    const ConnectionConfirmationOption& op, const ConnectionConfirmationOption& opSub) NN_NOEXCEPT
{
    NN_UNUSED(networkType);
    NN_UNUSED(networkInterfaceType);
    NN_UNUSED(internetAvailability);
    NN_UNUSED(op);
    NN_UNUSED(opSub);
}
void Dump(const nn::wlan::Ssid& ssid, const nn::wlan::MacAddress& macAddress, uint16_t channel, const nn::wlan::Security& security) NN_NOEXCEPT
{
    NN_UNUSED(ssid);
    NN_UNUSED(macAddress);
    NN_UNUSED(channel);
    NN_UNUSED(security);
}

#endif

}
}
}

