﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_Base64.h>

#include <nn/nn_SdkAssert.h>
#include <limits>

namespace nn
{
namespace nifm
{
namespace detail
{

    // NOTE:
    // ポインタにNULLを指定してはいけない
    // *numにdstに書き込んだバイト数がはいる
    // RFC2045 の形式に対応

    // NOTE: その他Base64エンコード方式の場合は、末尾にenum引数をつけた関数を追加して
    // エンコード方式を指定できるように対応していく。

namespace
{

const size_t DecodeTableSize = 128;

bool RFC2045EncodeImpl( size_t* pOutCount,
    char* dst, size_t sizeDst, const void* src, size_t sizeSrc,
    const char* chrTable, bool isCrLfRequired, bool isPaddingRequired) NN_NOEXCEPT
{
    const char* chr = chrTable;

    const unsigned char* p = reinterpret_cast<const unsigned char*>(src);
    const unsigned char* pEnd = p + sizeSrc;

    int column = 0;
    size_t i = 0;
    int c = 0;
    int clen = 0;

    while (p != pEnd)
    {
        c = c << 8 | *p;
        clen += 8;

        clen -= 6;
        if (isCrLfRequired && column == 76)
        {
            if (i == sizeDst) return false;
            dst[i++] = 0x0D;
            if (i == sizeDst) return false;
            dst[i++] = 0x0A;
            column = 0;
        }
        if (i == sizeDst) return false;
        dst[i++] = chr[(c >> clen) & 0x3F];
        ++column;

        if (clen >= 6)
        {
            clen -= 6;
            if (isCrLfRequired && column == 76)
            {
                if (i == sizeDst) return false;
                dst[i++] = 0x0D;
                if (i == sizeDst) return false;
                dst[i++] = 0x0A;
                column = 0;
            }
            if (i == sizeDst) return false;
            dst[i++] = chr[(c >> clen) & 0x3F];
            ++column;
        }

        ++p;
    }

    if (clen > 0)
    {
        c <<= 6 - clen;

        if (isCrLfRequired && column == 76)
        {
            if (i == sizeDst) return false;
            dst[i++] = 0x0D;
            if (i == sizeDst) return false;
            dst[i++] = 0x0A;
            column = 0;
        }
        if (i == sizeDst) return false;
        dst[i++] = chr[c & 0x3F];
        ++column;
    }

    if(isPaddingRequired)
    {
        while (column % 4 != 0)
        {
            if (i == sizeDst) return false;
            dst[i++] = chr[0x3F + 1];
            ++column;
        }
    }

    if (i == sizeDst)
    {
        return false;
    }

    *pOutCount = i;

    dst[i++] = '\0';

    return true;
}

bool RFC2045Encode(size_t* pOutCount, char* dst, size_t sizeDst, const void* src, size_t sizeSrc, Base64::Mode mode) NN_NOEXCEPT
{
    // 最後 [64] はパディング文字
    static const char ChrNormal[]     = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
    static const char ChrUrlSafe[]    = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_=";
    static const char ChrNmtokenPad[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789.-*";

    const char* chrTable = ChrNormal;
    bool isCrLfRequired = true;
    bool isPaddingRequired = true;

    switch (mode)
    {
    case Base64::Mode_Normal:
        chrTable = ChrNormal;
        isCrLfRequired = true;
        isPaddingRequired = true;
        break;
    case Base64::Mode_NormalNoLinefeed:
        chrTable = ChrNormal;
        isCrLfRequired = false;
        isPaddingRequired = true;
        break;
    case Base64::Mode_UrlSafe:
        chrTable = ChrUrlSafe;
        isCrLfRequired = false;
        isPaddingRequired = false;
        break;
    case Base64::Mode_NmtokenPad:
        chrTable = ChrNmtokenPad;
        isCrLfRequired = false;
        isPaddingRequired = true;
        break;
    default:
        NN_SDK_REQUIRES(false, "Unknown base64 mode.");
    }

    return RFC2045EncodeImpl(
        pOutCount,
        dst, sizeDst, src, sizeSrc,
        chrTable, isCrLfRequired, isPaddingRequired);
}

// パディング文字が来たとき、デコード最後に4の倍数の文字数でなかったときの端数データの処理
bool DecodeFractionData(size_t* num, uint8_t* buf, size_t sizeDst, int idx, const char* ch4) NN_NOEXCEPT
{
    switch (idx)
    {
    case 1:
        // 8bit分必要なのだが、6bitしかデータがない。
        return false;
    case 2:
        {
            if (*num + 1 > sizeDst)
                return false;

            *(buf + 0) = (ch4[0] << 2) | (ch4[1] >> 4);
            *num += 1;
        }
        break;
    case 3:
        {
            if (*num + 2 > sizeDst)
                return false;

            *(buf + 0) = (ch4[0] << 2) | (ch4[1] >> 4);
            *(buf + 1) = ((ch4[1] & 0x0F) << 4) | (ch4[2] >> 2);
            *num += 2;
        }
        break;
    default:
        // パディング文字は不要なケースだがエラーにはしない
        break;
    }
    return true;
}

bool RFC2045DecodeImpl(
    size_t* num, void* dst, size_t sizeDst, const char* src, size_t sizeSrc,
    const int8_t* table, bool isPaddingRequired) NN_NOEXCEPT
{
    // NOTE:
    // RFC2045をどの程度厳密に守らなければならないかが問題。
    //
    // この関数では、
    // 行の最大長に制限はない。
    // パディング文字が出現した場合、その後の文字は完全に無視される。その後の文字が正しいかどうかもチェックされていない。
    // LF, CR, 0x09, 0x20は読み飛ばされる。
    // それ以外の文字が出現した場合はスキップせずにfalseを返している。

    char ch4[4];
    int8_t ch;
    int8_t b;

    uint8_t* buf = reinterpret_cast<uint8_t*>(dst);
    const char* p = src;
    int idx = 0;
    size_t count = 0;
    *num = 0;


    while ((b = *p) != '\0' && count < sizeSrc)
    {
        if (b < 0)
        {
            // 指定された文字以外はエラー。
            return false;
        }

        ++p;
        ++count;
        ch = table[b];
        switch (ch)
        {
        case -1:
            // 指定された文字以外はエラー。
            return false;
        case -2:
            // 0x0A, 0x0D, 0x09, 0x20は単に読み飛ばす
            continue;
        case -3:
            // パディング文字
            return DecodeFractionData(num, buf, sizeDst, idx, ch4);
        default:
            ch4[idx++] = static_cast<char>( ch );
            if (idx == 4)
            {
                if (*num + 3 > sizeDst)
                    return false;

                *(buf + 0) = (ch4[0] << 2) | (ch4[1] >> 4);
                *(buf + 1) = ((ch4[1] & 0x0F) << 4) | (ch4[2] >> 2);
                *(buf + 2) = ((ch4[2] & 0x03) << 6) | ch4[3];

                *num += 3;
                buf += 3;
                idx = 0;
            }
            break;
        }
    }

    if(isPaddingRequired)
    {
        if (idx != 0)
            return false;
    }
    else
    {
        // パディング不要の場合、4の倍数でない端数のデータを最後に処理する
        return DecodeFractionData(num, buf, sizeDst, idx, ch4);
    }

    return true;
}

const int8_t DecodeTableNormal[DecodeTableSize] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, -2, -1, -1, -2, -1, -1, // 00-15
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 16-31
    -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63, // 32-47
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -3, -1, -1, // 48-63
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, // 64-79
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1, // 80-95
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, // 96-111
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1  // 112-127
};

// DecodeTableNormalとの相違点
//  '_'(95) を 63 のマッピングにして '/'(47) は禁止に
//  '-'(45) を 62 のマッピングにして '+'(43) は禁止に
const int8_t DecodeTableUrlSafe[DecodeTableSize] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, -2, -1, -1, -2, -1, -1, // 00-15
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 16-31
    -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, // 32-47
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -3, -1, -1, // 48-63
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, // 64-79
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, 63, // 80-95
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, // 96-111
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1  // 112-127
};

// DecodeTableNormalとの相違点
//  '.'(46) を 62 のマッピングにして '+'(43) は禁止に
//  '-'(45) を 63 のマッピングにして '/'(47) は禁止に
//  '*'(42) を padding にして '='(61) は禁止に
const int8_t DecodeTableNmtokenPad[DecodeTableSize] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, -2, -1, -1, -2, -1, -1, // 00-15
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 16-31
    -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -3, -1, -1, 63, 62, -1, // 32-47
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1, // 48-63
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, // 64-79
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1, // 80-95
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, // 96-111
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1  // 112-127
};

bool RFC2045Decode(size_t* num, void* dst, size_t sizeDst, const char* src, size_t sizeSrc, Base64::Mode mode) NN_NOEXCEPT
{
    const int8_t* table = DecodeTableNormal;
    bool isPaddingRequired = true;

    switch (mode)
    {
    case Base64::Mode_Normal:
        table = DecodeTableNormal;
        isPaddingRequired = true;
        break;
    case Base64::Mode_NormalNoLinefeed:
        table = DecodeTableNormal;
        isPaddingRequired = true;
        break;
    case Base64::Mode_UrlSafe:
        table = DecodeTableUrlSafe;
        isPaddingRequired = false;
        break;
    case Base64::Mode_NmtokenPad:
        table = DecodeTableNmtokenPad;
        isPaddingRequired = true;
        break;
    default:
        NN_SDK_REQUIRES(false, "Unknown base64 mode.");
    }

    return RFC2045DecodeImpl(
        num, dst, sizeDst, src, sizeSrc,
        table, isPaddingRequired);
}

} // namespace


/*---------------------------------------------------------------------------*/
Base64::Status
Base64::ToBase64String(size_t* pNum, char* pDst, size_t sizeDst, const void* pSrc, size_t sizeSrc, Mode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    bool isSuccess = RFC2045Encode(pNum, pDst, sizeDst, pSrc, sizeSrc, mode);
    if ( isSuccess )
    {
        return Base64::Status_Success;
    }
    else
    {
        return Base64::Status_BufferFull;
    }
}

/*---------------------------------------------------------------------------*/
Base64::Status
Base64::FromBase64String(size_t* pNum, void* pDst, size_t sizeDst, const char* pSrc, Mode mode) NN_NOEXCEPT
{
    // NULL終端で処理は終わるので、 sizeSrc は最大値を入れておく
    size_t sizeSrc = std::numeric_limits<size_t>::max();
    return FromBase64String(pNum, pDst, sizeDst, pSrc, sizeSrc, mode);
}

Base64::Status
Base64::FromBase64String(size_t* pNum, void* pDst, size_t sizeDst, const char* pSrc, size_t sizeSrc, Mode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNum);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    bool isSuccess = RFC2045Decode(pNum, pDst, sizeDst, pSrc, sizeSrc, mode);
    if ( isSuccess )
    {
        return Base64::Status_Success;
    }
    else
    {
        if ( *pNum + 3 > sizeDst )
        {
            return Base64::Status_BufferFull;
        }
        else
        {
            return Base64::Status_BadData;
        }
    }
}

/*---------------------------------------------------------------------------*/
}
}
}
