﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/service/nifm_RequestImpl.h>

#include <nn/nifm/detail/core/nifm_UserRequest.h>
#include <nn/nifm/detail/core/nifm_RequestManager.h>

#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/err/err_ErrorViewerAppletParam.h>

#include <nn/nifm/detail/core/connectionConfirmation/nifm_ConnectionTestClient.h>
#include <nn/nifm/detail/util/nifm_SettingsUtility.h>

#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/la/la_NifmToWifiWebAuthArgumentsWriter.h>
#include <nn/la/la_NifmToNetConnectArgumentsWriter.h>
#include <nn/oe/oe_LibraryAppletControlApis.h>

namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
    inline void ForwardOutBuffer(nn::sf::OutBuffer* pOutBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_LESS_EQUAL(size, pOutBuffer->GetSize());

        *pOutBuffer = pOutBuffer->MakePart(static_cast<ptrdiff_t>(size), pOutBuffer->GetSize() - size);
    }

    inline nn::Result WriteAndForwardOutBuffer(nn::sf::OutBuffer* pOutBuffer, const void* pBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);

        NN_RESULT_THROW_UNLESS(pOutBuffer->GetSize() >= size, ResultInvalidArgument());
        std::memcpy(pOutBuffer->GetPointerUnsafe(), pBuffer, size);
        ForwardOutBuffer(pOutBuffer, size);

        NN_RESULT_SUCCESS;
    }

    template<typename WRITER>
    nn::Result Export(nn::sf::OutBuffer outBuffer, uint32_t* pOutSize, const WRITER* pWriter) NN_NOEXCEPT
    {
        size_t fullSize = outBuffer.GetSize();

        size_t exportSize = pWriter->GetExportSize();
        NN_SDK_ASSERT_LESS(exportSize, std::numeric_limits<uint32_t>::max());

        uint32_t exportSize32 = static_cast<uint32_t>(exportSize);
        NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &exportSize32, sizeof(exportSize32)));

        NN_RESULT_THROW_UNLESS(outBuffer.GetSize() >= exportSize32, ResultInvalidArgument());
        pWriter->Export(outBuffer.GetPointerUnsafe(), outBuffer.GetSize());
        ForwardOutBuffer(&outBuffer, exportSize32);

        *pOutSize = static_cast<uint32_t>(fullSize - outBuffer.GetSize());

        NN_RESULT_SUCCESS;
    }
}

RequestImpl::SystemEventSet::SystemEventSet() NN_NOEXCEPT
    : m_SystemEvent1(nn::os::EventClearMode_AutoClear, true),
      m_SystemEvent2(nn::os::EventClearMode_AutoClear, true)
{
}

RequestImpl::SystemEventSet::~SystemEventSet() NN_NOEXCEPT
{
}

void RequestImpl::SystemEventSet::Signal() NN_NOEXCEPT
{
    // 数字が若いほうが先にシグナルされることを保証する
    m_SystemEvent1.Signal();
    m_SystemEvent2.Signal();
}

nn::os::NativeHandle RequestImpl::SystemEventSet::GetReadableHandle1() NN_NOEXCEPT
{
    return m_SystemEvent1.GetReadableHandle();
}

nn::os::NativeHandle RequestImpl::SystemEventSet::GetReadableHandle2() NN_NOEXCEPT
{
    return m_SystemEvent2.GetReadableHandle();
}


RequestImpl::RequestImpl(RequestManager* pRequestManager, const Capability& capability, RequirementPreset requirementPreset, ClientId clientId, nn::Bit64 processId) NN_NOEXCEPT
    : m_pRequestManager(nullptr),
      m_pUserRequest(nullptr),
      m_Capability(capability)
{
    UserRequest* pUserRequest;
    auto result = pRequestManager->CreateRequest(&pUserRequest, requirementPreset, &m_SystemEventSet, clientId, processId);

    if (result.IsFailure())
    {
        return;
    }

    m_pRequestManager = pRequestManager;
    m_pUserRequest = pUserRequest;
}

RequestImpl::~RequestImpl() NN_NOEXCEPT
{
    if (!IsAvailable())
    {
        return;
    }

    m_pRequestManager->DestroyRequest(m_pUserRequest);
}

nn::Result RequestImpl::Submit() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pRequestManager->Submit(m_pUserRequest));
}

nn::Result RequestImpl::Cancel() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pRequestManager->Cancel(m_pUserRequest));
}

nn::Result RequestImpl::GetRequestState(nn::sf::Out<IntRequestState> outRequestState) const NN_NOEXCEPT
{
    outRequestState.Set(static_cast<IntRequestState>(m_pUserRequest->GetRequestState()));
    NN_RESULT_SUCCESS;
}

nn::Result RequestImpl::GetSystemEventReadableHandles(
    nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle1,
    nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle2) NN_NOEXCEPT
{
    systemEventReadableHandle1.Set(nn::sf::NativeHandle(m_SystemEventSet.GetReadableHandle1(), false));
    systemEventReadableHandle2.Set(nn::sf::NativeHandle(m_SystemEventSet.GetReadableHandle2(), false));
    NN_RESULT_SUCCESS;
}

nn::Result RequestImpl::GetResult() const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->GetResult());
}

nn::Result RequestImpl::GetAdditionalInfo(nn::sf::Out<AdditionalInfo> outAdditionalInfo, nn::sf::Out<uint32_t> outRevision) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pUserRequest->GetAdditionalInfo(outAdditionalInfo.GetPointer(), outRevision.GetPointer()));
}

nn::Result RequestImpl::SetRequirement(const Requirement& requirement) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isRequirementSettable, ResultNoCapability());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetRequirement(requirement));
}

nn::Result RequestImpl::GetRequirement(nn::sf::Out<Requirement> outRequirement) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->GetRequirement(outRequirement.GetPointer()));
}

nn::Result RequestImpl::GetRevision(nn::sf::Out<uint32_t> outRevision) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    outRevision.Set(m_pUserRequest->GetRevision());

    NN_RESULT_SUCCESS;
}

nn::Result RequestImpl::SetRequirementPreset(int32_t requirementPreset) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isArbitraryPresetSettable ||
        requirementPreset == RequirementPreset_None ||
        requirementPreset == RequirementPreset_InternetBestEffort ||
        requirementPreset == RequirementPreset_InternetGeneric ||
        requirementPreset == RequirementPreset_LocalGeneric ||
        requirementPreset == RequirementPreset_NeighborDetection ||
        requirementPreset == RequirementPreset_Scan,
        ResultNoCapability());

    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    return m_pUserRequest->SetRequirementPreset(static_cast<RequirementPreset>(requirementPreset));
}

nn::Result RequestImpl::SetRequirementByRevision(uint32_t revision) NN_NOEXCEPT
{
    // TODO: SetRequirementPreset と同様の権限で実行可能
    NN_RESULT_THROW_UNLESS(m_Capability.isArbitraryPresetSettable, ResultNoCapability());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetRequirementByRevision(revision));
}

nn::Result RequestImpl::SetPriority(uint8_t priority) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetPriority(priority));
}

nn::Result RequestImpl::SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isArbitraryPrioritySettable, ResultNoCapability());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetRawPriority(rawPriority));
}

nn::Result RequestImpl::SetNetworkProfileId(nn::util::Uuid id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetNetworkProfileId(id));
}

nn::Result RequestImpl::SetRejectable(bool isRejectable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isRejectableFlagSettable, ResultNoCapability());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetRejectable(isRejectable));
}

nn::Result RequestImpl::SetConnectionConfirmationOption(int8_t connectionConfirmationOption) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(connectionConfirmationOption, ConnectionConfirmationOption_Count);
    NN_RESULT_THROW_UNLESS(connectionConfirmationOption < ConnectionConfirmationOption_Count, ResultOutOfRange());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetConnectionConfirmationOption(
        static_cast<ConnectionConfirmationOption>(connectionConfirmationOption)));
}

nn::Result RequestImpl::SetPersistent(bool isPersistent) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetPersistent(isPersistent));
}

nn::Result RequestImpl::SetInstant(bool isInstant) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetInstant(isInstant));
}

nn::Result RequestImpl::SetSustainable(bool isSustainable, uint8_t priority) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capability.isSustainableFlagSettable, ResultNoCapability());
    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultInternalError());   // TODO: [TORIAEZU]

    NN_RESULT_THROW(m_pUserRequest->SetSustainable(isSustainable, priority));
}

nn::Result RequestImpl::SetGreedy(bool isGreedy) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pUserRequest->SetGreedy(isGreedy));
}

nn::Result RequestImpl::SetSharable(bool isSharable) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pUserRequest->SetSharable(isSharable));
}

nn::Result RequestImpl::SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pUserRequest->SetKeptInSleep(isKeptInSleep));
}

nn::Result RequestImpl::RegisterSocketDescriptor(int32_t socketDescriptor) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(int) == sizeof(int32_t));

    NN_RESULT_DO(m_pUserRequest->RegisterSocketDescriptor(socketDescriptor));

    NN_RESULT_SUCCESS;
}

nn::Result RequestImpl::UnregisterSocketDescriptor(int32_t socketDescriptor) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(int) == sizeof(int32_t));

    NN_RESULT_DO(m_pUserRequest->UnregisterSocketDescriptor(socketDescriptor));

    NN_RESULT_SUCCESS;
}

nn::Result RequestImpl::GetAppletInfo(
    nn::sf::Out<Bit32> outAppletId,
    nn::sf::Out<Bit32> outLibraryAppletMode,
    nn::sf::Out<uint32_t> outSize,
    nn::sf::OutBuffer outBuffer,
    uint32_t colorIndex) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_STATIC_ASSERT(sizeof(nn::applet::AppletId) <= sizeof(Bit32));
    NN_STATIC_ASSERT(sizeof(nn::applet::LibraryAppletMode) <= sizeof(Bit32));

    nn::Result result = m_pUserRequest->GetResult();
    NN_RESULT_THROW_UNLESS(!(result <= ResultCanceled()), ResultErrorHandlingFailure());

    size_t fullSize = outBuffer.GetSize();

    nn::oe::SetExpectedThemeColor(static_cast<nn::oe::ThemeColorType>(colorIndex));

    // 疎通確認時に Web 認証アプレットが必要になった場合
    if (result <= ResultHotspotAuthenticationViaWebAuthAppletNeeded()) // Applet(WifiWebAuth)
    {
        uint32_t argumentSize;

        /** CommonArguments **/
        nn::la::CommonArgumentsWriter commonArgumentsWriter(0); // TODO: laVersion
        NN_RESULT_DO(Export<nn::la::CommonArgumentsWriter>(outBuffer, &argumentSize, &commonArgumentsWriter));
        ForwardOutBuffer(&outBuffer, argumentSize);

        /** NifmToWifiWebAuthArguments **/
        AdditionalInfo additionalInfo;
        uint32_t revision;
        NN_RESULT_DO(m_pUserRequest->GetAdditionalInfo(&additionalInfo, &revision));

        // Web 認証アプレットには固定の疎通確認 URL を渡し，再度リダイレクト応答を受け取ってもらう
        nn::la::NifmToWifiWebAuthArgumentsWriter wifiWebAuthWriter(ConnectionTestClient::ConnTestUrl, ConnectionTestClient::ConnTestUrl, additionalInfo.profileId, revision);
        NN_RESULT_DO(Export<nn::la::NifmToWifiWebAuthArgumentsWriter>(outBuffer, &argumentSize, &wifiWebAuthWriter));
        ForwardOutBuffer(&outBuffer, argumentSize);

        outAppletId.Set(nn::applet::AppletId_LibraryAppletWifiWebAuth);
        outLibraryAppletMode.Set(nn::applet::LibraryAppletMode_AllForeground);
    }
    else if (result <= ResultRequestRejected() && // Applet(NetConnect)
             !(result <= ResultDisconnected() || result <= ResultLocked()) // 除外される Result
             )
    {
        uint32_t argumentSize;

        /** CommonArguments **/
        nn::la::CommonArgumentsWriter commonArgumentsWriter(0); // TODO: laVersion
        NN_RESULT_DO(Export<nn::la::CommonArgumentsWriter>(outBuffer, &argumentSize, &commonArgumentsWriter));
        ForwardOutBuffer(&outBuffer, argumentSize);

        /** BootMode **/
        nn::la::netconnect::BootMode bootMode = nn::la::netconnect::BootMode_FromNifm;

        uint32_t bootModeSize = sizeof(bootMode);

        NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &bootModeSize, sizeof(bootModeSize)));
        NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &bootMode, sizeof(bootMode)));

        /** NifmToNetConnectArguments **/
        AdditionalInfo additionalInfo;
        uint32_t revision;
        NN_RESULT_DO(m_pUserRequest->GetAdditionalInfo(&additionalInfo, &revision));

        nn::la::NifmToNetConnectArgumentsWriter netConnectWriter(result, additionalInfo.profileId);
        NN_RESULT_DO(Export<nn::la::NifmToNetConnectArgumentsWriter>(outBuffer, &argumentSize, &netConnectWriter));
        ForwardOutBuffer(&outBuffer, argumentSize);

        outAppletId.Set(nn::applet::AppletId_LibraryAppletNetConnect);
        outLibraryAppletMode.Set(nn::applet::LibraryAppletMode_AllForeground);
    }
    else // Applet(Error)
    {
        uint32_t argumentSize;

        /** CommonArguments **/
        nn::la::CommonArgumentsWriter commonArgumentsWriter(0); // TODO: laVersion
        NN_RESULT_DO(Export<nn::la::CommonArgumentsWriter>(outBuffer, &argumentSize, &commonArgumentsWriter));
        ForwardOutBuffer(&outBuffer, argumentSize);

        /** outNifmBuffer **/
        nn::err::ErrorViewerStartupParamForSystemData param = nn::err::ErrorViewerStartupParamForSystemData();

        param.common.isJumpEnabled = false;
        param.isErrorCode = false;
        param.result = result;

        uint32_t paramSize = sizeof(param);

        NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &paramSize, sizeof(paramSize)));
        NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &param, sizeof(param)));

        outAppletId.Set(nn::applet::AppletId_LibraryAppletError);
        outLibraryAppletMode.Set(nn::applet::LibraryAppletMode_AllForeground);
    }

    /** outExtraBuffer **/
    // ...

    const int32_t sentinel = LibraryAppletArgumentBufferSentinel;
    NN_RESULT_DO(WriteAndForwardOutBuffer(&outBuffer, &sentinel, sizeof(sentinel)));

    outSize.Set(static_cast<uint32_t>(fullSize - outBuffer.GetSize()));

    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(outAppletId);
    NN_UNUSED(outLibraryAppletMode);
    NN_UNUSED(outSize);
    NN_UNUSED(outBuffer);
    NN_UNUSED(colorIndex);
    NN_RESULT_SUCCESS;
#endif
}

}
}
}
