﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/profile/nifm_WirelessUserSetting.h>

#include <nn/util/util_FormatString.h>

#include <algorithm>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <cwchar>
#include <cstdio>
#include <cstring>
#endif


namespace nn
{
namespace nifm
{
namespace detail
{

// SSID を文字列に変換，ASCII 外のデータが含まれる場合は全体を16進数表現
void WirelessUserSetting::ConvertSsidHexToString( char* pOutStringBuffer, int stringBufferSize, const Ssid& ssid ) NN_NOEXCEPT
{
    bool hasNonAscii = false;
    for( int i = 0; i < ssid.length; ++i )
    {
        if( ssid.hex[i] < 0x20 || 0x7e < ssid.hex[i] )
        {
            hasNonAscii = true;
            break;
        }
    }

    if( hasNonAscii )
    {
        int size = std::min<int>(stringBufferSize / 2, ssid.length);
        for( int i = 0; i < size; ++i )
        {
            nn::util::SNPrintf(&pOutStringBuffer[i * 2], 2, "%02X", ssid.hex[i]);
        }
        pOutStringBuffer[stringBufferSize > (2 * size) ? 2 * size : stringBufferSize - 1] = '\0';
    }
    else
    {
        nn::util::SNPrintf(pOutStringBuffer, stringBufferSize, "%s", ssid.hex);
        pOutStringBuffer[stringBufferSize > ssid.length ? ssid.length : stringBufferSize - 1] = '\0';
    }
}

WirelessUserSetting::WirelessUserSetting( const WirelessSettingData& setting ) NN_NOEXCEPT
{
    m_Setting = setting;

    SetSsidHex(setting.ssidConfig.ssid.hex, setting.ssidConfig.ssid.length);
    SetKeyMaterial(setting.security.sharedKey.keyMaterial, setting.security.sharedKey.length);
}

unsigned char* WirelessUserSetting::GetSsidHex( unsigned char* pOutBuf, int *pOutLength ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutBuf);
    std::memcpy( pOutBuf, m_Setting.ssidConfig.ssid.hex, m_Setting.ssidConfig.ssid.length );
    *pOutLength = m_Setting.ssidConfig.ssid.length;
    return pOutBuf;
}

char* WirelessUserSetting::GetSsidHexString( char* pOutStr ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStr);
    int len = 0;
    for( size_t i = 0; i < m_Setting.ssidConfig.ssid.length; ++i )
    {
        len += nn::util::SNPrintf( &pOutStr[i * 2], 3, "%02X", m_Setting.ssidConfig.ssid.hex[i] );
    }
    pOutStr[len] = '\0';
    return pOutStr;
}

Ssid* WirelessUserSetting::GetSsid( Ssid *pOutSsid ) const NN_NOEXCEPT
{
    *pOutSsid = m_Setting.ssidConfig.ssid;
    return pOutSsid;
}

bool WirelessUserSetting::IsNonBroadcastEnabled() const NN_NOEXCEPT
{
    return m_Setting.ssidConfig.nonBroadcast;
}

Authentication WirelessUserSetting::GetAuthentication() const NN_NOEXCEPT
{
    return m_Setting.security.authEncryption.authentication;
}

Encryption WirelessUserSetting::GetEncryption() const NN_NOEXCEPT
{
    return m_Setting.security.authEncryption.encryption;
}

char* WirelessUserSetting::GetKeyMaterial( char* pStr, int* pOutLength, int inLength ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    int copyLength = std::min<int>( inLength, m_Setting.security.sharedKey.length );
    memcpy( pStr, m_Setting.security.sharedKey.keyMaterial, copyLength );
    *pOutLength = copyLength;
    return pStr;
}

int WirelessUserSetting::GetKeyMaterialCount() const NN_NOEXCEPT
{
    return m_Setting.security.sharedKey.length;
}

void WirelessUserSetting::SetSsidHex(const unsigned char* pStr, int length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStr);

    int copyLength = std::min<int>(length, static_cast<int>(Ssid::HexSize));
    copyLength = std::max<int>(0, copyLength);
    std::memcpy( m_Setting.ssidConfig.ssid.hex, pStr, copyLength);
    m_Setting.ssidConfig.ssid.length = static_cast<uint8_t>(copyLength);
}

void WirelessUserSetting::SetNonBroadcastEnabled( bool isEnabled ) NN_NOEXCEPT
{
    m_Setting.ssidConfig.nonBroadcast = isEnabled;
}

void WirelessUserSetting::SetAuthentication( Authentication value ) NN_NOEXCEPT
{
    m_Setting.security.authEncryption.authentication = value;
}

void WirelessUserSetting::SetEncryption( Encryption value ) NN_NOEXCEPT
{
    m_Setting.security.authEncryption.encryption = value;
}

void WirelessUserSetting::SetKeyMaterial(const char* pStr, int length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStr);

    int copyLength = std::min<int>(length, static_cast<int>(SharedKey::KeyMaterialSize));
    copyLength = std::max<int>(0, copyLength);
    memcpy(m_Setting.security.sharedKey.keyMaterial, pStr, copyLength);
    m_Setting.security.sharedKey.length = static_cast<uint8_t>(copyLength);
}

bool WirelessUserSetting::IsSameSsidAuthEncrypt(const WirelessUserSetting& rh) const NN_NOEXCEPT
{
    return m_Setting.ssidConfig.ssid.length == rh.m_Setting.ssidConfig.ssid.length &&
           memcmp( m_Setting.ssidConfig.ssid.hex, rh.m_Setting.ssidConfig.ssid.hex, m_Setting.ssidConfig.ssid.length ) == 0 &&
           m_Setting.security.authEncryption.authentication == rh.m_Setting.security.authEncryption.authentication &&
           m_Setting.security.authEncryption.encryption == rh.m_Setting.security.authEncryption.encryption;
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
Result WirelessUserSetting::ConvertToXml( wchar_t *pOutXml, size_t size ) const NN_NOEXCEPT
{
    if( !pOutXml || size < XmlNecessarySize )
    {
        // TODO:Result分ける？
        NN_RESULT_THROW(ResultInvalidArgument());
    }

    size_t pos = 0;
    // TODO: size - pos の確認
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<?xml version=\"1.0\"?>\n<WLANProfile xmlns=\"http://www.microsoft.com/networking/WLAN/profile/v1\">\n");
    {
        pos += swprintf_s(&pOutXml[pos], size - pos, L"<name>temporary</name>\n");
    }
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<SSIDConfig>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<SSID>\n");
    {
        pos += swprintf_s(&pOutXml[pos], size - pos, L"<hex>");
        for( size_t i = 0; i < m_Setting.ssidConfig.ssid.length; ++i )
        {
            pos += swprintf_s(&pOutXml[pos], size - pos, L"%02x", m_Setting.ssidConfig.ssid.hex[i]);
        }
        pos += swprintf_s(&pOutXml[pos], size - pos, L"</hex>\n");
    }
    // <hex> をマスターにして <name> は不使用
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</SSID>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<nonBroadcast>%s</nonBroadcast>\n", m_Setting.ssidConfig.nonBroadcast ? L"true" : L"false" );
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</SSIDConfig>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<connectionType>ESS</connectionType>\n<connectionMode>manual</connectionMode>\n<autoSwitch>false</autoSwitch>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<MSM>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<security>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<authEncryption>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<authentication>");
    switch(m_Setting.security.authEncryption.authentication)
    {
    case Authentication_Open:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"open");
        break;
    case Authentication_Shared:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"shared");
        break;
    case Authentication_Wpa:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"WPA");
        break;
    case Authentication_WpaPsk:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"WPAPSK");
        break;
    case Authentication_Wpa2:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"WPA2");
        break;
    case Authentication_Wpa2Psk:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"WPA2PSK");
        break;
    default:
        break;
    }
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</authentication>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<encryption>");
    switch(m_Setting.security.authEncryption.encryption)
    {
    case Encryption_None:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"none");
        break;
    case Encryption_Wep:
        if(m_Setting.security.sharedKey.length == Wep40KeyLength ||
           m_Setting.security.sharedKey.length == Wep40KeyLengthAsHex)
        {
            pos += swprintf_s(&pOutXml[pos], size - pos, L"WEP40");
        }
        else if(m_Setting.security.sharedKey.length == Wep104KeyLength ||
                m_Setting.security.sharedKey.length == Wep104KeyLengthAsHex)
        {
            pos += swprintf_s(&pOutXml[pos], size - pos, L"WEP104");
        }
        break;
    case Encryption_Tkip:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"TKIP");
        break;
    case Encryption_Aes:
        pos += swprintf_s(&pOutXml[pos], size - pos, L"AES");
        break;
    default:
        break;
    }
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</encryption>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<useOneX>false</useOneX>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</authEncryption>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<sharedKey>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<keyType>");
    pos += swprintf_s(&pOutXml[pos], size - pos, m_Setting.security.authEncryption.encryption==Encryption_Wep ? L"networkKey" : L"passPhrase");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</keyType>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"<protected>false</protected>\n");
    {
        wchar_t buf[SharedKey::KeyMaterialSize];
        mbstowcs(buf, m_Setting.security.sharedKey.keyMaterial, SharedKey::KeyMaterialSize);
        pos += swprintf_s(&pOutXml[pos], size - pos, L"<keyMaterial>%s</keyMaterial>\n", buf);
    }
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</sharedKey>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</security>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</MSM>\n");
    pos += swprintf_s(&pOutXml[pos], size - pos, L"</WLANProfile>\n");

    NN_RESULT_SUCCESS;
} // NOLINT(readability/fn_size)
#endif

void WirelessUserSetting::Export(WirelessSettingData* pOutWirelessSettingData) const NN_NOEXCEPT
{
    *pOutWirelessSettingData = m_Setting;
}

}
}
}
