﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.win32.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_LockGuard.h>

#include <mutex>

#include <Iphlpapi.h>

namespace
{

nn::nifm::Authentication ConvertAuthenticationToNifm( DOT11_AUTH_ALGORITHM winAuth )
{
    switch( winAuth )
    {
    case DOT11_AUTH_ALGORITHM_OPEN_SYSTEM:
        return nn::nifm::Authentication_Open;
    case DOT11_AUTH_ALGORITHM_SHARED_KEY:
        return nn::nifm::Authentication_Shared;
    case DOT11_AUTH_ALGORITHM_WPA:
        return nn::nifm::Authentication_Wpa;
    case DOT11_AUTH_ALGORITHM_WPA_PSK:
        return nn::nifm::Authentication_WpaPsk;
    case DOT11_AUTH_ALGORITHM_RSNA:
        return nn::nifm::Authentication_Wpa2;
    case DOT11_AUTH_ALGORITHM_RSNA_PSK:
        return nn::nifm::Authentication_Wpa2Psk;
    case DOT11_AUTH_ALGORITHM_WPA_NONE:
        NN_FALL_THROUGH;
    default:
        return nn::nifm::Authentication_Invalid;
    }
}

nn::nifm::Encryption ConvertEncryptionToNifm( DOT11_CIPHER_ALGORITHM winCipher )
{
    switch( winCipher )
    {
    case DOT11_CIPHER_ALGO_NONE:
        return nn::nifm::Encryption_None;
    case DOT11_CIPHER_ALGO_TKIP:
        return nn::nifm::Encryption_Tkip;
    case DOT11_CIPHER_ALGO_CCMP:
        return nn::nifm::Encryption_Aes;
    case DOT11_CIPHER_ALGO_WEP40:
    case DOT11_CIPHER_ALGO_WEP104:
        return nn::nifm::Encryption_Wep;
    case DOT11_CIPHER_ALGO_WPA_USE_GROUP:
        NN_FALL_THROUGH;
    case DOT11_CIPHER_ALGO_WEP:
        NN_FALL_THROUGH;
    default:
        return nn::nifm::Encryption_Invalid;
    }
}

}

namespace nn
{
namespace nifm
{
namespace detail
{

const WCHAR WirelessInterface::profileBackupDirName[] = L"profiles";

nn::Result WirelessInterface::GetAllowedChannels(int16_t(&pOutChannels)[WirelessChannelsCountMax], int *pOutCount) NN_NOEXCEPT
{
    NN_UNUSED(pOutChannels);
    NN_UNUSED(pOutCount);

    NN_RESULT_THROW(ResultNotImplemented());
}

void WirelessInterface::GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) NN_NOEXCEPT
{
    static const NET_IF_MEDIA_CONNECT_STATE MediaConnectStateOrder[] = { MediaConnectStateConnected, MediaConnectStateDisconnected };

    NN_SDK_ASSERT(pOutMacAddresses != nullptr || inCount == 0);
    NN_SDK_ASSERT_NOT_NULL(pOutCount);

    *pOutCount = 0;

    PMIB_IF_TABLE2 pMibIfTable2;
    if( ::GetIfTable2( &pMibIfTable2 ) == NO_ERROR )
    {
        for( const auto mediaConnectState : MediaConnectStateOrder )
        {
            for( uint32_t i = 0; i < pMibIfTable2->NumEntries; ++i )
            {
                const MIB_IF_ROW2& mibIfRow2 = pMibIfTable2->Table[i];
                if( mibIfRow2.Type == IF_TYPE_IEEE80211 &&
                    mibIfRow2.InterfaceAndOperStatusFlags.HardwareInterface == TRUE &&
                    mibIfRow2.MediaConnectState == mediaConnectState )
                {
                    if( *pOutCount < inCount )
                    {
                        MacAddress& macAddress = pOutMacAddresses[*pOutCount];
                        memcpy( macAddress.data, mibIfRow2.PhysicalAddress, MacAddress::Size );
                    }
                    ++*pOutCount;
                }
            }
        }

        ::FreeMibTable( pMibIfTable2 );
    }
}

WirelessInterface::WirelessInterface() NN_NOEXCEPT
    : m_IsInitialized(false),
      m_Index(0),
      m_ClientHandle(NULL),
      m_MacAddress(InvalidMacAddress),
      m_Id(-1)  // TODO: TORIAEZU
{
}

WirelessInterface::~WirelessInterface() NN_NOEXCEPT
{
    if( m_ClientHandle != NULL )
    {
        ::WlanRegisterNotification(
                    m_ClientHandle,
                    WLAN_NOTIFICATION_SOURCE_NONE,
                    TRUE,
                    NULL,
                    NULL,
                    NULL,
                    NULL
        );

        ::CloseHandle(m_ClientHandle);
    }
}

nn::Result WirelessInterface::Initialize() NN_NOEXCEPT
{
    NN_RESULT_THROW(Initialize( InvalidMacAddress ));
}

nn::Result WirelessInterface::Initialize( const MacAddress& macAddress ) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_BlockingTaskMutex);

    if( !IsInitialized() )
    {
        // 無線インターフェースが見つからなくても初期化済みにして、関数が成功する状態にする
        m_IsInitialized = true;

        DWORD version;
        ::HANDLE clientHandle;

        if( ::WlanOpenHandle(WLAN_API_MAKE_VERSION(1, 0), NULL, &version, &clientHandle) == ERROR_SUCCESS )
        {
            m_ClientHandle = clientHandle;

            PMIB_IF_TABLE2 pMibIfTable2;
            if( ::GetIfTable2( &pMibIfTable2 ) == NO_ERROR )
            {
                for ( uint32_t i = 0; i < pMibIfTable2->NumEntries; ++i )
                {
                    const MIB_IF_ROW2& mibIfRow2 = pMibIfTable2->Table[i];
                    if( memcmp( macAddress.data, mibIfRow2.PhysicalAddress, MacAddress::Size ) == 0 &&
                        mibIfRow2.InterfaceAndOperStatusFlags.HardwareInterface == TRUE )
                    {
                        NN_RESULT_DO(m_IpConfiguration.Initialize(macAddress));

                        m_MacAddress = macAddress;
                        m_InterfaceGuid = mibIfRow2.InterfaceGuid;
                        m_Index = mibIfRow2.InterfaceIndex;

                        // コールバックの準備
                        NN_DETAIL_NIFM_RETURN_RESULT_IF_WINAPI_FAILED(
                            ::WlanRegisterNotification(
                                clientHandle,
                                WLAN_NOTIFICATION_SOURCE_ACM,
                                TRUE,
                                NotificationCallback,
                                this,
                                NULL,
                                NULL
                            )
                        );

                        break;
                    }
                }
                ::FreeMibTable( pMibIfTable2 );
            }
        }
    }

    NN_RESULT_SUCCESS;
}

bool WirelessInterface::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool WirelessInterface::IsAvailable() const NN_NOEXCEPT
{
    return IsInitialized() && m_ClientHandle != NULL;
}

MacAddress* WirelessInterface::GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT
{
    memcpy( pOutMacAddress->data, m_MacAddress.data, MacAddress::Size );
    return pOutMacAddress;
}

nn::Result WirelessInterface::UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInOutAccessPointData);

    DWORD connectionInfoSize = sizeof(WLAN_CONNECTION_ATTRIBUTES);
    PWLAN_CONNECTION_ATTRIBUTES pConnectionStatus = NULL;
    WLAN_OPCODE_VALUE_TYPE opCode = wlan_opcode_value_type_query_only;

    DWORD dwResult = ::WlanQueryInterface( m_ClientHandle,
                                           &m_InterfaceGuid,
                                           wlan_intf_opcode_current_connection,
                                           NULL,
                                           &connectionInfoSize,
                                           (PVOID*)&pConnectionStatus,
                                           &opCode );

    if( dwResult == ERROR_SUCCESS )
    {
        pInOutAccessPointData->rssi = ConvertLinkQualityToRssi( pConnectionStatus->wlanAssociationAttributes.wlanSignalQuality ); // -100 to -50
        pInOutAccessPointData->linkLevel = ConvertLinkQualityToLinkLevel( pConnectionStatus->wlanAssociationAttributes.wlanSignalQuality );

        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultWindowsApiFailed());
}

nn::Result WirelessInterface::Scan() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsEnabled(), ResultNetworkInterfaceNotAvailable());

    NN_UTIL_LOCK_GUARD(m_BlockingTaskMutex);

    // BG スキャンでイベントがシグナルしないように一時停止
    // TODO: BG スキャンの現在の状態を反映
    BOOL isBackgroundScanEnabled = FALSE;

    // Administrator 権限が必要
    // TODO: WlanSetInterface がブロッキングか確認
    DWORD errorBgScanDisable = ::WlanSetInterface(
        m_ClientHandle,
        &m_InterfaceGuid,
        ::wlan_intf_opcode_background_scan_enabled,
        sizeof(isBackgroundScanEnabled),
        &isBackgroundScanEnabled,
        NULL
    );

    if( errorBgScanDisable == ERROR_SUCCESS )
    {
        NN_DETAIL_NIFM_INFO("Background scan disabled.\n");
    }
    else
    {
        NN_DETAIL_NIFM_INFO("Could not disable background scan.\n");
    }

    m_AsyncTask.Initialize(AsyncTask::Type::Scan);
    m_AsyncTask.WaitForResponseOf(
        ::WlanScan(
            m_ClientHandle,
            &m_InterfaceGuid,
            NULL,
            NULL,
            NULL
        )
    );

    // スキャン失敗でも BG スキャンの設定は元に戻す

    if( errorBgScanDisable == ERROR_SUCCESS )
    {
        isBackgroundScanEnabled = TRUE;

        NN_DETAIL_NIFM_RETURN_RESULT_IF_WINAPI_FAILED(
            ::WlanSetInterface(
                m_ClientHandle,
                &m_InterfaceGuid,
                ::wlan_intf_opcode_background_scan_enabled,
                sizeof(isBackgroundScanEnabled),
                &isBackgroundScanEnabled,
                NULL
            )
        );

        NN_DETAIL_NIFM_INFO("Background scan enabled.\n");
    }

    NN_RESULT_DO(m_AsyncTask.GetResult());

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_UNUSED(timeSpan);

    nn::Result result = ResultSuccess();

    // TODO: 更新との排他を図る
    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    {
        // ローカル通信用のダミーアクセスポイントの追加
        if (pAccessPointList->CreateAndAdd<LocalAccessPoint>(this) == nullptr)
        {
            NN_DETAIL_NIFM_WARN("Accesspoint list is full. (Local)\n");
            NN_RESULT_SUCCESS;
        }
        // すれいちがい通信用のダミーアクセスポイントの追加
        if (pAccessPointList->CreateAndAdd<NeighborDetectionAccessPoint>(this) == nullptr)
        {
            NN_DETAIL_NIFM_WARN("Accesspoint list is full. (Neighbor)\n");
            NN_RESULT_SUCCESS;
        }
    }

    // インターネット通信用のアクセスポイントの列挙

    ::PWLAN_BSS_LIST pWlanBssList = NULL;
    ::PWLAN_AVAILABLE_NETWORK_LIST pWlanAvalableNetworkList = NULL;

    NN_UTIL_SCOPE_EXIT
    {
        if( pWlanBssList != NULL )
        {
            ::WlanFreeMemory(pWlanBssList);
        }

        if( pWlanAvalableNetworkList != NULL )
        {
            ::WlanFreeMemory(pWlanAvalableNetworkList);
        }
    };

    NN_DETAIL_NIFM_TRACE("GetAccessPointList\n");

    NN_DETAIL_NIFM_RETURN_RESULT_IF_WINAPI_FAILED(
        ::WlanGetNetworkBssList(
            m_ClientHandle,
            &m_InterfaceGuid,
            NULL,
            ::dot11_BSS_type_infrastructure,
            FALSE,  // TODO: ?
            NULL,
            &pWlanBssList
        )
    );

    NN_DETAIL_NIFM_RETURN_RESULT_IF_WINAPI_FAILED(
        ::WlanGetAvailableNetworkList(
            m_ClientHandle,
            &m_InterfaceGuid,
            0,
            NULL,
            &pWlanAvalableNetworkList
        )
    );

    NN_DETAIL_NIFM_TRACE("GetAccessPointList:%d\n", pWlanBssList->dwNumberOfItems);
    for( size_t i=0; i<pWlanBssList->dwNumberOfItems; ++i )
    {
        WirelessAccessPoint temporaryAp(this);
        WLAN_BSS_ENTRY *e = &(pWlanBssList->wlanBssEntries[i]);
        temporaryAp.SetSsid( e->dot11Ssid.ucSSID, e->dot11Ssid.uSSIDLength );
        MacAddress tmpMacAddress;
        std::memcpy( tmpMacAddress.data, e->dot11Bssid, MacAddress::Size );
        temporaryAp.SetBssid( tmpMacAddress );
        temporaryAp.SetRssi( ConvertLinkQualityToRssi(e->uLinkQuality) );
        temporaryAp.SetLinkLevel( ConvertLinkQualityToLinkLevel(e->uLinkQuality) );
        temporaryAp.SetChannel( 0 );    // Windows では取得も指定もできない

        for( size_t j=0; j<pWlanAvalableNetworkList->dwNumberOfItems; ++j )
        {
            WLAN_AVAILABLE_NETWORK *n = &(pWlanAvalableNetworkList->Network[j]);

            // SSID が同じでセキュリティ設定が異なる AP のビーコンの区別ができないので
            // とりあえずすべての可能性で格納する

            if( e->dot11Ssid.uSSIDLength == n->dot11Ssid.uSSIDLength &&
                std::memcmp(e->dot11Ssid.ucSSID, n->dot11Ssid.ucSSID, e->dot11Ssid.uSSIDLength) == 0 )
            {
                Authentication auth = ConvertAuthenticationToNifm(n->dot11DefaultAuthAlgorithm);
                Encryption enc = ConvertEncryptionToNifm(n->dot11DefaultCipherAlgorithm);

                // GroupEncryption が抽出できないので個別の Encryption と同じ値を入れる
                if (auth == Authentication_Invalid || enc == Encryption_Invalid)
                {
                    NN_DETAIL_NIFM_WARN("Unknown auth/cipher algorithm(%d/%d). Use \"Open/None\" instead.\n",
                        n->dot11DefaultAuthAlgorithm, n->dot11DefaultCipherAlgorithm);
                    temporaryAp.SetWepAuthenticationAndEncryption(Authentication_Open, Encryption_None);
                }
                else if (auth == Authentication_Wpa || auth == Authentication_Wpa2 ||
                    auth == Authentication_WpaPsk || auth == Authentication_Wpa2Psk)
                {
                    temporaryAp.SetWpaAuthenticationAndEncryption(auth, enc, enc);
                }
                else
                {
                    temporaryAp.SetWepAuthenticationAndEncryption(auth, enc);
                }

                if (pAccessPointList->CreateAndAdd<WirelessAccessPoint>(temporaryAp) == nullptr)
                {
                    NN_DETAIL_NIFM_WARN("Accesspoint list is full.\n");
                    NN_RESULT_SUCCESS;
                }
            }
        }
    }

    NN_RESULT_THROW(result);
} // NOLINT(readability/fn_size)


nn::Result WirelessInterface::ConnectImpl(
    const WirelessAccessPoint& wirelessAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_INFO("WirelessInterface::ConnectImpl(const WirelessAccessPoint&, ...)\n");

    NN_UTIL_LOCK_GUARD(m_BlockingTaskMutex);

    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultNetworkInterfaceNotAvailable());

    Ssid ssid;
    wirelessAccessPoint.GetSsid( &ssid );
    char ssidStr[Ssid::HexSize + 1] = {};
    memcpy( ssidStr, ssid.hex, ssid.length );
    NN_DETAIL_NIFM_INFO( "Connecting to %.32s\n", ssidStr );

    // Windows 上で、無線のネットワークインタフェースによる接続が確立している（MediaConnectSatteConnected）場合、
    // 接続試行を行わずに成功を返し、現在の Windows 接続を維持します。
    // そうでない場合、接続試行を行わずに失敗を返します。
    NN_UNUSED(networkProfile);

    MIB_IF_ROW2 mibIfRow2;
    ::SecureZeroMemory((PVOID)&mibIfRow2, sizeof(mibIfRow2));
    mibIfRow2.InterfaceIndex = m_Index;
    DWORD error = ::GetIfEntry2(&mibIfRow2);

    NN_RESULT_THROW_UNLESS(error == NO_ERROR, ResultNotConnected());
    NN_RESULT_THROW_UNLESS(mibIfRow2.Type == IF_TYPE_IEEE80211 && mibIfRow2.MediaConnectState == ::MediaConnectStateConnected, ResultNotConnected());

    PWLAN_CONNECTION_ATTRIBUTES pConnectInfo = NULL;
    DWORD connectInfoSize = sizeof(WLAN_CONNECTION_ATTRIBUTES);
    DWORD dwResult = WlanQueryInterface(m_ClientHandle, &m_InterfaceGuid, ::wlan_intf_opcode_current_connection, NULL,
        &connectInfoSize, (PVOID*)&pConnectInfo, NULL);

    NN_RESULT_THROW_UNLESS(dwResult == ERROR_SUCCESS, ResultNotConnected());

    Ssid tmpSsid;
    networkProfile.GetWirelessSetting().GetSsid(&tmpSsid);
    WLAN_ASSOCIATION_ATTRIBUTES association = pConnectInfo->wlanAssociationAttributes;
    WLAN_SECURITY_ATTRIBUTES security = pConnectInfo->wlanSecurityAttributes;

    NN_RESULT_THROW_UNLESS(association.dot11Ssid.uSSIDLength == tmpSsid.length, ResultNotConnected());
    NN_RESULT_THROW_UNLESS(memcmp( association.dot11Ssid.ucSSID, tmpSsid.hex, tmpSsid.length ) == 0, ResultNotConnected());
    NN_RESULT_THROW_UNLESS(ConvertAuthenticationToNifm( security.dot11AuthAlgorithm ) == networkProfile.GetWirelessSetting().GetAuthentication(), ResultNotConnected());
    NN_RESULT_THROW_UNLESS(ConvertEncryptionToNifm( security.dot11CipherAlgorithm ) == networkProfile.GetWirelessSetting().GetEncryption(), ResultNotConnected());

    auto result = InitializeInternetConnection(networkProfile, aggregatedRequest);

    if (result.IsFailure())
    {
        DisconnectImpl(wirelessAccessPoint);
    }

    NN_RESULT_THROW(result);
} // NOLINT(impl/function_size)

nn::Result WirelessInterface::DisconnectImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(wirelessAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const WirelessAccessPoint&, ...)\n");

    NN_UTIL_LOCK_GUARD(m_BlockingTaskMutex);

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(wirelessAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const WirelessAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(WirelessAccessPoint* pInOutWirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInOutWirelessAccessPoint);

    // ステルスの AP だとビーコン情報に SSID が入っていないので、プロファイルから埋める
    Ssid ssid;
    networkProfile.GetWirelessSetting().GetSsid(&ssid);
    pInOutWirelessAccessPoint->SetSsid(ssid);

    // WEP の認証設定はビーコンからは判断できないので、プロファイルから埋める
    if (pInOutWirelessAccessPoint->GetEncryption() == Encryption_Wep)
    {
        NN_SDK_ASSERT_EQUAL(pInOutWirelessAccessPoint->GetEncryption(), networkProfile.GetWirelessSetting().GetEncryption());

        pInOutWirelessAccessPoint->SetWepAuthenticationAndEncryption(networkProfile.GetWirelessSetting().GetAuthentication(), pInOutWirelessAccessPoint->GetEncryption());
    }
}

// Local

nn::Result WirelessInterface::ConnectImpl(
    const LocalAccessPoint& localAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ConnectImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::DisconnectImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(LocalAccessPoint* pInOutLocalAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutLocalAccessPoint);
    NN_UNUSED(networkProfile);
}

// NeighborDetection

nn::Result WirelessInterface::ConnectImpl(
    const NeighborDetectionAccessPoint& neighborDetectionAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ConnectImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::DisconnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(NeighborDetectionAccessPoint* pInOutNeighborDetectionAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutNeighborDetectionAccessPoint);
    NN_UNUSED(networkProfile);
}

bool WirelessInterface::IsMediaConnected() const NN_NOEXCEPT
{
    bool isMediaConnected = false;

    PMIB_IF_TABLE2 pMibIfTable2;
    ::GetIfTable2( &pMibIfTable2 );

    for ( ULONG i = 0; i < pMibIfTable2->NumEntries; ++i )
    {
        const MIB_IF_ROW2& mibIfRow2 = pMibIfTable2->Table[i];
        if ( memcmp( &m_InterfaceGuid, &mibIfRow2.InterfaceGuid, sizeof(m_InterfaceGuid) ) == 0 )
        {
            isMediaConnected = !mibIfRow2.InterfaceAndOperStatusFlags.NotMediaConnected;
            break;
        }
    }

    ::FreeMibTable( pMibIfTable2 );

    return isMediaConnected;
}

nn::Result WirelessInterface::WaitForIpChange( uint32_t timeoutMS ) NN_NOEXCEPT
{
    OVERLAPPED overlap;
    DWORD ret;

    HANDLE hand = WSACreateEvent();
    overlap.hEvent = WSACreateEvent();

    ret = NotifyAddrChange(&hand, &overlap);

    if ( (ret != NO_ERROR && WSAGetLastError() != WSA_IO_PENDING) ||
        WaitForSingleObject(overlap.hEvent, timeoutMS) != WAIT_OBJECT_0 )
    {
        if( IsMediaConnected() )
        {
            m_AsyncTask.Initialize(AsyncTask::Type::Disconnect);
            m_AsyncTask.WaitForResponseOf(
                ::WlanDisconnect(
                    m_ClientHandle,
                    &m_InterfaceGuid,
                    NULL
                ),
                nn::TimeSpan::FromSeconds(3)    // wlan_notification_acm_disconnected が来たり来なかったりするので時限付で待つ
            );
        }
        NN_RESULT_THROW(ResultWindowsApiFailed());
    }

    NN_RESULT_SUCCESS;
}

void WirelessInterface::CopyCurrentProfileNameTo( WCHAR* profileName ) NN_NOEXCEPT
{
    PWLAN_CONNECTION_ATTRIBUTES pConnectInfo = NULL;
    DWORD connectInfoSize = sizeof(WLAN_CONNECTION_ATTRIBUTES);

    WlanQueryInterface( m_ClientHandle, &m_InterfaceGuid, ::wlan_intf_opcode_current_connection, NULL,
                        &connectInfoSize, (PVOID*)&pConnectInfo, NULL );

    nn::util::Strlcpy<WCHAR>( profileName, pConnectInfo->strProfileName, 256 );
}

void WirelessInterface::ConnectByProfileName( WCHAR* profileName ) NN_NOEXCEPT
{
    ::WLAN_CONNECTION_PARAMETERS wlanConnectionParameters = {
        ::wlan_connection_mode_profile,
        profileName,
        NULL,
        NULL,
        dot11_BSS_type_infrastructure,
        WLAN_CONNECTION_HIDDEN_NETWORK
    };

    m_AsyncTask.Initialize(AsyncTask::Type::Connect);
    m_AsyncTask.WaitForResponseOf(
        ::WlanConnect(
            m_ClientHandle,
            &m_InterfaceGuid,
            &wlanConnectionParameters,
            NULL
        )
    );
}

int WirelessInterface::ConvertLinkQualityToRssi(LONG linkQuality) const NN_NOEXCEPT
{
    // https://msdn.microsoft.com/en-us/library/windows/desktop/ms706828(v=vs.85).aspx
    // A percentage value that represents the signal quality of the network.
    // WLAN_SIGNAL_QUALITY is of type ULONG. This member contains a value between
    // 0 and 100. A value of 0 implies an actual RSSI signal strength of -100 dbm.
    // A value of 100 implies an actual RSSI signal strength of -50 dbm. You can
    // calculate the RSSI signal strength value for wlanSignalQuality values
    // between 1 and 99 using linear interpolation.
    return linkQuality / 2 - 100;
}

LinkLevel WirelessInterface::ConvertLinkQualityToLinkLevel(LONG linkQuality) const NN_NOEXCEPT
{
    int rssi = ConvertLinkQualityToRssi(linkQuality);

    const int rssiThreashold1 = -99;
    const int rssiThreashold2 = -87;
    const int rssiThresshold3 = -77;

    if( rssi < rssiThreashold1 )
    {
        return LinkLevel_0;
    }
    else if( rssi < rssiThreashold2 )
    {
        return LinkLevel_1;
    }
    else if( rssi < rssiThresshold3 )
    {
        return LinkLevel_2;
    }
    else
    {
        return LinkLevel_3;
    }
}


VOID WINAPI WirelessInterface::NotificationCallback(::PWLAN_NOTIFICATION_DATA pData, PVOID pContext) NN_NOEXCEPT
{
    reinterpret_cast<WirelessInterface*>(pContext)->NotificationCallbackImpl(pData);
}

void WirelessInterface::NotificationCallbackImpl(::PWLAN_NOTIFICATION_DATA pData) NN_NOEXCEPT
{
    if (pData->NotificationSource == WLAN_NOTIFICATION_SOURCE_ACM)
    {
        switch( pData->NotificationCode )
        {
        case ::wlan_notification_acm_scan_complete:
            NN_DETAIL_NIFM_INFO("NotificationCallbackImpl: wlan_notification_acm_scan_complete\n");
            // BG スキャン完了でもここに到達する
            if( m_AsyncTask.GetType() == AsyncTask::Type::Scan )
            {
                m_AsyncTask.Complete(nn::ResultSuccess());
            }
            break;

        case ::wlan_notification_acm_scan_fail:
            NN_DETAIL_NIFM_INFO("NotificationCallbackImpl: wlan_notification_acm_scan_fail\n");

            if( m_AsyncTask.GetType() == AsyncTask::Type::Scan )
            {
                m_AsyncTask.Complete(ResultWindowsApiFailed()); // TODO:
            }
            break;

        case ::wlan_notification_acm_connection_complete:
            NN_DETAIL_NIFM_INFO("NotificationCallbackImpl: wlan_notification_acm_connection_complete\n");
            // 自動接続でもここに到達する
            if( m_AsyncTask.GetType() == AsyncTask::Type::Connect )
            {
                PWLAN_CONNECTION_NOTIFICATION_DATA pConData = reinterpret_cast<PWLAN_CONNECTION_NOTIFICATION_DATA>(pData->pData);
                if( pConData->wlanReasonCode == WLAN_REASON_CODE_SUCCESS )
                {
                    NN_SDK_ASSERT(GetNetworkResourceState() == NetworkResourceState::ToBePrepared, "ResourceState=%d", GetNetworkResourceState());
                    m_AsyncTask.Complete(nn::ResultSuccess());
                }
            }
            break;

        case ::wlan_notification_acm_connection_attempt_fail:
            NN_DETAIL_NIFM_INFO("NotificationCallbackImpl: wlan_notification_acm_connection_attempt_fail\n");

            if( m_AsyncTask.GetType() == AsyncTask::Type::Connect )
            {
                NN_SDK_ASSERT(GetNetworkResourceState() == NetworkResourceState::ToBePrepared, "ResourceState=%d", GetNetworkResourceState());
                m_AsyncTask.Complete(ResultWindowsApiFailed()); // TODO:
            }
            break;

        case ::wlan_notification_acm_disconnected:
            NN_DETAIL_NIFM_INFO("NotificationCallbackImpl: wlan_notification_acm_disconnected\n");
            // 能動的切断のみハンドリング
            // ただし WlanDisconnect を呼んだときはコールバックが来たり来なかったりする模様
            if( m_AsyncTask.GetType() == AsyncTask::Type::Disconnect )
            {
                m_AsyncTask.Complete(nn::ResultSuccess());
            }

            NotifyLinkDisconnected(ResultDisconnected());   // TODO
            break;

        default:
            break;
        }
    }
}

bool WirelessInterface::IsLinkUp() const NN_NOEXCEPT
{
    MIB_IF_ROW2 mibIfRow2;
    ::SecureZeroMemory((PVOID)&mibIfRow2, sizeof(mibIfRow2));
    mibIfRow2.InterfaceIndex = m_Index;
    DWORD error = ::GetIfEntry2(&mibIfRow2);

    return error == NO_ERROR &&
           mibIfRow2.Type == IF_TYPE_IEEE80211 &&
           mibIfRow2.MediaConnectState == ::MediaConnectStateConnected;
}

nn::Result WirelessInterface::GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutScanChannels);

    *pOutCount = InvalidScanChannelCount;

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::PutToSleepImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::WakeUpImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}
}
}

