﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterfaceManagerBase.horizon.h>

#include <nn/nifm/detail/util/nifm_FwdbgSettings.h>

#include <nn/util/util_LockGuard.h>

namespace nn { namespace eth {
#include <nn/eth/eth_Result.public.h>
}}

#include <new>
#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

void EthernetInterfaceManagerBase::InterfaceGroupEventCallback::ExecuteImpl() NN_NOEXCEPT
{
    m_pEthernetInterfaceManagerBase->Renew();
}

void EthernetInterfaceManagerBase::StandByTimerEventCallback::ExecuteImpl() NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("No USB Ethernet adaptors are found.\n");
    m_pEthernetInterfaceManagerBase->GiveUpEthernetInterface();
}

void EthernetInterfaceManagerBase::GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutMacAddresses != nullptr || inCount > 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    uint32_t outCount;
    nn::Result result = m_InterfaceGroupHandler.GetInterfaceCount(&outCount);

    if( result.IsFailure() )
    {
        *pOutCount = 0;
        return;
    }

    nn::eth::InterfaceList interfaceList;
    result = m_InterfaceGroupHandler.GetInterfaceList(&interfaceList);

    if( result.IsFailure() )
    {
        *pOutCount = 0;
        return;
    }

    // GetInterfaceCount と GetInterfaceList のあいだに増減があった場合を考慮
    outCount = std::max(outCount, interfaceList.adapterCount);

    if( sizeof(outCount) >= sizeof(int) && outCount >= static_cast<uint32_t>(INT_MAX) )
    {
        NN_DETAIL_NIFM_WARN("*pOutCount is too large and capped to INT_MAX.\n");
        outCount = static_cast<uint32_t>(INT_MAX);
    }

    if( sizeof(interfaceList.adapterCount) >= sizeof(int) && interfaceList.adapterCount >= static_cast<uint32_t>(INT_MAX) )
    {
        NN_DETAIL_NIFM_WARN("interfaceList.adapterCount is too large and capped to INT_MAX.\n");
        interfaceList.adapterCount = static_cast<uint32_t>(INT_MAX);
    }

    *pOutCount = outCount;

    const int adapterCount = interfaceList.adapterCount;
    const int copyCount = std::min(adapterCount, inCount);

    for( int i = 0; i < copyCount; ++i )
    {
        std::memcpy(pOutMacAddresses[i].data, interfaceList.adapterInfo[i].interfaceMacAddress, MacAddress::Size);
    }

    const int zeroSetCount = std::min(*pOutCount, inCount);

    // InterfaceGroupHandler がハンドリングしない分には無効な MAC アドレスを入れておく
    for( int i = copyCount; i < zeroSetCount; ++i )
    {
        std::memset(pOutMacAddresses[i].data, 0, MacAddress::Size); // TODO: InvalidMacAddress
    }
}

EthernetInterfaceManagerBase::EthernetInterfaceManagerBase() NN_NOEXCEPT
    : m_InterfaceGroupHandler(),
      m_InterfaceGroupHandlerInitializer(m_InterfaceGroupHandler, nn::os::EventClearMode_AutoClear),
      m_InterfaceGroupEvent(*m_InterfaceGroupHandler.GetSystemEventPointer()),
      m_InterfaceGroupEventCallback(this),
      m_InterfaceGroupEventHandler(m_InterfaceGroupEvent),
      m_IsStandingBy(false),
      m_StandByTimerEvent(nn::os::EventClearMode_AutoClear),
      m_StandByTimerEventCallback(this),
      m_StandByTimerEventHandler(m_StandByTimerEvent),
      m_IsEnabled(true)
{
    m_InterfaceGroupEventHandler.Register(&m_InterfaceGroupEventCallback);
    m_StandByTimerEventHandler.Register(&m_StandByTimerEventCallback);

    m_PluggingEventHandler.Add(m_InterfaceGroupEventHandler);
    m_PluggingEventHandler.Add(m_StandByTimerEventHandler);

    StandByForEthernetInterface();
}

EthernetInterfaceManagerBase::~EthernetInterfaceManagerBase() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_EthernetInterfaceList.size() == 0);     // 派生クラスが RemoveAll していれば通る（していなくても通ることもある）
}

bool EthernetInterfaceManagerBase::Renew() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::eth::InterfaceList interfaceList;
    m_InterfaceGroupHandler.GetInterfaceList(&interfaceList);

    NN_DETAIL_NIFM_INFO("Count of USB-Ether interfaces: %u\n", interfaceList.adapterCount);

    // nifm が認識していて eth が認識していないインターフェースを削除
    for (auto&& itr = m_EthernetInterfaceList.begin(); itr != m_EthernetInterfaceList.end(); /* erase の都合でインクリメントは個別におこなう */)
    {
        auto& ethernetInterface = *itr;

        if (!ethernetInterface.IsPresent())
        {
            RemoveCallback(ethernetInterface);
            m_EthernetInterfaceList.erase(itr++);

            ethernetInterface.~EthernetInterface();
            m_EthernetInterfaceHeap.Free(&ethernetInterface);

            continue;   // erase 時に itr をインクリメントしているのでそのまま次のイテレーションへ
        }

        ++itr;
    }

    bool isNewInterfaceAdded = false;

    // eth が認識していて nifm が認識していないインターフェースを追加
    for (uint32_t i = 0; i < interfaceList.adapterCount; ++i)
    {
        nn::eth::InterfaceInfo& interfaceInfo = interfaceList.adapterInfo[i];   // TODO: const にしたいが eth のインターフェースの都合でできていない

        bool isRegistered = false;

        for (const auto& ethernetInterface : m_EthernetInterfaceList)
        {
            MacAddress macAddress;
            ethernetInterface.GetMacAddress(&macAddress);

            if (std::memcmp(&macAddress, interfaceInfo.interfaceMacAddress, nn::nifm::MacAddress::Size) == 0)
            {
                isRegistered = true;
                break;
            }
        }

        if (!isRegistered)
        {
            auto pEthernetInterface = m_EthernetInterfaceHeap.Create();

            if (pEthernetInterface == nullptr)
            {
                NN_DETAIL_NIFM_WARN("USB Ether adapter full.\n");
                break;  // メモリが足りない場合は、これ以上まわしても同じ
            }
            else
            {
                auto result = pEthernetInterface->Initialize(&m_InterfaceGroupHandler, &interfaceInfo);

                if (result.IsSuccess())
                {
                    pEthernetInterface->SetEnabled(m_IsEnabled);
                    m_EthernetInterfaceList.push_back(*pEthernetInterface);
                    AppendCallback(*pEthernetInterface);

                    isNewInterfaceAdded = true;
                }
                else
                {
                    NN_DETAIL_NIFM_WARN("EthernetInterface::Initialize() failed (%03d-%04d).\n", result.GetModule(), result.GetDescription());

                    m_EthernetInterfaceHeap.Delete(pEthernetInterface);
                }
            }
        }
    }

    if (isNewInterfaceAdded)
    {
        GiveUpEthernetInterface();
    }

    return isNewInterfaceAdded;
}

EthernetInterface* EthernetInterfaceManagerBase::Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    // この関数の呼び出し中はインターフェースの増減が発生する操作をしない約束なので、ロックは不要

    for (auto&& ethernetInterface : m_EthernetInterfaceList)
    {
        auto pEthernetInterface = ethernetInterface.Find(pMultiWaitHolder);

        if (pEthernetInterface != nullptr)
        {
            return pEthernetInterface;
        }
    }

    return nullptr;
}

nn::Result EthernetInterfaceManagerBase::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_IsEnabled = isEnabled;

    for (auto&& ethernetInterface : m_EthernetInterfaceList)
    {
        NN_RESULT_DO(ethernetInterface.SetEnabled(isEnabled));
    }

    NN_RESULT_SUCCESS;
}

bool EthernetInterfaceManagerBase::IsEnabled() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_IsEnabled;
}

void EthernetInterfaceManagerBase::RemoveAll() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for( auto&& itr = m_EthernetInterfaceList.begin(); itr != m_EthernetInterfaceList.end(); /* インクリメントは erase 時におこなう */ )
    {
        auto&& ethernetInterface = *itr;

        RemoveCallback(ethernetInterface);
        m_EthernetInterfaceList.erase(itr++);

        m_EthernetInterfaceHeap.Delete(&ethernetInterface);
    }
}

bool EthernetInterfaceManagerBase::IsStandingBy() NN_NOEXCEPT
{
    return m_IsStandingBy;
}

nn::Result EthernetInterfaceManagerBase::PutToSleep() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterfaceManagerBase::WakeUp() NN_NOEXCEPT
{
    StandByForEthernetInterface();

    NN_RESULT_SUCCESS;
}

void EthernetInterfaceManagerBase::StandByForEthernetInterface() NN_NOEXCEPT
{
    NN_DETAIL_NIFM_TRACE("EthernetInterfaceManagerBase::StandByForEthernetInterface(): %lld [ms]\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

    m_StandByTimerEvent.Stop();
    m_StandByTimerEvent.Clear();
    m_IsStandingBy = true;
    m_StandByTimerEvent.StartOneShot(nn::TimeSpan::FromMilliSeconds(FwdbgSettings::GetSingleton().GetEthernetAdapterStandbyTime().GetMilliSeconds()));
}

void EthernetInterfaceManagerBase::GiveUpEthernetInterface() NN_NOEXCEPT
{
    NN_DETAIL_NIFM_TRACE("EthernetInterfaceManagerBase::GiveUpEthernetInterface(): %lld [ms]\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

    m_StandByTimerEvent.Stop();
    m_StandByTimerEvent.Clear();
    m_IsStandingBy = false;
}

}
}
}
