﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/connectionConfirmation/nifm_ConnectionTestClient.horizon.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <algorithm>

namespace nn
{
namespace nifm
{
namespace detail
{

const char ConnectionTestClient::ConnTestUrl[] = "http://ctest.cdn.nintendo.net/";


ConnectionTestClient::HttpHeaderReaderForConnTest::HttpHeaderReaderForConnTest() NN_NOEXCEPT
    : m_IsOrganizationValidated(false),
      m_pLocation(nullptr)
{
}

ConnectionTestClient::HttpHeaderReaderForConnTest::~HttpHeaderReaderForConnTest() NN_NOEXCEPT
{
}

void ConnectionTestClient::HttpHeaderReaderForConnTest::Parse(const char* pInLine, size_t size) NN_NOEXCEPT
{
    // pLine が null 文字で終端されていないことに注意

    int count = static_cast<int>(size) + 1;

#if defined(NN_DETAIL_NIFM_LOG_ENABLED)
    {
        char buffer[1024];
        nn::util::Strlcpy(buffer, pInLine, std::min(count, 1024));
        NN_DETAIL_NIFM_TRACE_V2("%s", buffer);
    }
#endif

    Field httpHeaderFiled;
    if (ParseResponseHeaderLine(&httpHeaderFiled, pInLine, count))
    {
        if ((nn::util::Strncmp(httpHeaderFiled.name, "X-Organization", sizeof("X-Organization")) == 0) &&
            (nn::util::Strncmp(httpHeaderFiled.value, "Nintendo", sizeof("Nintendo")) == 0))
        {
            m_IsOrganizationValidated = true;
        }
    }
}

void ConnectionTestClient::HttpHeaderReaderForConnTest::Clear() NN_NOEXCEPT
{
    m_IsOrganizationValidated = false;
    m_pLocation = nullptr;
}

size_t ConnectionTestClient::ResponseHeaderLineCallbackStatic(char* pInLine, size_t size, size_t nmemb, void* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    return reinterpret_cast<ConnectionTestClient*>(pContext)->ResponseHeaderLineCallback(pInLine, size, nmemb);
}

size_t ConnectionTestClient::ResponseBodyWriteCallbackStatic(void *pData, size_t size, size_t nmemb, void *pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    return reinterpret_cast<ConnectionTestClient*>(pContext)->ResponseBodyWriteCallback(pData, size, nmemb);
}

size_t ConnectionTestClient::ResponseHeaderLineCallback(char* pInLine, size_t size, size_t nmemb) NN_NOEXCEPT
{
    size_t count = size * nmemb;
    m_HttpHeaderReader.Parse(pInLine, count);

    if (m_pHttpResponse == nullptr)
    {
        return 0;
    }

    if (count > 0)
    {
        auto writeSize = m_pHttpResponse->Add(pInLine, count);

        NN_UNUSED(writeSize);
        NN_DETAIL_NIFM_INFO_V2("[%d/%d] %s", writeSize, m_pHttpResponse->GetLength(), &m_pHttpResponse->GetPointer()[m_pHttpResponse->GetLength() - writeSize]);
    }

    return count;
}

size_t ConnectionTestClient::ResponseBodyWriteCallback(void *pData, size_t size, size_t nmemb) NN_NOEXCEPT
{
    size_t count = size * nmemb;

    if (m_pHttpResponse == nullptr)
    {
        return 0;
    }

    if (count > 0)
    {
        auto writeSize = m_pHttpResponse->Add(reinterpret_cast<char*>(pData), count);

        NN_UNUSED(writeSize);
        NN_DETAIL_NIFM_INFO_V2("[%d/%d] %s", writeSize, m_pHttpResponse->GetLength(), &m_pHttpResponse->GetPointer()[m_pHttpResponse->GetLength() - writeSize]);

        // バッファに書き込めない応答が返った場合，（SSID リスト経由の接続なら）先頭 Size 分だけでも Nas に投げる必要があるのでエラーを返さない
    }

    return count;
}

ConnectionTestClient::ConnectionTestClient() NN_NOEXCEPT
{
    SetSequencePeriod(SequencePeriod::ConnectionTest);
}

ConnectionTestClient::~ConnectionTestClient() NN_NOEXCEPT
{
}

nn::Result ConnectionTestClient::ConfirmInternetConnection(const char* pUrl) NN_NOEXCEPT
{
    Clear();
    m_HttpHeaderReader.Clear();

    // 共通の設定
    SetCurlCommonOpt();

    // 独自の設定
    SetRequestUrl(pUrl);
    SetRequesetMethod(HttpRequestMethod::Get);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_DISABLE_CACHE, true);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_FOLLOWLOCATION, false);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_HEADERFUNCTION, ResponseHeaderLineCallbackStatic);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_HEADERDATA, this);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_WRITEFUNCTION, ResponseBodyWriteCallbackStatic);
    NIFM_DETAIL_CURL_EASY_SETOPT(m_pCurl, CURLOPT_WRITEDATA, this);

    NN_RESULT_DO(Connect());

    switch (m_StatusCode)
    {
    case 200:
        NN_RESULT_THROW_UNLESS(m_HttpHeaderReader.m_IsOrganizationValidated, ResultConnectionTestResponseInvalidOrganization());
        break;

    case 301:
        curl_easy_getinfo(m_pCurl, CURLINFO_REDIRECT_URL, &m_HttpHeaderReader.m_pLocation);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusMovedPermanently());
        break;
    case 302:
        curl_easy_getinfo(m_pCurl, CURLINFO_REDIRECT_URL, &m_HttpHeaderReader.m_pLocation);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusFound());
        break;
    case 303:
        curl_easy_getinfo(m_pCurl, CURLINFO_REDIRECT_URL, &m_HttpHeaderReader.m_pLocation);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusSeeOther());
        break;
    case 307:
        curl_easy_getinfo(m_pCurl, CURLINFO_REDIRECT_URL, &m_HttpHeaderReader.m_pLocation);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusTemporaryRedirect());
        break;
    case 308:
        curl_easy_getinfo(m_pCurl, CURLINFO_REDIRECT_URL, &m_HttpHeaderReader.m_pLocation);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusPermanentRedirect());
        break;

    case 407:
        NN_RESULT_THROW(ResultConnectionTestHttpStatusProxyAuthenticationRequired());     // TODO

    default:
        NN_DETAIL_NIFM_WARN_V3("ConfirmInternetConnection: StatusCode=%d\n", m_StatusCode);
        NN_RESULT_THROW(ResultConnectionTestHttpStatusError());     // TODO
    }

    NN_RESULT_SUCCESS;
}

const char* ConnectionTestClient::GetRedirectUrlPointer() const NN_NOEXCEPT
{
    return m_HttpHeaderReader.m_pLocation;
}

}
}
}
