﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>

#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>

#include <locale>
#include <cstring>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
    bool IsWpaSupported(const WpaInfo& wpaInfo, Encryption *pOutEncryption, Encryption* pOutGroupEncryption)
    {
        if (wpaInfo.groupDataCipherSuite.Test<CipherSuiteFlag::AesCcmp>())
        {
            *pOutGroupEncryption = Encryption_Aes;
            if (wpaInfo.cipherSuites.Test<CipherSuiteFlag::AesCcmp>())
            {
                *pOutEncryption = Encryption_Aes;
                return true;
            }
            else if (wpaInfo.cipherSuites.Test<CipherSuiteFlag::Tkip>())
            {
                *pOutEncryption = Encryption_Tkip;
                return true;
            }
        }
        else if (wpaInfo.groupDataCipherSuite.Test<CipherSuiteFlag::Tkip>())
        {
            *pOutGroupEncryption = Encryption_Tkip;
            if (wpaInfo.cipherSuites.Test<CipherSuiteFlag::AesCcmp>())
            {
                *pOutEncryption = Encryption_Aes;
                return true;
            }
            else if (wpaInfo.cipherSuites.Test<CipherSuiteFlag::Tkip>())
            {
                *pOutEncryption = Encryption_Tkip;
                return true;
            }
        }

        return false;
    }
}

WirelessAccessPoint::WirelessAccessPoint(WirelessInterface* pWirelessInterface) NN_NOEXCEPT
    : AccessPointBase(pWirelessInterface)
{
    std::memset(&m_AccessPointExt, 0, sizeof(m_AccessPointExt));
}

WirelessAccessPoint::~WirelessAccessPoint() NN_NOEXCEPT
{
}

nn::Result WirelessAccessPoint::ConnectImpl(
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    NN_RESULT_THROW(static_cast<WirelessInterface*>(m_pNetworkInterfaceBase)->ConnectImpl(*this, networkProfile, aggregatedRequest));
}

nn::Result WirelessAccessPoint::DisconnectImpl() const NN_NOEXCEPT
{
    NN_RESULT_THROW(static_cast<WirelessInterface*>(m_pNetworkInterfaceBase)->DisconnectImpl(*this));
}

nn::Result WirelessAccessPoint::ReleaseImpl() const NN_NOEXCEPT
{
    NN_RESULT_THROW(static_cast<WirelessInterface*>(m_pNetworkInterfaceBase)->ReleaseImpl(*this));
}

void WirelessAccessPoint::UpdateAccessPoint(
    const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    static_cast<WirelessInterface*>(m_pNetworkInterfaceBase)->UpdateAccessPoint(this, networkProfile);
}

int WirelessAccessPoint::GetPriority() const NN_NOEXCEPT
{
    // RSSI の範囲は高々 [-128,0)
    return (LinkLevel_Count - m_AccessPointExt.linkLevel) * 1000 - m_AccessPointExt.rssi;
}

nn::Result WirelessAccessPoint::TryExport(AccessPointData* pOutAccessPointData) const NN_NOEXCEPT
{
    GetSsid(&pOutAccessPointData->ssid);
    GetBssid(&pOutAccessPointData->bssid);
    pOutAccessPointData->rssi = GetRssi();
    pOutAccessPointData->linkLevel = GetLinkLevel();
    pOutAccessPointData->channel = GetChannel();
    pOutAccessPointData->authentication = GetAuthentication();
    pOutAccessPointData->encryption = GetEncryption();
    pOutAccessPointData->groupEncryption = GetGroupEncryption();
    pOutAccessPointData->isSupported = m_AccessPointExt.isSupported;

    // 非サポートの認証・暗号方式の AP は isSupported で判定可能

    NN_RESULT_SUCCESS;
}

AccessPointBase* WirelessAccessPoint::CopyTo(void* p, size_t size) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(p);
    NN_SDK_ASSERT_GREATER_EQUAL(size, sizeof(*this));
    NN_UNUSED(size);

    return new(p) WirelessAccessPoint(*this);
}

size_t WirelessAccessPoint::GetSize() const NN_NOEXCEPT
{
    return sizeof(*this);
}

bool WirelessAccessPoint::IsPossiblyAvailableImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT
{
    NN_UNUSED(aggregatedRequest);

    // TODO: フラグを見たりとかいろいろ

    if (networkProfile.GetNetworkInterfaceType() == NetworkInterfaceType_Ieee80211)
    {
        Ssid profileSsid;
        networkProfile.GetWirelessSetting().GetSsid(&profileSsid);
        bool nonBroadcast = networkProfile.GetWirelessSetting().IsNonBroadcastEnabled();

        if (!IsPossiblyEqual(profileSsid, nonBroadcast))
        {
            return false;
        }

        Authentication profAuth = networkProfile.GetWirelessSetting().GetAuthentication();
        Encryption profEnc = networkProfile.GetWirelessSetting().GetEncryption();

        if (!IsSupported(profAuth, profEnc))
        {
            return false;
        }
    }

    return true;
}

bool WirelessAccessPoint::GetTopSecurity(Authentication *pOutAuthentiCation, Encryption* pOutEncryption, Encryption* pOutGroupEncryption) const NN_NOEXCEPT
{
    // WPA2
    if (IsWpaSupported(m_AccessPointExt.wpa2, pOutEncryption, pOutGroupEncryption))
    {
        if (m_AccessPointExt.wpa2.akmSuites.Test<AkmSuiteFlag::Eap>())
        {
            *pOutAuthentiCation = Authentication_Wpa2;
            return true;
        }
        else if (m_AccessPointExt.wpa2.akmSuites.Test<AkmSuiteFlag::Psk>())
        {
            *pOutAuthentiCation = Authentication_Wpa2Psk;
            return true;
        }
        return false;
    }

    // WPA
    if (IsWpaSupported(m_AccessPointExt.wpa, pOutEncryption, pOutGroupEncryption))
    {
        if (m_AccessPointExt.wpa.akmSuites.Test<AkmSuiteFlag::Eap>())
        {
            *pOutAuthentiCation = Authentication_Wpa;
            return true;
        }
        else if (m_AccessPointExt.wpa.akmSuites.Test<AkmSuiteFlag::Psk>())
        {
            *pOutAuthentiCation = Authentication_WpaPsk;
            return true;
        }
        return false;
    }

    // WEP
    if (m_AccessPointExt.privacy.Test<PrivacyFlag::Shared_Wep>())
    {
        *pOutAuthentiCation = Authentication_Shared;
        *pOutEncryption = Encryption_Wep;
        *pOutGroupEncryption = Encryption_Wep;
        return true;
    }
    else if (m_AccessPointExt.privacy.Test<PrivacyFlag::Open_Wep>())
    {
        *pOutAuthentiCation = Authentication_Open;
        *pOutEncryption = Encryption_Wep;
        *pOutGroupEncryption = Encryption_Wep;
        return true;
    }
    else if (m_AccessPointExt.privacy.Test<PrivacyFlag::Unknown_Wep>())
    {
        *pOutAuthentiCation = Authentication_Unknown;
        *pOutEncryption = Encryption_Wep;
        *pOutGroupEncryption = Encryption_Wep;
        return true;
    }
    else if (m_AccessPointExt.privacy.Test<PrivacyFlag::Open_None>())
    {
        *pOutAuthentiCation = Authentication_Open;
        *pOutEncryption = Encryption_None;
        *pOutGroupEncryption = Encryption_None;
        return true;
    }

    return false;
}

Ssid* WirelessAccessPoint::GetSsid(Ssid *pOutSsid) const NN_NOEXCEPT
{
    std::memcpy( pOutSsid->hex, m_AccessPointExt.ssid.hex, m_AccessPointExt.ssid.length );
    pOutSsid->length = m_AccessPointExt.ssid.length;
    return pOutSsid;
}

MacAddress* WirelessAccessPoint::GetBssid(MacAddress *pOutBssid) const NN_NOEXCEPT
{
    std::memcpy( pOutBssid, &m_AccessPointExt.bssid, MacAddress::Size );
    return pOutBssid;
}

int32_t WirelessAccessPoint::GetRssi() const NN_NOEXCEPT
{
    return m_AccessPointExt.rssi;
}

LinkLevel WirelessAccessPoint::GetLinkLevel() const NN_NOEXCEPT
{
    return m_AccessPointExt.linkLevel;
}

int16_t WirelessAccessPoint::GetChannel() const NN_NOEXCEPT
{
    return m_AccessPointExt.channel;
}

Authentication WirelessAccessPoint::GetAuthentication() const NN_NOEXCEPT
{
    // 互換性のために最も強い認証方式を返す

    Authentication a;
    Encryption e;
    Encryption g;

    if (GetTopSecurity(&a, &e, &g))
    {
        return a;
    }

    return Authentication_Invalid;
}

Encryption WirelessAccessPoint::GetEncryption() const NN_NOEXCEPT
{
    // 互換性のために最も強い暗号方式を返す

    Authentication a;
    Encryption e;
    Encryption g;

    if (GetTopSecurity(&a, &e, &g))
    {
        return e;
    }

    return Encryption_Invalid;
}

Encryption WirelessAccessPoint::GetGroupEncryption() const NN_NOEXCEPT
{
    // 互換性のために最も強い暗号方式を返す

    Authentication a;
    Encryption e;
    Encryption g;

    if (GetTopSecurity(&a, &e, &g))
    {
        return g;
    }

    return Encryption_Invalid;
}

void WirelessAccessPoint::SetSsid(const Ssid &rSsid) NN_NOEXCEPT
{
    memcpy(m_AccessPointExt.ssid.hex, rSsid.hex, rSsid.length );
    m_AccessPointExt.ssid.length = rSsid.length;
}

void WirelessAccessPoint::SetSsid(const unsigned char* pSsid, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT( length <= Ssid::HexSize );
    memcpy(m_AccessPointExt.ssid.hex, pSsid, length );
    m_AccessPointExt.ssid.length = static_cast<uint8_t>(length);
}

void WirelessAccessPoint::SetBssid(const MacAddress &rBssid) NN_NOEXCEPT
{
    memcpy(m_AccessPointExt.bssid.data, rBssid.data, MacAddress::Size );
}

void WirelessAccessPoint::SetRssi(int32_t signalStrength) NN_NOEXCEPT
{
    m_AccessPointExt.rssi = signalStrength;
}

void WirelessAccessPoint::SetLinkLevel(LinkLevel linkLevel) NN_NOEXCEPT
{
    m_AccessPointExt.linkLevel = linkLevel;
}

void WirelessAccessPoint::SetChannel(int16_t channel) NN_NOEXCEPT
{
    m_AccessPointExt.channel = channel;
}

void WirelessAccessPoint::SetWpaAuthenticationAndEncryption(const Authentication authentication, const Encryption encryption, const Encryption groupEncryption) NN_NOEXCEPT
{
    WpaInfo* pWpaInfo = nullptr;

    switch(authentication)
    {
    case Authentication_WpaPsk:
        m_AccessPointExt.wpa.akmSuites.Set<AkmSuiteFlag::Psk>(true);
        pWpaInfo = &m_AccessPointExt.wpa;
        break;
    case Authentication_Wpa2Psk:
        m_AccessPointExt.wpa2.akmSuites.Set<AkmSuiteFlag::Psk>(true);
        pWpaInfo = &m_AccessPointExt.wpa2;
        break;
    case Authentication_Wpa:
        m_AccessPointExt.wpa.akmSuites.Set<AkmSuiteFlag::Eap>(true);
        pWpaInfo = &m_AccessPointExt.wpa;
        break;
    case Authentication_Wpa2:
        m_AccessPointExt.wpa2.akmSuites.Set<AkmSuiteFlag::Eap>(true);
        pWpaInfo = &m_AccessPointExt.wpa2;
        break;
    default:
        NN_DETAIL_NIFM_INFO("Unsupported authentication algorithm(%d).\n", authentication);
        return;
    }

    NN_SDK_ASSERT_NOT_EQUAL(pWpaInfo, nullptr);

    switch (encryption)
    {
    case Encryption_Tkip:
        pWpaInfo->cipherSuites.Set<CipherSuiteFlag::Tkip>(true);
        switch (groupEncryption)
        {
        case Encryption_Tkip:
            pWpaInfo->groupDataCipherSuite.Set<CipherSuiteFlag::Tkip>(true);
            break;
        case Encryption_Aes:
            pWpaInfo->groupDataCipherSuite.Set<CipherSuiteFlag::AesCcmp>(true);
            break;
        default:
            break;
        }
        break;
    case Encryption_Aes:
        pWpaInfo->cipherSuites.Set<CipherSuiteFlag::AesCcmp>(true);
        switch (groupEncryption)
        {
        case Encryption_Tkip:
            pWpaInfo->groupDataCipherSuite.Set<CipherSuiteFlag::Tkip>(true);
            break;
        case Encryption_Aes:
            pWpaInfo->groupDataCipherSuite.Set<CipherSuiteFlag::AesCcmp>(true);
            break;
        default:
            break;
        }
        break;
    default:
        break;
    }

    // WPA-PSK/WPA2-PSK(AES) のみをサポート
    m_AccessPointExt.isSupported = m_AccessPointExt.isSupported ? true :
        pWpaInfo->akmSuites.Test<AkmSuiteFlag::Psk>() && pWpaInfo->cipherSuites.Test<CipherSuiteFlag::AesCcmp>();
}

void WirelessAccessPoint::SetWepAuthenticationAndEncryption(const Authentication authentication, const Encryption encryption) NN_NOEXCEPT
{
    switch (authentication)
    {
    case Authentication_Open:
        switch (encryption)
        {
        case Encryption_Wep:
            m_AccessPointExt.privacy.Set<PrivacyFlag::Open_Wep>(true);
            break;
        case Encryption_None:
            m_AccessPointExt.privacy.Set<PrivacyFlag::Open_None>(true);
            break;
        default:
            NN_DETAIL_NIFM_WARN_V3("Invalid cipher suite (%d)/Open.\n", encryption);
            break;
        }
        break;
    case Authentication_Shared:
        switch (encryption)
        {
        case Encryption_Wep:
            m_AccessPointExt.privacy.Set<PrivacyFlag::Shared_Wep>(true);
            break;
        default:
            NN_DETAIL_NIFM_WARN_V3("Invalid cipher suite (%d)/Shared.\n", encryption);
            break;
        }
        break;
    case Authentication_Unknown:
        switch (encryption)
        {
        case Encryption_Wep:
            m_AccessPointExt.privacy.Set<PrivacyFlag::Unknown_Wep>(true);
            break;
        default:
            NN_DETAIL_NIFM_WARN_V3("Invalid cipher suite (%d)/Unknown.\n", encryption);
            break;
        }
        break;
    default:
        NN_DETAIL_NIFM_WARN_V3("Unsupported authentication algorithm(%d).\n", authentication);
        break;
    }

    // WEP 形式はすべてサポート
    m_AccessPointExt.isSupported = m_AccessPointExt.isSupported ? true : m_AccessPointExt.privacy.IsAnyOn();
}

bool WirelessAccessPoint::operator==(const AccessPointBase& rh) const NN_NOEXCEPT
{
    return rh == *this;
}

bool WirelessAccessPoint::operator==(const WirelessAccessPoint& rh) const NN_NOEXCEPT
{
return IsEqual(rh.m_AccessPointExt.ssid) &&
IsEqual(rh.m_AccessPointExt.bssid);
}

bool WirelessAccessPoint::IsSupported(Authentication authentication, Encryption encryption) const NN_NOEXCEPT
{
    switch (authentication)
    {
    case Authentication_Open:
        switch (encryption)
        {
        case nn::nifm::Encryption_Wep:
            return m_AccessPointExt.privacy.Test<PrivacyFlag::Open_Wep>() ||
                m_AccessPointExt.privacy.Test<PrivacyFlag::Unknown_Wep>();
        case nn::nifm::Encryption_None:
            return m_AccessPointExt.privacy.Test<PrivacyFlag::Open_None>();
        default:
            return false;
        }
        break;
    case Authentication_Shared:
        switch (encryption)
        {
        case nn::nifm::Encryption_Wep:
            return m_AccessPointExt.privacy.Test<PrivacyFlag::Shared_Wep>() ||
                m_AccessPointExt.privacy.Test<PrivacyFlag::Unknown_Wep>();
        default:
            return false;
        }
        break;
    case Authentication_WpaPsk:
        switch (encryption)
        {
        case nn::nifm::Encryption_Aes:
            return m_AccessPointExt.wpa.akmSuites.Test<AkmSuiteFlag::Psk>() &&
                m_AccessPointExt.wpa.cipherSuites.Test<CipherSuiteFlag::AesCcmp>();
        default:
            return false;
        }
        break;
    case Authentication_Wpa2Psk:
        switch (encryption)
        {
        case nn::nifm::Encryption_Aes:
            return m_AccessPointExt.wpa2.akmSuites.Test<AkmSuiteFlag::Psk>() &&
                m_AccessPointExt.wpa2.cipherSuites.Test<CipherSuiteFlag::AesCcmp>();
        default:
            return false;
        }
        break;
    default:
        return false;
    }
}

bool WirelessAccessPoint::IsSupported() const NN_NOEXCEPT
{
    return m_AccessPointExt.isSupported;
}

bool WirelessAccessPoint::IsStealth() const NN_NOEXCEPT
{
    return IsStealth(m_AccessPointExt.ssid);
}

bool WirelessAccessPoint::IsStealth(const Ssid& ssid) NN_NOEXCEPT
{
    // CTR と同様の判定
    return (ssid.length == 0 ||
            ssid.hex[0] == '\0' ||
            (ssid.length == 1 && ssid.hex[0] == ' '));
}

bool WirelessAccessPoint::IsEqual(const Ssid& ssid) const NN_NOEXCEPT
{
    return m_AccessPointExt.ssid.length == ssid.length &&
           memcmp(m_AccessPointExt.ssid.hex, ssid.hex, ssid.length ) == 0;
}

bool WirelessAccessPoint::IsPossiblyEqual(const Ssid& ssid, bool nonBroadcast) const NN_NOEXCEPT
{
    if( IsEqual(ssid) )
    {
        return true; // SSID が完全一致
    }

    if( IsStealth() && nonBroadcast )
    {
        return true; // SSID が一致するかもしれない（ステルス AP かつ、ステルス対応の接続設定）
    }

    return false;
}

bool WirelessAccessPoint::IsEqual(const MacAddress& bssid) const NN_NOEXCEPT
{
    return memcmp(m_AccessPointExt.bssid.data, bssid.data, MacAddress::Size ) == 0;
}

}
}
}
