﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointBase.h>


namespace nn
{
namespace nifm
{
namespace detail
{

const nn::TimeSpan AccessPointListBase::DefaultLifetime = nn::TimeSpan::FromDays(10 * 365);     // 通常の利用で失効しないくらい長い時間

AccessPointListBase::AccessPointListBase(HeapBase& heap, nn::TimeSpan lifetime) NN_NOEXCEPT
    : m_Heap(heap),
      m_Lifetime(lifetime)
{
    // ここで m_Heap を操作してはならない
}

AccessPointListBase::~AccessPointListBase() NN_NOEXCEPT
{
    // ここで m_Heap を操作してはならない
}

void AccessPointListBase::Clear() NN_NOEXCEPT
{
    AccessPointIntrusiveList::clear();
    m_Heap.Clear();
}

void AccessPointListBase::Commit() NN_NOEXCEPT
{
    splice(begin(), m_TempList);
}

bool AccessPointListBase::DeleteOldest() NN_NOEXCEPT
{
    auto itr = rbegin();

    if (itr == rend())
    {
        return false;
    }

    auto p = &*itr;

    erase((++itr).base());
    m_Heap.Delete(p);

    return true;
}

void AccessPointListBase::ExportTo(AccessPointListBase* pAccessPointList) const NN_NOEXCEPT
{
    auto expirationTime = nn::os::GetSystemTick().ToTimeSpan() - m_Lifetime;

    for (const auto& accessPoint : *this)
    {
        if (accessPoint.GetTimestamp() >= expirationTime)
        {
            size_t size = accessPoint.GetSize();
            void* p = pAccessPointList->m_Heap.Allocate(size);

            if (p == nullptr)
            {
                break;  // TODO: error
            }

            static_cast<AccessPointIntrusiveList*>(pAccessPointList)->push_back(*accessPoint.CopyTo(p, size));
        }
    }
}

void AccessPointListBase::AddToListImpl(AccessPointBase& accessPoint, AccessPointIntrusiveList& list) NN_NOEXCEPT
{
    list.push_back(accessPoint);
}

void AccessPointListBase::UpdateListImpl(AccessPointBase& accessPoint, AccessPointIntrusiveList& list) NN_NOEXCEPT
{
    // まずオブジェクトをコンストラクトしないと重複判定ができないので、
    // 追加時に重複を除去する

    // 新しいものは先頭に足されるので、更新対象となる要素は先頭側から調べた方がはやい
    for (auto itr = begin(); itr != end(); ++itr)
    {
        if (*itr == accessPoint)
        {
            auto p = &*itr;
            itr = --erase(itr);
            m_Heap.Delete(p);

            break;
        }
    }

    AddToListImpl(accessPoint, list);
}

void AccessPointListBase::push_back(AccessPointBase& value) NN_NOEXCEPT
{
    NN_UNUSED(value);
    NN_SDK_ASSERT(false, "push_back is obsoleted.\n");
}

void AccessPointListBase::push_front(AccessPointBase& value) NN_NOEXCEPT
{
    NN_UNUSED(value);
    NN_SDK_ASSERT(false, "push_front is obsoleted.\n");
}

void AccessPointListBase::pop_back() NN_NOEXCEPT
{
    NN_SDK_ASSERT(false, "pop_back is obsoleted.\n");
}

void AccessPointListBase::pop_front() NN_NOEXCEPT
{
    NN_SDK_ASSERT(false, "pop_front is obsoleted.\n");
}

void AccessPointListBase::clear() NN_NOEXCEPT
{
    NN_SDK_ASSERT(false, "clear is obsoleted.\n");
}

}
}
}
