﻿/*--------------------------------------------------------------------------------*
  Project: CrossRoad
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cerrno>
#include <cstdlib>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <climits>      // for RSIZE_MAX
#include <Windows.h>    // for HANDLE, CONDITION_VARIABLE, ....
#endif

#include <nn/nn_SdkLog.h>

namespace nn { namespace ngc { namespace detail {

//
// Error Type
//
typedef int errno_t;  // TR 24731-1

#if defined(NN_BUILD_CONFIG_ABI_LP64)
// if long, unsinged long are 64 bits long
typedef int64_t ngc_long_compatible_t;
typedef uint64_t ngc_ulong_compatible_t;
#else
typedef int32_t ngc_long_compatible_t;
typedef uint32_t ngc_ulong_compatible_t;
#endif

#ifndef RSIZE_MAX
# if defined(NN_BUILD_CONFIG_CPU_X64) || defined(NN_OS_CPU_ARM_AARCH64)
#  define RSIZE_MAX 0x7FFFFFFFFFFFFFFFLL    // NOLINT(preprocessor/const)
# else
#  define RSIZE_MAX 0x7FFFFFFFL             // NOLINT(preprocessor/const)
# endif
#endif

// Scatter/Gather buffer
struct FdIoVec {
    void* iovBase;
    size_t iovLen;
};

// for (0..count) { swapendian(p[count]); }
nn::ngc::detail::errno_t SwapEndian16(uint16_t* p, size_t count);
// for (0..count) { swapendian(p[count]); }
nn::ngc::detail::errno_t SwapEndian32(uint32_t* p, size_t count);
// for (0..count) { swapendian(p[count]); }
nn::ngc::detail::errno_t SwapEndian64(uint64_t* p, size_t count);

// memcpy_s
static inline
nn::ngc::detail::errno_t MemCpy(void* __restrict pDst, size_t dstSize,
                                const void* __restrict pSrc, size_t srcSize)
{
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC) || defined(__STDC_LIB_EXT1__)
    return memcpy_s(pDst, dstSize, pSrc, srcSize);
#else
    if (dstSize < srcSize)
    {
        memset(pDst, 0, dstSize);
        return ERANGE;
    }
    memcpy(pDst, pSrc, srcSize);
    return 0;
#endif
}

// memmove_s
static inline
nn::ngc::detail::errno_t MemMove(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize)
{
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC) || defined(__STDC_LIB_EXT1__)
    return memmove_s(pDst, dstSize, pSrc, srcSize);
#else
    if (dstSize < srcSize) return ERANGE;
    memmove(pDst, pSrc, srcSize);
    return 0;
#endif
}

static inline
nn::ngc::detail::errno_t MemSet(void* buf, int ch, size_t n)
{
    memset(buf, ch, n);
    return 0;
}

// popcnt
static int PopCnt16(uint16_t x);
static int PopCnt32(uint32_t x);
static int PopCnt64(uint64_t x);

extern const unsigned char PopCntArray[];
static inline int PopCnt32(uint32_t x)
{
    return PopCntArray[(x >> 24) & 0xFF] + PopCntArray[(x >> 16) & 0xFF] +
        PopCntArray[(x >> 8) & 0xFF] + PopCntArray[(x) & 0xFF];
}
static inline int PopCnt64(uint64_t x)
{
    return PopCntArray[(x >> 56) & 0xFF] + PopCntArray[(x >> 48) & 0xFF] +
        PopCntArray[(x >> 40) & 0xFF] + PopCntArray[(x >> 32) & 0xFF] +
        PopCntArray[(x >> 24) & 0xFF] + PopCntArray[(x >> 16) & 0xFF] +
        PopCntArray[(x >> 8) & 0xFF] + PopCntArray[(x) & 0xFF];
}
static inline int PopCnt16(uint16_t x)
{
    return PopCntArray[(x >> 8) & 0xFF] + PopCntArray[(x) & 0xFF];
}

// clz
// CountLeadingZero32(0x80000000) -> 0, CountLeadingZero32(1) -> 31
static int CountLeadingZero32(uint32_t x);
// CountTrailingZero32(0x80000000) -> 31, CountTrailingZero32(1) -> 0
static int CountTrailingZero32(uint32_t x);
// CountLeadingZero64(INT64_MIN) -> 0, CountLeadingZero64(1) -> 63
static int CountLeadingZero64(uint64_t x);
// CountTrailingZero64(INT64_MIN) -> 63, CountTrailingZero64(1) -> 0
static int CountTrailingZero64(uint64_t x);

#if defined(_MSC_VER)
static inline int CountLeadingZero32(uint32_t x)
{
    DWORD cnt;
    return _BitScanReverse(&cnt, x) ? (int)(31 - cnt) : 32;  // NOLINT
    // return (int)(__lzcnt(x));  // needs haswell+
}
static inline int CountTrailingZero32(uint32_t x)
{
    DWORD cnt;
    return _BitScanForward(&cnt, x) ? cnt : 32;
}
static inline int CountLeadingZero64(uint64_t x)
{
#if defined(NN_BUILD_CONFIG_CPU_X64) || defined(NN_OS_CPU_ARM_AARCH64)
    DWORD cnt;
    return _BitScanReverse64(&cnt, x) ? (int)(63 - cnt) : 64;  // NOLINT
    // return (int)(__lzcnt64(x));  // needs haswell+
#else
    DWORD cnt;
    DWORD dw = (DWORD)(x >> 32);
    if (_BitScanReverse(&cnt, dw))
    {
        return (int)(31 - cnt);  // NOLINT
    }
    else
    {
        dw = (DWORD)(x);
        return _BitScanReverse(&cnt, dw) ?
            (int)(63 - cnt) : 64;  // NOLINT
    }
#endif
}
static inline int CountTrailingZero64(uint64_t x)
{
#if defined(NN_BUILD_CONFIG_CPU_X64) || defined(NN_OS_CPU_ARM_AARCH64)
    DWORD cnt;
    return _BitScanForward64(&cnt, x) ? cnt : 64;
#else
    DWORD cnt;
    DWORD dw = (DWORD)(x);
    if (_BitScanForward(&cnt, dw))
    {
        return (int)(cnt);  // NOLINT
    }
    else
    {
        dw = (DWORD)(x >> 32);
        return _BitScanForward(&cnt, dw) ?
            (int)(32 + cnt) : 64;  // NOLINT
    }
#endif
}
#else
static inline int CountLeadingZero32(uint32_t x)
{
    return x != 0 ? __builtin_clz(x) : 32;
}
static inline int CountTrailingZero32(uint32_t x)
{
    return x != 0 ? __builtin_ctz(x) : 32;
}
static inline int CountLeadingZero64(uint64_t x)
{
    return x != 0 ? __builtin_clzll(x) : 64;
}
static inline int CountTrailingZero64(uint64_t x)
{
    return x != 0 ? __builtin_ctzll(x) : 64;
}
#endif

// nn::ngc::detail の動的確保の一元管理
// 動的確保は nn::ngc ライブラリでは使われず、 nn::ngc 用のシステムデータ生成ツールにおいてのみ利用される
static inline void* AllocateMemoryNgc(size_t size)
{
    return std::malloc(size);
}

static inline void FreeMemoryNgc(void* ptr)
{
    std::free(ptr);
}

static inline void* ReallocateMemoryNgc(void* ptr, size_t newSize)
{
    if(newSize == 0)
    {
        std::free(ptr);
        return NULL;
    }
    void* p = std::realloc(ptr, newSize);
    return p;
}

}}} // nn::ngc::detail
