﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <utility>

#include <nn/nn_Abort.h>

#include "./ngc_Sbv.h"
#include "./ngc_SbvPrivate.h"
#include "./ngc_BinaryReader.h"
#include "./ngc_BinaryWriter.h"
#include "./ngc_ReallocVec.h"

namespace nn { namespace ngc { namespace detail {

static const int LvA = 256; // 一つ目の補助データの区切り
static const int LvB = 32;  // 二つ目の補助データの区切り

using ::nn::ngc::detail::SelectPos;

BitVector32::~BitVector32() NN_NOEXCEPT
{
    this->Reset();
}

bool BitVector32::Init(uint32_t size) NN_NOEXCEPT
{
    if (size == 0)
    {
        m_BitVec.Reset();
        m_Size = 0;
        return true;
    }
    // ブロック数を算出 ( 1 ブロック 32bit )
    uint32_t blk = GetBlockCount<uint32_t>(size);
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(uint32_t) * blk);
        m_BitVec.Reset((pTmp) ? new (pTmp)uint32_t[blk] : NULL, m_pAllocator);
    }
    else
    {
        m_BitVec.Reset(new (std::nothrow) uint32_t[blk]);
    }
    if (!m_BitVec)
    {
        return false;
    }
    m_Size = size;
    memset(m_BitVec.Get(), 0, blk * sizeof(m_BitVec[0]));
    return true;
}

bool BitVector32::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void BitVector32::ReleaseAllocator() NN_NOEXCEPT
{
    this->Reset();
}

void BitVector32::Reset() NN_NOEXCEPT
{
    m_BitVec.Reset();
    m_Size = 0;
    m_pAllocator = NULL;
}

size_t BitVector32::MemSize() const NN_NOEXCEPT
{
    size_t bv = sizeof(m_BitVec[0]) * GetBlockCount<uint32_t>(m_Size);
    return bv + sizeof(m_BitVec) + sizeof(m_Size) + sizeof(m_pAllocator);
}

bool BitVector32::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!w->Write(m_Size))
    {
        return false;
    }
    if (m_Size > 0 && !w->WriteArray(m_BitVec.Get(), GetBlockCount<uint32_t>(m_Size)))
    {
        return false;
    }
    return true;
}

bool BitVector32::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!r->Read(&m_Size))
    {
        return false;
    }
    if (m_Size == 0)
    {
        m_BitVec.Reset();
        return true;
    }
    size_t blk = GetBlockCount<uint32_t>(m_Size);
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(uint32_t) * blk);
        m_BitVec.Reset(pTmp ? new (pTmp)uint32_t[blk] : NULL, m_pAllocator);
    }
    else
    {
        m_BitVec.Reset(new (std::nothrow) uint32_t[blk]);
    }
    if (!m_BitVec)
    {
        return false;
    }
    return blk == r->ReadArray(m_BitVec.Get(), blk);
}

SbvRank::~SbvRank() NN_NOEXCEPT
{
    this->Reset();
}

bool SbvRank::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void SbvRank::ReleaseAllocator() NN_NOEXCEPT
{
    this->Reset();
    m_pAllocator = NULL;
}

// LV-Aの配列には累積のpopCountが256ビット毎に入っている
// LV-Bの配列には256ビットブロック内のpopCountが32ビット毎に入っている。
// O(1)でランク計算が可能
//
// NOTE: LvAとLvBをインターリーブしてキャッシュミスを減らすという
// やり方もあるが、実験してみたところ遅かった。
// 計算量が若干増えたことがキャッシュのメリットを上回ったようだ。
bool SbvRank::Build(const uint32_t* pBv, uint32_t size) NN_NOEXCEPT
{
    this->Reset();  // for ビットベクトルをいじって計算しなおしたくなった場合
    if (m_pAllocator)
    {
        void* pLvA = m_pAllocator->Allocate(sizeof(uint32_t) * (size / LvA + 1));
        m_LvA.Reset(pLvA ? new (pLvA)uint32_t[size / LvA + 1] : NULL, m_pAllocator);
        if (!m_LvA)
        {
            return false;
        }

        void* pLvB = m_pAllocator->Allocate(sizeof(uint8_t) * (size / LvB + 1));
        m_LvB.Reset(pLvB ? new (pLvB)uint8_t[size / LvB + 1] : NULL, m_pAllocator);
    }
    else
    {
        m_LvA.Reset(new (std::nothrow) uint32_t[size / LvA + 1]);
        if (!m_LvA)
        {
            return false;
        }

        m_LvB.Reset(new (std::nothrow) uint8_t[size / LvB + 1]);
    }
    if (!m_LvB)
    {
        m_LvA.Reset();
        return false;
    }
    memset(m_LvA.Get(), 0, (size / LvA + 1) * sizeof(m_LvA[0]));
    memset(m_LvB.Get(), 0, (size / LvB + 1) * sizeof(m_LvB[0]));
    uint32_t numBlock = GetBlockCount<uint32_t>(size);
    ::nn::ngc::detail::BuildRankDictionary(m_LvA.Get(), m_LvB.Get(), numBlock, pBv);
    return true;
}

void SbvRank::Reset() NN_NOEXCEPT
{
    m_LvA.Reset();
    m_LvB.Reset();
}

size_t SbvRank::MemSize(uint32_t size) const NN_NOEXCEPT
{
    return sizeof(m_LvA) + sizeof(m_LvB) + sizeof(m_LvA[0]) * (size / LvA + 1) +
           sizeof(m_LvB[0]) * (size / LvB + 1) + sizeof(m_pAllocator);
}

bool SbvRank::Export(BinaryWriter* w, uint32_t size) const NN_NOEXCEPT
{
    if (size > 0)
    {
        if (!w->WriteArray(m_LvA.Get(), (size / LvA + 1)))
        {
            return false;
        }
        if (!w->WriteArray(m_LvB.Get(), (size / LvB + 1)))
        {
            return false;
        }
    }
    return true;
}

bool SbvRank::Import(BinaryReader* r, uint32_t size) NN_NOEXCEPT
{
    if (size == 0) return true;
    Resetter<SbvRank> rst(this);
    size_t cntLvA = size / LvA + 1;
    size_t cntLvB = size / LvB + 1;
    if (m_pAllocator)
    {
        void* pLvA = m_pAllocator->Allocate(sizeof(uint32_t) * cntLvA);
        m_LvA.Reset(pLvA ? new (pLvA)uint32_t[cntLvA] : NULL, m_pAllocator);
        if (!m_LvA)
        {
            return false;
        }

        void* pLvB = m_pAllocator->Allocate(sizeof(uint8_t) * cntLvB);
        m_LvB.Reset(pLvB ? new (pLvB)uint8_t[cntLvB] : NULL, m_pAllocator);
    }
    else
    {
        m_LvA.Reset(new (std::nothrow) uint32_t[cntLvA]);
        if (!m_LvA)
        {
            return false;
        }

        m_LvB.Reset(new (std::nothrow) uint8_t[cntLvB]);
    }
    if (!m_LvB)
    {
        return false;
    }
    return cntLvA == r->ReadArray(m_LvA.Get(), cntLvA) &&
           cntLvB == r->ReadArray(m_LvB.Get(), cntLvB) && rst.Ok();
}

SbvSelect::~SbvSelect() NN_NOEXCEPT
{
    this->Reset();
}

bool SbvSelect::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    m_ClumpArray.SetAllocator(m_pAllocator);
    m_Blocks.SetAllocator(m_pAllocator);
    m_Asserted.SetAllocator(m_pAllocator);
    m_RankForBlocks.SetAllocator(m_pAllocator);
    m_RankForAsserted.SetAllocator(m_pAllocator);
    return true;
}

void SbvSelect::ReleaseAllocator() NN_NOEXCEPT
{
    m_ClumpArray.ReleaseAllocator();
    m_Blocks.ReleaseAllocator();
    m_Asserted.ReleaseAllocator();
    m_RankForBlocks.ReleaseAllocator();
    m_RankForAsserted.ReleaseAllocator();
    this->Reset();
    m_pAllocator = NULL;
}

bool SbvSelect::Build(const uint32_t* pBv, uint32_t size) NN_NOEXCEPT
{
    this->Reset();  // for ビットベクトルをいじって計算しなおしたくなった場合
    this->SetAllocator(m_pAllocator);
    // qsize: 1になっているビットの数
    // rsize: 0でないブロックの数
    // clump_array_size: 0ブロックの塊の数
    // を設定する。
    uint32_t qsize = 0;
    uint32_t rsize = 0;
    uint32_t clump_array_size = 0;
    bool prev_blk_notzero = false;
    size_t numBlk = GetBlockCount<uint32_t>(size);
    for (size_t i = 0; i < numBlk; ++i)
    {
        uint32_t blk = pBv[i];
        if (blk != 0)
        {
            // 0ブロックの終わりかどうかチェック
            if (!prev_blk_notzero)
            {
                ++clump_array_size;
            }
            prev_blk_notzero = true;
            ++rsize;
        }
        else
        {
            prev_blk_notzero = false;
        }
        qsize += PopCnt32(blk);
    }
    if (!prev_blk_notzero)
    {
        ++clump_array_size;
    }

    // ビットベクトル領域のアロケート
    if (!m_Blocks.Init(rsize) || !m_Asserted.Init(qsize) || !m_ClumpArray.Init(clump_array_size))
    {
        Reset();
        return false;
    }

    // ビットベクトルの設定
    // R: 0でないブロックの性質を記述。keyでkey番目の非0ブロックを指す
    //    R[key] = 1のときそこが0でないブロックの開始地点
    //    つまり、R[key]=1ならばkey番目の非0のブロックは0でないブロックの開始地点
    //    rank(key)はkeyで指定されるブロックまでに存在する0ブロックの塊の数になる。

    // Q: 1であるビットの性質を記述。keyでkey番目の1であるビットを指す
    //    Q[key] = 1のときそこがブロック内の最初の1であるビットである。
    //    rank(key)は1であるビットが存在するブロックの数になる。

    // clumpArray: 0ブロック個数の累積値が格納されている。
    //             idxはN個目の0ブロックの塊であることを示している。
    prev_blk_notzero = false;
    uint32_t rcount = 0;
    uint32_t qcount = 0;
    uint32_t ccount = 0;
    unsigned int cumClump = 0;
    for (size_t i = 0; i < numBlk; ++i)
    {
        uint32_t blk = pBv[i];
        if (blk != 0)
        {
            if (!prev_blk_notzero)
            {
                // 0でないブロックの開始地点
                // rcount目のブロックは0でないブロックの開始地点である。
                m_Blocks.TurnOn(rcount);
                // ここまでの0ブロックの総数を記録
                m_ClumpArray[ccount++] = cumClump;
            }
            ++rcount;
            // qcount目の1ビットはブロック内の最初の1bitであるからm_Assertedに記録しておく。
            m_Asserted.TurnOn(qcount);
            qcount += PopCnt32(blk);
            prev_blk_notzero = true;
        }
        else
        {
            ++cumClump;
            prev_blk_notzero = false;
        }
    }
    if (!m_RankForBlocks.Build(m_Blocks.Data(), m_Blocks.Size()) || !m_RankForAsserted.Build(m_Asserted.Data(), m_Asserted.Size()))
    {
        Reset();
        return false;
    }
    return true;
}

int32_t SbvSelect::Select(const Set& set, uint32_t nth) const NN_NOEXCEPT
{
    const uint32_t* bv = set.GetBitVector();

    if (nth >= m_Asserted.Size())
    {
        return -1;
    }
    // nth番目の1までに存在する1を含むブロックの数 - 1
    uint32_t blockRank = m_RankForAsserted.Rank1(m_Asserted.Data(), nth) - 1;

    // nth番目の1ビットがあるブロックまでに
    // いくつの0ブロックチャンクがあったか？
    uint32_t clumpNum = m_RankForBlocks.Rank1(m_Blocks.Data(), blockRank) - 1;

    // nth番目の1がどのブロックなのかが決まる。
    // bv[selectP]内にnth番目の1ビットがある。
    uint32_t selectP = m_ClumpArray[clumpNum] + blockRank;

    // bv[selectP]以前にある1ビットの数を引く
    // bv[selectP]内のremain番目の1ビットの位置が答え
    uint32_t remain = nth;
    if (selectP > 0)
    {
        remain -= set.Rank1(selectP * SbvBlockSize - 1);
    }
    // ポジションを求める
    // selectPosでブロック内のremain個目の1の場所(remain=0-31)
    uint32_t bit = bv[selectP];
    int32_t rval = selectP * SbvBlockSize + SelectPos(bit, remain);
    return rval;
}

void SbvSelect::Reset() NN_NOEXCEPT
{
    m_ClumpArray.Reset();
    m_Blocks.Reset();
    m_Asserted.Reset();
    m_RankForBlocks.Reset();
    m_RankForAsserted.Reset();
}

size_t SbvSelect::MemSize() const NN_NOEXCEPT
{
    size_t sizeC = m_ClumpArray.MemSize();
    size_t sizeR = m_Blocks.MemSize();
    size_t sizeQ = m_Asserted.MemSize();
    size_t rankR = m_RankForBlocks.MemSize(m_Blocks.Size());
    size_t rankQ = m_RankForAsserted.MemSize(m_Asserted.Size());
    return rankR + rankQ + sizeC + sizeR + sizeQ + sizeof(m_pAllocator);
}

bool SbvSelect::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_ClumpArray.Export(w))
    {
        return false;
    }
    if (!m_Blocks.Export(w))
    {
        return false;
    }
    if (!m_Asserted.Export(w))
    {
        return false;
    }
    if (!m_RankForBlocks.Export(w, m_Blocks.Size()))
    {
        return false;
    }
    if (!m_RankForAsserted.Export(w, m_Asserted.Size()))
    {
        return false;
    }
    return true;
}

bool SbvSelect::Import(BinaryReader* r) NN_NOEXCEPT
{
    Resetter<SbvSelect> rst(this);
    if (!m_ClumpArray.Import(r))
    {
        return false;
    }
    if (!m_Blocks.Import(r) || !m_Asserted.Import(r))
    {
        return false;
    }
    return m_RankForBlocks.Import(r, m_Blocks.Size()) && m_RankForAsserted.Import(r, m_Asserted.Size()) && rst.Ok();
}

// ビットベクトルと Rank を持った構造体
struct Set::SetPrivate
{
    detail::BitVector32 bvec;
    detail::SbvRank rank;
};

struct Set::SetPrivateDeleter
{
    void operator()(SetPrivate* ptr)
    {
        if (ptr)
        {
            NN_ABORT_UNLESS(m_pAllocator);
            ptr->bvec.ReleaseAllocator();
            ptr->rank.ReleaseAllocator();
            ptr->bvec.~BitVector32();
            ptr->rank.~SbvRank();
            m_pAllocator->Free(ptr);
        }
    }
    NN_IMPLICIT SetPrivateDeleter(WorkBufAllocator* pAllocator) : m_pAllocator(pAllocator) {}
private:
    WorkBufAllocator* m_pAllocator;
};

Set::~Set() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call Set::ReleaseAllocator() before calling Set::~Set()\n");
        NN_ABORT();

    }
    else if(m_Prv)
    {
        delete m_Prv;
    }
}

bool Set::Init(uint32_t bvSize) NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(SetPrivate));
        std::unique_ptr<SetPrivate, SetPrivateDeleter> impl(pTmp ? new (pTmp) SetPrivate() : NULL, m_pAllocator);
        if (!impl)
        {
            return false;
        }
        impl->bvec.SetAllocator(m_pAllocator);
        impl->bvec.SetAllocator(m_pAllocator);
        if (!impl->bvec.Init(bvSize))
        {
            return false;
        }
        if (m_Prv)
        {
            m_Prv->bvec.ReleaseAllocator();
            m_Prv->rank.ReleaseAllocator();
            m_Prv->bvec.~BitVector32();
            m_Prv->rank.~SbvRank();
            m_pAllocator->Free(m_Prv);
        }
        m_Prv = impl.release();
    }
    else
    {
        std::unique_ptr<SetPrivate> impl(new (std::nothrow) SetPrivate());
        if (!impl)
        {
            return false;
        }
        if (!impl->bvec.Init(bvSize))
        {
            return false;
        }
        if (m_Prv)
        {
            delete m_Prv;
        }
        m_Prv = impl.release();
    }
    return true;
}

bool Set::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void Set::ReleaseAllocator() NN_NOEXCEPT
{
    if (!m_pAllocator)
    {
        return;
    }
    if (m_Prv) {
        m_Prv->bvec.ReleaseAllocator();
        m_Prv->rank.ReleaseAllocator();
        m_Prv->bvec.~BitVector32();
        m_Prv->rank.~SbvRank();
        m_pAllocator->Free(m_Prv);
        m_Prv = NULL;
    }
    m_pAllocator = NULL;
}

bool Set::TurnOn(uint32_t idx) NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    return m_Prv->bvec.TurnOn(idx);
}

bool Set::TurnOff(uint32_t idx) NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    return m_Prv->bvec.TurnOff(idx);
}


bool Set::Build() NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    detail::SbvRank rank(m_pAllocator);
    if (!rank.Build(m_Prv->bvec.Data(), m_Prv->bvec.Size()))
    {
        rank.ReleaseAllocator();
        return false;
    }
    rank.Swap(m_Prv->rank);
    return true;
}

bool Set::Has(uint32_t idx) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    return m_Prv->bvec[idx];
}

uint32_t Set::Rank1(uint32_t idx) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    // The values beyond the bit-vector are all 0.
    uint32_t pos = idx < m_Prv->bvec.Size() ? idx : m_Prv->bvec.Size() - 1;
    return m_Prv->rank.Rank1(m_Prv->bvec.Data(), pos);
}

int32_t Set::Select1(uint32_t nth) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return -1;
    }
    return m_Prv->rank.Select1(m_Prv->bvec.Data(), m_Prv->bvec.Size(), nth);
}

int32_t Set::Select0(uint32_t nth) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return -1;
    }
    return m_Prv->rank.Select0(m_Prv->bvec.Data(), m_Prv->bvec.Size(), nth);
}

size_t Set::MemSize() const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return 0;
    }
    size_t rankSize = m_Prv->rank.MemSize(m_Prv->bvec.Size());
    size_t bvSize = m_Prv->bvec.MemSize();
    return rankSize + bvSize + sizeof(m_pAllocator);
}

uint32_t Set::GetBitVectorSize() const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return 0;
    }
    return m_Prv->bvec.Size();
}

const uint32_t* Set::GetBitVector() const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return NULL;
    }
    return m_Prv->bvec.Data();
}

void Set::Reset() NN_NOEXCEPT
{
    if (m_Prv)
    {
        m_Prv->bvec.Reset();
        m_Prv->rank.Reset();
    }
}

bool Set::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        // 初期化がされていない場合は空のデータをエクスポート
        detail::BitVector32 bvec;
        detail::SbvRank rank;
        if (!bvec.Export(w))
        {
            return false;
        }
        if (!rank.Export(w, 0))
        {
            return false;
        }
    }
    else
    {
        if (!m_Prv->bvec.Export(w))
        {
            return false;
        }
        if (!m_Prv->rank.Export(w, m_Prv->bvec.Size()))
        {
            return false;
        }
    }
    return true;
}

bool Set::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!m_Prv)
    {
        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(SetPrivate));
            m_Prv = pTmp ? new (pTmp) SetPrivate() : NULL;
            if (m_Prv)
            {
                m_Prv->bvec.SetAllocator(m_pAllocator);
                m_Prv->rank.SetAllocator(m_pAllocator);
            }
        }
        else
        {
            m_Prv = new (std::nothrow) SetPrivate();
        }
        if (!m_Prv)
        {
            return false;
        }
    }
    detail::Resetter<Set> rst(this);
    if (!m_Prv->bvec.Import(r) || !m_Prv->rank.Import(r, m_Prv->bvec.Size()))
    {
        return false;
    }
    rst.Ok();

    return true;
}

struct Sbv::SbvPrivate
{
    detail::SbvSelect select1;
};

struct Sbv::SbvPrivateDeleter
{
    void operator()(SbvPrivate* ptr)
    {
        if (ptr)
        {
            NN_ABORT_UNLESS(m_pAllocator);
            ptr->select1.ReleaseAllocator();
            ptr->select1.~SbvSelect();
            m_pAllocator->Free(ptr);
        }
    }
    NN_IMPLICIT SbvPrivateDeleter(WorkBufAllocator* pAllocator) : m_pAllocator(pAllocator) {}
private:
    WorkBufAllocator* m_pAllocator;
};

Sbv::~Sbv() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call Sbv::ReleaseAllocator() before calling Sbv::~Sbv()\n");
        NN_ABORT();
    }
    else if(m_Prv)
    {
        delete m_Prv;
    }
    m_pAllocator = NULL;
}

bool Sbv::Init(uint32_t bvSize) NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(SbvPrivate));
        std::unique_ptr<SbvPrivate, SbvPrivateDeleter> impl(pTmp ? new (pTmp) SbvPrivate() : NULL, m_pAllocator);
        if (!impl)
        {
            return false;
        }
        Set s;
        s.SetAllocator(m_pAllocator);
        if (!s.Init(bvSize))
        {
            return false;
        }
        if (m_Prv)
        {
            m_Prv->select1.ReleaseAllocator();
            m_Prv->select1.~SbvSelect();
            m_pAllocator->Free(m_Prv);
        }
        m_Prv = impl.release();
        m_Set.Swap(s);
        s.ReleaseAllocator();
    }
    else
    {
        std::unique_ptr<SbvPrivate> impl(new (std::nothrow) SbvPrivate());
        if (!impl)
        {
            return false;
        }
        Set s;
        if (!s.Init(bvSize))
        {
            return false;
        }
        delete m_Prv;
        m_Prv = impl.release();
        m_Set.Swap(s);
    }
    return true;
}

bool Sbv::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    m_Set.SetAllocator(m_pAllocator);
    return true;
}

void Sbv::ReleaseAllocator() NN_NOEXCEPT
{
    if (!m_pAllocator)
    {
        return;
    }
    else if(m_Prv)
    {
        m_Prv->select1.ReleaseAllocator();
        m_Prv->select1.~SbvSelect();
        m_pAllocator->Free(m_Prv);
        m_Prv = NULL;
    }
    m_Set.ReleaseAllocator();
    m_pAllocator = NULL;
}

size_t Sbv::MemSize() const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return 0;
    }
    size_t set = m_Set.MemSize();
    size_t select1 = m_Prv->select1.MemSize();
    return set + select1 + sizeof(m_pAllocator);
}

bool Sbv::Build() NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return false;
    }
    detail::SbvSelect mySelect(m_pAllocator);
    if (!m_Set.Build() || !mySelect.Build(GetBitVector(), GetBitVectorSize()))
    {
        mySelect.ReleaseAllocator();
        return false;
    }
    mySelect.Swap(m_Prv->select1);
    return true;
}

int32_t Sbv::Select1(uint32_t nth) const NN_NOEXCEPT
{
    if (!m_Prv)
    {
        return -1;
    }
    return m_Prv->select1.Select(m_Set, nth);
}

void Sbv::Reset() NN_NOEXCEPT
{
    if (m_Prv)
    {
        m_Prv->select1.Reset();
    }
    m_Set.Reset();
}

bool Sbv::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_Set.Export(w))
    {
        return false;
    }
    if (m_Prv)
    {
        if (!m_Prv->select1.Export(w))
        {
            return false;
        }
    }
    else
    {
        detail::SbvSelect sel;
        if (!sel.Export(w))
        {
            return false;
        }
    }
    return true;
}

bool Sbv::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!m_Prv)
    {
        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(SbvPrivate));
            m_Prv = pTmp ? new (pTmp) SbvPrivate() : NULL;
            if (m_Prv)
            {
                m_Prv->select1.SetAllocator(m_pAllocator);
            }
        }
        else
        {
            m_Prv = new (std::nothrow) SbvPrivate();
        }
        if (!m_Prv) return false;
    }
    if (!m_Set.Import(r) || !m_Prv->select1.Import(r))
    {
        this->Reset();
        return false;
    }
    return true;
}

SparseSet::~SparseSet() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call SparseSet::ReleaseAllocator() before calling SparseSet::~SparseSet()\n");
        NN_ABORT();
    }
    this->Reset();
}

SparseSet::SparseSet(SparseSet&& rhs) NN_NOEXCEPT :
    m_BitvectorSize(rhs.m_BitvectorSize), m_LowestIdx(rhs.m_LowestIdx),
    m_HighestIdx(rhs.m_HighestIdx), m_NumOnBits(rhs.m_NumOnBits),
    m_Width(rhs.m_Width), m_LowerBitsLength(rhs.m_LowerBitsLength),
    m_pLowerBits(rhs.m_pLowerBits), m_Sbv(std::move(rhs.m_Sbv))
{
    rhs.m_pLowerBits = NULL;
    m_pAllocator = rhs.m_pAllocator;
    rhs.m_pAllocator = NULL;
}

SparseSet& SparseSet::operator=(SparseSet&& rhs) NN_NOEXCEPT
{
    m_BitvectorSize = rhs.m_BitvectorSize;
    m_LowestIdx = rhs.m_LowestIdx;
    m_HighestIdx = rhs.m_HighestIdx;
    m_NumOnBits = rhs.m_NumOnBits;
    m_Width = rhs.m_Width;
    m_LowerBitsLength = rhs.m_LowerBitsLength;
    m_pLowerBits = rhs.m_pLowerBits;
    rhs.m_pLowerBits = NULL;
    m_Sbv = std::move(rhs.m_Sbv);
    m_pAllocator = rhs.m_pAllocator;
    rhs.m_pAllocator = NULL;
    return *this;
}

bool SparseSet::Init(uint64_t bvSize) NN_NOEXCEPT
{
    if (bvSize == 0)
    {
        return false;
    }
    // 32bit+5bit分のインデックスを保持可能。それ以上はNG
    if (bvSize >= 0x2000000000ULL)
    {
        return false;
    }
    // テンポラリとして普通のビットベクトルを作る。
    // TurnOnでビットを立てていく。
    m_BitvectorSize = bvSize;

    // このsizeはとても大きくなるが、現在のところ対策せず。
    uint64_t size = (bvSize + detail::SbvBlockSize - 1) / detail::SbvBlockSize;
    size_t AllocSize;
#if !defined(NN_BUILD_CONFIG_ADDRESS_64)
    // 32bitだと結局newできないから
    if (size >= 0x100000000ULL)
    {
        return false;
    }
    AllocSize = static_cast<size_t>(size);
#else
    AllocSize = size;
#endif
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(uint32_t) * AllocSize);
        m_pLowerBits = pTmp ? new (pTmp) uint32_t[AllocSize] : NULL;
    }
    else
    {
        m_pLowerBits = new (std::nothrow) uint32_t[AllocSize];
    }
    if (!m_pLowerBits)
    {
        return false;
    }
    memset(m_pLowerBits, 0, AllocSize * sizeof(m_pLowerBits[0]));
    return true;
}

bool SparseSet::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    m_Sbv.SetAllocator(m_pAllocator);
    return true;
}

void SparseSet::ReleaseAllocator() NN_NOEXCEPT
{
    if (m_pLowerBits)
    {
        m_pAllocator->Free(m_pLowerBits);
        m_pLowerBits = NULL;
    }
    m_Sbv.ReleaseAllocator();
    m_pAllocator = NULL;
}

bool SparseSet::TurnOn(uint64_t idx) NN_NOEXCEPT
{
    // Build前に実行される。
    if (idx >= m_BitvectorSize)
    {
        return false;
    }
    size_t n = static_cast<size_t>(idx / detail::SbvBlockSize);
    size_t r = static_cast<size_t>(idx % detail::SbvBlockSize);
    if (!(m_pLowerBits[n] & (1 << r)))
    {
        m_pLowerBits[n] |= 1 << r;
        ++m_NumOnBits;  // ONビットの数でエンコード方法が変わるので記録
        if (idx < m_LowestIdx)
        {
            m_LowestIdx = idx;
        }
        if (idx > m_HighestIdx)
        {
            m_HighestIdx = idx;
        }
    }
    return true;
}

inline unsigned int GetBitWidth(uint64_t value) NN_NOEXCEPT
{
    return static_cast<unsigned int>(64 - CountLeadingZero64(value));
    // unsigned int r = 0;
    // while (value >> r) ++r;
    // return r;
}

bool SparseSet::Build() NN_NOEXCEPT
{
    // InitされていないかBuild後なら失敗
    if (!m_pLowerBits)
    {
        return false;
    }
    if (m_NumOnBits == 0)
    {
        m_Width = 0;
        if (m_pAllocator)
        {
            m_pAllocator->Free(m_pLowerBits);
        }
        else
        {
            delete[] m_pLowerBits;
        }
        m_pLowerBits = NULL;
        m_LowerBitsLength = 0;
        return true;
    }
    // ONになっているビットのインデックスを上下2つに分割して、
    // 上の方の桁は別のビットマップのSelect操作の結果として取り出せるように保存、
    // 残りの桁はそのままの形で並べていく。
    // 上の方の桁を表現するビットベクトルの大きさはONとなっているビット数にフィット
    // するようになっているのでデータ量を節約できる。
    PodManager<uint32_t> bv(m_pLowerBits, m_pAllocator);
    m_pLowerBits = NULL;
    uint64_t bvSize = m_BitvectorSize;

    // ONビットの上位ビットを切り出すための幅はONビットの数に依存する。
    unsigned int high_width = GetBitWidth(static_cast<int>(m_NumOnBits * 1.44f));
    // 下位ビットの幅
    int width = GetBitWidth(bvSize) - high_width;
    if (width >= 32) {
        high_width += width - 31;
        width = 31;
    }
    m_Width = width;
    m_Sbv.Init(1 << (high_width + 1));  // +1するのは必ずSbvに格納できるようにするため

    ReallocVec<uint32_t> B(m_pAllocator);
    uint64_t idx = 0;
    const unsigned int mask = (1 << width) - 1;
    unsigned int data = 0;
    const size_t bvElemBitCount = sizeof(bv.Get()[0]) * 8;
    const uint64_t bmax = (bvSize + bvElemBitCount - 1) / bvElemBitCount;
    for (uint64_t b = 0; b < bmax; ++b)
    {
        if (bv.Get()[b] == 0) continue;
        uint64_t i = b * bvElemBitCount;
        for (size_t j = 0; j < bvElemBitCount; ++j, ++i)
        {
            if (detail::GetBit(bv.Get(), i))
            {
                // m_Sbvのサイズはm_NumOnBits + (1 << high_width)以上なので入る
                uint32_t sbv_on = static_cast<uint32_t>(i >> width);

                // 上位ビットの値が被った場合でも大丈夫なように加工する。
                // sbv_onは単調増加。idxと合成することでsbv_onが被っても大丈夫になる。
                // idxは最大でもm_NumOnBitsの数にしかならないのでm_Sbvに格納できる
                m_Sbv.TurnOn(static_cast<uint32_t>(sbv_on + idx));

                // valに下位ビットの値を取り出す
                unsigned int val = static_cast<unsigned int>(i & mask);
                uint64_t pos = idx * width;  // ビット列が格納される場所(posビット目)
                unsigned int pos_offset = static_cast<unsigned int>(pos % SbvBlockSize);
                data |= val << pos_offset;
                if (pos_offset + width >= SbvBlockSize)
                {
                    if (!B.PushBack(data))
                    {
                        return false;
                    }
                    // valはwidth幅のデータしかない
                    data = (val >> (SbvBlockSize - pos_offset));
                }
                ++idx;
            }
        }
    }
    B.PushBack(data);  // 下位ビットデータの書き出し忘れの防止

    if (!m_Sbv.Build())
    {
        goto SS_BUILD_FAIL;
    }
    if (!B.Empty())
    {
        if (m_pAllocator)
        {
            void* bitsPtr = m_pAllocator->Allocate(sizeof(uint32_t) * B.Size());
            m_pLowerBits = bitsPtr ? new (bitsPtr) uint32_t[B.Size()] : NULL;
        }
        else
        {
            m_pLowerBits = new (std::nothrow) uint32_t[B.Size()];
        }
        if (!m_pLowerBits)
        {
            goto SS_BUILD_FAIL;
        }
        MemCpy(m_pLowerBits, B.Size() * sizeof(m_pLowerBits[0]), &*B.Begin(), B.Size() * sizeof(m_pLowerBits[0]));
        m_LowerBitsLength = static_cast<uint32_t>(B.Size());
    }
    else
    {
        if (m_pAllocator)
        {
            m_pAllocator->Free(m_pLowerBits);
        }
        else
        {
            delete[] m_pLowerBits;
        }
        m_pLowerBits = NULL;
        m_LowerBitsLength = 0;
    }
    if (m_pAllocator)
    {
        B.ReleaseAllocator();
    }
    return true;
SS_BUILD_FAIL:
    NN_SDK_ASSERT(!m_pLowerBits);
    m_pLowerBits = bv.Release();
    m_Width = 0;
    if (m_pAllocator)
    {
        B.ReleaseAllocator();
    }
    return false;
}   // NOLINT(impl/function_size)

void SparseSet::Reset() NN_NOEXCEPT
{
    m_NumOnBits = m_LowerBitsLength = 0;
    m_BitvectorSize = m_HighestIdx = 0;
    m_LowestIdx = 0xFFFFFFFF;
    if (m_pAllocator)
    {
        m_pAllocator->Free(m_pLowerBits);
    }
    else if(m_pLowerBits)
    {
        delete[] m_pLowerBits;
    }
    m_pLowerBits = NULL;
    m_Sbv.Reset();
    //m_pAllocator = NULL;
}

size_t SparseSet::MemSize() const NN_NOEXCEPT
{
    size_t mh = m_Sbv.MemSize();
    size_t mb = sizeof(m_pLowerBits[0]) * m_LowerBitsLength;
    return sizeof(m_BitvectorSize) + sizeof(m_NumOnBits) + sizeof(m_LowestIdx) +
           sizeof(m_HighestIdx) + sizeof(m_LowerBitsLength) + sizeof(m_pLowerBits) + sizeof(m_Width) + mh + mb +
           sizeof(m_pAllocator);
}

bool SparseSet::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!w->Write(m_BitvectorSize))
    {
        return false;
    }
    if (!w->Write(m_LowestIdx))
    {
        return false;
    }
    if (!w->Write(m_HighestIdx))
    {
        return false;
    }
    if (!w->Write(m_NumOnBits))
    {
        return false;
    }
    if (!w->Write(m_Width))
    {
        return false;
    }
    if (!w->Write(m_LowerBitsLength))
    {
        return false;
    }
    if (m_LowerBitsLength > 0 && !w->WriteArray(m_pLowerBits, m_LowerBitsLength))
    {
        return false;
    }
    if (!m_Sbv.Export(w))
    {
        return false;
    }
    return true;
}

bool SparseSet::Import(BinaryReader* r) NN_NOEXCEPT
{
    detail::Resetter<SparseSet> rst(this);
    if (!r->Read(&m_BitvectorSize))
    {
        return false;
    }
    if (!r->Read(&m_LowestIdx))
    {
        return false;
    }
    if (!r->Read(&m_HighestIdx))
    {
        return false;
    }
    if (!r->Read(&m_NumOnBits))
    {
        return false;
    }
    if (!r->Read(&m_Width))
    {
        return false;
    }
    if (!r->Read(&m_LowerBitsLength))
    {
        return false;
    }
    if (m_LowerBitsLength > 0)
    {
        if (m_pAllocator)
        {
            void* bitsPtr = m_pAllocator->Allocate(sizeof(uint32_t) * m_LowerBitsLength);
            m_pLowerBits = bitsPtr ? new (bitsPtr) uint32_t[m_LowerBitsLength] : NULL;
        }
        else
        {
            m_pLowerBits = new (std::nothrow) uint32_t[m_LowerBitsLength];
        }
        if (!m_pLowerBits)
        {
            return false;
        }
        if (m_LowerBitsLength != r->ReadArray(m_pLowerBits, m_LowerBitsLength))
        {
            return false;
        }
    }
    else
    {
        if (m_pAllocator)
        {
            m_pAllocator->Free(m_pLowerBits);
        }
        else
        {
            delete[] m_pLowerBits;
        }
        m_pLowerBits = NULL;
    }
    if (!m_Sbv.Import(r))
    {
        return false;
    }
    rst.Ok();
    return true;
}

// Rank操作を行います。[0..idx] 内に含まれる 1 の数を返します。
// Select1Ex() を利用した二分探索で見つける
uint32_t SparseSet::Rank1(uint64_t idx) const NN_NOEXCEPT
{
    if (m_NumOnBits == 0)
    {
        return 0;
    }
    if (idx < m_LowestIdx)
    {
        return 0;
    }
    if (idx == m_LowestIdx)
    {
        return 1;
    }
    if (idx >= m_HighestIdx)
    {
        return m_NumOnBits;
    }
    uint32_t low = 0;
    uint32_t high = m_NumOnBits - 1;
    while (low + 1 < high)
    {
        // select(low) <= idx < select(high)を維持しつつ
        // low + 1 == highでストップ
        int diff = high - low;
        uint32_t mid = low + diff / 2;
        int64_t val = this->Select1Ex(mid);
        NN_SDK_ASSERT(val >= 0);
        if (idx < static_cast<uint64_t>(val))
        {
            high = mid;
        }
        else
        {
            low = mid;
        }
    }
    return low + 1;
}

int32_t SparseSet::Select1(uint32_t nth) const NN_NOEXCEPT
{
    if (nth >= m_NumOnBits)
    {
        return -1;
    }
    int high = m_Sbv.Select1(nth);
    NN_SDK_ASSERT(high >= 0);
    high -= static_cast<int>(nth);
    high <<= m_Width;
    int low = GetBits(m_Width * nth, m_Width);
    return high + low;
}

int64_t SparseSet::Select1Ex(uint32_t nth) const NN_NOEXCEPT
{
    if (nth >= m_NumOnBits)
    {
        return -1;
    }
    int64_t high = m_Sbv.Select1(nth);
    NN_SDK_ASSERT(high >= 0);
    high -= nth;
    high <<= m_Width;
    int low = GetBits(m_Width * nth, m_Width);
    return high + low;
}

void SparseSet::Swap(SparseSet& rhs) NN_NOEXCEPT
{
    using std::swap;
    swap(m_BitvectorSize, rhs.m_BitvectorSize);
    swap(m_LowestIdx, rhs.m_LowestIdx);
    swap(m_HighestIdx, rhs.m_HighestIdx);
    swap(m_NumOnBits, rhs.m_NumOnBits);
    swap(m_Width, rhs.m_Width);
    swap(m_LowerBitsLength, rhs.m_LowerBitsLength);
    swap(m_pLowerBits, rhs.m_pLowerBits);
    m_Sbv.Swap(rhs.m_Sbv);
    WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
    rhs.m_pAllocator = m_pAllocator;
    m_pAllocator = pAllocatorTmp;
}

namespace {

// values 内の要素のうち最大と最小のものを求め
// その中間値を返します。
uint32_t CalcBaseValue(const uint32_t* values, size_t n) NN_NOEXCEPT
{
    uint32_t valMax = 0ULL;
    uint32_t valMin = 0xFFFFFFFFULL;
    for (size_t i = 0; i < n; ++i)
    {
        if (valMax < values[i]) valMax = values[i];
        if (valMin > values[i]) valMin = values[i];
    }
    uint64_t result = 1;
    result += valMax;
    result += valMin;
    result /= 2;

    // 同じビット幅ならマイナスの方が1つレンジが広いので切り上げ
    return static_cast<uint32_t>(result);
}

// diff 値を 2 進数表現したときに必要なビット幅を返します。
uint8_t CalcWidth(int32_t diff) NN_NOEXCEPT
{
    if (diff == 0)
    {
        return 1;
    }
    if (diff > 0)
    {
        int w = 1;
        while (static_cast<uint32_t>(diff) >= (1UL << w))
        {
            ++w;
        }
        return static_cast<uint8_t>(w + 1);
    }
    else
    {
        int w = 31;
        while (w >= 0 && (diff & (1 << w)))
        {
            --w;
        }
        return static_cast<uint8_t>(w + 2);
    }
}

}  // namespace

CompressedArray::~CompressedArray() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call CompressedArray::ReleaseAllocator() before calling CompressedArray::~CompressedArray()\n");
        NN_ABORT();
    }
    else
    {
        if (m_pHeader)
        {
            FreeMemoryNgc(m_pHeader);
        }
        if (m_pData)
        {
            FreeMemoryNgc(m_pData);
        }
    }
}

CompressedArray::CompressedArray(CompressedArray&& rhs) NN_NOEXCEPT
{
    m_pHeader = rhs.m_pHeader;
    rhs.m_pHeader = NULL;
    m_pData = rhs.m_pData;
    rhs.m_pData = NULL;
    m_HeaderSize = rhs.m_HeaderSize;
    m_HeaderCapacity = rhs.m_HeaderCapacity;
    m_DataSize = rhs.m_DataSize;
    m_DataCapacity = rhs.m_DataCapacity;
    for (size_t i = 0; i < UnitSize; ++i)
    {
        m_pWork[i] = rhs.m_pWork[i];
    }
    m_pAllocator = rhs.m_pAllocator;
    rhs.m_pAllocator = NULL;
}

CompressedArray& CompressedArray::operator=(CompressedArray&& rhs) NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        m_pAllocator->Free(m_pHeader);
        m_pAllocator->Free(m_pData);
    }
    else
    {
        FreeMemoryNgc(m_pHeader);
        FreeMemoryNgc(m_pData);
    }

    m_pHeader = rhs.m_pHeader;
    rhs.m_pHeader = NULL;
    m_pData = rhs.m_pData;
    rhs.m_pData = NULL;
    m_HeaderSize = rhs.m_HeaderSize;
    m_HeaderCapacity = rhs.m_HeaderCapacity;
    m_DataSize = rhs.m_DataSize;
    m_DataCapacity = rhs.m_DataCapacity;
    for (size_t i = 0; i < UnitSize; ++i)
    {
        m_pWork[i] = rhs.m_pWork[i];
    }
    m_pAllocator = rhs.m_pAllocator;
    rhs.m_pAllocator = NULL;
    return *this;
}

void CompressedArray::Swap(CompressedArray& rhs) NN_NOEXCEPT
{
    using std::swap;
    swap(m_pHeader, rhs.m_pHeader);
    swap(m_pData, rhs.m_pData);
    swap(m_HeaderSize, rhs.m_HeaderSize);
    swap(m_HeaderCapacity, rhs.m_HeaderCapacity);
    swap(m_DataSize, rhs.m_DataSize);
    swap(m_DataCapacity, rhs.m_DataCapacity);
    swap(m_WorkIdx, rhs.m_WorkIdx);
    for (size_t i = 0; i < UnitSize; ++i) {
        swap(m_pWork[i], rhs.m_pWork[i]);
    }
    WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
    rhs.m_pAllocator = m_pAllocator;
    m_pAllocator = pAllocatorTmp;
}

bool CompressedArray::Build() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_WorkIdx == UnitSize);
    // Dataのindexを保存
    uint32_t baseidx = m_DataSize;  // 今回の圧縮における圧縮後のデータの始点
    // ベース値を求める
    // work_ 内の各要素のうち最大と最小のものの中央値が入ります
    uint32_t base = CalcBaseValue(m_pWork, UnitSize);
    m_WorkIdx = 0;
    uint8_t k = 0;  // base と各要素の差分を表現するのに必要な最大ビット数
    for (size_t i = 0; i < UnitSize; ++i)
    {
        uint8_t tmp = CalcWidth(m_pWork[i] - base);
        if (k < tmp) k = tmp;
    }
    // k-bitの値をつめていく。kは差分を格納するのに十分大きいビット幅
    uint32_t mask = (1 << k) - 1;
    uint32_t data = 0;

    // MiniReallocOut で m_pData を管理する
    // MiniReallocOut は m_pData, m_DataSize, m_DataCapacity を操作するだけのヘルパ関数
    // Prepare することで m_pData 位置のポインタが指すメモリサイズを拡張してくれる
    nn::ngc::detail::MiniReallocOut<uint32_t> dataout(&m_pData, &m_DataSize, &m_DataCapacity, m_pAllocator);

    // m_pData を UNIT_SIZE + 1 分拡張
    if (!dataout.Prepare(UnitSize + 1))
    {
        return false;
    }

    // data_ に値を圧縮して格納
    // 32 ビット幅の data に、圧縮後の値をできるだけ詰め込む
    // 1 つの値を表すのに base ビットだけあればよいので、
    // 32 ビットの中に 32 / base 個の値の情報が入れられる
    for (size_t i = 0; i < UnitSize; ++i)
    {
        int32_t diff = m_pWork[i] - base;   // ベースからの差分
        size_t ofs = (i * k) % 32;          // 差分をずらして格納するが、どれくらいずらす必要があるか

        // data に圧縮後の値を入れる。各値の base からの差分が 32 bit 内にぴっちり収められる
        data |= (diff & mask) << ofs;
        // 32 bit の data からいくつかの bit があふれた場合
        if (ofs + k >= 32)
        {
            dataout.Append(data);   // 満タンの data を m_pData に反映
            // あふれた部分を新たに data に格納
            size_t w2 = ofs + k - 32;
            uint32_t mask2 = (1 << w2) - 1;
            data = (diff >> (k - w2)) & mask2;
        }
    }
    dataout.Append(data);
    NN_SDK_ASSERT(k > 0 && k <= 32);

    // 現在の baseidx と base の値を m_pHeader, m_HeaderSize, m_HeaderCapacity に保存
    nn::ngc::detail::MiniReallocOut<uint32_t>
        headerout(&m_pHeader, &m_HeaderSize, &m_HeaderCapacity, m_pAllocator);
    if (!headerout.Prepare(2))
    {
        return false;
    }
    // baseidx の後半 5 ビットを間借りして k と baseidx を 32 ビット内に一緒に格納
    // 0 < k <= 32 なので -1 すると 5 bit で事足りる
    // NOTE: baseidx が 256M 超えたらどうする・・？
    headerout.Append(baseidx | ((k - 1) << 27));
    headerout.Append(base);
    return true;
}

uint32_t CompressedArray::operator[](size_t idx) const NN_NOEXCEPT
{
    size_t bidx = idx / UnitSize;
    size_t ridx = idx % UnitSize;
    // 範囲外なら0
    if (bidx >= m_HeaderSize / 2)
    {
        // まだ圧縮されていない領域だったら work_ から値を返して終了
        return (bidx < this->Size()) ? m_pWork[ridx] : 0;
    }
    uint32_t baseidx = m_pHeader[bidx * 2] & 0x07FFFFFF;
    uint32_t k = 1 + (m_pHeader[bidx * 2] >> 27);
    // [ridx * k, +k)までのビットを取得(符号付き整数)
    int32_t diff;
    {
        size_t data_idx = (ridx * k) / 32;
        size_t data_ofs = (ridx * k) % 32;
        // 圧縮されている値を data_ から取り出す
        uint64_t val_u = m_pData[baseidx + data_idx];
        // 32bit 境界をまたいでいる場合はその次のところから余りのビットを取ってくる
        if (data_ofs + k > 32)
        {
            uint64_t tmp = m_pData[baseidx + data_idx + 1];
            val_u |= tmp << 32;
        }
        // 取り出した部分のうち不要な部分を落とす
        diff = static_cast<int32_t>((val_u >> data_ofs) & ((1 << k) - 1));
        // 算術シフトで符号をよみがえらせる
        diff <<= 32 - k;
        diff >>= 32 - k;
    }
    // baseを足す
    uint32_t base = m_pHeader[bidx * 2 + 1];
    return base + diff;
}

bool CompressedArray::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void CompressedArray::ReleaseAllocator() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        m_pAllocator->Free(m_pHeader);
        m_pAllocator->Free(m_pData);
        m_pHeader = NULL;
        m_pData = NULL;
    }
    m_pAllocator = NULL;
}

void CompressedArray::Reset() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        m_pAllocator->Free(m_pHeader);
        m_pAllocator->Free(m_pData);
    }
    else
    {
        FreeMemoryNgc(m_pHeader);
        FreeMemoryNgc(m_pData);
    }
    m_pHeader = NULL;
    m_HeaderCapacity = m_HeaderSize = 0;
    m_pData = NULL;
    m_DataCapacity = m_DataSize = 0;
    m_WorkIdx = 0;
}

size_t CompressedArray::MemSize() const NN_NOEXCEPT
{
    return sizeof(m_pHeader[0]) * m_HeaderSize + sizeof(m_pData[0]) * m_DataSize +
           sizeof(m_pHeader) + sizeof(m_pData) + sizeof(m_pWork) + sizeof(m_WorkIdx) + sizeof(m_pAllocator);
}

bool CompressedArray::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!w->Write(m_HeaderSize))
    {
        return false;
    }
    if (m_HeaderSize != 0 && !w->WriteArray(&m_pHeader[0], m_HeaderSize))
    {
        return false;
    }
    if (!w->Write(m_DataSize))
    {
        return false;
    }
    if (m_DataSize != 0 && !w->WriteArray(&m_pData[0], m_DataSize))
    {
        return false;
    }
    if (!w->Write(m_WorkIdx))
    {
        return false;
    }
    if (m_WorkIdx > 0 && !w->WriteArray(&m_pWork[0], m_WorkIdx))
    {
        return false;
    }
    return true;
}

bool CompressedArray::Import(BinaryReader* r) NN_NOEXCEPT
{
    detail::Resetter<CompressedArray> rst(this);

    uint32_t hdrLen;
    if (!r->Read(&hdrLen))
    {
        return false;
    }
    nn::ngc::detail::MiniReallocOut<uint32_t>
        headerout(&m_pHeader, &m_HeaderSize, &m_HeaderCapacity, m_pAllocator);
    if (!headerout.Resize(hdrLen))
    {
        return false;
    }
    if (hdrLen > 0 && hdrLen != r->ReadArray(&m_pHeader[0], hdrLen))
    {
        return false;
    }

    uint32_t dataLen;
    if (!r->Read(&dataLen))
    {
        return false;
    }

    nn::ngc::detail::MiniReallocOut<uint32_t> dataout(&m_pData, &m_DataSize, &m_DataCapacity, m_pAllocator);
    if (!dataout.Resize(dataLen))
    {
        return false;
    }

    if (dataLen > 0 && dataLen != r->ReadArray(&m_pData[0], dataLen))
    {
        return false;
    }

    if (!r->Read(&m_WorkIdx))
    {
        return false;
    }
    if (m_WorkIdx >= UnitSize)
    {
        return false;
    }
    if (m_WorkIdx != r->ReadArray(&m_pWork[0], m_WorkIdx))
    {
        return false;
    }

    rst.Ok();
    return true;
}

}}} // nn::ngc::detail
