﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include "./ngc_ErrnoT.h"

namespace nn { namespace ngc { namespace detail {
// code snippets:
//   InputStream& stream = 'stream that inherits InputStream';
//   if (!stream.Read(dataptr, nbyte)) { error = stream.GetErrorValue(); ..... }
//   while ((c = stream.Read()) >= 0) { c is [0x00 - 0xFF] }
//   if (!stream) { error = stream.GetErrorValue(); ..... }
class  InputStream
{
public:
    InputStream() NN_NOEXCEPT : m_Pos(0),
                                m_IsBufCurIdx(0),
                                m_IsBufEndIdx(0),
                                m_pIsBuf(NULL),
                                m_IsBufSize(0),
                                m_Errno(0) {}
    virtual ~InputStream() NN_NOEXCEPT
    {
        // NOTE:
        // cannot call Close_() from this destructor.
        // the destructor of the derived class has to call Close_() if needed.
    }
    // returns true if there has been no errors
    // you can also write the codes such that 'if (!!stream) { ... }'
    errno_t GetErrorValue() const NN_NOEXCEPT
    {
        return m_Errno;
    }
    // use Pos64() if you need a 64bit value on 32bit arch
    size_t Pos() const NN_NOEXCEPT
    {
        return static_cast<size_t>(Pos64());
    }
    uint64_t Pos64() const NN_NOEXCEPT
    {
        return m_Pos + m_IsBufCurIdx;
    }
    bool IsEos() NN_NOEXCEPT
    {
        if (*this && m_IsBufCurIdx == m_IsBufEndIdx)
        {
            CheckBuffer();
            return *this && m_IsBufCurIdx == m_IsBufEndIdx;
        }
        return false;
    }
    // < 0 if EOS or error, otherwise returns a byte and ++pos
    int Read() NN_NOEXCEPT
    {
        if (m_IsBufCurIdx == m_IsBufEndIdx)
        {
            CheckBuffer();
            if (m_IsBufCurIdx == m_IsBufEndIdx)
            {
                return -1;
            }
        }
        return m_pIsBuf[m_IsBufCurIdx++];
    }
    // < 0 if EOS or error, otherwise returns a byte and pos is unchanged
    int Peek() NN_NOEXCEPT
    {
        if (m_IsBufCurIdx == m_IsBufEndIdx)
        {
            CheckBuffer();
            if (m_IsBufCurIdx == m_IsBufEndIdx)
            {
                return -1;
            }
        }
        return m_pIsBuf[m_IsBufCurIdx];
    }
    // returns skipped number of bytes
    size_t Skip(size_t nbytes) NN_NOEXCEPT;
    size_t Read(void* ptr, size_t nbytes) NN_NOEXCEPT
    {
        if (nbytes > RSIZE_MAX)
        {  // INT01-C
            this->SetError(EINVAL);
            return 0;
        }
        if (m_IsBufCurIdx + nbytes <= m_IsBufEndIdx)
        {
            MemCpy(ptr, nbytes, &m_pIsBuf[m_IsBufCurIdx], nbytes);
            m_IsBufCurIdx += nbytes;
            return nbytes;
        }
        return this->Read_(ptr, nbytes);
    }
    bool Close() NN_NOEXCEPT;

public:
    inline NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return (m_Errno == 0);
    }

protected:
    void ResetBuffer(void* p, size_t nbytes) NN_NOEXCEPT
    {
        m_pIsBuf = reinterpret_cast<unsigned char*>(p);
        m_IsBufSize = static_cast<int>(nbytes);
    }
    void SetError(errno_t e) const NN_NOEXCEPT
    {
        if (m_Errno == 0)
        {
            m_Errno = e;
        }
    }

private:
    virtual size_t FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT = 0;
    virtual bool Close_() NN_NOEXCEPT = 0;
    virtual void* GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT;
    size_t Read_(void* ptr, size_t nbytes) NN_NOEXCEPT;

    // By default, Skip_() reads data on m_pIsBuf.
    // you can override it if you can implement more efficient code.
    virtual size_t Skip_(size_t nbytes) NN_NOEXCEPT;

private:
    void CheckBuffer() NN_NOEXCEPT;
    bool GetWorkBuffer() NN_NOEXCEPT;

private:
    uint64_t m_Pos;
    size_t m_IsBufCurIdx;
    size_t m_IsBufEndIdx;
    unsigned char* m_pIsBuf;
    size_t m_IsBufSize;
    mutable ErrnoT m_Errno;
    // m_Errno is changed only from derived classes.
    // FillBuffer_() or Close_() changes m_Errno in other words.

    InputStream(const InputStream&) = delete;
    void operator=(const InputStream&) = delete;
};

class  NullInputStream final : public InputStream
{
public:
    NullInputStream() NN_NOEXCEPT;
    virtual ~NullInputStream() NN_NOEXCEPT {}

private:
    char m_pDunnyBuf[256];
    virtual size_t FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT override;
    virtual bool Close_() NN_NOEXCEPT override
    {
        return true;
    }
    virtual size_t Skip_(size_t nbytes) NN_NOEXCEPT override;
};

template<size_t N>
class MiniBufIn
{
public:
    NN_STATIC_ASSERT(N >= 16);
    static const size_t BufSize = N;
    inline MiniBufIn() NN_NOEXCEPT
    {
        m_pCur = m_pEnd = &m_pBuf[0];
    }
    template<class IS>
    inline bool Prefetch(IS* stream, size_t n) NN_NOEXCEPT
    {
        if (m_pCur + n > m_pEnd)
        {
            NN_SDK_ASSERT(n <= N);
            size_t rem = m_pEnd - m_pCur;
            MemMove(reinterpret_cast<void*>(&m_pBuf[0]), N, m_pCur, rem);
            size_t nread = stream->Read(&m_pBuf[rem], N - rem);
            m_pCur = &m_pBuf[0];
            m_pEnd = &m_pBuf[0] + rem + nread;
            return (nread + rem >= n);
        }
        return true;
    }
    template<class IS>
    inline int Peek(IS* stream) NN_NOEXCEPT
    {
        if (m_pCur == m_pEnd)
        {
            size_t nread = stream->Read(&m_pBuf[0], N);
            if (nread == 0)
            {
                return -1;
            }
            m_pCur = &m_pBuf[0];
            m_pEnd = &m_pBuf[0] + nread;
        }
        return *m_pCur;
    }
    template<class IS>
    inline int Read(IS* stream) NN_NOEXCEPT
    {
        int c;
        if (m_pCur == m_pEnd)
        {
            size_t nread = stream->Read(&m_pBuf[0], N);
            if (nread == 0)
            {
                return -1;
            }
            m_pCur = &m_pBuf[0] + 1;
            m_pEnd = &m_pBuf[0] + nread;
            return m_pBuf[0];
        }
        else
        {
            c = *m_pCur;
            ++m_pCur;
            return c;
        }
    }

    inline void Advance(size_t n) NN_NOEXCEPT
    {
        // Prefetch(stream, n) must be successful beforehand
        m_pCur += n;
    }
    inline const uint8_t& operator[](size_t n) NN_NOEXCEPT
    {
        // Prefetch(stream, n + 1) must be successful beforehand
        return m_pCur[n];
    }
    template<class IS>
    inline
    size_t ReadBytes(IS* stream, void* data, size_t nbytes) NN_NOEXCEPT
    {
        if (m_pCur + nbytes <= m_pEnd)
        {
            MemCpy(data, nbytes, m_pCur, nbytes);
            m_pCur += nbytes;
            return nbytes;
        }
        uint8_t* p = reinterpret_cast<uint8_t*>(data);
        size_t rem = m_pEnd - m_pCur;
        MemCpy(reinterpret_cast<void*>(p), nbytes, m_pCur, rem);
        m_pCur = m_pEnd = &m_pBuf[0];
        size_t nread = stream->Read(p + rem, nbytes - rem);
        return nread + rem;
    }
    template<class IS, class T>
    size_t Read(IS* stream, T* data, size_t count) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(*data) <= N);
        NN_STATIC_ASSERT(std::is_pod<T>::value);
        size_t nbytes = sizeof(*data) * count;
        size_t nread = ReadBytes(stream, data, nbytes);
        if (nread == nbytes)
        {
            return count;
        }
        size_t mod = nread % sizeof(*data);
        uint8_t* p = reinterpret_cast<uint8_t*>(data);
        MemCpy(reinterpret_cast<void*>(&m_pBuf[0]), N,
                    p + nread - mod, mod);
        m_pEnd = &m_pBuf[mod];
        return nread / sizeof(*data);
    }
    inline size_t size() const NN_NOEXCEPT
    {
        return m_pEnd - m_pCur;
    }

private:
    uint8_t* m_pCur;
    uint8_t* m_pEnd;
    uint8_t m_pBuf[N];
};

}}} // nn::ngc::detail
