﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_Sbv.h"
#include "./ngc_ReallocVec.h"

namespace nn { namespace ngc { namespace detail {

/**
 * @brief   各種操作を O(1) で行うことができるコンパクトなツリー構造です。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 */
class  Bp final
{
public:
    Bp() NN_NOEXCEPT : m_pPrv(NULL), m_pAllocator(NULL) {}
    ~Bp() NN_NOEXCEPT;
    inline Bp(Bp&& rhs) : m_pPrv(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline Bp& operator=(Bp&& rhs) NN_NOEXCEPT
    {
        Bp tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    void Swap(Bp& rhs) NN_NOEXCEPT
    {
        BpPrivate* tmp = rhs.m_pPrv;
        rhs.m_pPrv = m_pPrv;
        m_pPrv = tmp;
        nn::ngc::detail::WorkBufAllocator* tmpAllocator = m_pAllocator;
        m_pAllocator = rhs.m_pAllocator;
        rhs.m_pAllocator = tmpAllocator;
    }
    /**
     * @brief       ツリーオブジェクトから括弧木を生成します。
     * @param[in]   root        根ノード
     * @param[in]   numNodes    ノード数
     * @details     引数となるTREEITER 型は以下のメンバ関数を実装している必要があります。
     *                  - const void* NodePtr();
     *                      - ノードのポインタを返す
     *                  - const void* ParentNodePtr();
     *                      - 親ノードのポインタを返す
     *                  - bool MoveNext();
     *                      - 深さ優先で次のノードに移動
     *              ノードIDは深さ優先順で0からついていきます。
     */
    template <class TREEITER>
    bool Init(TREEITER& root, uint32_t numNodes) NN_NOEXCEPT;  // NOLINT
    bool InitDirect(const uint32_t* bv, uint32_t bvSize) NN_NOEXCEPT;

    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;

    /**
     * @brief       ビット列の位置からノードIDを取得します。
     * @param[in]   pos 括弧の位置を表すインデックス
     * @return      pos に対応するノードIDが存在する場合は非負整数。存在しなければ-1。
     */
    int ToNodeId(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       ノードIDからビット列の位置を取得します。
     * @param[in]   nodeid  ノードID
     * @return      nodeId が存在する場合対応する非負整数。存在しなければ-1。
     */
    int ToPos(uint32_t nodeid) const NN_NOEXCEPT;
    /**
     * @brief       閉じ括弧に対応する開き括弧の位置を取得します。
     * @param[in]   pos 閉じ括弧の位置
     * @return      閉じ括弧の位置を指定すると開き括弧の位置を返します。 指定した位置に開き括弧がある場合は引数と同じ値を返します。
     */
    int FindOpen(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       開き括弧に対応する閉じ括弧の位置を取得します。
     * @param[in]   pos 開き括弧の位置
     * @return      対応する閉じ括弧の位置。存在しなければ-1。
     *              開き括弧の位置を指定すると閉じ括弧の位置を返します。 指定した位置に閉じ括弧がある場合は引数と同じ値を返します。
     */
    int FindClose(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       指定された括弧を直接包む括弧組の開き括弧の位置を取得します。
     * @param[in]   pos 括弧の位置
     * @return      対応する開き括弧の位置。存在しなければ-1。
     * @details     括弧の位置を指定すると、その括弧を包む一番範囲の狭い括弧（の開き括弧）の位置を返します。 木においては親ノードを取得する操作に対応します。
     */
    int Enclose(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       指定された括弧の中にある最初の開き括弧の位置を取得します。
     * @param[in]   pos 括弧の位置
     * @return      子ノードに対応する開き括弧の位置。存在しなければ-1。
     * @details     木において最初の子にアクセスする操作に対応します。
     */
    int FirstChild(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       指定された括弧の中にある最後の閉じ括弧の位置を取得します。
     * @param[in]   pos 括弧の位置
     * @return      子ノードに対応する閉じ括弧の位置。存在しなければ-1。
     * @details     木において最後の子にアクセスする操作に対応します。
     */
    int LastChild(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       指定された括弧に対応するノードの次の兄弟ノードに対応する括弧の位置を取得します。
     * @param[in]   pos 括弧の位置
     * @return      次の兄弟ノードの開き括弧の位置
     * @details     木においては次の兄弟にアクセスする操作に対応します。
     */
    int NextSibling(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       指定された括弧に対応するノードの前の兄弟ノードに対応する括弧の位置を取得します。
     * @param[in]   pos 括弧の位置
     * @return      前の兄弟ノードの閉じ括弧の位置
     * @details     木においては前の兄弟にアクセスする操作に対応します。
     */
    int PrevSibling(uint32_t pos) const NN_NOEXCEPT;
    /**
     * @brief       親ノードとなる括弧の位置を返します。
     * @param[in]   pos 括弧の位置
     * @return      親ノードに対応する開き括弧の位置。存在しなければ-1。
     */
    int Parent(uint32_t pos) const NN_NOEXCEPT
    {
        return this->Enclose(pos);
    }
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     * @details m_pAllocator も NULL にします。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    Set* Get1stLevelSet() NN_NOEXCEPT;
    bool Build(const uint32_t* bv, uint32_t bvsize) NN_NOEXCEPT;

private:
    struct BpPrivate;
    struct BpPrivateDeleter;
    BpPrivate* m_pPrv;
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;

    Bp(const Bp&) = delete;
    void operator=(const Bp&) = delete;
};

template <class TREEITER>
bool Bp::Init(TREEITER& iter, uint32_t numNodes) NN_NOEXCEPT
{   // NOLINT
    // TREEITER has following methods:
    //   const void* NodePtr();        // ptr to node
    //   const void* ParentNodePtr();  // ptr to parent node
    //   bool MoveNext();  // move to the next node by depth first search
    if (numNodes >= 0x80000000)
    {
        return false;
    }
    if (m_pPrv)
    {
        return false;
    }
    Set* bv = Get1stLevelSet();  // bv = &bp_.b_
    if (!bv)
    {
        this->Reset();
        return false;
    }
    bv->Init(numNodes * 2);

    uint32_t idx = 0;
    ReallocVec<const void*> stack(m_pAllocator);
    do {
        const void* parent = iter.ParentNodePtr();
        const void* node = iter.NodePtr();
        if (stack.Empty() || stack.Back() == parent)
        {
            bv->TurnOn(idx++);
            if (!stack.PushBack(node))
            {
                this->Reset();
                stack.ReleaseAllocator();
                return false;
            }
        }
        else
        {
            while (stack.Back() != parent)
            {
                stack.PopBack();
                ++idx;
            }
            bv->TurnOn(idx++);
            if (!stack.PushBack(node))
            {
                this->Reset();
                stack.ReleaseAllocator();
                return false;
            }
        }
    } while (iter.MoveNext());

    if (!this->Build(bv->GetBitVector(), bv->GetBitVectorSize()))
    {
        this->Reset();
        stack.ReleaseAllocator();
        return false;
    }
    stack.ReleaseAllocator();
    return true;
}

}}} // nn::ngc::detail

namespace std {

template<> inline void swap<::nn::ngc::detail::Bp>(::nn::ngc::detail::Bp& lhs, ::nn::ngc::detail::Bp& rhs)
{
    lhs.Swap(rhs);
}

}
