﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <algorithm>
#include <cassert>
#include <utility>

#include <nn/nn_Abort.h>

#include "./ngc_Bp.h"
#include "./ngc_SbvPrivate.h"
#include "./ngc_BinaryReader.h"
#include "./ngc_BinaryWriter.h"

#ifdef near
#undef near
#endif
#ifdef far
#undef far
#endif

using std::pair;
using std::make_pair;
using std::sort;

namespace nn { namespace ngc { namespace detail {

namespace {

// far な括弧のペアを格納するための構造体
struct FarPair {
    size_t first;   // 開括弧の位置
    size_t second;  // 閉括弧の位置
};

// ソート時に使われるソート規則
// 開括弧の位置で昇順ソート
inline bool MyLess1(const FarPair& l, const FarPair& r) NN_NOEXCEPT
{
    return l.first < r.first;
}
// 閉括弧の位置で昇順ソート
inline bool MyLess2(const FarPair& l, const FarPair& r) NN_NOEXCEPT
{
    return l.second < r.second;
}

}  // namespace

//namespace succinct {

namespace {

inline int FindNearClose32(uint32_t blk, size_t nth) NN_NOEXCEPT
{
    NN_SDK_ASSERT(blk & (1 << nth));
    // 開カッコの場合
    // [nth + 1, 31]までをチェック
    int level = 1;
#define FN_CASE_OPEN(nth)      \
    case nth:                  \
        if (blk & (1U << nth)) \
            ++level;           \
        else if (--level == 0) \
        return nth
    switch (nth + 1)
    {
        FN_CASE_OPEN(1);
        FN_CASE_OPEN(2);
        FN_CASE_OPEN(3);
        FN_CASE_OPEN(4);
        FN_CASE_OPEN(5);
        FN_CASE_OPEN(6);
        FN_CASE_OPEN(7);
        FN_CASE_OPEN(8);
        FN_CASE_OPEN(9);
        FN_CASE_OPEN(10);
        FN_CASE_OPEN(11);
        FN_CASE_OPEN(12);
        FN_CASE_OPEN(13);
        FN_CASE_OPEN(14);
        FN_CASE_OPEN(15);
        FN_CASE_OPEN(16);
        FN_CASE_OPEN(17);
        FN_CASE_OPEN(18);
        FN_CASE_OPEN(19);
        FN_CASE_OPEN(20);
        FN_CASE_OPEN(21);
        FN_CASE_OPEN(22);
        FN_CASE_OPEN(23);
        FN_CASE_OPEN(24);
        FN_CASE_OPEN(25);
        FN_CASE_OPEN(26);
        FN_CASE_OPEN(27);
        FN_CASE_OPEN(28);
        FN_CASE_OPEN(29);
        FN_CASE_OPEN(30);
        FN_CASE_OPEN(31);
    default:
        break;
    }
#undef FN_CASE_OPEN
    return -1;
}

/**
 * @brief   blk 内の　nth 位置にある閉括弧に対する開括弧の位置を返します。
 *          見つからなかった場合は -1 を返します。
 */
inline int FindNearOpen32(uint32_t blk, size_t nth) NN_NOEXCEPT {
    NN_SDK_ASSERT(!(blk & (1 << nth)));
    // 閉カッコの場合
    // [0, nth - 1]までをチェック
    int level = 1;
#if 1
#define FN_CASE_CLOSE(nth)        \
    case nth:                     \
        if (!(blk & (1U << nth))) \
            ++level;              \
        else if (--level == 0)    \
        return nth
    switch (static_cast<int>(nth - 1)) {
        FN_CASE_CLOSE(30);
        FN_CASE_CLOSE(29);
        FN_CASE_CLOSE(28);
        FN_CASE_CLOSE(27);
        FN_CASE_CLOSE(26);
        FN_CASE_CLOSE(25);
        FN_CASE_CLOSE(24);
        FN_CASE_CLOSE(23);
        FN_CASE_CLOSE(22);
        FN_CASE_CLOSE(21);
        FN_CASE_CLOSE(20);
        FN_CASE_CLOSE(19);
        FN_CASE_CLOSE(18);
        FN_CASE_CLOSE(17);
        FN_CASE_CLOSE(16);
        FN_CASE_CLOSE(15);
        FN_CASE_CLOSE(14);
        FN_CASE_CLOSE(13);
        FN_CASE_CLOSE(12);
        FN_CASE_CLOSE(11);
        FN_CASE_CLOSE(10);
        FN_CASE_CLOSE(9);
        FN_CASE_CLOSE(8);
        FN_CASE_CLOSE(7);
        FN_CASE_CLOSE(6);
        FN_CASE_CLOSE(5);
        FN_CASE_CLOSE(4);
        FN_CASE_CLOSE(3);
        FN_CASE_CLOSE(2);
        FN_CASE_CLOSE(1);
        FN_CASE_CLOSE(0);
    default:
        break;
    }
#undef FN_CASE_CLOSE
#else
    for (int i = static_cast<int>(nth - 1); i >= 0; --i) {
        // 8bitずつくらいでテーブルにするのもよい
        level = (blk & (1 << i)) ? level - 1 : level + 1;
        if (level == 0) return i;
    }
#endif
    return -1;
}

}  // namespace

/**
 * @brief   Bp の実体
 *          BP として有効な作成済みビット列に対して補助データを作成します。
 *              - ビット列用 Set
 *              - pioneer のみをビット列 SparseSet
 *              - pioneer のみの BP_ へのポインタ
 *          また作成した補助データに対していくつかの位置を定数時間で返す機能を提供します。
 */
class Bp_ {
    static const size_t BlockSize = SbvBlockSize;

public:
    Bp_() NN_NOEXCEPT;
    explicit Bp_(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    ~Bp_() NN_NOEXCEPT;
    bool SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    void ReleaseAllocator() NN_NOEXCEPT;
    bool Build(size_t M) NN_NOEXCEPT;

    /**
     * @brief   閉じ括弧に対応する開き括弧の位置を取得します。
     */
    int FindOpen(uint32_t p) const NN_NOEXCEPT;
    /**
     * @brief   開き括弧に対応する閉じ括弧の位置を取得します。
     */
    int FindClose(uint32_t p) const NN_NOEXCEPT;
    /**
     * @brief   指定された括弧を直接包む括弧組の開き括弧の位置を取得します。
     */
    int Enclose(uint32_t p) const NN_NOEXCEPT;
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    Set m_Bits;             // BP 表現されたビット列。Set で管理。
    SparseSet m_Pioneer;    // m_Bits のうち pioneer の位置だけを保存したビット列。 SparseSet で管理
    Bp_* m_NextLevel;       // m_Pioneer を圧縮した pioneer だけのビット列に対する BP_ へのポインタ
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;

    friend class Bp;
    Bp_(const Bp_& rhs) NN_NOEXCEPT;
    Bp_& operator=(const Bp_& rhs) NN_NOEXCEPT;
};

Bp_::Bp_() NN_NOEXCEPT : m_Bits(), m_Pioneer(), m_NextLevel(NULL), m_pAllocator(NULL)
{
    NN_SDK_ASSERT(32 == SbvBlockSize);
}

Bp_::Bp_(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Bits(), m_Pioneer(), m_NextLevel(NULL)
{
    NN_SDK_ASSERT(32 == SbvBlockSize);
    NN_ABORT_UNLESS(SetAllocator(pAllocator));
}

Bp_::~Bp_() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call Bp_::ReleaseAllocator() before calling Bp_::~Bp_()\n");
        NN_ABORT();
    }
    else
    {
        delete m_NextLevel;
    }
}

bool Bp_::SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    m_Bits.SetAllocator(m_pAllocator);
    m_Pioneer.SetAllocator(m_pAllocator);
    return true;
}

void Bp_::ReleaseAllocator() NN_NOEXCEPT
{
    if (!m_pAllocator)
    {
        return;
    }
    if (m_NextLevel)
    {
        m_NextLevel->ReleaseAllocator();
        m_NextLevel->~Bp_();
        m_pAllocator->Free(m_NextLevel);
        m_NextLevel = NULL;
    }
    m_Bits.ReleaseAllocator();
    m_Pioneer.ReleaseAllocator();
    m_pAllocator = NULL;
}

size_t Bp_::MemSize() const NN_NOEXCEPT
{
    size_t b = m_Bits.MemSize();
    size_t p = m_Pioneer.MemSize();
    size_t r = m_NextLevel ? m_NextLevel->MemSize() : 0;
    return b + p + r + sizeof(m_NextLevel) + sizeof(m_pAllocator);
}

bool Bp_::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_Bits.Export(w))
    {
        return false;
    }
    if (m_NextLevel)
    {
        const uint8_t mark01 = 0x01;
        if (!w->Write(mark01))
        {
            return false;
        }
        if (!m_Pioneer.Export(w))
        {
            return false;
        }
        if (!m_NextLevel->Export(w))
        {
            return false;
        }
    }
    else
    {
        const uint8_t mark00 = 0x00;
        if (!w->Write(mark00))
        {
            return false;
        }
    }
    return true;
}

bool Bp_::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!m_Bits.Import(r))
    {
        return false;
    }
    unsigned char tmp;
    if (!r->Read(&tmp))
    {
        return false;
    }
    if (tmp != 0x00)
    {
        if (!m_Pioneer.Import(r))
        {
            return false;
        }
        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(Bp_));
            m_NextLevel = pTmp ? new (pTmp) Bp_(m_pAllocator) : NULL;
            if (m_NextLevel)
            {
                m_NextLevel->SetAllocator(m_pAllocator);
            }
        }
        else
        {
            m_NextLevel = new (std::nothrow) Bp_();
        }
        if (!m_NextLevel)
        {
            return false;
        }

        if (!m_NextLevel->Import(r))
        {
            return false;
        }
    }
    return true;
}

bool Bp_::Build(size_t M) NN_NOEXCEPT
{
    // 括弧の列を32個(32 bit)ずつのブロックに分割して格納していく。
    // ここで、括弧を以下のように分類する。
    // 1) 対応する括弧がブロック内にあるもの(nearと呼ぶ)
    // 2) 対応する括弧がブロック内にないもの(farと呼ぶ)
    // 2-1) ブロック内の最初のfarな開括弧と対応する閉括弧
    //      (opening pioneerと呼ぶ)
    // 2-2) ブロック内を後ろからみたときの最初のfarな閉括弧と対応する開括弧
    //      (closing pioneerと呼ぶ)
    // ここで、括弧列内のopening/closing pioneerをマークしたビットベクトルと、
    // 括弧列からopening/closing pioneerだけを取り出した
    // 新たな括弧列を再帰的に作成していく。
    // ここで、新たな括弧列もBpの要件を見たしていることがわかる。
    // 括弧列の長さは段階ごとに急速に縮小していくので、この括弧列を再帰的に
    // 参照する操作はO(1)で動作することになる(再帰の深さの伸びは括弧列の長さ
    // の伸びに比べて無視できるほど小さい)。
    if (!m_Bits.Build()) return false;
    uint32_t n = m_Bits.GetBitVectorSize();

    // 括弧列からopening/closing pioneerだけを取り出した新たな括弧列
    typedef ReallocVec<FarPair> FarPairVec;

    // farである開括弧と閉括弧のベクタを作成する。
    bool foundPioneer = false;
    ReallocVec<size_t> stack(m_pAllocator);
    FarPairVec farPairVec(m_pAllocator);
    for (uint32_t i = 0; i < n; ++i)
    {
        if (m_Bits.Has(i))
        {
            // 該当ビットが 1 なら stack に突っ込む
            if (!stack.PushBack(i))
            {
                stack.ReleaseAllocator();
                farPairVec.ReleaseAllocator();
                return false;
            }
        }
        else
        {
            NN_SDK_ASSERT(!stack.Empty());
            size_t idxO = stack.Back();    // 末尾の要素を取得
            stack.PopBack();                // 末尾の要素を削除
            // この時点で i には 0 の位置が、 idxO にはそれに対応する 1 の位置が入っている
            // この二つは対応する開括弧と閉括弧
            // nearでないかどうかチェック
            size_t blkC = i / M;
            size_t blkO = idxO / M;
            if (blkC != blkO)
            {
                // 同じブロックにないのでfarである。
                FarPair tmp = { idxO, i};
                if (!farPairVec.PushBack(tmp))
                {
                    stack.ReleaseAllocator();
                    farPairVec.ReleaseAllocator();
                    return false;
                }
            }
        }
    }
    NN_SDK_ASSERT(stack.Empty());
    stack.ReleaseAllocator();
    ReallocVec<size_t>(m_pAllocator).Swap(stack);

    // 開括弧の場所が昇順になるようソート
    sort(farPairVec.Begin(), farPairVec.End(), MyLess1);
    {
        //
        // Opening Pioneerをみつける
        //
        size_t blkPioneer = ~0U;
        FarPairVec::const_iterator it;
        FarPairVec::const_iterator itEnd = farPairVec.End();
        for (it = farPairVec.Begin(); it != itEnd; ++it)
        {
            size_t blkO = it->first / M;   // 開括弧の所属ブロック
            // 開括弧の所属ブロックが初出→opening pioneer である
            if (blkO != blkPioneer)
            {
                blkPioneer = blkO;
                if (!foundPioneer)
                {
                    foundPioneer = true;
                    if (!m_Pioneer.Init(m_Bits.GetBitVectorSize()))
                    {
                        stack.ReleaseAllocator();
                        farPairVec.ReleaseAllocator();
                        return false;
                    }
                }
                // opening pioneerの場所をマーク。対応する閉括弧もpioneerである。
                m_Pioneer.TurnOn(it->first);
                m_Pioneer.TurnOn(it->second);
            }
        }
    }

    // 閉括弧の場所が昇順になるようにソート
    sort(farPairVec.Begin(), farPairVec.End(), MyLess2);
    {
        //
        // Closing Pioneerをみつける
        //
        size_t blkPioneer = ~0U;
        FarPairVec::const_reverse_iterator it;
        FarPairVec::const_reverse_iterator itEnd = farPairVec.Rend();
        for (it = farPairVec.Rbegin(); it != itEnd; ++it)
        {
            size_t blkC = it->second / M;
            if (blkC != blkPioneer)
            {
                blkPioneer = blkC;
                if (!foundPioneer)
                {
                    foundPioneer = true;
                    if (!m_Pioneer.Init(m_Bits.GetBitVectorSize()))
                    {
                        stack.ReleaseAllocator();
                        farPairVec.ReleaseAllocator();
                        return false;
                    }
                }
                // closing pioneerの場所をマーク。対応する開括弧もpioneerである。
                m_Pioneer.TurnOn(it->second);
                m_Pioneer.TurnOn(it->first);
            }
        }
    }
    // 再帰の前にメモリを開放しておく。
    farPairVec.ReleaseAllocator();
    FarPairVec(m_pAllocator).Swap(farPairVec);

    // もしpioneerがあれば次のレベルを作成しなくてはならない。
    if (foundPioneer)
    {
        // 再帰的にpioneerのみのBPを作成していく。
        if (!m_Pioneer.Build())
        {
            stack.ReleaseAllocator();
            farPairVec.ReleaseAllocator();
            return false;
        }

        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(Bp_));
            m_NextLevel = pTmp ? new (pTmp) Bp_(m_pAllocator) : NULL;
        }
        else
        {
            m_NextLevel = new (std::nothrow) Bp_();
        }

        if (!m_NextLevel)
        {
            stack.ReleaseAllocator();
            farPairVec.ReleaseAllocator();
            return false;
        }
        // 立っているビットの全数（=ノードの数 * 2）
        uint32_t numBit1 = m_Pioneer.Rank1(m_Bits.GetBitVectorSize());
        NN_SDK_ASSERT(numBit1 % 2 == 0);
        // 次のレベルは b_ の中の pioneer だけが書かれた BP になる（pioneer じゃないものは省かれるのでサイズは小さくなる）
        m_NextLevel->m_Bits.Init(numBit1);
        size_t cnt = 0;
        for (uint32_t i = 0; i < numBit1; ++i)
        {
            // pioneerのみのBPを作成する。
            int select1 = m_Pioneer.Select1(i);
            NN_SDK_ASSERT(select1 >= 0);
            if (m_Bits.Has(select1))
            {
                m_NextLevel->m_Bits.TurnOn(i);
                ++cnt;
            }
        }
        NN_SDK_ASSERT(cnt == numBit1 / 2);
        stack.ReleaseAllocator();
        farPairVec.ReleaseAllocator();
        return m_NextLevel->Build(M);
    }

    stack.ReleaseAllocator();
    farPairVec.ReleaseAllocator();

    // 最終的に pioneer なしの BP が作成される
    // 全要素の BP → 1つ前の pioneer だけの BP → 1つ前の pioneer だけの BP → ...
    return true;
}   // NOLINT(impl/function_size)

// 閉じ括弧に対応する開き括弧の位置を取得します。
// p 閉括弧の位置
int Bp_::FindOpen(uint32_t p) const NN_NOEXCEPT
{
    // FINDOPEN(p, B): p=閉かっこの場所
    //  FindNear(p): 見つかればそれが答え
    //  p' = SUCC(p, P): // p'は直後のpioneer
    //  j = FINDOPEN(RANK1(p', P), Bp)
    //  q' = SELECT1(j, P)
    //  diff = [p, p')の閉じ括弧の過剰量 > 0
    //  diffを使ってq'からqを求める
    //   qはカッコの差分が一致する最も右側のカッコ

    // p がいるブロックのインデックス
    size_t blkIdxP = p / BlockSize;
    // p がいるブロックの値
    uint32_t blkP = m_Bits.GetBitVector()[blkIdxP];
    // p のブロック中のインデックス（0始まり）
    size_t r = p % BlockSize;

    int near = FindNearOpen32(blkP, r);
    if (near >= 0)
    {
        // ブロック内に対応する開カッコが存在
        return near + static_cast<int>(p & ~(BlockSize - 1));
    }

    // --- この時点で p は far であることが確定 --- //

    // 一番近い大きいインデックスにあるpioneer(p自身かもしれない)を探す
    // 大きいインデックス ... ブロックごとにインデックスを割り振った場合の数字のことだと思う
    // → p/BLK_SIZE のインデックスのブロックにいる opening/closing pioneer を探す（p の直後の pioneer）
    // これは必ず閉じカッコになる(でなければ矛盾する)
    int pp;
    uint32_t rankP = m_Pioneer.Rank1(p);
    NN_SDK_ASSERT(rankP > 0);
    NN_SDK_ASSERT(m_NextLevel);
    if (m_Pioneer.Has(p))
    {
        pp = static_cast<int>(p);
        rankP -= 1;  // m_NextLevel.m_Bitsでの自身の位置
    }
    else
    {
        if (rankP == 0 || !m_NextLevel)
        {
            return -1;  // 起こりえない
        }
        pp = m_Pioneer.Select1(rankP);
        NN_SDK_ASSERT(pp >= 0);
        if (pp < 0)
        {
            return -1;  // 起こりえない
        }
    }
    // pp = p もしくは p の直後の pioneer の位置(p')

    // 下のレベル(pioneerだけで構成されるBP)での結果を求める
    int j = m_NextLevel->FindOpen(rankP);
    NN_SDK_ASSERT(j >= 0);
    if (j < 0)
    {
        return -1;  // 起こりえない
    }

    // 下のレベルでの結果を現在のレベルでの結果に変換
    int qq = m_Pioneer.Select1(j);
    NN_SDK_ASSERT(qq >= 0);
    if (qq < 0)
    {
        return -1;  // 起こりえない
    }

    if (p == static_cast<size_t>(pp))
    {
        return qq;  // p=ppならばq=qq
    }

    // ---- この時点で p は pioneer でないことが確定 ---- //

    // ここでp'とpおよびq',qは同じブロックである
    // 同じでなければp,qはpioneerになってしまい矛盾する
    // pとp'の間でどれだけ深くなるか計算

    // p と p' の間にある far な閉括弧の数（p'含む）
    int diff = 1;
    // p' の存在するブロック中のインデックス
    size_t rr = pp % BlockSize;
    for (size_t i = rr - 1; i > r; --i)
    {
        diff += (blkP & (1 << i)) ? -1 : 1;
    }
    NN_SDK_ASSERT(diff >= 1);

    // q, q'の属するブロック
    // q がいるブロックのインデックス
    size_t blkIdxQ = qq / BlockSize;
    // q がいるブロックの値(32bit)
    uint32_t blkQ = m_Bits.GetBitVector()[blkIdxQ];
    // q' のブロック中のインデックス
    int r_qq = qq % BlockSize;
    int ans = -1;
    while (r_qq < 31)
    {
        ++r_qq;
        if (blkQ & (1 << r_qq))
        {
            --diff;
            if (diff == 0) ans = r_qq;  // 最も大きい位置にある開きカッコを記録
        }
        else
        {
            ++diff;
        }
    }
    NN_SDK_ASSERT(ans != -1);
    return static_cast<int>(ans + (qq & ~(BlockSize - 1)));
}

int Bp_::FindClose(uint32_t p) const NN_NOEXCEPT
{
    // FINDCLOSE(p, B): p=開かっこの場所
    //  FindNearClose(p): ブロック内に対応する閉じカッコが見つかればそれが答え
    //  p' = PRED(p, P);  // p'は直前のpioneer
    //  j = FINDCLOSE(RANK1(p', P), Bp)
    //  q' = SELECT1(j, P)
    //  diff = [p',p)の開かっこの過剰量 > 0
    //  diffを使ってq'からqを求める
    //   qはカッコの差分が一致する最も左側のカッコ
    size_t blkIdxP = p / BlockSize;
    uint32_t blkP = m_Bits.GetBitVector()[blkIdxP];
    size_t r = p % BlockSize;
    int near = FindNearClose32(blkP, r);
    if (near >= 0)
    {
        // ブロック内に対応する閉じカッコが存在
        return near + static_cast<int>(p & ~(BlockSize - 1));
    }
    // 一番近い小さいインデックスにあるpioneer(p自身かもしれない)を探す
    // これは必ず開きカッコになる(でなければ矛盾する)
    int pp;
    uint32_t rankP = m_Pioneer.Rank1(p);
    NN_SDK_ASSERT(rankP > 0);
    NN_SDK_ASSERT(m_NextLevel);
    if (m_Pioneer.Has(p))
    {
        pp = static_cast<int>(p);
    }
    else
    {
        if (rankP == 0 || !m_NextLevel)
        {
            return -1;  // 起こりえない
        }
        pp = m_Pioneer.Select1(rankP - 1);
        NN_SDK_ASSERT(pp >= 0);
        if (pp < 0)
        {
            return -1;  // 起こりえない
        }
    }

    // 下のレベル(pioneerだけで構成されるBP)での結果を求める
    int j = m_NextLevel->FindClose(rankP - 1);
    NN_SDK_ASSERT(j >= 0);
    if (j < 0)
    {
        return -1;  // 起こりえない
    }

    // 下のレベルでの結果を現在のレベルでの結果に変換
    int qq = m_Pioneer.Select1(j);
    NN_SDK_ASSERT(qq >= 0);
    if (qq < 0)
    {
        return -1;  // 起こりえない
    }

    if (p == static_cast<size_t>(pp))
    {
        return qq;  // p=ppならばq=qq
    }
    // ここでp'とpおよびq',qは同じブロックである
    // 同じでなければp,qはpioneerになってしまい矛盾する
    // pとp'の間でどれだけ深くなるか計算
    int diff = 1;
    size_t rr = pp % BlockSize;
    for (size_t i = rr + 1; i < r; ++i)
    {
        diff += (blkP & (1 << i)) ? 1 : -1;
    }
    NN_SDK_ASSERT(diff >= 1);

    // q, q'の属するブロック
    size_t blkIdxQ = qq / BlockSize;
    uint32_t blkQ = m_Bits.GetBitVector()[blkIdxQ];
    int r_qq = qq % BlockSize;
    int ans = -1;
    while (r_qq > 0)
    {
        --r_qq;
        if (blkQ & (1 << r_qq))
        {
            ++diff;
        }
        else
        {
            --diff;
            if (diff == 0)
            {
                ans = r_qq;  // 最も小さい位置にある閉じカッコを記録
            }
        }
    }
    NN_SDK_ASSERT(ans != -1);
    return static_cast<int>(ans + (qq & ~(BlockSize - 1)));
}

// 指定された括弧を直接包む括弧組の開き括弧の位置を取得します。
// 自分の親ノードを表す括弧のうち開括弧の位置を取得します。
// p 開括弧の位置
int Bp_::Enclose(uint32_t p) const NN_NOEXCEPT {
    // ENCLOSE(p, B): pは開きかっこ
    //  encloseの開きカッコか閉じカッコがpと同じブロック
    //    ブロック内の左を見ていきあればそれが答え
    //    FINDCLOSE(p)の右を見ていきあればFINDOPEN
    //  c = SUCC(p, P)
    //  if c is close paren -> p' = FINDOPEN(c)
    //  else ->
    //    j = ENCLOSE(RANK1(c, P), Bp)
    //    p' = SELECT1(j, P)
    //  q = SUCC(p' + 1, P)
    //  if blk(q) == blk(p') -> ans = qよりも前にある最も近くのfar paren
    //  else ans = blk(p')の最も後ろにあるfar paren

    // 同じブロック内にpを包む開カッコか閉カッコがあるかどうかをチェック
    {
        uint32_t blkIdxP = p / BlockSize;               // p がいるブロックのインデックス
        uint32_t blkP = m_Bits.GetBitVector()[blkIdxP]; // p がいるブロックの値(32bit)
        int r = static_cast<int>(p % BlockSize);        // p のブロック中のインデックス
        // 同じブロックにpを包む開きカッコがあるかどうかチェック
        int diff = 1;
        int rr = r;
        while (diff != 0 && --rr >= 0)
        {
            diff += (blkP & (1 << rr)) ? -1 : 1;
        }
        if (diff == 0)
        {
            return static_cast<int>(blkIdxP * BlockSize + rr);  // 同じブロックに存在
        }
        // 同じブロックにpを包む閉じカッコがあるかどうかチェック
        diff = 2;
        rr = r;
        while (diff != 0 && ++rr < 32)
        {
            diff += (blkP & (1 << rr)) ? 1 : -1;
        }
        if (diff == 0)
        {
            return this->FindOpen(blkIdxP * BlockSize + rr);
        }
    }

    // pと同じかより後ろ(内側)にあるpioneerを探す
    uint32_t c;
    if (m_Pioneer.Has(p))
    {
        c = p;
    }
    else
    {
        uint32_t rank1 = m_Pioneer.Rank1(p);
        int tmp = m_Pioneer.Select1(rank1);
        NN_SDK_ASSERT(tmp >= 0);
        if (tmp < 0) return -1;
        c = static_cast<uint32_t>(tmp);
    }

    // pを包む最内のpioneer(の開カッコ)を求める -> pp
    size_t pp;
    if (!m_Bits.Has(c))
    {
        // もし閉カッコならばFINDOPEN
        int tmp = this->FindOpen(c);
        NN_SDK_ASSERT(tmp >= 0);
        if (tmp < 0)
        {
            return -1;
        }
        pp = static_cast<size_t>(tmp);
    }
    else
    {
        // もし開カッコならば再帰的に求める
        // cを包む最内のpioneerがそれである
        NN_SDK_ASSERT(m_NextLevel);
        int j = m_NextLevel->Enclose(m_Pioneer.Rank1(c) - 1);
        NN_SDK_ASSERT(j >= 0);
        if (j < 0)
        {
            return -1;
        }
        pp = m_Pioneer.Select1(j);
    }

    // ppの次のpioneerを求める
    // 求めるノードは両者の間にある。
    size_t q;
    {
        int tmp = m_Pioneer.Select1(m_Pioneer.Rank1(pp));
        NN_SDK_ASSERT(tmp >= 0);
        if (tmp < 0)
        {
            return -1;
        }
        q = static_cast<size_t>(tmp);
    }

    // [pp q)でppと同じブロック内にあることになる。
    size_t r = pp % BlockSize;
    size_t blk_idx = pp / BlockSize;
    uint32_t blk = m_Bits.GetBitVector()[blk_idx];
    int diff = 0;
    if (pp / BlockSize == q / BlockSize)
    {
        // pp,qが同じブロック内にある場合
        // qよりも左にある最も近くのfarカッコが答え
        size_t r_q = q % BlockSize;
        for (size_t ans = r_q - 1; ans >= r; --ans)
        {
            diff += (blk & (1 << ans)) ? 1 : -1;
            if (diff > 0)
            {
                return static_cast<int>(blk_idx * BlockSize + ans);
            }
        }
    }
    else
    {
        // pp以降の最も右にあるfarカッコが答え
        for (size_t ans = 31; ans >= r; --ans)
        {
            diff += (blk & (1 << ans)) ? 1 : -1;
            if (diff > 0)
            {
                return static_cast<int>(blk_idx * BlockSize + ans);
            }
        }
    }
    NN_SDK_ASSERT(0);
    return -1;
}

// Bp の内部実装を定義している構造体
struct Bp::BpPrivate {
    Bp_* bp;
    detail::SbvSelect select1;
};

struct Bp::BpPrivateDeleter
{
    void operator()(BpPrivate* ptr)
    {
        if (ptr)
        {
            NN_ABORT_UNLESS(m_pAllocator);
            ptr->bp->ReleaseAllocator();
            ptr->select1.ReleaseAllocator();
            ptr->bp->~Bp_();
            ptr->select1.~SbvSelect();
            m_pAllocator->Free(ptr);
            m_pAllocator = NULL;
        }
    }
    explicit BpPrivateDeleter(nn::ngc::detail::WorkBufAllocator* pAllocator) : m_pAllocator(pAllocator) {}
private:
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;
};

Bp::~Bp() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call Bp::ReleaseAllocator() before calling Bp::~Bp()\n");
        NN_ABORT();
    }
    this->Reset();
}

bool Bp::InitDirect(const uint32_t* bv, uint32_t bvSize) NN_NOEXCEPT
{
    Set* s = Get1stLevelSet();
    s->Init(bvSize);
    for (uint32_t i = 0; i < bvSize; ++i)
    {
        if (detail::GetBit(bv, i)) s->TurnOn(i);
    }
    return this->Build(bv, bvSize);
}

bool Bp::SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

void Bp::ReleaseAllocator() NN_NOEXCEPT
{
    this->Reset();
    m_pAllocator = NULL;
}

Set* Bp::Get1stLevelSet() NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        if (m_pAllocator)
        {
            void* pTmp = m_pAllocator->Allocate(sizeof(BpPrivate));
            std::unique_ptr<BpPrivate, BpPrivateDeleter> impl(pTmp ? new (pTmp) BpPrivate() : NULL, BpPrivateDeleter(m_pAllocator));
            if (!impl)
            {
                return NULL;
            }
            pTmp = m_pAllocator->Allocate(sizeof(Bp_));
            impl->bp = pTmp ? new (pTmp) Bp_(m_pAllocator) : NULL;
            if (!impl->bp)
            {
                return NULL;
            }
            m_pPrv = impl.release();
            // 実装クラスのインスタンスにもアロケータを設定
            m_pPrv->bp->SetAllocator(m_pAllocator);
            m_pPrv->select1.SetAllocator(m_pAllocator);
        }
        else
        {
            std::unique_ptr<BpPrivate> impl(new (std::nothrow) BpPrivate());
            if (!impl)
            {
                return NULL;
            }
            impl->bp = new (std::nothrow) Bp_();
            if (!impl->bp)
            {
                return NULL;
            }
            m_pPrv = impl.release();
        }
    }
    return &m_pPrv->bp->m_Bits;
}

bool Bp::Build(const uint32_t* bv, uint32_t bvsize) NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return false;
    }
    return m_pPrv->bp->Build(32) && m_pPrv->select1.Build(bv, bvsize);
}

size_t Bp::MemSize() const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return 0;
    }
    size_t s = m_pPrv->select1.MemSize();
    return s + m_pPrv->bp->MemSize() + sizeof(m_pPrv->bp);
}

int Bp::FindOpen(uint32_t pos) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return -1;
    }
    if (pos >= m_pPrv->bp->m_Bits.GetBitVectorSize())
    {
        return -1;
    }
    // 自分が開括弧だったらそのまま返す
    if (m_pPrv->bp->m_Bits.Has(pos))
    {
        return static_cast<int>(pos);
    }
    return m_pPrv->bp->FindOpen(pos);
}

int Bp::FindClose(uint32_t pos) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return -1;
    }
    if (pos >= m_pPrv->bp->m_Bits.GetBitVectorSize())
    {
        return -1;
    }
    // 自分が閉括弧だったらそのまま返す
    if (!m_pPrv->bp->m_Bits.Has(pos))
    {
        return static_cast<int>(pos);
    }
    return m_pPrv->bp->FindClose(pos);
}

int Bp::Enclose(uint32_t pos) const NN_NOEXCEPT
{
    int p = this->FindOpen(pos);
    if (p <= 0)
    {
        return -1;  // rootでも親はいない。
    }
    return m_pPrv->bp->Enclose(p);
}

int Bp::ToPos(uint32_t nodeId) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return -1;
    }
    return m_pPrv->select1.Select(m_pPrv->bp->m_Bits, nodeId);
}

int Bp::ToNodeId(uint32_t pos) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return -1;
    }
    int p = this->FindOpen(pos);
    if (p < 0)
    {
        return -1;
    }
    size_t rank1 = m_pPrv->bp->m_Bits.Rank1(p);
    return rank1 > 0 ? static_cast<int>(rank1 - 1) : -1;
}

// 指定された括弧の中にある最初の開き括弧の位置を取得します。
int Bp::FirstChild(uint32_t pos) const NN_NOEXCEPT
{
    int p = this->FindOpen(pos);
    if (p < 0)
    {
        return -1;
    }
    NN_SDK_ASSERT(static_cast<size_t>(p) < m_pPrv->bp->m_Bits.GetBitVectorSize() - 1);
    // p + 1が1ならばそれがchild
    // さもなければ子を持たない
    return m_pPrv->bp->m_Bits.Has(p + 1) ? p + 1 : -1;
}

// 指定された括弧の中にある最後の閉じ括弧の位置を取得します。
int Bp::LastChild(uint32_t pos) const NN_NOEXCEPT
{
    int p = this->FindClose(pos);
    if (p < 0)
    {
        return -1;
    }
    NN_SDK_ASSERT(p > 0);
    // p - 1が0ならばそれがchild
    // さもなくば子をもたない
    return !m_pPrv->bp->m_Bits.Has(p - 1) ? p - 1 : -1;
}

// 指定された括弧に対応するノードの次の兄弟ノードに対応する括弧の位置を取得します。
int Bp::NextSibling(uint32_t pos) const NN_NOEXCEPT
{
    int p = this->FindClose(pos);
    if (p < 0)
    {
        return -1;
    }
    // p + 1が1ならばそれが次の兄弟
    // さもなくば存在しない。
    return m_pPrv->bp->m_Bits.Has(p + 1) ? p + 1 : -1;
}

// 指定された括弧に対応するノードの前の兄弟ノードに対応する括弧の位置を取得します。
int Bp::PrevSibling(uint32_t pos) const NN_NOEXCEPT
{
    int p = this->FindOpen(pos);
    if (p <= 0)
    {
        return -1;  // rootの場合も確定する
    }
    // p - 0が0ならばそれが前の兄弟の閉じ括弧
    // さもなくば存在しない。
    return !m_pPrv->bp->m_Bits.Has(p - 1) ? p - 1 : -1;
}

void Bp::Reset() NN_NOEXCEPT
{
    if (m_pPrv)
    {
        if (m_pPrv->bp)
        {
            if (m_pAllocator)
            {
                m_pPrv->bp->ReleaseAllocator();
                m_pPrv->bp->~Bp_();
                m_pAllocator->Free(m_pPrv->bp);
            }
            else
            {
                delete m_pPrv->bp;
            }
        }
        if (m_pAllocator)
        {
            m_pPrv->select1.ReleaseAllocator();
            m_pPrv->select1.~SbvSelect();
            m_pAllocator->Free(m_pPrv);
            m_pAllocator = NULL;
        }
        else
        {
            delete m_pPrv;
        }
        m_pPrv = NULL;
    }
    m_pAllocator = NULL;
}

bool Bp::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!m_pPrv)
    {
        return false;
    }
    if (!m_pPrv->bp->Export(w))
    {
        return false;
    }
    if (!m_pPrv->select1.Export(w))
    {
        return false;
    }
    return true;
}

bool Bp::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!this->Get1stLevelSet())
    {
        this->Reset();
        return false;
    }
    if( !m_pPrv->bp->Import(r) || !m_pPrv->select1.Import(r))
    {
        this->Reset();
        return false;
    }
    return true;
}

}}} // nn::ngc::detail
