﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "./ngc_ErrnoT.h"
#include "./ngc_OutputStream.h"

namespace nn { namespace ngc { namespace detail {

class  BinaryWriter final
{
public:
    enum EndianSetting
    {
        EndianSetting_Default = 0,
        EndianSetting_EndianLittle,
        EndianSetting_EndianBig
    };

public:
    BinaryWriter() NN_NOEXCEPT : m_Stream(NULL),
                                 m_Errno(-1),
                                 m_SwapEndian(false) {}
    ~BinaryWriter() NN_NOEXCEPT
    {
        this->Close();
    }
    errno_t Init(EndianSetting endian) NN_NOEXCEPT;
    errno_t Init() NN_NOEXCEPT
    {
        return Init(EndianSetting_Default);
    }
    errno_t Open(OutputStream* stream) NN_NOEXCEPT;

#define NGC_BINW(tp) return this->Write_(static_cast<tp>(x))
    bool Write(char x) NN_NOEXCEPT
    {
        NGC_BINW(uint8_t);
    }
    bool Write(signed char x) NN_NOEXCEPT
    {
        NGC_BINW(uint8_t);
    }
    bool Write(unsigned char x) NN_NOEXCEPT
    {
        NGC_BINW(uint8_t);
    }
    bool Write(short x) NN_NOEXCEPT
    {
        NGC_BINW(uint16_t);
    }  // NOLINT
    bool Write(unsigned short x) NN_NOEXCEPT
    {
        NGC_BINW(uint16_t);
    }  // NOLINT
    bool Write(int x) NN_NOEXCEPT
    {
        NGC_BINW(uint32_t);
    }
    bool Write(unsigned int x) NN_NOEXCEPT
    {
        NGC_BINW(uint32_t);
    }
    bool Write(long x) NN_NOEXCEPT
    {
        NGC_BINW(ngc_ulong_compatible_t);
    }  // NOLINT
    bool Write(unsigned long x) NN_NOEXCEPT
    {
        NGC_BINW(ngc_ulong_compatible_t);
    }  // NOLINT
    bool Write(long long x) NN_NOEXCEPT
    {
        NGC_BINW(uint64_t);
    }  // NOLINT
    bool Write(unsigned long long x) NN_NOEXCEPT
    {
        NGC_BINW(uint64_t);
    }  // NOLINT
    bool Write(float x) NN_NOEXCEPT
    {
        NGC_BINW(float);
    }  // NOLINT
    bool Write(double x) NN_NOEXCEPT
    {
        NGC_BINW(double);
    }  // NOLINT
#undef NGC_BINW

    template <class T>
    bool Write(T x) NN_NOEXCEPT;

    bool WriteArray(const unsigned char* x, size_t n) NN_NOEXCEPT;
    bool WriteArray(const unsigned short* x, size_t n) NN_NOEXCEPT;  // NOLINT
    bool WriteArray(const unsigned int* x, size_t n) NN_NOEXCEPT;
    bool WriteArray(const unsigned long long* x, size_t n) NN_NOEXCEPT;  // NOLINT
    bool WriteArray(const unsigned long* x, size_t n) NN_NOEXCEPT {  // NOLINT
#ifdef NN_BUILD_CONFIG_ABI_LP64
        return this->WriteArray(reinterpret_cast<const unsigned long long*>(x), n);  // NOLINT
#else
        return this->WriteArray(reinterpret_cast<const unsigned int*>(x), n);
#endif
    }

    bool WriteArray(const float* x, size_t n) NN_NOEXCEPT;
    bool WriteArray(const double* x, size_t n) NN_NOEXCEPT;
    bool WriteArray(const signed char* x, size_t n) NN_NOEXCEPT
    {
        return this->WriteArray(reinterpret_cast<const unsigned char*>(x), n);
    }
    bool WriteArray(const char* x, size_t n) NN_NOEXCEPT
    {
        return this->WriteArray(reinterpret_cast<const unsigned char*>(x), n);
    }
    bool WriteArray(const short* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
        return this->WriteArray(reinterpret_cast<const unsigned short*>(x), n);  // NOLINT
    }
    bool WriteArray(const int* x, size_t n) NN_NOEXCEPT
    {
        return this->WriteArray(reinterpret_cast<const unsigned int*>(x), n);
    }
    bool WriteArray(const long* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
#ifdef NN_BUILD_CONFIG_ABI_LP64
        return this->WriteArray(reinterpret_cast<const unsigned long long*>(x), n);  // NOLINT
#else
        return this->WriteArray(reinterpret_cast<const unsigned int*>(x), n);
#endif
    }
    bool WriteArray(const long long* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
        return this->WriteArray(reinterpret_cast<const unsigned long long*>(x), n);  // NOLINT
    }
    template <class T>
    bool WriteArray(const T* x, size_t n) NN_NOEXCEPT;
    template <class T, size_t N>
    bool WriteArray(const T(&a)[N]) NN_NOEXCEPT;
    bool Flush() NN_NOEXCEPT;
    bool Close() NN_NOEXCEPT;
    void SetError(errno_t e) NN_NOEXCEPT
    {
        if (m_Errno == 0) m_Errno = e;
    }
    errno_t GetErrorValue() const NN_NOEXCEPT
    {
        return m_Errno;
    }
    OutputStream* GetStream() NN_NOEXCEPT
    {
        return m_Stream;
    }

public:
    inline NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return (GetErrorValue() == 0);
    }

private:
    bool Write_(uint8_t x) NN_NOEXCEPT;
    bool Write_(uint16_t x) NN_NOEXCEPT;
    bool Write_(uint32_t x) NN_NOEXCEPT;
    bool Write_(uint64_t x) NN_NOEXCEPT;
    bool Write_(float x) NN_NOEXCEPT;
    bool Write_(double x) NN_NOEXCEPT;
    bool WriteStream() NN_NOEXCEPT;

private:
    OutputStream* m_Stream;
    ErrnoT m_Errno;
    bool m_SwapEndian;
    detail::MiniBufOut<512> m_MiniBuf;

    BinaryWriter(const BinaryWriter&) = delete;
    void operator=(const BinaryWriter&) = delete;
};

template <class T>
inline bool BinaryWriter::Write(T x) NN_NOEXCEPT
{
    return this->Write(x);
}

template <class T>
inline bool BinaryWriter::WriteArray(const T* x, size_t n) NN_NOEXCEPT
{
    return this->WriteArray(x, n);
}

template <class T, size_t N>
inline bool BinaryWriter::WriteArray(const T (&a)[N]) NN_NOEXCEPT
{
    return this->WriteArray(&a[0], N);
}

}}} // nn::ngc::detail
