﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "./ngc_ErrnoT.h"
#include "./ngc_InputStream.h"

namespace nn { namespace ngc { namespace detail {

class  BinaryReader final
{
public:
    enum EndianSetting
    {
        EndianSetting_Default = 0,
        EndianSetting_EndianLittle,
        EndianSetting_EndianBig
    };

public:
    BinaryReader() NN_NOEXCEPT : m_Stream(NULL),
                                 m_Errno(-1),
                                 m_SwapEndian(false) {}
    ~BinaryReader() NN_NOEXCEPT
    {
        this->Close();
    }
    errno_t Init(EndianSetting endian) NN_NOEXCEPT;
    errno_t Init() NN_NOEXCEPT
    {
        return Init(EndianSetting_Default);
    }
    errno_t Open(InputStream* stream) NN_NOEXCEPT;

#define NGC_BINR(tp) return this->Read_(reinterpret_cast<tp*>(x))

    bool Read(char* x) NN_NOEXCEPT
    {
        NGC_BINR(uint8_t);
    }
    bool Read(signed char* x) NN_NOEXCEPT
    {
        NGC_BINR(uint8_t);
    }
    bool Read(unsigned char* x) NN_NOEXCEPT
    {
        NGC_BINR(uint8_t);
    }
    bool Read(short* x) NN_NOEXCEPT
    {
        NGC_BINR(uint16_t);
    }  // NOLINT
    bool Read(unsigned short* x) NN_NOEXCEPT
    {
        NGC_BINR(uint16_t);
    }  // NOLINT
    bool Read(int* x) NN_NOEXCEPT
    {
        NGC_BINR(uint32_t);
    }
    bool Read(unsigned int* x) NN_NOEXCEPT
    {
        NGC_BINR(uint32_t);
    }
    bool Read(long* x) NN_NOEXCEPT
    {
        NGC_BINR(ngc_ulong_compatible_t);
    }  // NOLINT
    bool Read(unsigned long* x) NN_NOEXCEPT
    {
        NGC_BINR(ngc_ulong_compatible_t);
    }  // NOLINT
    bool Read(long long* x) NN_NOEXCEPT
    {
        NGC_BINR(uint64_t);
    }  // NOLINT
    bool Read(unsigned long long* x) NN_NOEXCEPT
    {
        NGC_BINR(uint64_t);
    }  // NOLINT
    bool Read(float* x) NN_NOEXCEPT
    {
        NGC_BINR(float);
    }  // NOLINT
    bool Read(double* x) NN_NOEXCEPT
    {
        NGC_BINR(double);
    }  // NOLINT

    template <class T>
    bool Read(T* x) NN_NOEXCEPT;

    int Peek() NN_NOEXCEPT
    {
        return m_MiniBuf.Peek(m_Stream);
    }
    bool Skip(size_t n) NN_NOEXCEPT
    {
        size_t rem = m_MiniBuf.size();
        if (rem <= n)
        {
            m_MiniBuf.Advance(n);
            return true;
        }
        else
        {
            m_MiniBuf.Advance(rem);
            n -= rem;
            return this->Skip_(n);
        }
    }
#undef NGC_BINR

    size_t ReadArray(unsigned char* x, size_t n) NN_NOEXCEPT;
    size_t ReadArray(unsigned short* x, size_t n) NN_NOEXCEPT;  // NOLINT
    size_t ReadArray(unsigned int* x, size_t n) NN_NOEXCEPT;
    size_t ReadArray(unsigned long long* x, size_t n) NN_NOEXCEPT;  // NOLINT
    size_t ReadArray(unsigned long* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
#ifdef NN_BUILD_CONFIG_ABI_LP64
        return this->ReadArray(reinterpret_cast<unsigned long long*>(x), n);  // NOLINT
#else
        return this->ReadArray(reinterpret_cast<unsigned int*>(x), n);
#endif
    }

    size_t ReadArray(float* x, size_t n) NN_NOEXCEPT;
    size_t ReadArray(double* x, size_t n) NN_NOEXCEPT;

    size_t ReadArray(signed char* x, size_t n) NN_NOEXCEPT
    {
        return this->ReadArray(reinterpret_cast<unsigned char*>(x), n);
    }
    size_t ReadArray(char* x, size_t n) NN_NOEXCEPT
    {
        return this->ReadArray(reinterpret_cast<unsigned char*>(x), n);
    }
    size_t ReadArray(short* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
        return this->ReadArray(reinterpret_cast<unsigned short*>(x), n);  // NOLINT
    }
    size_t ReadArray(int* x, size_t n) NN_NOEXCEPT
    {
        return this->ReadArray(reinterpret_cast<unsigned int*>(x), n);
    }
    size_t ReadArray(long* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
#ifdef NN_BUILD_CONFIG_ABI_LP64
        return this->ReadArray(reinterpret_cast<unsigned long long*>(x), n);  // NOLINT
#else
        return this->ReadArray(reinterpret_cast<unsigned int*>(x), n);
#endif
    }
    size_t ReadArray(long long* x, size_t n) NN_NOEXCEPT
    {  // NOLINT
        return this->ReadArray(reinterpret_cast<unsigned long long*>(x), n);  // NOLINT
    }

    template <class T>
    size_t ReadArray(T* x, size_t n) NN_NOEXCEPT;
    template <class T, size_t N>
    size_t ReadArray(T (&a)[N]) NN_NOEXCEPT;
    bool Close() NN_NOEXCEPT
    {
        // Do not m_Stream->Close(). only detach it from BinaryReader.
        m_Stream = NULL;
        return true;
    }
    void SetError(errno_t e) NN_NOEXCEPT
    {
        if (m_Errno == 0)
        {
            m_Errno = e;
        }
    }
    errno_t GetErrorValue() const NN_NOEXCEPT
    {
        return m_Errno;
    }

public:
    inline NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return (GetErrorValue() == 0);
    }

private:
    bool Skip_(size_t n) NN_NOEXCEPT;
    bool Read_(uint8_t* x) NN_NOEXCEPT;
    bool Read_(uint16_t* x) NN_NOEXCEPT;
    bool Read_(uint32_t* x) NN_NOEXCEPT;
    bool Read_(uint64_t* x) NN_NOEXCEPT;
    bool Read_(float* x) NN_NOEXCEPT
    {
        union {
            float f;
            uint32_t i;
        } v;
        if (!this->Read_(&v.i))
        {
            return false;
        }
        *x = v.f;
        return true;
    }
    bool Read_(double* x) NN_NOEXCEPT
    {
        union {
            double f;
            uint64_t i;
        } v;
        if (!this->Read_(&v.i))
        {
            return false;
        }
        *x = v.f;
        return true;
    }

private:
    InputStream* m_Stream;
    ErrnoT m_Errno;
    bool m_SwapEndian;
    detail::MiniBufIn<512> m_MiniBuf;

    BinaryReader(const BinaryReader&) = delete;
    void operator=(const BinaryReader&) = delete;
};

template <class T>
inline bool BinaryReader::Read(T* x) NN_NOEXCEPT
{
    return this->Read(x);
}

template <class T>
inline size_t BinaryReader::ReadArray(T* x, size_t n) NN_NOEXCEPT
{
    return this->ReadArray(x, n);
}

template <class T, size_t N>
inline size_t BinaryReader::ReadArray(T (&a)[N]) NN_NOEXCEPT
{
    return this->ReadArray(&a[0], N);
}

}}} // nn::ngc::detail
